// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief SND pi+pi-pi0eta cross section
  class SND_2019_I1726419 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(SND_2019_I1726419);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {

      // Initialise and register projections
      declare(FinalState(), "FS");
      declare(UnstableParticles(), "UFS");
      for (size_t ix=0; ix<4; ++ix) {
        book(_sigma[ix], 1, 1, 1+ix);
      }
      for (const string& en : _sigma[0].binning().edges<0>()) {
        double eval = stod(en)*GeV;
        if (isCompatibleWithSqrtS(eval)) {
          _sqs = en; break;
        }
      }
      raiseBeamErrorIf(_sqs.empty());
    }


    void findChildren(const Particle& p, map<long,int>& nRes, int& ncount) const {
      for (const Particle& child : p.children()) {
        if (child.children().empty()) {
          --nRes[child.pid()];
          --ncount;
        }
        else findChildren(child,nRes,ncount);
      }
    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {

      const FinalState& fs = apply<FinalState>(event, "FS");

      map<long,int> nCount;
      int ntotal(0);
      for (const Particle& p : fs.particles()) {
        nCount[p.pid()] += 1;
        ++ntotal;
      }
      const FinalState& ufs = apply<FinalState>(event, "UFS");
      bool found = false, foundOmegaPhi = false;
      for (const Particle& p : ufs.particles(Cuts::pid==221)) {
        map<long,int> nRes = nCount;
        int ncount = ntotal;
        findChildren(p,nRes,ncount);
        // eta pi+pi-
        if (ncount==3) {
          bool matched = true;
          for (const auto& val : nRes) {
            if (abs(val.first)==211 || val.first==111 ) {
              if (val.second !=1) {
                matched = false;
                break;
              }
            }
            else if (val.second!=0) {
              matched = false;
              break;
            }
          }
          if (matched) {
            _sigma[0]->fill(_sqs);
            found = true;
          }
        }
        for (const Particle& p2 : ufs.particles()) {
          if (p2.pid()!=223 && p2.pid()!=333) continue;
          map<long,int> nResB = nRes;
          int ncountB = ncount;
          findChildren(p2,nResB,ncountB);
          if (ncountB!=0) continue;
          bool matched2 = true;
          for (const auto& val : nResB) {
            if (val.second!=0) {
              matched2 = false;
              break;
            }
          }
          if (matched2) {
            if (p2.pid()==223) {
              _sigma[1]->fill(_sqs);
            }
            else if (p2.pid()==333 && sqrtS()>1.6*GeV) {
              _sigma[2]->fill(_sqs);
            }
            foundOmegaPhi=true;
          }
        }
      }
      if (found && !foundOmegaPhi)  _sigma[3]->fill(_sqs);
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      scale(_sigma, crossSection()/nanobarn/sumOfWeights());
    }

    /// @}


    /// @name Histograms
    /// @{
    BinnedHistoPtr<string> _sigma[4];
    string _sqs = "";
    /// @}


  };


  RIVET_DECLARE_PLUGIN(SND_2019_I1726419);
}
