// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/ChargedFinalState.hh"

namespace Rivet {


  /// @brief Charged particle spectra between 2.2 and 4.8 GeV
  class BESII_2004_I622224 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(BESII_2004_I622224);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      const ChargedFinalState fs;
      declare(fs, "FS");

      size_t ih = 0;
      for (double eVal : allowedEnergies()) {
        ++ih;
        const string en = toString(round(eVal/MeV));
        if (isCompatibleWithSqrtS(eVal, 1e-3))  _sqs = en;
        book(_h[en], ih, 1, 1);
        book(_c[en], "TMP/Weight_"+en);
      }
      raiseBeamErrorIf(_sqs.empty());
    }

    string map2string(const double xi) const {
      const size_t idx = axis.index(xi) - 1;
      if (idx < edges.size())  return edges[idx];
      return "OTHER";
    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {
      if (edges.empty())  edges = _h[_sqs]->xEdges();
      const ChargedFinalState& fs = apply<ChargedFinalState>(event, "FS");
      if (fs.particles().size()==2 &&
          abs(fs.particles()[0].pid())==13 &&
          abs(fs.particles()[1].pid())==13) vetoEvent;
      for (const Particle& p : fs.particles()) {
        const Vector3 mom3 = p.p3();
        double pp = mom3.mod();
        double xi = -log(2.*pp/sqrtS());
        _h[_sqs]->fill(map2string(xi));
      }
      _c[_sqs]->fill();
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      for (auto& item : _h) {
        scale(item.second, 1./dbl(*_c[item.first]));
        for (auto& b : item.second->bins()) {
          const size_t idx = b.index();
          b.scaleW(1./axis.width(idx));
        }
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    map<string,BinnedHistoPtr<string>> _h;
    map<string,CounterPtr> _c;
    vector<string> edges;
    string _sqs = "";
    YODA::Axis<double> axis{0.05, 0.45, 0.6, 0.7, 0.8, 0.9, 1.0,
                            1.1, 1.2, 1.3, 1.4, 1.5, 1.6, 1.7, 1.8, 1.9, 2.0,
                            2.1, 2.2, 2.3, 2.4, 2.5, 2.6, 2.7, 2.8, 2.9, 3.0, 3.5};
    /// @}


  };


  RIVET_DECLARE_PLUGIN(BESII_2004_I622224);

}
