﻿// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

using System.IO;
using System.Linq;
using System.Text;
using Cake.Core.IO;
using Cake.Testing;
using NSubstitute;
using Xunit;

namespace Cake.Core.Tests.Unit.IO
{
    public sealed class FileExtensionsTests
    {
        public sealed class TheOpenMethod
        {
            public sealed class WithFileMode
            {
                [Fact]
                public void Should_Throw_If_File_Is_Null()
                {
                    // Given, When
                    var result = Record.Exception(() => FileExtensions.Open(null, FileMode.Create));

                    // Then
                    AssertEx.IsArgumentNullException(result, "file");
                }

                [Theory]
                [InlineData(FileMode.Append, FileAccess.Write)]
                [InlineData(FileMode.Create, FileAccess.ReadWrite)]
                [InlineData(FileMode.CreateNew, FileAccess.ReadWrite)]
                [InlineData(FileMode.Open, FileAccess.ReadWrite)]
                [InlineData(FileMode.OpenOrCreate, FileAccess.ReadWrite)]
                [InlineData(FileMode.Truncate, FileAccess.ReadWrite)]
                public void Should_Open_With_Specified_File_Mode_And_Infer_File_Access(FileMode mode, FileAccess access)
                {
                    // Given
                    var file = Substitute.For<IFile>();

                    // When
                    file.Open(mode);

                    // Then
                    file.Received(1).Open(mode, access, FileShare.None);
                }
            }

            public sealed class WithFileModeAndFileAccess
            {
                [Fact]
                public void Should_Throw_If_File_Is_Null()
                {
                    // Given, When
                    var result = Record.Exception(() => FileExtensions.Open(null, FileMode.Create, FileAccess.Write));

                    // Then
                    AssertEx.IsArgumentNullException(result, "file");
                }

                [Theory]
                [InlineData(FileMode.Append, FileAccess.Write)]
                [InlineData(FileMode.Create, FileAccess.ReadWrite)]
                [InlineData(FileMode.CreateNew, FileAccess.ReadWrite)]
                [InlineData(FileMode.Open, FileAccess.ReadWrite)]
                [InlineData(FileMode.OpenOrCreate, FileAccess.ReadWrite)]
                [InlineData(FileMode.Truncate, FileAccess.ReadWrite)]
                public void Should_Open_With_Specified_File_Mode_And_Infer_File_Access(FileMode mode, FileAccess access)
                {
                    // Given
                    var file = Substitute.For<IFile>();

                    // When
                    file.Open(mode, access);

                    // Then
                    file.Received(1).Open(mode, access, FileShare.None);
                }
            }
        }

        public sealed class TheOpenReadMethod
        {
            [Fact]
            public void Should_Throw_If_File_Is_Null()
            {
                // Given, When
                var result = Record.Exception(() => FileExtensions.OpenRead(null));

                // Then
                AssertEx.IsArgumentNullException(result, "file");
            }

            [Fact]
            public void Should_Open_Stream_With_Expected_FileMode_And_FileAccess()
            {
                // Given
                var file = Substitute.For<IFile>();

                // When
                file.OpenRead();

                // Then
                file.Received(1).Open(FileMode.Open, FileAccess.Read, FileShare.Read);
            }
        }

        public sealed class TheOpenWriteMethod
        {
            [Fact]
            public void Should_Throw_If_File_Is_Null()
            {
                // Given, When
                var result = Record.Exception(() => FileExtensions.OpenWrite(null));

                // Then
                AssertEx.IsArgumentNullException(result, "file");
            }

            [Fact]
            public void Should_Open_Stream_With_Expected_FileMode_And_FileAccess()
            {
                // Given
                var file = Substitute.For<IFile>();

                // When
                file.OpenWrite();

                // Then
                file.Received(1).Open(FileMode.Create, FileAccess.Write, FileShare.None);
            }
        }

        public sealed class TheReadLinesMethod
        {
            [Fact]
            public void Should_Throw_If_File_Is_Null()
            {
                // Given, When
                var result = Record.Exception(() => FileExtensions.ReadLines(null, Encoding.UTF8));

                // Then
                AssertEx.IsArgumentNullException(result, "file");
            }

            [Fact]
            public void Should_Return_Empty_List_If_File_Contains_No_Lines()
            {
                // Given
                var environment = FakeEnvironment.CreateUnixEnvironment();
                var fileSystem = new FakeFileSystem(environment);
                var file = fileSystem.CreateFile("text.txt");

                // When
                var result = file.ReadLines(Encoding.UTF8).ToList();

                // Then
                Assert.Empty(result);
            }

            [Fact]
            public void Should_Read_File_With_Single_Line_Correctly()
            {
                // Given
                var environment = FakeEnvironment.CreateUnixEnvironment();
                var fileSystem = new FakeFileSystem(environment);
                var file = fileSystem.CreateFile("text.txt").SetContent("Hello World");

                // When
                var result = file.ReadLines(Encoding.UTF8).ToList();

                // Then
                Assert.Single(result);
            }

            [Fact]
            public void Should_Read_File_With_Multiple_Lines_Correctly()
            {
                // Given
                var environment = FakeEnvironment.CreateUnixEnvironment();
                var fileSystem = new FakeFileSystem(environment);
                var content = new StringBuilder();
                content.AppendLine("1");
                content.AppendLine("2");
                content.AppendLine("3");
                var file = fileSystem.CreateFile("text.txt").SetContent(content.ToString());

                // When
                var result = file.ReadLines(Encoding.UTF8).ToList();

                // Then
                Assert.Equal(3, result.Count);
                Assert.Equal("1", result[0]);
                Assert.Equal("2", result[1]);
                Assert.Equal("3", result[2]);
            }
        }

        public sealed class TheIsClrAssemblyMethod
        {
            private static readonly byte[] _validPeAndClrAssemblyBytes =
            {
                0x4d, 0x5a, 0x90, 0x00, 0x03, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0xff, 0xff, 0x00, 0x00, 0xb8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00,
                0x0e, 0x1f, 0xba, 0x0e, 0x00, 0xb4, 0x09, 0xcd, 0x21, 0xb8, 0x01, 0x4c, 0xcd, 0x21, 0x54, 0x68, 0x69, 0x73, 0x20, 0x70, 0x72, 0x6f, 0x67, 0x72, 0x61, 0x6d, 0x20, 0x63, 0x61, 0x6e, 0x6e, 0x6f,
                0x74, 0x20, 0x62, 0x65, 0x20, 0x72, 0x75, 0x6e, 0x20, 0x69, 0x6e, 0x20, 0x44, 0x4f, 0x53, 0x20, 0x6d, 0x6f, 0x64, 0x65, 0x2e, 0x0d, 0x0d, 0x0a, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x50, 0x45, 0x00, 0x00, 0x4c, 0x01, 0x03, 0x00, 0x8a, 0x54, 0x4a, 0xf5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe0, 0x00, 0x22, 0x20, 0x0b, 0x01, 0x30, 0x00, 0x00, 0x84, 0x00, 0x00,
                0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x82, 0xa2, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x20, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,
                0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x60, 0x85,
                0x00, 0x00, 0x10, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x2e, 0xa2, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x88, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0xe0, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x94, 0xa1, 0x00, 0x00, 0x38, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x20, 0x00, 0x00, 0x48,
            };

            private static readonly byte[] _validPeButNotClrAssemblyBytes =
            {
                0x4d, 0x5a, 0x90, 0x00, 0x03, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0xff, 0xff, 0x00, 0x00, 0xb8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x01, 0x00, 0x00,
                0x0e, 0x1f, 0xba, 0x0e, 0x00, 0xb4, 0x09, 0xcd, 0x21, 0xb8, 0x01, 0x4c, 0xcd, 0x21, 0x54, 0x68, 0x69, 0x73, 0x20, 0x70, 0x72, 0x6f, 0x67, 0x72, 0x61, 0x6d, 0x20, 0x63, 0x61, 0x6e, 0x6e, 0x6f,
                0x74, 0x20, 0x62, 0x65, 0x20, 0x72, 0x75, 0x6e, 0x20, 0x69, 0x6e, 0x20, 0x44, 0x4f, 0x53, 0x20, 0x6d, 0x6f, 0x64, 0x65, 0x2e, 0x0d, 0x0d, 0x0a, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0xab, 0x04, 0x28, 0x30, 0xef, 0x65, 0x46, 0x63, 0xef, 0x65, 0x46, 0x63, 0xef, 0x65, 0x46, 0x63, 0x71, 0xc5, 0x81, 0x63, 0xea, 0x65, 0x46, 0x63, 0xa9, 0x34, 0xa7, 0x63, 0xc2, 0x65, 0x46, 0x63,
                0xa9, 0x34, 0xa6, 0x63, 0x71, 0x65, 0x46, 0x63, 0xa9, 0x34, 0x99, 0x63, 0xe4, 0x65, 0x46, 0x63, 0x32, 0x9a, 0x88, 0x63, 0xed, 0x65, 0x46, 0x63, 0x32, 0x9a, 0x8d, 0x63, 0xe0, 0x65, 0x46, 0x63,
                0xef, 0x65, 0x47, 0x63, 0x43, 0x65, 0x46, 0x63, 0x5a, 0xfb, 0xa2, 0x63, 0x4d, 0x65, 0x46, 0x63, 0x5a, 0xfb, 0x9a, 0x63, 0xee, 0x65, 0x46, 0x63, 0xe2, 0x37, 0x9d, 0x63, 0xee, 0x65, 0x46, 0x63,
                0xef, 0x65, 0xd1, 0x63, 0xee, 0x65, 0x46, 0x63, 0x5a, 0xfb, 0x98, 0x63, 0xee, 0x65, 0x46, 0x63, 0x52, 0x69, 0x63, 0x68, 0xef, 0x65, 0x46, 0x63, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x50, 0x45, 0x00, 0x00, 0x64, 0x86, 0x06, 0x00, 0xe2, 0xac, 0x47, 0x59, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf0, 0x00, 0x22, 0x20,
                0x0b, 0x02, 0x0c, 0x00, 0x00, 0x6e, 0x0d, 0x00, 0x00, 0x08, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x87, 0x0c, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x00,
                0x00, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1a, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0, 0x11, 0x00, 0x00, 0x04, 0x00, 0x00,
                0xfe, 0x24, 0x12, 0x00, 0x03, 0x00, 0x60, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0xd9, 0x0f, 0x00, 0x8a, 0x62, 0x00, 0x00, 0x8c, 0x3b, 0x10, 0x00, 0xa0, 0x00, 0x00, 0x00,
                0x00, 0x90, 0x11, 0x00, 0xb8, 0x05, 0x00, 0x00, 0x00, 0xb0, 0x10, 0x00, 0xb4, 0xd5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x11, 0x00, 0x3c, 0x09, 0x00, 0x00,
                0xa0, 0x85, 0x0d, 0x00, 0x38, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x90, 0xcd, 0x0e, 0x00, 0x70, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x0d, 0x00, 0x20, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00,
            };

            private static readonly byte[] _invalidPeBytes =
            {
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x13, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            };

            [Fact]
            public void Should_Return_True_When_File_Is_Valid_PE_And_Is_Clr_Assembly()
            {
                // Given
                var file = Substitute.For<IFile>();

                // When
                file.Exists.Returns(true);
                file.Length.Returns(_validPeAndClrAssemblyBytes.Length);
                file.OpenRead().Returns(new MemoryStream(_validPeAndClrAssemblyBytes));

                // Then
                Assert.True(FileExtensions.IsClrAssembly(file));
            }

            [Fact]
            public void Should_Return_False_When_File_Is_Valid_PE_But_Not_A_Clr_Assembly()
            {
                // Given
                var file = Substitute.For<IFile>();

                // When
                file.Exists.Returns(true);
                file.Length.Returns(_validPeButNotClrAssemblyBytes.Length);
                file.OpenRead().Returns(new MemoryStream(_validPeButNotClrAssemblyBytes));

                // Then
                Assert.False(FileExtensions.IsClrAssembly(file));
            }

            [Fact]
            public void Should_Return_False_When_File_Is_Not_A_Valid_PE()
            {
                // Given
                var file = Substitute.For<IFile>();

                // When
                file.Exists.Returns(true);
                file.Length.Returns(_invalidPeBytes.Length);
                file.OpenRead().Returns(new MemoryStream(_invalidPeBytes));

                // Then
                Assert.False(FileExtensions.IsClrAssembly(file));
            }

            [Fact]
            public void Should_Return_False_When_File_Is_MacOS_MachO_dylib()
            {
                // Given
                var file = Substitute.For<IFile>();

                // When
                file.Exists.Returns(true);
                file.Path.Returns(new string("fullname.dylib"));
                file.Length.Returns(_invalidPeBytes.Length);

                // Then
                Assert.False(FileExtensions.IsClrAssembly(file));
            }
        }
    }
}