// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

using System.Collections.Generic;
using Cake.Core.IO;
using Cake.Core.Tooling;

namespace Cake.Common.Tools.WiX.Heat
{
    /// <summary>
    /// Contains settings used by <see cref="HeatRunner"/>.
    /// </summary>
    public sealed class HeatSettings : ToolSettings
    {
        /// <summary>
        /// Gets or sets the WiX extensions to use.
        /// </summary>
        public IEnumerable<string> Extensions { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether [no logo].
        /// </summary>
        /// <value>
        ///   <c>true</c> if [no logo]; otherwise, <c>false</c>.
        /// </value>
        public bool NoLogo { get; set; }

        /// <summary>
        /// Gets or sets the suppress specific warnings.
        /// </summary>
        /// <value>
        /// The suppress specific warnings.
        /// </value>
        public IEnumerable<string> SuppressSpecificWarnings { get; set; }

        /// <summary>
        /// Gets or sets the treat specific warnings as errors.
        /// </summary>
        /// <value>
        /// The treat specific warnings as errors.
        /// </value>
        public IEnumerable<string> TreatSpecificWarningsAsErrors { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether output is verbose.
        /// </summary>
        /// <value>
        ///   <c>true</c> if verbose; otherwise, <c>false</c>.
        /// </value>
        public bool Verbose { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether to auto generate <see cref="System.Guid"/> at compile time.
        /// </summary>
        /// <value>
        /// <c>true</c> if [autogenerated unique identifier]; otherwise, <c>false</c>.
        /// </value>
        public bool AutogeneratedGuid { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether all components are given a guid.
        /// </summary>
        /// <value>
        /// <c>true</c> if [generate unique identifier]; otherwise, <c>false</c>.
        /// </value>
        public bool GenerateGuid { get; set; }

        /// <summary>
        /// Gets or sets the output file.
        /// </summary>
        /// <value>
        /// The output file.
        /// </value>
        public FilePath OutputFile { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether to suppress the
        /// generation of fragments for directories and components.
        /// </summary>
        /// <value>
        ///   <c>true</c> if [suppress fragments]; otherwise, <c>false</c>.
        /// </value>
        public bool SuppressFragments { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether to suppress unique identifiers
        /// for files, components, and directories.
        /// </summary>
        /// <value>
        ///   <c>true</c> if [suppress unique ids]; otherwise, <c>false</c>.
        /// </value>
        public bool SuppressUniqueIds { get; set; }

        /// <summary>
        /// Gets or sets the transform to apply to harvested files.
        /// </summary>
        /// <value>
        /// The transform.
        /// </value>
        public string Transform { get; set; }

        /// <summary>
        /// Gets or sets the file.
        /// </summary>
        /// <value>
        /// The file.
        /// </value>
        public FilePath File { get; set; }

        /// <summary>
        /// Gets or sets the name of the component group.
        /// </summary>
        /// <value>
        /// The name of the component group.
        /// </value>
        public string ComponentGroupName { get; set; }

        /// <summary>
        /// Gets or sets the directory reference identifier for generated directory elements.
        /// </summary>
        /// <value>
        /// The directory reference identifier.
        /// </value>
        public string DirectoryReferenceId { get; set; }

        /// <summary>
        /// Gets or sets the preprocessor variable.
        /// </summary>
        /// <value>
        /// The preprocessor variable.
        /// </value>
        public string PreprocessorVariable { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether generate binder variables instead
        /// of preprocessor variables.
        /// </summary>
        /// <value>
        /// <c>true</c> if [generate binder variables]; otherwise, <c>false</c>.
        /// </value>
        public bool GenerateBinderVariables { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether the COM elements.
        /// </summary>
        /// <value>
        ///   <c>true</c> if [suppress COM]; otherwise, <c>false</c>.
        /// </value>
        public bool SuppressCom { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether [suppress registry].
        /// </summary>
        /// <value>
        ///   <c>true</c> if [suppress registry]; otherwise, <c>false</c>.
        /// </value>
        public bool SuppressRegistry { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether [suppress root directory].
        /// </summary>
        /// <value>
        /// <c>true</c> if [suppress root directory]; otherwise, <c>false</c>.
        /// </value>
        public bool SuppressRootDirectory { get; set; }

        /// <summary>
        /// Gets or sets the configuration to set when harvesting the project.
        /// </summary>
        /// <value>
        /// The configuration.
        /// </value>
        public string Configuration { get; set; }

        /// <summary>
        /// Gets or sets the overridden directory identifier for generated directory elements.
        /// </summary>
        /// <value>
        /// The directory identifier.
        /// </value>
        public string DirectoryId { get; set; }

        /// <summary>
        /// Gets or sets the type of elements to generate.
        /// </summary>
        /// <value>
        /// The generate.
        /// </value>
        public WiXGenerateType Generate { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether to generate guids without curly braces.
        /// </summary>
        /// <value>
        /// <c>true</c> if generate guids without curly braces; otherwise, <c>false</c>.
        /// </value>
        public bool GenerateGuidWithoutBraces { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether to keep empty directories.
        /// </summary>
        /// <value>
        /// <c>true</c> if keep empty directories; otherwise, <c>false</c>.
        /// </value>
        public bool KeepEmptyDirectories { get; set; }

        /// <summary>
        /// Gets or sets the platform to set when harvesting the project.
        /// </summary>
        /// <value>
        /// The platform.
        /// </value>
        public string Platform { get; set; }

        /// <summary>
        /// Gets or sets the output group of Visual Studio project.
        /// </summary>
        /// <value>
        /// The output group.
        /// </value>
        public WiXOutputGroupType? OutputGroup { get; set; }

        /// <summary>
        /// Gets or sets the overridden project name to use in variables.
        /// </summary>
        /// <value>
        /// The name of the project.
        /// </value>
        public string ProjectName { get; set; }

        /// <summary>
        /// Gets or sets the template to use when harvesting.
        /// </summary>
        /// <value>
        /// The template.
        /// </value>
        public WiXTemplateType? Template { get; set; }

        /// <summary>
        /// Gets or sets the indentation multiplier, overrides default of 4.
        /// </summary>
        /// <value>
        /// The indent.
        /// </value>
        public int? Indent { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether to suppress VB6 COM registration entries.
        /// </summary>
        /// <value>
        ///   <c>true</c> if suppress VB6 COM registration entries; otherwise, <c>false</c>.
        /// </value>
        public bool SuppressVb6Com { get; set; }
    }
}