/******************************************************************************
*******************************************************************************
**
**    Copyright 1997 Kenneth W. Preslan <kpreslan@lcse.umn.edu>
**              1998-1999 Grant Erickson <gerickson@brocade.com>
**
**    University of Minnesota
**    Department of Electrical and Computer Engineering
**
**    Brocade Communications Systems, Inc.
** 
**    Module name: fwdl.C
**    Date:        1998/09/30
**
**    Description:
**      This modules implements device generic code for the Seagate disk 
**      firmware download tool.
**
*******************************************************************************
******************************************************************************/

/*
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *  
 *    This program is distributed in the hope that it will be useful,  
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the   
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>

#include <sys/types.h>

#include <sys/ioctl.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/sysmacros.h>

#ifdef linux
#include "fwdl-linux.h"
#define	FWDL_SCSI_OPS	linuxOps
#endif

#ifdef sgi
#include "fwdl-prisa.h"
#define	FWDL_SCSI_OPS	prisaOps
#endif

#ifdef sun
#include "fwdl-sun.h"
#define	FWDL_SCSI_OPS	sunOps
#endif


/* Preprocessor Defines */

#if !defined(TRUE) || ((TRUE) != 1)
#define TRUE    1
#endif

#if !defined(FALSE) || ((FALSE) != 0)
#define FALSE   0
#endif

#define	MAJOR_VERSION	1
#define MINOR_VERSION	2
#define UPDATE_LEVEL	3

#ifdef DEBUG
#define OPTION_STRING   "dhV"
#else
#define OPTION_STRING   "hV"
#endif

#define VERY_SURE


// Function Macros


// Type Definitions


// Global Variables

#ifdef DEBUG
int               debug          = FALSE;
#endif
static const char       *programName    = NULL;
static const char	*deviceName	= NULL;
static const char	*firmwareFile	= NULL;

senseData_t global_sdata;	// Global structure for SCSI sense data

static char devTypes[][16] = {
        "Disk",    "Tape", 
	"Printer", "Processor", 
	"WORM",    "CD-ROM",
        "Scanner", "Optical", 
	"Jukebox", "Comm", 
	"Unknown"
};

#define NPDT (sizeof devTypes / sizeof devTypes[0])


// Function Prototypes

void	 decodeOptions(int argc, char *const argv[]);
void	 printUsage(int status);
void	 printVersion(void);

void	*readFirmware(const char *fwFile, unsigned int &fwSize);

void	 printInquiryData(inquiryData_t *data);



int main(int argc, char *argv[])
{
  unsigned char	*fwData = NULL;
  unsigned int	 fwSize;
  inquiryData_t  deviceData;
  scsi_ops_t	*ops;
  int		 i, fd;
  char		 response[256];;
  char		 success = FALSE;


  programName = argv[0];

  decodeOptions(argc, argv);

  // Read in the firmware

  if ((fwData = (unsigned char *)readFirmware(firmwareFile, fwSize)) == NULL)
    {
      fprintf(stderr, "%s: Failed to read in firmware data.\n", programName);
      return(EXIT_FAILURE);
    }

  // Connect SCSI operations

  ops = &FWDL_SCSI_OPS;

  if (ops == NULL) {
	  fprintf(stderr,
		  "%s: Failed to connect platform-specific SCSI operations.\n",
		  programName);
	  return(EXIT_FAILURE);
  }

  // Open the device

  if ((fd = ops->scsiOpen(deviceName, 0)) < 0)
    {
      fprintf(stderr, "%s: Failed to open \"%s\"\n", programName, deviceName);
      if (fwData)
	free(fwData);
      return(EXIT_FAILURE);
    }

  // Issue a couple of test unit ready commands

  for (i = 0; (i < 3) && (success == 0); i++)
    {
      success = ops->scsiTstUnitRdy(fd);
    }

  if (success != 0)
    {
      fprintf(stderr, "The drive is not ready to accept the firmware.\n");
      if (fwData)
	free(fwData);
      ops->scsiClose(fd);
      return(EXIT_FAILURE);
    }
  
  printf("Gathering inquiry data from the drive...");

  if (ops->scsiInquiry(fd, &deviceData)  != 0)
    {
      fprintf(stderr, "Failed to get inquiry data.\n");
      if (fwData)
	free(fwData);
      ops->scsiClose(fd);
      return(EXIT_FAILURE);
    }
  
  printf("done\n");

  printInquiryData(&deviceData);

#ifdef VERY_SURE
  fprintf(stderr, "About to update drive firmware. This could render the \
drive unusable.\n\nAre you certain you want to continue? [yN] ");

  scanf("%s", response);

  if (tolower(response[0]) != 'y')
    {
      fprintf(stderr, "Update canceled.\n");
      exit(EXIT_FAILURE);
    }

  fprintf(stderr, "About to update firmware.");
  for (i = 1; i <= 5; i++)
    {
      fprintf(stderr, "..%d.", i);
      sleep(1);
    }
  fprintf(stderr, "..updating firmware...\n");
#endif

  // Download the firmware

  ops->scsiWriteFW(fd, fwData, fwSize);

  // Close the device

  ops->scsiClose(fd);

  // Free the memory allocated for the firmware.

  if (fwData)
    free(fwData);

  fprintf(stderr, "done.\n");

  return(EXIT_SUCCESS);
}


/******************************************************************************
*******************************************************************************
**
** void decodeOptions()
**
** Description:
**   This routine steps through the command-line arguments, parsing out
**   recognzied options.
**
** NOTE: This routine alters a number of global variables.
**
** Input(s):
**    argc - Number of arguments on the command line.
**  **argv - Pointer to an array of strings containing command-line arguments.
**
*******************************************************************************
******************************************************************************/

void decodeOptions(int argc, char *const argv[])
{
  int   errFlag = 0;
  int   optchar;
  
  while((optchar = getopt(argc, argv, OPTION_STRING)) != EOF) {
    switch (optchar) {
    case 'd':
#ifdef DEBUG
      debug = TRUE;
      fprintf(stderr, "Debugging output enabled.\n");
#else
      fprintf(stderr, "This version was not built with debugging support.\n");
#endif
      break;
    case 'h':
      printUsage(EXIT_SUCCESS);
      break;
    case 'V':
      printVersion();
      exit(0);
      break;
    case '?':
    default:
      errFlag++;
    }
  }

  if ((deviceName = argv[optind++]) == NULL || deviceName[0] == '\0') {
    errFlag++;
  }

  if ((firmwareFile = argv[optind++]) == NULL || firmwareFile[0] == '\0') {
    errFlag++;
  }

  if (errFlag) {
    printUsage(EXIT_FAILURE);
  }

#ifdef DEBUG
  if (debug) {
    fprintf(stderr, "Device name: %s\n", deviceName);
    fprintf(stderr, "Firmware: %s\n", firmwareFile);
  }
#endif

  return;
}


/******************************************************************************
*******************************************************************************
**
** void printUsage()
**
** Description:
**   This routine prints out the proper command line usage for this program.
**
** Input(s):
**  status - Flag determining what usage information will be printed and what
**           the exit status of the program will be after the information is
**           printed.
**
*******************************************************************************
******************************************************************************/

void printUsage(int status)
{

#ifdef DEBUG
  printf("Usage: %s [-dhV] <target disk> <file>\n", programName);
#else
  printf("Usage: %s [-hV] <target disk> <file>\n", programName);
#endif

  if (status != EXIT_SUCCESS)
    printf("Try `%s -h' for more information.\n", programName);

  if (status != EXIT_FAILURE)
    {
      printf("  target disk  Target disk device.\n");
      printf("  file         Firmware data to be downloaded to the drive.\n");
#ifdef DEBUG
      printf("  -d           Enable debugging code.\n");
#endif
      printf("  -h           Print this help, then exit.\n");
      printf("  -V           Print program version information, then exit.\n");
    }
  exit(status);
}


/******************************************************************************
*******************************************************************************
**
** void printVersion()
**
** Description:
**   This routine prints out version and compilation information for this
**   program.
**
*******************************************************************************
******************************************************************************/

void printVersion(void)
{
  printf("\
fwdl - Downloads firmware to a Seagate disk.\n\
Version: %d.%d.%d\n\
Built: %s\n\
Copyright (c) 1997-1999 University of Minnesota\n\
Copyright (c) 1998-1999 Brocade Communications Systems, Inc.\n",
	 MAJOR_VERSION, MINOR_VERSION, UPDATE_LEVEL, __DATE__);

  return;
}


/******************************************************************************
*******************************************************************************
**
** void *readFirmware()
**
** Description:
**   This routine reads the firmware specified into a dynamically allocated
**   buffer and returns a pointer to it and the size of the buffer.
**
** Input(s):
**  *fwFile - Name of the firmware file.
**  &fwSize - Storage for the size of the firmware file read in.
**
** Output(s):
**  &fwSize - Size of the firmware file read in.
**
** Returns:
**   Pointer to the firmware buffer if OK, NULL on error.
**
*******************************************************************************
******************************************************************************/

void *readFirmware(const char *fwFile, unsigned int &fwSize)
{
  struct stat    fwStats;
  unsigned char	*fwData = NULL;
  size_t	 resid = 0;
  FILE		*fp = NULL;
    
  // Get the size of the firmware file

  if (stat(firmwareFile, &fwStats) < 0)
    {
      fprintf(stderr, "Could not get the size of \"%s\"\n", fwFile);
      return(NULL);
    }
  
  fwSize = fwStats.st_size;

#ifdef DEBUG
  if (debug) {
    fprintf(stderr, "Firmware size: %d\n", fwSize);
  }
#endif

  // Allocate memory

  if ((fwData = (unsigned char *)valloc(fwSize*sizeof(unsigned char))) == NULL)
    {
      fprintf(stderr, "Could not allocate enough memory.\n");
      return(NULL);
    }

  // Get the firmware data

  if ((fp = fopen(fwFile, "rb")) == NULL)
    {
      fprintf(stderr, "Error opening the firmware file \"%s\"\n", 
	      firmwareFile);
      if (fwData)
	free(fwData);
      return(NULL);
    }
  
  if ((resid = fread(fwData, sizeof(unsigned char), fwSize, fp)) < fwSize)
    {
      fprintf(stderr, "Failed to read %d bytes of the firmware.\n", 
	      fwSize - resid);
      fclose(fp);
      if (fwData) 
	free(fwData);
      return(NULL);
    }
  return(fwData);
}


/******************************************************************************
*******************************************************************************
**
** void printInquiryData()
**
** Description:
**   This routine prints out the inquiry data for a SCSI device in nice,
**   formatted manner.
**
** Input(s):
**  *data - Pointer to an inquiry data structure.
**
*******************************************************************************
******************************************************************************/

void printInquiryData(inquiryData_t *data)
{
  int i;

  printf("Device Type:\t\t%s\n", 
	 devTypes[(data->deviceType < NPDT) ? data->deviceType : NPDT - 1]);
  printf("Removable:\t\t%x\n", data->rmbDtm);
  printf("ISO Version:\t\t%x\n", data->version);
  printf("Response Data Format:\t%x\n", data->rdf);
  printf("Additional Length:\t%x\n", data->addLength);
  printf("Option Bits:\t\t%x\n", data->optionBits);
  
  printf("Vendor ID:\t\t");
  for (i = 0; i < 8; i++) 
    {
      printf("%c", data->vendorID[i]);
    }
  printf("\nProduct ID:\t\t");
  for (i = 0; i < 16; i++)
    {
      printf("%c", data->productID[i]);
    }
  printf("\nRevision Level:\t\t");
  for (i = 0; i < 4; i++)
    {      
      printf("%c", data->prodRevLevel[i]);
    }
  printf("\nVendor Specific:\t\t");
  for (i = 0; i < 20; i++)
    {       
      printf("%c", data->vendorSpecific[i]);
    }
  printf("\n\n");

  return;
}
