#!/usr/bin/env python

import six
import unittest

from erlastic import decode, encode
from erlastic.compat import *
from erlastic.types import *

erlang_term_binaries = [
    # nil
    ([], list, six.b('\x83j')),
    # binary
    (six.b('foo'), six.binary_type, six.b('\x83m\x00\x00\x00\x03foo')),
    # atom
    (Atom("foo"), Atom, six.b('\x83d\x00\x03foo')),
    # atom true
    (True, bool, six.b('\x83d\x00\x04true')),
    # atom false
    (False, bool, six.b('\x83d\x00\x05false')),
    # atom none
    (None, type(None), six.b('\x83d\x00\x04none')),
    # small integer
    (123, six.integer_types, six.b('\x83a{')),
    # integer
    (12345, six.integer_types, six.b('\x83b\x00\x0009')),
    # float
    (1.2345, float, six.b('\x83c1.23449999999999993072e+00\x00\x00\x00\x00\x00')),
    # tuple
    ((Atom("foo"), b"test", 123), tuple, six.b('\x83h\x03d\x00\x03foom\x00\x00\x00\x04testa{')),
    # list
    ([1024, b"test", 4.096], list, six.b('\x83l\x00\x00\x00\x03b\x00\x00\x04\x00m\x00\x00\x00\x04testc4.09600000000000008527e+00\x00\x00\x00\x00\x00j')),
    ([102, 111, 111], list, six.b('\x83l\x00\x00\x00\x03afaoaoj')),
    # small big
    (12345678901234567890, six.integer_types, six.b('\x83n\x08\x00\xd2\n\x1f\xeb\x8c\xa9T\xab')),
    # large big
    (123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890,
     six.integer_types, six.b('\x83o\x00\x00\x01D\x00\xd2\n?\xce\x96\xf1\xcf\xacK\xf1{\xefa\x11=$^\x93\xa9\x88\x17\xa0\xc2\x01\xa5%\xb7\xe3Q\x1b\x00\xeb\xe7\xe5\xd5Po\x98\xbd\x90\xf1\xc3\xddR\x83\xd1)\xfc&\xeaH\xc31w\xf1\x07\xf3\xf33\x8f\xb7\x96\x83\x05t\xeci\x9cY"\x98\x98i\xca\x11bY=\xcc\xa1\xb4R\x1bl\x01\x86\x18\xe9\xa23\xaa\x14\xef\x11[}O\x14RU\x18$\xfe\x7f\x96\x94\xcer?\xd7\x8b\x9a\xa7v\xbd\xbb+\x07X\x94x\x7fI\x024.\xa0\xcc\xde\xef:\xa7\x89~\xa4\xafb\xe4\xc1\x07\x1d\xf3cl|0\xc9P`\xbf\xab\x95z\xa2DQf\xf7\xca\xef\xb0\xc4=\x11\x06*:Y\xf58\xaf\x18\xa7\x81\x13\xdf\xbdTl4\xe0\x00\xee\x93\xd6\x83V\xc9<\xe7I\xdf\xa8.\xf5\xfc\xa4$R\x95\xef\xd1\xa7\xd2\x89\xceu!\xf8\x08\xb1Zv\xa6\xd9z\xdb0\x88\x10\xf3\x7f\xd3sc\x98[\x1a\xac6V\x1f\xad0)\xd0\x978\xd1\x02\xe6\xfbH\x149\xdc).\xb5\x92\xf6\x91A\x1b\xcd\xb8`B\xc6\x04\x83L\xc0\xb8\xafN+\x81\xed\xec?;\x1f\xab1\xc1^J\xffO\x1e\x01\x87H\x0f.ZD\x06\xf0\xbak\xaagVH]\x17\xe6I.B\x14a2\xc1;\xd1+\xea.\xe4\x92\x15\x93\xe9\'E\xd0(\xcd\x90\xfb\x10')),
    # reference
    (Reference('nonode@nohost', [33, 0, 0], 0), Reference, six.b('\x83r\x00\x03d\x00\rnonode@nohost\x00\x00\x00\x00!\x00\x00\x00\x00\x00\x00\x00\x00')),
    # port
    (Port('nonode@nohost', 455, 0), Port, six.b('\x83fd\x00\rnonode@nohost\x00\x00\x01\xc7\x00')),
    # pid
    (PID('nonode@nohost', 31, 0, 0), PID, six.b('\x83gd\x00\rnonode@nohost\x00\x00\x00\x1f\x00\x00\x00\x00\x00')),
    # function export
    (Export('jobqueue', 'stats', 0), Export, six.b('\x83qd\x00\x08jobqueued\x00\x05statsa\x00')),
]

erlang_term_decode = [
    ## ext_string is an optimized way to send list of bytes, so not bijective (no erlang representation), only decode
    (pack_bytes([102, 111, 111]), six.binary_type, six.b('\x83k\x00\x03foo'))
]


erlang_term_encode = [
    ## python3 choice : encode string as binary utf-8, so not bijective (decoded binary utf-8 is of type "six.binary_type"), only encode
    ("foo", six.string_types, six.b('\x83m\x00\x00\x00\x03foo'))
]

class ErlangTestCase(unittest.TestCase):
    def testDecode(self):
        for python, expected_type, erlang in erlang_term_binaries + erlang_term_decode:
            decoded = decode(erlang)
            self.assertEqual(python, decoded)
            self.assertTrue(isinstance(decoded, expected_type))

    def testEncode(self):
        for python, expected_type, erlang in erlang_term_binaries + erlang_term_encode:
            encoded = encode(python)
            self.assertEqual(erlang, encoded)

if __name__ == '__main__':
    unittest.main()
