/*
 * Copyright (c) 2005-2008 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.jvnet.substance.utils;

import java.awt.Color;
import java.awt.Component;

import javax.swing.JComponent;
import javax.swing.JTabbedPane;
import javax.swing.plaf.UIResource;

import org.jvnet.lafwidget.animation.FadeKind;
import org.jvnet.substance.SubstanceLookAndFeel;
import org.jvnet.substance.combo.SubstanceComboBoxButton;
import org.jvnet.substance.painter.decoration.DecorationAreaType;
import org.jvnet.substance.painter.decoration.SubstanceDecorationUtilities;
import org.jvnet.substance.scroll.SubstanceScrollButton;
import org.jvnet.substance.theme.*;

public class SubstanceThemeUtilities {
	public static SubstanceTheme getDecorationTheme(Component component) {
		Component c = component;
		SubstanceTheme globalTheme = SubstanceLookAndFeel.getTheme();
		while (c != null) {
			DecorationAreaType decorationType = SubstanceDecorationUtilities
					.getImmediateDecorationType(c);
			if (decorationType != null) {
				if (globalTheme.toUseDecorationPainter(decorationType)) {
					return globalTheme.getDecorationTheme(decorationType);
				}
			}
			// if (c instanceof JPopupMenu) {
			// Component invoker = ((JPopupMenu) c).getInvoker();
			// if (c != invoker)
			// return getDecorationTheme(invoker);
			// }
			c = c.getParent();
		}
		return null;
	}

	public static SubstanceTheme getNonColorizedTheme(Component component,
			boolean toReturnCurrent) {
		SubstanceTheme notColorized = null;
		if (component != null) {
			Component comp = component;
			// while (comp != null) {
			if (comp instanceof JComponent) {
				JComponent jcomp = (JComponent) comp;
				Object controlThemeObj = jcomp
						.getClientProperty(SubstanceLookAndFeel.THEME_PROPERTY);
				if (controlThemeObj != null) {
					if (controlThemeObj instanceof String) {
						notColorized = SubstanceTheme
								.getTheme((String) controlThemeObj);
					}
					if (controlThemeObj instanceof ThemeInfo) {
						notColorized = SubstanceTheme
								.createInstance((ThemeInfo) controlThemeObj);
					}
					if (controlThemeObj instanceof SubstanceTheme) {
						notColorized = ((SubstanceTheme) controlThemeObj);
					}
				}
				// Object decorationAreaObj = jcomp.getClientProperty(key)
			}
		}

		if (notColorized == null)
			if (toReturnCurrent)
				return SubstanceLookAndFeel.getTheme();

		return notColorized;
	}

	public static SubstanceTheme getTheme(Component component) {
		SubstanceTheme theme = getDecorationTheme(component);
		if (theme == null) {
			theme = getNonColorizedTheme(component, true);
		}
		if (theme == null) {
			theme = SubstanceLookAndFeel.getTheme();
		}

		return getColorizedTheme(component, theme);
	}

	public static SubstanceTheme getColorizedTheme(Component component,
			SubstanceTheme theme, ColorizationSupport support) {
		if (component != null) {
			// Support for enhancement 256 - colorizing
			// controls.
			Color bk = support.getBackground(component);
			Color fg = support.getForeground(component);
			if (component instanceof SubstanceTitleButton) {
				if ((fg != null) && (bk != null)) {
					// guard for issue 322 - these are null when JavaHelp
					// window is printed.
					fg = SubstanceColorUtilities.getInterpolatedColor(fg, bk,
							0.5);
				}
			}
			if (bk instanceof UIResource)
				bk = null;
			if (fg instanceof UIResource) {
				fg = null;
			}
			if ((bk != null) || (fg != null)) {
				double colorization = SubstanceCoreUtilities
						.getColorizationFactor(component);
				if (colorization > 0.0) {
					return SubstanceShiftTheme.getShiftedTheme(theme, bk,
							colorization, fg, colorization);
				}
			}
		}
		return theme;
	}

	public static interface ColorizationSupport {
		public Color getBackground(Component component);

		public Color getForeground(Component component);
	}

	public static SubstanceTheme getColorizedTheme(Component component,
			SubstanceTheme theme) {
		ColorizationSupport support = new ColorizationSupport() {
			private boolean toTakeFromParent(Component component) {
				return (component.getParent() != null)
						&& ((component instanceof SubstanceScrollButton)
								|| (component instanceof SubstanceSpinnerButton)
								|| (component instanceof SubstanceComboBoxButton) || (component instanceof SubstanceTitleButton));
			}

			public Color getBackground(Component component) {
				return toTakeFromParent(component) ? component.getParent()
						.getBackground() : component.getBackground();
			}

			public Color getForeground(Component component) {
				return toTakeFromParent(component) ? component.getParent()
						.getBackground() : component.getBackground();
			}
		};

		if (component != null) {
			// Support for enhancement 256 - colorizing
			// controls. Special case - scroll, combo and spinner buttons
			// that should take the background from parent.
			boolean takeFromParent = (component instanceof SubstanceScrollButton)
					|| (component instanceof SubstanceSpinnerButton)
					|| (component instanceof SubstanceComboBoxButton)
					|| (component instanceof SubstanceTitleButton);
			takeFromParent = takeFromParent && (component.getParent() != null);
			Color bk = takeFromParent ? component.getParent().getBackground()
					: component.getBackground();
			Color fg = takeFromParent ? component.getParent().getForeground()
					: component.getForeground();
			if (component instanceof SubstanceTitleButton) {
				if ((fg != null) && (bk != null)) {
					// guard for issue 322 - these are null when JavaHelp
					// window is printed.
					fg = SubstanceColorUtilities.getInterpolatedColor(fg, bk,
							0.5);
				}
			}
			if (bk instanceof UIResource)
				bk = null;
			if (fg instanceof UIResource) {
				fg = null;
			}
			if ((bk != null) || (fg != null)) {
				double colorization = SubstanceCoreUtilities
						.getColorizationFactor(component);
				if (colorization > 0.0) {
					return SubstanceShiftTheme.getShiftedTheme(theme, bk,
							colorization, fg, colorization);
				}
			}
		}
		return theme;
	}

	public static SubstanceTheme getNonColorizedTheme(JTabbedPane jtp,
			int tabIndex) {
		SubstanceTheme compTheme = getNonColorizedTheme(jtp
				.getComponentAt(tabIndex), false);
		if (compTheme != null)
			return compTheme;
		SubstanceTheme paneTheme = getNonColorizedTheme(jtp, true);
		return paneTheme;
	}

	public static SubstanceTheme getTheme(final JTabbedPane jtp,
			final int tabIndex, ComponentState componentState) {
		Component component = jtp.getComponent(tabIndex);

		SubstanceTheme nonColorized = getNonColorizedTheme(jtp, tabIndex);
		if (SubstanceCoreUtilities
				.isControlAlwaysPaintedActive(component, true)) {
			if (componentState.isKindActive(FadeKind.ENABLE))
				componentState = ComponentState.SELECTED;
			else
				componentState = ComponentState.DISABLED_SELECTED;
		}
		SubstanceTheme colorized = getColorizedTheme(component, nonColorized,
				new ColorizationSupport() {
					public Color getBackground(Component component) {
						return jtp.getBackgroundAt(tabIndex);
					}

					public Color getForeground(Component component) {
						return jtp.getForegroundAt(tabIndex);
					}
				});
		return colorized.getTheme(component, componentState);
	}

	/**
	 * Returns the theme of the component.
	 * 
	 * @param component
	 *            Component.
	 * @param componentState
	 *            Component state.
	 * @return Component theme.
	 */
	public static SubstanceTheme getTheme(Component component,
			ComponentState componentState) {
		return getTheme(component, componentState, false);
	}

	/**
	 * Returns the theme of the component.
	 * 
	 * @param component
	 *            Component.
	 * @param componentState
	 *            Component state.
	 * @return Component theme.
	 */
	public static SubstanceTheme getTheme(Component component,
			ComponentState componentState, boolean toIgnoreHighlights) {
		if ((component != null)
				&& SubstanceCoreUtilities.isControlAlwaysPaintedActive(
						component, true)
				&& !componentState.isKindActive(FadeKind.PRESS)
				&& componentState.isKindActive(FadeKind.ENABLE)) {
			componentState = ComponentState.ACTIVE;
		}
		SubstanceTheme theme = null;

		boolean isControlActive = componentState.isKindActive(FadeKind.ENABLE)
				&& (componentState != ComponentState.DEFAULT);

		// special case - if the component is marked as flat and
		// it is in the default state, get the theme of the parent
		if (SubstanceCoreUtilities.hasFlatAppearance(component, false)
				&& (componentState == ComponentState.DEFAULT)) {
			component = component.getParent();
		}

		boolean toUseDecorationTheme = false;
		DecorationAreaType decorationType = SubstanceDecorationUtilities
				.getDecorationType(component);
		if (decorationType != null) {
			toUseDecorationTheme = SubstanceLookAndFeel.getTheme()
					.toUseDecorationThemeOnActiveControls(decorationType);
		}

		if (!isControlActive || toUseDecorationTheme) {
			theme = getTheme(component);
		} else {
			// if the component is in active state and participates in
			// rollover, selection, press or arm transition, we need to
			// ignore the decoration theme.
			theme = getNonColorizedTheme(component, true);
			if (theme != null) {
				theme = getColorizedTheme(component, theme);
			}
		}
		if (theme != null) {
			return theme
					.getTheme(component, componentState, toIgnoreHighlights);
		}
		return SubstanceLookAndFeel.getTheme();
	}

	/**
	 * Returns the highlight theme of the component.
	 * 
	 * @param component
	 *            Component.
	 * @param componentState
	 *            Component state.
	 * @return Component highlight theme.
	 */
	public static SubstanceTheme getHighlightTheme(Component component,
			ComponentState componentState) {
		SubstanceTheme nonColorized = getNonColorizedTheme(component, true);
		SubstanceTheme theme = getColorizedTheme(component, nonColorized);
		if (theme != null)
			return theme.getHighlightTheme(component, componentState);
		return null;
	}

	/**
	 * Returns the alpha channel of the highlight theme of the component.
	 * 
	 * @param component
	 *            Component.
	 * @param componentState
	 *            Component state.
	 * @return Highlight theme alpha channel.
	 */
	public static float getHighlightAlpha(Component component,
			ComponentState componentState) {
		SubstanceTheme nonColorized = getNonColorizedTheme(component, true);
		SubstanceTheme theme = getColorizedTheme(component, nonColorized);
		if (theme != null)
			return theme.getHighlightThemeAlpha(component, componentState);
		return 0.0f;
	}

	private static class ConstantThemeWrapper extends SubstanceTheme {
		private SubstanceTheme delegate;

		public ConstantThemeWrapper(SubstanceTheme delegate) {
			super(delegate.getColorScheme(), "Wrapper "
					+ delegate.getDisplayName(), delegate.getKind());
			this.delegate = delegate;
		}

		@Override
		public SubstanceTheme getActiveTheme() {
			return this.delegate;
		}

		@Override
		public SubstanceTheme getActiveTitlePaneTheme() {
			return this.delegate;
		}

		@Override
		public SubstanceTheme getBorderTheme() {
			return this.delegate;
		}

		@Override
		public SubstanceTheme getDecorationTheme(
				DecorationAreaType decorationType) {
			return this.delegate;
		}

		@Override
		public SubstanceTheme getDefaultTheme() {
			return this.delegate;
		}

		@Override
		public SubstanceTheme getDefaultTitlePaneTheme() {
			return this.delegate;
		}

		@Override
		public SubstanceTheme getDisabledTheme() {
			return this.delegate.getDisabledTheme();
		}

		@Override
		public SubstanceTheme getFirstTheme() {
			return this.delegate;
		}

		@Override
		public SubstanceTheme getHighlightTheme(Component comp,
				ComponentState componentState) {
			return this.delegate;
		}

		@Override
		public SubstanceTheme getSecondTheme() {
			return this.delegate;
		}

		@Override
		public SubstanceTheme getTheme(Component comp,
				ComponentState componentState) {
			if (!componentState.isKindActive(FadeKind.ENABLE))
				return this.delegate.getDisabledTheme();
			return this.delegate;
		}

		@Override
		public SubstanceTheme getTheme(Component comp,
				ComponentState componentState, boolean toIgnoreHighlights) {
			if (!componentState.isKindActive(FadeKind.ENABLE))
				return this.delegate.getDisabledTheme();
			return this.delegate;
		}

		@Override
		public float getThemeAlpha(Component comp, ComponentState componentState) {
			return this.delegate.getThemeAlpha(comp, componentState);
		}

		@Override
		public SubstanceTheme getWatermarkTheme() {
			return this.delegate;
		}
	}

	public static SubstanceTheme getConstantTheme(SubstanceTheme theme) {
		return new ConstantThemeWrapper(theme);
	}
}
