/*
 * Copyright (c) 2005-2008 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.substance.theme;

import org.jvnet.substance.color.BlendBiColorScheme;

/**
 * Blend theme. Blend theme is based on two original themes and the blend
 * factor, blending the colors from both themes for the painting. Here is how
 * you can create and set a blend theme:<br>
 * <br>
 * 
 * <code>
 * SubstanceTheme blend = new SubstanceBlendBiTheme(<br>
 * &nbsp;&nbsp;SubstanceAquaTheme(), SubstanceBottleGreenTheme(), 0.4);<br>
 * SubstanceLookAndFeel.setCurrentTheme(blend);<br>
 * for (Frame frame : Frame.getFrames()) {<br>
 * &nbsp;&nbsp;SwingUtilities.updateComponentTreeUI(frame);<br>
 * }
 * </code> <br>
 * This class is part of officially supported API.
 * 
 * @author Kirill Grouchnikov
 */
public class SubstanceBlendBiTheme extends SubstanceTheme {
	/**
	 * The first original theme.
	 */
	private SubstanceTheme originalFirstTheme;

	/**
	 * The second original theme.
	 */
	private SubstanceTheme originalSecondTheme;

	/**
	 * Likeness to the {@link #originalFirstTheme}. Values close to 0.0 will
	 * create theme that closely matches the {@link #originalSecondTheme}.
	 * Values close to 1.0 will create theme that closely matches the
	 * {@link #originalFirstTheme}.
	 */
	private double firstThemeLikeness;

	/**
	 * Creates a new blended theme.
	 * 
	 * @param originalFirstTheme
	 *            The first original theme.
	 * @param originalSecondTheme
	 *            The second original theme.
	 * @param firstThemeLikeness
	 *            Likeness to the first theme. Values close to 0.0 will create
	 *            theme that closely matches the second original theme. Values
	 *            close to 1.0 will create theme that closely matches the first
	 *            original scheme.
	 */
	public SubstanceBlendBiTheme(SubstanceTheme originalFirstTheme,
			SubstanceTheme originalSecondTheme, double firstThemeLikeness) {
		super(new BlendBiColorScheme(originalFirstTheme.getColorScheme(),
				originalSecondTheme.getColorScheme(), firstThemeLikeness),
				"Blended " + originalFirstTheme.getDisplayName() + " & "
						+ originalSecondTheme.getDisplayName() + " "
						+ firstThemeLikeness,
				(firstThemeLikeness >= 0.5) ? originalFirstTheme.getKind()
						: originalSecondTheme.getKind());
		this.originalFirstTheme = originalFirstTheme;
		this.originalSecondTheme = originalSecondTheme;
		this.firstThemeLikeness = firstThemeLikeness;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#tint(double)
	 */
	@Override
	public SubstanceTheme tint(double tintFactor) {
		return new SubstanceBlendBiTheme(this.originalFirstTheme
				.tint(tintFactor), this.originalSecondTheme.tint(tintFactor),
				this.firstThemeLikeness);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#tone(double)
	 */
	@Override
	public SubstanceTheme tone(double toneFactor) {
		return new SubstanceBlendBiTheme(this.originalFirstTheme
				.tone(toneFactor), this.originalSecondTheme.tone(toneFactor),
				this.firstThemeLikeness);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#shade(double)
	 */
	@Override
	public SubstanceTheme shade(double shadeFactor) {
		return new SubstanceBlendBiTheme(this.originalFirstTheme
				.shade(shadeFactor), this.originalSecondTheme
				.shade(shadeFactor), this.firstThemeLikeness);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#saturate(double, boolean)
	 */
	@Override
	public SubstanceTheme saturate(double saturateFactor,
			boolean toSaturateEverything) {
		return new SubstanceBlendBiTheme(this.originalFirstTheme.saturate(
				saturateFactor, toSaturateEverything), this.originalSecondTheme
				.saturate(saturateFactor, toSaturateEverything),
				this.firstThemeLikeness);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#invert()
	 */
	@Override
	public SubstanceTheme invert() {
		return new SubstanceBlendBiTheme(this.originalFirstTheme.invert(),
				this.originalSecondTheme.invert(), this.firstThemeLikeness);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#negate()
	 */
	@Override
	public SubstanceTheme negate() {
		return new SubstanceBlendBiTheme(this.originalFirstTheme.negate(),
				this.originalSecondTheme.negate(), this.firstThemeLikeness);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#hueShift(double)
	 */
	@Override
	public SubstanceTheme hueShift(double hueShiftFactor) {
		return new SubstanceBlendBiTheme(this.originalFirstTheme
				.hueShift(hueShiftFactor), this.originalSecondTheme
				.hueShift(hueShiftFactor), this.firstThemeLikeness);
	}

	/**
	 * Returns the first original theme.
	 * 
	 * @return The first original theme.
	 */
	public SubstanceTheme getOriginalFirstTheme() {
		return this.originalFirstTheme;
	}

	/**
	 * Returns the second original theme.
	 * 
	 * @return The second original theme.
	 */
	public SubstanceTheme getOriginalSecondTheme() {
		return this.originalSecondTheme;
	}
}
