/*
 * Copyright (c) 2007-2011 by The Broad Institute, Inc. and the Massachusetts Institute of
 * Technology.  All Rights Reserved.
 *
 * This software is licensed under the terms of the GNU Lesser General Public License (LGPL),
 * Version 2.1 which is available at http://www.opensource.org/licenses/lgpl-2.1.php.
 *
 * THE SOFTWARE IS PROVIDED "AS IS." THE BROAD AND MIT MAKE NO REPRESENTATIONS OR
 * WARRANTES OF ANY KIND CONCERNING THE SOFTWARE, EXPRESS OR IMPLIED, INCLUDING,
 * WITHOUT LIMITATION, WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, NONINFRINGEMENT, OR THE ABSENCE OF LATENT OR OTHER DEFECTS, WHETHER
 * OR NOT DISCOVERABLE.  IN NO EVENT SHALL THE BROAD OR MIT, OR THEIR RESPECTIVE
 * TRUSTEES, DIRECTORS, OFFICERS, EMPLOYEES, AND AFFILIATES BE LIABLE FOR ANY DAMAGES
 * OF ANY KIND, INCLUDING, WITHOUT LIMITATION, INCIDENTAL OR CONSEQUENTIAL DAMAGES,
 * ECONOMIC DAMAGES OR INJURY TO PROPERTY AND LOST PROFITS, REGARDLESS OF WHETHER
 * THE BROAD OR MIT SHALL BE ADVISED, SHALL HAVE OTHER REASON TO KNOW, OR IN FACT
 * SHALL KNOW OF THE POSSIBILITY OF THE FOREGOING.
 */

package org.broad.igv.util.sigma;

import jsc.independentsamples.MannWhitneyTest;
import net.sf.samtools.util.CloseableIterator;
import org.apache.commons.math.stat.StatUtils;
import org.broad.igv.sam.Alignment;
import org.broad.igv.sam.PairedAlignment;
import org.broad.igv.sam.reader.AlignmentQueryReader;
import org.broad.igv.sam.reader.SamQueryReaderFactory;
import org.broad.igv.util.collections.DoubleArrayList;
import org.broad.igv.util.stats.Distribution;

import java.io.FileWriter;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author jrobinso
 * @date Feb 16, 2011
 */
public class KCCN3Analysis {
    static String chr = "chr1";
    static int start1 = 153108824;
    static int end1 = 153108866;
    static int start2 = 153108938;
    static int end2 = 153108980;

    public static void main(String[] args) throws IOException {

        String[] affectedBams = {
                "http://iwww.broadinstitute.org/igvdata/sigma/aggregation/G8552/259/v2/259.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/aggregation/G4202/265/v2/265.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/aggregation/G4200/266/v4/266.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/aggregation/G4203/352/v6/352.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/aggregation/G4204/414/v5/414.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/aggregation/G8551/4/v1/4.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/aggregation/G4198/8/v3/8.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/aggregation/G4206/13/v4/13.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/aggregation/G4205/15/v3/15.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/aggregation/G8556/564/v2/564.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/aggregation/G8556/564/v2/564.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/aggregation/G8558/563/v3/563.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/aggregation/G8558/563/v3/563.bam"
        };

        String[] exomeBams = {
                "http://iwww.broadinstitute.org/igvdata/sigma/exome/259/v1/259.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/exome/265/v4/265.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/exome/266/v1/266.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/exome/352/v2/352.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/exome/375/v2/375.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/exome/414/v3/414.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/exome/15/v3/15.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/exome/8/v3/8.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/exome/564/v2/564.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/exome/701/v1/701.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/exome/4/v2/4.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/exome/13/v1/13.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/exome/563/v1/563.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/exome/566/v1/566.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/exome/469/v1/469.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/exome/479/v1/479.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/exome/491/v1/491.bam"
        };

        String[] normalBams = {
                "http://iwww.broadinstitute.org/igvdata/sigma/aggregation/G4201/375/v2/375.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/bams/controls/bwa_validation/bwa_hg18_control1.bam",
                "http://iwww.broadinstitute.org/igvdata/sigma/bams/controls/bwa_validation/bwa_hg18_control2.bam"
        };


        PrintWriter samWriter = new PrintWriter(new FileWriter("left_exome.sam"));


        PrintWriter pw = new PrintWriter(new FileWriter("kccn3_exome.txt")); // new PrintWriter(new FileWriter( + ".dist.txt"));
        run(exomeBams, pw, samWriter);

        //PrintWriter pw = new PrintWriter(new FileWriter("kccn3_wgs.txt")); // new PrintWriter(new FileWriter( + ".dist.txt"));
        //run(affectedBams, pw);

        pw.println();
        pw.println();

        //System.out.println("Normals");
        //pw.println("Normals");
        //run(normalBams, pw);
        //pw.println();
        //pw.println();

        pw.close();
        samWriter.close();


    }

    private static void run(String[] bams, PrintWriter pw, PrintWriter samWriter) throws IOException {
        Distribution allDist = new Distribution(700);
        Distribution spanningDist = new Distribution(700);
        Distribution leftDist = new Distribution(700);
        Distribution rightDist = new Distribution(700);

        DoubleArrayList allISizes = new DoubleArrayList(12000);
        DoubleArrayList rightISizes = new DoubleArrayList(1000);
        DoubleArrayList leftISizes = new DoubleArrayList(1000);
        DoubleArrayList spanningISizes = new DoubleArrayList(1000);

        int totalPairs = 0;
        int spanningPairs = 0;
        int leftPairs = 0;
        int rightPairs = 0;
        double avgTotalLength = 0;
        double avgSpanningLength = 0;
        double avgLeftLength = 0;
        double avgRightLength = 0;


        // For each sample
        for (String bamfile : bams) {
            Map<String, PairedAlignment> pairCache = new HashMap();
            List<PairedAlignment> pairs = new ArrayList(15000);
            AlignmentQueryReader reader = SamQueryReaderFactory.getReader(bamfile, true);

            int qStart = start1 - 10000;
            int qEnd = end2 + 10000;
            CloseableIterator<Alignment> iter = reader.query(chr, qStart, qEnd, false);

            while (iter.hasNext()) {
                Alignment al = iter.next();
                if (al.isPaired() && al.getMate().isMapped() && al.isProperPair()) {
                    PairedAlignment pair = pairCache.get(al.getReadName());
                    if (pair == null) {
                        pairCache.put(al.getReadName(), new PairedAlignment(al));
                    } else {
                        pairCache.remove(al.getReadName());
                        pair.setSecondAlignment(al);
                        pairs.add(pair);
                    }
                }
            }
            iter.close();
            reader.close();

            DoubleArrayList allISizesForThisSample = new DoubleArrayList(10000);
            for (PairedAlignment pair : pairs) {
                int iSize = Math.abs(pair.getInferredInsertSize());
                allISizesForThisSample.add(iSize);
            }
            double[] sampleISizes = allISizesForThisSample.toArray();
            int sampleMedian = (int) StatUtils.percentile(sampleISizes, 50);
            System.out.println(bamfile + "  MEDIAN = " + sampleMedian + "  (" + sampleISizes.length + ")");
            if (sampleISizes.length == 0) {
                continue;
            }

            for (PairedAlignment pair : pairs) {

                Alignment leftAlignment = pair.getFirstAlignment();
                Alignment rightAlignment = pair.getSecondAlignment();

                totalPairs++;
                avgTotalLength += pair.getEnd() - pair.getStart();
                int iSize = sampleMedian - Math.abs(pair.getInferredInsertSize());

                allDist.addDataPoint(iSize);
                allISizes.add(iSize);
                if (pair.getStart() < start1 && pair.getEnd() > end2) {
                    spanningDist.addDataPoint(iSize);
                    spanningPairs++;
                    avgSpanningLength += pair.getEnd() - pair.getStart();
                    spanningISizes.add(iSize);
                }
                if (pair.getStart() < start1 && rightAlignment.getEnd() > end1 && rightAlignment.getStart() < start2) {
                    samWriter.println(leftAlignment.toString());
                    samWriter.println(rightAlignment.toString());
                    leftDist.addDataPoint(iSize);
                    leftPairs++;
                    avgLeftLength += pair.getEnd() - pair.getStart();
                    leftISizes.add(iSize);

                }
                if (pair.getEnd() > end2 && leftAlignment.getEnd() > end1 && leftAlignment.getStart() < start2) {

                    rightDist.addDataPoint(iSize);
                    rightPairs++;
                    avgRightLength += pair.getEnd() - pair.getStart();
                    rightISizes.add(iSize);
                }

            }
        }


        avgTotalLength /= totalPairs;
        pw.println("total pairs = " + totalPairs + "   " + avgTotalLength);
        avgSpanningLength /= spanningPairs;
        avgLeftLength /= leftPairs;
        avgRightLength /= rightPairs;

        pw.println("spanning pairs = " + spanningPairs + "   " + avgSpanningLength);
        pw.println("left pairs = " + leftPairs + "   " + avgLeftLength);
        pw.println("right pairs = " + rightPairs + "   " + avgRightLength);

        pw.println("All");
        allDist.print(pw);
        pw.println();

        pw.println("Left");
        leftDist.print(pw);
        pw.println();

        pw.println("Right");
        rightDist.print(pw);


        double allMean = StatUtils.mean(allISizes.toArray());
        double rightMean = StatUtils.mean(rightISizes.toArray());
        double leftMean = StatUtils.mean(leftISizes.toArray());
        // double spanningMean = StatUtils.mean(spanningISizes.toArray());
        double allMedian = StatUtils.percentile(allISizes.toArray(), 50);
        double rightMedian = StatUtils.percentile(rightISizes.toArray(), 50);
        double leftMedian = StatUtils.percentile(leftISizes.toArray(), 50);
        //double spanningMedian = StatUtils.percentile(spanningISizes.toArray(), 50);
        pw.println("All mean =              " + allMean + "    median =  " + allMedian + "  (" + allISizes.toArray().length + ")");
        pw.println("Right mean =            " + rightMean + "    median =  " + rightMedian + "  (" + rightISizes.toArray().length + ")");
        pw.println("Left mean =             " + leftMean + "    median =  " + leftMedian + "  (" + leftISizes.toArray().length + ")");
        //pw.println("Spanning mean = " + spanningMean + "    median =  " + spanningMedian + "  (" + spanningISizes.toArray().length + ")");


        MannWhitneyTest test = new MannWhitneyTest(allISizes.toArray(), rightISizes.toArray());
        pw.println("MannWhitneyTest  (All vs Right)");
        pw.println("  Significance probability: " + test.exactSP());
        pw.println("  Test statistic (U):             " + test.getStatistic());

        MannWhitneyTest test2 = new MannWhitneyTest(allISizes.toArray(), leftISizes.toArray());
        pw.println("MannWhitneyTest  (All vs Left)");
        pw.println("  Significance probability: " + test2.exactSP());
        pw.println("  Test statistic (U):             " + test2.getStatistic());

        //MannWhitneyTest test3 = new MannWhitneyTest(allISizes.toArray(), spanningISizes.toArray());
        //pw.println("MannWhitneyTest  (All vs Spanning)");
        //pw.println("  Significance probability: " + test3.exactSP());
        //pw.println("  Test statistic (U):             " + test3.getStatistic());


        //MannWhitneyTest test4 = new MannWhitneyTest(leftISizes.toArray(), rightISizes.toArray());
        //pw.println("MannWhitneyTest  (Left vs Right)");
        //pw.println("  Significance probability: " + test4.exactSP());
        //pw.println("  Test statistic (U):             " + test4.getStatistic());
    }

    /*
               for (AlignmentInterval.Row row : alignmentRows) {
               for (Alignment al : row.alignments) {
                   if (al instanceof PairedAlignment) {
                       PairedAlignment pair = (PairedAlignment) al;
                       alignments.add(pair.firstAlignment);
                       if (pair.secondAlignment != null) {
                           alignments.add(pair.secondAlignment);
                       }
                   } else {
                       alignments.add(al);
                   }
               }
           }
    */
}

