# $Id: Page.pm,v 1.12 2011-06-27 15:56:06 sb23 Exp $

package EnsEMBL::Web::Controller::Page;

### Prints the main web page - header, footer, navigation etc, and non dynamically loaded content.
### Deals with Command modules if required.

use strict;

use base qw(EnsEMBL::Web::Controller);

sub request   { return 'page'; }
sub cacheable { return 1;      }

sub init {
  my $self    = shift;
  my $hub     = $self->hub;
  my $request = $self->request;
  
  $self->clear_cached_content;
  
  my $cached = $self->get_cached_content($request); # Page retrieved from cache
  
  if (!$cached) {
    my $redirect = $self->builder->create_objects($hub->factorytype, $request) eq 'redirect'; # Build objects before checking configuration - they are needed by Configuration.pm
    $self->configure;                                                                         # Set hub->components before checking configuration
    return if $self->update_configuration_from_url || $redirect;                              # Redirect if a configuration has been updated or object parameters have changed
  }
  
  $self->update_user_history if $hub->user;
  
  return if $cached;
  
  $self->page->initialize; # Adds the components to be rendered to the page module
  $self->render_page;
}

sub render_page {
  my $self = shift;
  $self->SUPER::render_page if $self->access_ok && !$self->process_command;
}

sub update_configuration_from_url {
  ### Checks for shared data and updated config settings from the URL parameters
  ### If either exist, returns 1 to force a redirect to the updated page
  ### This function is only called during main page (EnsEMBL::Web::Magic::stuff) requests
  
  my $self      = shift;
  my $r         = $self->r;
  my $input     = $self->input;
  my $hub       = $self->hub;
  my $session   = $hub->session;
  my @share_ref = $input->param('share_ref');
  my $new_url;
  
  if (@share_ref) {
    $session->receive_shared_data(@share_ref); # This should push a message onto the message queue
    $input->delete('share_ref');
    $new_url = 1;
  }
  
  $new_url += $hub->get_viewconfig($_)->update_from_url($r) for @{$self->configuration->get_configurable_components}; # This should push a message onto the message queue
  
  if ($new_url) {
    $input->param('time', time); # Add time to cache-bust the browser
    (my $query_string = $input->query_string) =~ s/%3A/:/g; # Unescape : in query string generated by $input->query_string
    $input->redirect($r->uri . "?$query_string"); # If something has changed then we redirect to the new page
    return 1;
  }
}

sub process_command {
  ### Handles Command modules and the Framework-based database frontend. 
  ### Once the command has been processed, a redirect to a Component page will occur.
  
  my $self    = shift;
  my $command = $self->command;
  my $action  = $self->action;
  
  return unless $command || $action eq 'Wizard';
  
  my $object  = $self->object;
  my $page    = $self->page;
  my $builder = $self->builder;
  my $hub     = $self->hub;
  my $node    = $self->node;
  
  if ($command eq 'db_frontend') {
    my $type     = $self->type;
    my $function = $self->function || 'Display';

    # Look for all possible modules for this URL, in order of specificity and likelihood
    my @classes = (
      "EnsEMBL::Web::Component::${type}::${action}::$function",
      "EnsEMBL::Web::Command::${type}::${action}::$function",
      "EnsEMBL::Web::Component::DbFrontend::$function",
      "EnsEMBL::Web::Command::DbFrontend::$function"
    );

    foreach my $class (@classes) {
      if ($self->dynamic_use($class)) {
        if ($class =~ /Command/) {
          my $command_module = $class->new({
            object => $object,
            hub    => $hub,
            page   => $page,
            node   => $node
          });
          
          my $rtn = $command_module->process;
          
          return defined $rtn ? $rtn : 1;
        } else {
          $self->SUPER::render_page;
        }
      }
    }
  } else {
    # Normal command module
    my $class = $action eq 'Wizard' ? 'EnsEMBL::Web::Command::Wizard' : $command;
    
    if ($class && $self->dynamic_use($class)) {
      my $command_module = $class->new({
        object => $object,
        hub    => $hub,
        page   => $page,
        node   => $node
      });
      
      my $rtn = $command_module->process;
      
      return defined $rtn ? $rtn : 1;
    }
  }
}

sub access_ok {
  ### Checks if the given Command module is allowed, and forces a redirect if it isn't
  
  my $self = shift;
  
  my $filter = $self->not_allowed($self->hub);
  
  if ($filter) {
    my $url = $filter->redirect_url;
    
    # Double-check that a filter name is being passed, since we have the option 
    # of using the default URL (current page) rather than setting it explicitly
    $url .= ($url =~ /\?/ ? ';' : '?') . 'filter_module=' . $filter->name       unless $url =~ /filter_module/;
    $url .= ($url =~ /\?/ ? ';' : '?') . 'filter_code='   . $filter->error_code unless $url =~ /filter_code/;
    
    $self->page->ajax_redirect($url);
    
    return 0;
  }
  
  return 1;
}

1;