/**********************************************************************
zyGrib: meteorological GRIB file viewer
Copyright (C) 2008-2010 - Jacques Zaninetti - http://www.zygrib.org

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
***********************************************************************/

#include <QApplication>
#include <QTextStream>
#include <QDomDocument>

#include <cassert>

#include "FileLoaderGRIB.h"
#include "Util.h"
#include "Version.h"
#include "sha1/sha1.h"

//-------------------------------------------------------------------------------
FileLoaderGRIB::FileLoaderGRIB()
{
    step = 0;
    
    idFileName = -1;    // id des requêtes à -1, car ev. reçu lors du "setHost()"
    idGribContent = -1;
    
    zygriblog = "a07622b82b18524d2088c9b272bb3feeb0eb1737";
    zygribpwd = "61c9b2b17db77a27841bbeeabff923448b0f6388";
    
    http = new QHttp(this);
    assert(http);

	host = "www.zygrib.org";
	scriptpath = "/noaa/";
	scriptstock = "313O562/";
	
    http->setHost(host);


    connect(http, SIGNAL(done(bool)), this, SLOT(slotDone(bool)));

    connect(http, SIGNAL(requestFinished (int, bool)),
            this, SLOT(requestFinished (int, bool)));
    
    connect(http, SIGNAL(dataReadProgress (int, int)),
            this, SLOT(dataReadProgress (int, int)));
}
//-------------------------------------------------------------------------------
FileLoaderGRIB::~FileLoaderGRIB () {
    if (http != NULL)
        delete http;
}

//-------------------------------------------------------------------------------
void FileLoaderGRIB::stop () {
    http->abort();
}

//-------------------------------------------------------------------------------
void FileLoaderGRIB::dataReadProgress (int done , int  total) {
    emit signalGribReadProgress(step, done, total);
}

//-------------------------------------------------------------------------------
void FileLoaderGRIB::getGribFile(
        float x0, float y0, float x1, float y1,
        float resolution, int interval, int days,
        bool wind, bool pressure, bool rain,
        bool cloud, bool temp, bool humid, bool isotherm0,
		bool tempPot, bool tempMin, bool tempMax, bool snowDepth,
		bool snowCateg, bool frzRainCateg,
		bool CAPEsfc, 
		bool altitudeData200,
		bool altitudeData300,
		bool altitudeData500,
		bool altitudeData700,
		bool altitudeData850
	)
{
    QString page;
    
    fileLoaded = false;
    statusErrorString = "";
    
    //----------------------------------------------------------------
    // Etape 1 : Demande la création du fichier Grib (nom en retour)
    //----------------------------------------------------------------
    QString parameters = "";
    if (wind) {
        parameters += "W";
    }
    if (pressure) {
        parameters += "P";
    }
    if (rain) {
        parameters += "R";
    }
    if (cloud) {
        parameters += "C";
    }
    if (temp) {
        parameters += "T";
    }
    if (humid) {
        parameters += "H";
    }
	if (isotherm0) {
		parameters += "I";
    }    
    if (tempPot) {
        parameters += "t";
    }
    if (tempMin) {
        parameters += "m";
    }
    if (tempMax) {
        parameters += "M";
    }
    if (snowDepth) {
        parameters += "S";
    }
    if (snowCateg) {
        parameters += "s";
    }
    if (frzRainCateg) {
        parameters += "Z";
    }
    if (CAPEsfc) {
        parameters += "c";
    }
	
    if (altitudeData200) parameters += "2";
    if (altitudeData300) parameters += "3";
    if (altitudeData500) parameters += "5";
    if (altitudeData700) parameters += "7";
    if (altitudeData850) parameters += "8";

//parameters += "r";	// PRATE
    
    if (parameters != "")
    {
        if (Util::getSetting("httpUseProxy", false).toBool())
        {
            http->setProxy(
                Util::getSetting("httpProxyHostname", "").toString(),
                Util::getSetting("httpProxyPort", 0).toInt(),
                Util::getSetting("httpProxyUsername", "").toString(),
                Util::getSetting("httpProxyUserPassword", "").toString()
            );
        }
        else {
            http->setProxy("",0,"","");
        }
        
        step = 1;
        emit signalGribSendMessage(
        		tr("Make file on server... Please wait..."));
        emit signalGribReadProgress(step, 0, 0);
		
		QString phpfilename;
		
		phpfilename = scriptpath+"getzygribfile2.php?";
        QTextStream(&page) << phpfilename
                           << "but=prepfile"
                           << "&la1=" << floor(y0)
                           << "&la2=" << ceil(y1)
                           << "&lo1=" << floor(x0)
                           << "&lo2=" << ceil(x1)
                           << "&res=" << resolution
                           << "&hrs=" << interval
                           << "&jrs=" << days
                           << "&par=" << parameters
                           << "&l=" << zygriblog
                           << "&m=" << zygribpwd
                           << "&client=" << Version::getCompleteName()
                           ;
		
		if (ioBuffer.isOpen()) {
			ioBuffer.close();
			ioBuffer.setBuffer(NULL);
		}

		QHttpRequestHeader header("GET", page);
		header.setValue("Host", host);
		header.setValue("User-Agent", Version::getAppName()+"/"+Version::getVersion());
		idFileName = http->request(header, 0,  &ioBuffer);
        
        //idFileName = http->get(page, &ioBuffer);
		//printf("SEND idFileName=%d\n", idFileName);
    }
    // Suite de la séquence de récupération dans requestFinished()
}

//-------------------------------------------------------------------------------
void FileLoaderGRIB::requestFinished ( int id, bool error )
{
//printf("requestFinished id=%d error=%d\n", id,error);
    QString page;
    if (error) {
        emit signalGribLoadError(http->errorString());
    }
    else if (id==idFileName)
    {
        //-------------------------------------------
        // Retour de l'étape 1 : préparation du fichier
        //-------------------------------------------
        QString strbuf = ioBuffer.buffer();
        QStringList lsbuf = strbuf.split("\n");

		QString status;
        for (int i=0; i < lsbuf.size(); i++)
        {
            QStringList lsval = lsbuf.at(i).split(":");
            if (lsval.size() >= 2) {
                if (lsval.at(0) == "status")
                    status = lsval.at(1);
                else if (lsval.at(0) == "file") 
                    fileName = QString(lsval.at(1)).replace(".grb","%20");
                else if (lsval.at(0) == "size")
                    fileSize = lsval.at(1).toInt();
                else if (lsval.at(0) == "checksum")
                    checkSumSHA1 = lsval.at(1);
            }
        }
        
        //--------------------------------------------------
        // Etape 2 : Demande le contenu du fichier Grib
        //--------------------------------------------------
        if (status == "ok") {
            step = 2;
            QString s;
            s = tr("Total size : ") + s.sprintf("%d",fileSize/1024) + " ko";
            emit signalGribSendMessage(s);
            QTextStream(&page) << scriptpath
			<<"313O562/"<<fileName;
            ioBuffer.close();
            ioBuffer.setBuffer(NULL);
            emit signalGribStartLoadData();            
            idGribContent = http->get(page, &ioBuffer);
        }
        else if (idFileName != -1)
        {
    		fileLoaded = false;
    		idFileName = -1;
    		statusErrorString = status;
        }
    }
    else if (id==idGribContent)
    {
    	fileLoaded = true;
    }
}



//-------------------------------------------------------------------------------
void FileLoaderGRIB::slotDone (bool error)
{
//printf("slotDone() error=%d\n", error);

	if (! fileLoaded  || error) {
		emit signalGribLoadError(statusErrorString);
		return;
	}
	
	//--------------------------------------------------
	// Reçu le contenu du fichier Grib
	//--------------------------------------------------
	arrayContent = ioBuffer.buffer();
	if (arrayContent.size() == 0) {
		//printf("arrayContent.size() == 0\n");
		return;
	}
	
	//--------------------------------------------------
	// Vérifie le checksum
	//--------------------------------------------------
	emit signalGribSendMessage(tr("CheckSum control"));
	SHA1 sha1;
	unsigned char * digest;
	sha1.addBytes( arrayContent.data(), arrayContent.size() );
	digest = sha1.getDigest();
	assert( digest );
	QString s, strsha1 = "";
	for (int i=0; i<20; i++) {
		strsha1 += s.sprintf("%02x", digest[i]);
	}
	free( digest );
	if (strsha1 == checkSumSHA1)
	{
		//--------------------------------------------------
		// Signale la fin du téléchargement
		//--------------------------------------------------
		emit signalGribSendMessage(tr("Finish")
						+ QString(" %1 ko").arg(arrayContent.size()/1024));
		emit signalGribDataReceived(&arrayContent, fileName.replace("%20",".grb"));
	}
	else {
		emit signalGribLoadError(tr("Bad checksum."));
	}
}


