/***************************************************************************
 *
 * Copyright (c) 2000, 2001, 2002, 2003, 2004, 2005 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * Note that this permission is granted for only version 2 of the GPL.
 *
 * As an additional exemption you are allowed to compile & link against the
 * OpenSSL libraries as published by the OpenSSL project. See the file
 * COPYING for details.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: main.c,v 1.27 2004/07/01 16:55:25 bazsi Exp $
 *
 * Author  : bazsi
 * Auditor : 
 * Last version : 
 * Notes   :
 *
 ***************************************************************************/

#include "zorpctl.h"
#include "szig.h"

#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <signal.h>

#include <time.h>
#include <wait.h>
#include <syslog.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <fcntl.h>
#include <popt.h>
#include <stdarg.h>
#include <errno.h>
#include <sys/stat.h>

#include <grp.h>
#include <pwd.h>

#define ZORP ZORP_LIBDIR    "/zorp"

#if 1
#define ZORP_INSTANCES_CONF ZORP_SYSCONFDIR "/instances.conf"
#define ZORP_ZORPCTL_CONF   ZORP_SYSCONFDIR "/zorpctl.conf"
#else
#define ZORP_INSTANCES_CONF  "instances.conf"
#define ZORP_ZORPCTL_CONF    "zorpctl.conf"
#endif

#define UNUSED __attribute__((unused))

#define MAX_ZORPCTL_LINE_LENGTH 4096

#define MAX(a,b) ((a) < (b) ? (b) : (a))

/* Refer a string or the constant "(null)" if it's NULL
 * This is the default behaviour on Linux when printf()ing NULL, but on Solaris
 * it triggers a SIGSEGV
 */
#define SAFE_STR(n) ((n) ? (n) : "(null)")
#define SAFE_STR_EMPTY(n) ((n) ? (n) : "")

typedef struct _ZInstance ZInstance;

struct _ZInstance
{
  ZInstance *next;
  char *name;
  char **zorp_argv;
  int zorp_argc;
  char **zorpctl_argv;
  int zorpctl_argc;
  
  int auto_restart;
  int fd_limit;
  int process_limit;
  int enable_core;
  int no_auto_start;
};

typedef struct _ZCommand ZCommand;
struct _ZCommand
{
  char *cmd;
  int (*func)(int argc, char *argv[]);
  char *help;
};

ZInstance *instances;
extern ZCommand commands[];

int auto_restart = 1;
int auto_restart_time_threshold = 60;
int auto_restart_max_count = 3;
int auto_restart_delay = 3;
int start_check_timeout = 5;
int start_wait_timeout = 60;
int stop_check_delay = 1;
int stop_check_timeout = 3;
int process_limit_reserve = 64;
int process_limit_threshold = -1;
int process_limit_min = 256;
int process_limit_sum = 0;
int fd_limit_threshold = 64;
int fd_limit_min = 1024;
int check_perms = 1;
char *zorp_append_args = NULL;
char *zorpctl_append_args = NULL;
char *proc_title;
char *pidfile_dir = "/var/run/zorp", *pidfile_dir_owner = NULL, *pidfile_dir_group = NULL, *pidfile_dir_perm = NULL;
char *config_dir = ZORP_SYSCONFDIR, *config_dir_owner = "root", *config_dir_group = "zorp", *config_dir_perm = "0750";
char delayed_errors[16384];
int sigalarm_received = 0;

static inline void
chomp(char *line)
{
  int len = strlen(line);
  
  if (line[len-1] == '\n')
    line[len-1] = 0;
}

static inline char *
strduplen(char *line, int len)
{
  char *res;
  res = malloc(len + 1);
  strncpy(res, line, len+1);
  res[len] = 0;
  return res;
}

static void
z_sigalarm_handler(int signo UNUSED)
{
  sigalarm_received = 1;
  signal(SIGALRM, z_sigalarm_handler);
}

static inline void
z_alarm_request(int seconds)
{
  sigalarm_received = 0;
  alarm(seconds);
}

static inline int
z_alarm_fired(void)
{
  return sigalarm_received;
}

static inline void
z_setup_signals(void)
{
  siginterrupt(SIGALRM, 1);
  signal(SIGALRM, z_sigalarm_handler);
}

void
z_error(int delay, const char *format, ...)
{
  char buf[1024];
  va_list args;
  
  va_start(args, format);
  if (delay)
    {
      vsnprintf(buf, sizeof(buf), format, args);
      strncat(delayed_errors, buf, sizeof(delayed_errors));
    }
  else
    {
      vfprintf(stderr, format, args);
    }
  va_end(args);
}

void
z_dump_errors(void)
{
  if (delayed_errors[0])
    {
      fprintf(stderr, "\nThe following errors occurred so far:\n%s\n", delayed_errors);
      delayed_errors[0] = 0;
    }
}

int
z_instance_running(ZInstance *inst, pid_t *pid, int *stale)
{
  FILE *pidfile;
  char buf[256];
  
  *stale = 0;
  snprintf(buf, sizeof(buf), "%s/zorp-%s.pid", pidfile_dir, inst->name);
  pidfile = fopen(buf, "r");
  if (!pidfile)
    {
      return 0;
    }
  fgets(buf, sizeof(buf), pidfile);
  *pid = atoi(buf);
  fclose(pidfile);
  if (!(*pid))
    {
      /* invalid pid, pid 0 */
      *stale = 1;
      return 0;
    }
  if (kill(*pid, SIGCONT) == 0)
    {
      /* still running */
      return 1;
    }
  /* stale pidfile */
  *stale = 1;
  return 0;
}

void 
z_instance_remove_stale_pidfile(ZInstance *inst)
{
  char buf[256];
  
  snprintf(buf, sizeof(buf), "%s/zorp-%s.pid", pidfile_dir, inst->name);
  unlink(buf);
}

void
z_instance_free(ZInstance *inst)
{
  free(inst->name);
  if (inst->zorp_argv)
    free(inst->zorp_argv);
  if (inst->zorpctl_argv)
    free(inst->zorpctl_argv);
  free(inst);
}

int
z_check_instance_name(char *name)
{
  int i;
  for (i = 0; name[i]; i++)
    {
      if (!isalnum(name[i]) && name[i] != '_')
        return 0;
    }
  return isalpha(name[0]);
}

int
z_check_dir(char *dir, char *owner, char *group, char *perm, int create)
{
  struct passwd *pw = getpwnam(owner);
  struct group *gr = group ? getgrnam(group) : NULL;
  uid_t uid = -1;
  gid_t gid = -1;
  mode_t mode = perm ? strtol(perm, NULL, 8) : 0700;
  struct stat st;
  
  if (pw)
    uid = pw->pw_uid;
  if (gr)
    gid = gr->gr_gid;
  if (gid == (gid_t) -1 && pw)
    gid = pw->pw_gid;
  
  if (uid == (gid_t) -1 || gid == (gid_t) -1)
    {
      z_error(0, "Owner/group not found, owner='%s', group='%s'\n", 
                 SAFE_STR(owner), SAFE_STR(group));
      return 0;
    }
  if (stat(dir, &st) < 0)
    {
      if (create)
        {
          if (mkdir(dir, mode) >= 0 && chown(dir, uid, gid) >= 0)
            return 1;
          z_error(0, "Error creating directory, dir='%s', uid='%d', gid='%d', mode='%o', error='%s'\n", dir, uid, gid, perm, strerror(errno));
        }
      return 0;
    }
  if (owner && st.st_uid != uid)
    return 0;
  if (group && st.st_gid != gid)
    return 0;
  if (perm && ((st.st_mode & 07777) != mode))
    return 0;
  return 1;
}

void
z_check_pidfile_dir(void)
{
  if (pidfile_dir_owner || pidfile_dir_group || pidfile_dir_perm)
    {
      z_check_dir(pidfile_dir, pidfile_dir_owner, pidfile_dir_group, pidfile_dir_perm, 1);
    }
}

int
z_check_config_dir(void)
{
  if (check_perms)
    {
      if (!z_check_dir(config_dir, config_dir_owner, config_dir_group, config_dir_perm, 0))
        {
          z_error(0, "Config directory has invalid permissions, expected: dir='%s', owner='%s', group='%s', perm='%s'\n", config_dir, SAFE_STR(config_dir_owner), SAFE_STR(config_dir_group), SAFE_STR(config_dir_perm));
          return 0;
        }
    }
  return 1;
}

int
z_parse_args(ZInstance *inst, char *zorp_args, char *zorpctl_args)
{
  static struct poptOption zorpctl_options[] = 
  {
    { "auto-restart", 'A', POPT_ARG_NONE, NULL, 'A', NULL, NULL },
    { "no-auto-restart", 'a', POPT_ARG_NONE, NULL, 'a', NULL, NULL },
    { "no-auto-start", 'S', POPT_ARG_NONE, NULL, 'S', NULL, NULL },
    { "fd-limit", 'f', POPT_ARG_INT, NULL, 'f', NULL, NULL },
    { "process-limit", 'p', POPT_ARG_INT, NULL, 'p', NULL, NULL },
    { "enable-core", 'c', POPT_ARG_NONE, NULL, 'c', NULL, NULL },
    { NULL, 0, 0, NULL, 0, NULL, NULL }
  };
  poptContext ctx;
  int i, threads = 100, opt;
  char buf[MAX_ZORPCTL_LINE_LENGTH + 256];
  const char    *leftover_arg;
  
  /* the length of ZORP is assumed to be less than 256 chars */
  sprintf(buf, "%s --as %s %s %s", ZORP, inst->name, zorp_args, SAFE_STR_EMPTY(zorp_append_args));
  if (poptParseArgvString(buf, &inst->zorp_argc, (const char ***) &inst->zorp_argv) < 0)
    {
      z_error(1, "Invalid zorp argument list: %s\n", zorp_args);
      return 0;
    }
  sprintf(buf, "zorpctl %s %s", SAFE_STR_EMPTY(zorpctl_args), SAFE_STR_EMPTY(zorpctl_append_args));
  if (poptParseArgvString(buf, &inst->zorpctl_argc, (const char ***) &inst->zorpctl_argv) < 0)
    {
      z_error(1, "Invalid zorpctl argument list: %s\n", zorpctl_args);
      return 0;
    }
  
  for (i = 1; i < inst->zorp_argc; i++)
    {
      if (strcmp(inst->zorp_argv[i], "--threads") == 0)
        {
          if (i + 1 < inst->zorp_argc)
            {
              threads = atoi(inst->zorp_argv[i+1]);
            }
          else
            threads = 100;
          process_limit_sum += threads + 5;
          break;
        }
    }
    
  inst->auto_restart = auto_restart;
  inst->fd_limit = MAX(fd_limit_threshold * threads + 64, fd_limit_min);
  inst->process_limit = MAX(process_limit_min, threads + 5);
 
  if (!inst->zorpctl_argv)
    return 1;
  ctx = poptGetContext("zorpctl", inst->zorpctl_argc, (const char **) inst->zorpctl_argv, zorpctl_options, 0);
  while ((opt = poptGetNextOpt(ctx)) > 0)
    {
      switch (opt)
        {
        case 'A':
          inst->auto_restart = 1;
          break;
        case 'a':
          inst->auto_restart = 0;
          break;
        case 'f':
          inst->fd_limit = atoi(poptGetOptArg(ctx));
          break;
        case 'p':
          inst->process_limit = atoi(poptGetOptArg(ctx));
          break;
        case 'c':
          inst->enable_core = 1;
          break;
        case 'S':
          inst->no_auto_start = 1;
          break;
        }
    }
  
  if (opt < -1)
    {
      z_error(1, "Invalid argument (%s) encountered, error: %s\n",
              poptBadOption(ctx, POPT_BADOPTION_NOALIAS), poptStrerror(opt));
      poptFreeContext(ctx);
      return 0;
  }
            
  leftover_arg = poptPeekArg(ctx);
  if (leftover_arg)
    {
      z_error(1, "Junk found at the end of the arg list: %s\n", leftover_arg);
      poptFreeContext(ctx);
      return 0;
    }
  poptFreeContext(ctx);
  return 1;
}

int
z_parse_instances(void)
{
  FILE *instf;
  char line[MAX_ZORPCTL_LINE_LENGTH];
  ZInstance *inst, **last = &instances;
  int len, i;
  char *dashdash, *zorp_args, *zorpctl_args;

  instances = NULL;  
  instf = fopen(ZORP_INSTANCES_CONF, "r");
  if (!instf)
    {
      /*LOG
       *This message indicates that Zorp was unable to open the instances file.
       *Check the permissions of your instances file.
       */
      z_error(1, "Error opening instances file: %s\n", ZORP_INSTANCES_CONF);
      return 0;
    }
  while (fgets(line, sizeof(line), instf))
    {
      chomp(line);
      if (line[0] == 0 || line[0] == '#')
        continue;
      len = strlen(line);
      inst = calloc(sizeof(ZInstance), 1);
      i = 0;
      while (i < len && !isspace(line[i]))
        i++;
      inst->name = strduplen(line, i);
      if (!z_check_instance_name(inst->name))
        {
          z_error(1, "Invalid instance name: %s\n", inst->name);
          z_instance_free(inst);
          return 0;
        }
      while (i < len && isspace(line[i]))
        i++;
      dashdash = strstr(&line[i], " -- ");
      if (dashdash)
        {
          zorp_args = strduplen(&line[i], dashdash - &line[i]);
          zorpctl_args = strdup(dashdash + 4);
        }
      else
        {
          zorp_args = strdup(&line[i]);
          zorpctl_args = NULL;
        }
      
      if (!z_parse_args(inst, zorp_args, zorpctl_args))
        {
          z_error(1, "Invalid argument list at instance: %s\n", inst->name);
          z_instance_free(inst);
          return 0;
        }
      free(zorp_args);
      free(zorpctl_args);
      
      inst->next = *last;
      *last = inst;
      last = &inst->next;
    }
  return 1;
}

ZInstance *
z_search_instance(char *name)
{
  ZInstance *p;
  for (p = instances; p; p = p->next)
    {
      if (strcmp(name, p->name) == 0)
        return p;
    }
  return NULL;
}

int
z_parse_config_line_int(char *var, char *name, char *value, int *result)
{
  if (strcmp(name, var) == 0)
    {
      char quote = value[0];
      char *end;
      
      if (quote == '"' || quote == '\'')
        {
          *result = strtol(value + 1, &end, 10);
          if (*end != quote)
            return 0;
        }
      else
        {
          *result = strtol(value, &end, 10);
          if (*end != 0)
            return 0;
        }
      return 1;
    }
  return 0;
}

int
z_parse_config_line_str(char *var, char *name, char *value, char **result)
{
  if (strcmp(name, var) == 0)
    {
      char quote = value[0];
      
      if (quote == '"' || quote == '\'')
        {
          char *p, *dst;
          
          *result = malloc(strlen(value));
          dst = *result;
          for (p = value + 1; *p; p++)
            {
              if (*p == '\\')
                {
                  *dst = *(p+1);
                  p++;
                }
              else if (*p == quote)
                {
                  break;
                }
              else
                {
                  *dst = *p;
                }
              dst++;
            }
          if (*p != 0 && *(p+1))
            {
              /* invalid quotation marks */
              free(*result);
              *result = NULL;
              return 0;
            }
          *dst = 0;
        }
      else
        *result = strdup(value);
      return 1;
    }
  return 0;
}

int
z_parse_config(void)
{
  FILE *cfgf;
  char line[256], *value, *name, *eq;
  int lineno = 0;
  
  cfgf = fopen(ZORP_ZORPCTL_CONF, "r");
  if (!cfgf)
    return 1;
    
  while (fgets(line, sizeof(line), cfgf))
    {
      lineno++;
      chomp(line);
      if (line[0] == 0 || line[0] == '#')
        continue;
      eq = strchr(line, '=');
      if (!eq)
        {
          z_error(0, "Invalid zorpctl.conf line at %d: %s\n", lineno, line);
          return 0;
        }
      *eq = 0;
      value = eq + 1;
      name = line;
      
      if (!(z_parse_config_line_int("AUTO_RESTART", name, value, &auto_restart) ||
            z_parse_config_line_int("AUTO_RESTART_TIME_THRESHOLD", name, value, &auto_restart_time_threshold) ||
            z_parse_config_line_int("AUTO_RESTART_MAX_COUNT", name, value, &auto_restart_max_count) ||
            z_parse_config_line_int("AUTO_RESTART_DELAY", name, value, &auto_restart_delay) ||
            z_parse_config_line_int("START_CHECK_TIMEOUT", name, value, &start_check_timeout) || 
            z_parse_config_line_int("START_WAIT_TIMEOUT", name, value, &start_wait_timeout) || 
            z_parse_config_line_int("PROCESS_LIMIT_THRESHOLD", name, value, &process_limit_threshold) ||
            z_parse_config_line_int("STOP_CHECK_DELAY", name, value, &stop_check_delay) ||
            z_parse_config_line_int("STOP_CHECK_TIMEOUT", name, value, &stop_check_timeout) ||
            z_parse_config_line_int("PROCESS_LIMIT_RESERVE", name, value, &process_limit_reserve) ||
	    z_parse_config_line_int("PROCESS_LIMIT_MIN", name, value, &process_limit_min) ||
	    z_parse_config_line_int("FD_LIMIT_THRESHOLD", name, value, &fd_limit_threshold) ||
            z_parse_config_line_int("FD_LIMIT_MIN", name, value, &fd_limit_min) ||
            z_parse_config_line_int("CHECK_PERMS", name, value, &check_perms) ||
            z_parse_config_line_str("CONFIG_DIR", name, value, &config_dir) ||
            z_parse_config_line_str("CONFIG_DIR_OWNER", name, value, &config_dir_owner) ||
            z_parse_config_line_str("CONFIG_DIR_GROUP", name, value, &config_dir_group) ||
            z_parse_config_line_str("CONFIG_DIR_MODE", name, value, &config_dir_perm) ||
            z_parse_config_line_str("APPEND_ARGS", name, value, &zorp_append_args) ||
            z_parse_config_line_str("ZORP_APPEND_ARGS", name, value, &zorp_append_args) ||
            z_parse_config_line_str("ZORPCTL_APPEND_ARGS", name, value, &zorpctl_append_args) ||
            z_parse_config_line_str("PIDFILE_DIR", name, value, &pidfile_dir) ||
            z_parse_config_line_str("PIDFILE_DIR_OWNER", name, value, &pidfile_dir_owner) ||
            z_parse_config_line_str("PIDFILE_DIR_GROUP", name, value, &pidfile_dir_group) ||
            z_parse_config_line_str("PIDFILE_DIR_MODE", name, value, &pidfile_dir_perm)))
        {
          z_error(0, "Unknown zorpctl.conf directive at %d: %s\n", lineno, line);
        }
    }
  if (process_limit_threshold != -1)
    {
      z_error(0, "The use of PROCESS_LIMIT_THRESHOLD in zorpctl.conf is deprecated, its value is ignored\n");
    }
  return 1;
}

int 
z_get_counter(ZInstance *inst, const char *var_name)
{
  ZSzigContext *ctx;
  char result[256];
  int success, thread_count;
  
  ctx = z_szig_context_new(inst->name);
  if (!ctx)
    return -1;
  
  success = z_szig_get_value(ctx, var_name, result, sizeof(result));
  z_szig_context_destroy(ctx);

  if (!success)
    return -1;
  thread_count = strtol(result, NULL, 10);
  return thread_count;
}

int
z_get_thread_count(ZInstance *inst)
{
  return z_get_counter(inst, "zorp.stats.threads_running");
}

int
z_get_thread_number(ZInstance *inst)
{
  return z_get_counter(inst, "zorp.stats.thread_number");
}

void
z_setup_env(void)
{
  static char pythonpath[512], ldlibpath[512];
  char *oldpath;
  
  oldpath = getenv("PYTHONPATH");
  snprintf(pythonpath, sizeof(pythonpath), "PYTHONPATH=%s:%s", ZORP_SYSCONFDIR, oldpath ? oldpath : "");
  
  oldpath = getenv("LD_LIBRARY_PATH");
  snprintf(ldlibpath, sizeof(ldlibpath), "LD_LIBRARY_PATH=%s:%s", ZORP_LIBDIR, oldpath ? oldpath : "");

  putenv(pythonpath);
  putenv(ldlibpath);
}

void
z_setup_limits(ZInstance *inst)
{
  struct rlimit limit;
  
#ifdef RLIMIT_NPROC
  limit.rlim_cur = limit.rlim_max = MAX(inst->process_limit, process_limit_sum + process_limit_reserve);
  setrlimit(RLIMIT_NPROC, &limit);
#endif
  limit.rlim_cur = limit.rlim_max = inst->fd_limit;  
  setrlimit(RLIMIT_NOFILE, &limit);
  if (inst->enable_core)
    {
      limit.rlim_cur = limit.rlim_max = RLIM_INFINITY;  
      setrlimit(RLIMIT_CORE, &limit);
  
      chdir("/var/run/zorp");
    }
}

void
z_setup_fds(void)
{
  int devnullfd;
  
  devnullfd = open("/dev/null", O_RDONLY);
  dup2(devnullfd, 0);
  close(devnullfd);
  devnullfd = open("/dev/null", O_WRONLY);
  dup2(devnullfd, 1);
  dup2(devnullfd, 2);
  close(devnullfd);
  
  chdir("/");
}

int
z_safe_start_instance(ZInstance *inst)
{
  pid_t child;
  int status;
  int res = 1;
  
  if (inst->auto_restart)
    {
      int result[2];
      char buf[16];
      
      if (start_check_timeout)
        {
          if (pipe(result) < 0)
            {
              res = 0;
              goto exit;
            }
        }
      if (fork() == 0)
        {
          int bad_restarts = 0;
          time_t prev_restart = time(NULL), now;
          pid_t child;
          char **new_zorp_argv;
          int rc;
          int result_sent = 0;
          
          close(result[0]);
          
          new_zorp_argv = malloc((inst->zorp_argc + 2) * sizeof(char *));
          memcpy(new_zorp_argv, inst->zorp_argv, (inst->zorp_argc + 1) * sizeof(char *));
          inst->zorp_argv = new_zorp_argv;
          inst->zorp_argv[inst->zorp_argc] = "--foreground";
          inst->zorp_argv[inst->zorp_argc + 1] = NULL;
          inst->zorp_argc++;
          
          openlog("zorpctl", LOG_NDELAY | LOG_PID, LOG_DAEMON);
          snprintf(proc_title, 64, "zorpctl supervising %s", inst->name);
          setsid();
          while (1)
            {
              if ((child = fork()) == 0)
                {
                  z_setup_env();
                  z_setup_limits(inst);
                  execvp(ZORP, inst->zorp_argv);
                }
              else
                {
                  z_setup_fds();
                  if (!result_sent && start_check_timeout)
                    {
                      /* FIXME: make this controllable by the user */
                      z_alarm_request(start_check_timeout);
                      while (waitpid(child, &rc, 0) < 0)
                        {
                          if (z_alarm_fired())
                            {
                              /* ok, some seconds passed, Zorp did not exit, take it as a successful startup */
                              write(result[1], "0", 1);
                              close(result[1]);
                              result_sent = 1;
                              waitpid(child, &rc, 0);
                              break;
                            }
                        }
                    }
                  else
                    waitpid(child, &rc, 0);
                  
                  if (WIFSIGNALED(rc) || (WIFEXITED(rc) && WEXITSTATUS(rc) != 0))
                    {
                      if (!WIFSIGNALED(rc) || WTERMSIG(rc) != SIGKILL)
                        {
                          now = time(NULL);
                          
                          if (now - prev_restart < auto_restart_time_threshold)
                            {
                              if (!result_sent && start_check_timeout)
                                {
                                  /* Zorp has just been started and it exited before the alarm above fired, bad */
                                  write(result[1], "1", 1);
                                  close(result[1]);
                                  result_sent = 1;
                                  syslog(LOG_INFO, "Zorp exited immediately with non-zero exit status after starting up, exiting...\n");
                                  exit(0);
                                }
                              bad_restarts++;
                              prev_restart = now;
                              if (bad_restarts > auto_restart_max_count)
                                {
                                  syslog(LOG_INFO, "Excessive number of restarts, more than %d within %d seconds, exiting...\n", 
                                         auto_restart_max_count, auto_restart_time_threshold);
                                  exit(0);
                                }
                            }
                          syslog(LOG_INFO, "Zorp exited due to a signal/failure (instance='%s', exitcode=%d), restarting...\n", SAFE_STR(inst->name), rc);
                          sleep(auto_restart_delay);
                        }
                      else
                        {
                          syslog(LOG_INFO, "Zorp was killed (instance='%s', exitcode=%d), not restarting...\n", SAFE_STR(inst->name), rc);
                          exit(0);
                        }
                    }
                  else
                    {
                      /* exited normally */
                      syslog(LOG_INFO, "Zorp exited gracefully (instance='%s', exitcode=%d), not restarting...\n", SAFE_STR(inst->name), rc);
                      exit(0);
                    }
                }
            }
        }
      if (start_check_timeout)
        {
          close(result[1]);
      
          if (read(result[0], buf, sizeof(buf)) < 0)
            {
              z_error(1, "Error reading result pipe from child, instance='%s', error='%s'\n", inst->name, strerror(errno));
              res = 0;
            }
          else 
            {
              res = buf[0] == '0';
            }
          close(result[0]);
        }
    }
  else
    {
      child = fork();
      if (child == 0)
        {
          z_setup_env();
          z_setup_limits(inst);
          setsid();
          execvp(ZORP, inst->zorp_argv);
          exit(1);
        }
      if (start_wait_timeout)
        {
          z_alarm_request(start_wait_timeout);
          if (waitpid(child, &status, 0) < 0)
            {
              res = 0;
              if (z_alarm_fired())
                {
                  /* timed out */
                  z_error(1, "Timeout waiting for Zorp instance to start up, instance='%s'\n", inst->name);
                }
            }
          else if (status != 0)
            { 
              z_error(1, "Zorp instance startup failed, instance='%s', rc='%d'\n", inst->name, status);
              res = 0;
            }
        }
    }
 exit:
  return res;
}

int
z_process_start_instance(ZInstance *inst, void *user_data UNUSED)
{
  int stale;
  pid_t pid;
  
  if (z_instance_running(inst, &pid, &stale))
    {
      return 0;
    }
  if (stale)
    z_instance_remove_stale_pidfile(inst);

  if (!inst->no_auto_start)
    return z_safe_start_instance(inst);
  return 1;
}

int
z_process_force_start_instance(ZInstance *inst, void *user_data UNUSED)
{
  inst->no_auto_start = 0;
  return z_process_start_instance(inst, user_data);
}

int
z_process_stop_instance(ZInstance *inst, void *user_data)
{
  int stale, killed;
  int signo = (long) user_data;
  time_t prev_check, now;
  pid_t pid;
  
  if (!z_instance_running(inst, &pid, &stale))
    {
      return 0;
    }
  if (stale)
    {
      z_instance_remove_stale_pidfile(inst);
    }
  else
    {
      kill(pid, signo);
      
      prev_check = now = time(NULL);
      for (killed = 0; (now - prev_check) < stop_check_timeout; now = time(NULL))
        {
          if ((killed = !z_instance_running(inst, &pid, &stale)))
              break;
          sleep(stop_check_delay);
        }
      if (!killed)
        {
          z_error(1, "Zorp instance did not exit in time (instance='%s', pid='%d', signo='%d', timeout='%d')\n", SAFE_STR(inst->name), pid, signo, stop_check_timeout);
          return 0;
        }
      if (signo == SIGKILL)
        z_instance_remove_stale_pidfile(inst);
    }

  return 1;
}

int
z_process_restart_instance(ZInstance *inst, void *user_data)
{
  z_process_stop_instance(inst, user_data);
  sleep(2);
  return z_process_start_instance(inst, user_data);
}

int
z_process_signal_instance(ZInstance *inst, void *user_data)
{
  int stale;
  int signo = (long) user_data;
  pid_t pid;
  
  if (!z_instance_running(inst, &pid, &stale))
    {
      return 0;
    }
  kill(pid, signo);
  return 1;
}

int
z_process_status_instance(ZInstance *inst, void *user_data UNUSED)
{
  int stale;
  pid_t pid;
  
  printf("Instance %s: ", inst->name);
  if (!z_instance_running(inst, &pid, &stale))
    {
      if (stale)
        {
          printf("stale pidfile, pid %d\n", pid);
        }
      else
        {
          printf("not running\n");
        }
      return 1;
    }
  printf("running, %d threads active, pid %d\n", z_get_thread_count(inst), pid);
  return 1;  
}

int
z_process_gui_status_instance(ZInstance *inst, void *user_data UNUSED)
{
  int stale;
  pid_t pid;
  ZSzigContext *ctx;
  char *qry_strings[] = 
  {
    "zorp.stats.threads_running",
    "zorp.stats.thread_number",
    "zorp.stats.thread_avg1",
    "zorp.stats.thread_avg5", 
    "zorp.stats.thread_avg15",
    NULL
  };
  int i;
  
  printf("\"%s\";", inst->name);
  if (!z_instance_running(inst, &pid, &stale))
    {
      if (stale)
        {
          printf("\"missing\";\"%d\";;;;\n", pid);
        }
      else
        {
          printf("\"stopped\";;;;;\n");
        }
      return 1;
    }

  ctx = z_szig_context_new(inst->name);
  if (!ctx)
    {
      printf("\"noszig\";\"%d\";;;;\n", pid);
      return 1;
    }
  printf("\"running\";\"%d\"", pid);
  for (i = 0; qry_strings[i]; i++)
    {
      char result[256];
      long val;
      
      z_szig_get_value(ctx, qry_strings[i], result, sizeof(result));
      val = strtol(result, NULL, 10);
      printf(";\"%ld\"", val);
    }
  printf("\n");
  return 1;  
}

void
z_szig_walk(ZSzigContext *ctx, const char *root)
{
  char result[16384];
  
  z_szig_get_value(ctx, root, result, sizeof(result));
  printf("%s: %s\n", root, result);
  z_szig_get_child(ctx, root, result, sizeof(result));
  if (strcmp(result, "None") != 0)
    z_szig_walk(ctx, result);
  z_szig_get_sibling(ctx, result, result, sizeof(result));
  while (strcmp(result, "None") != 0)
    {
      z_szig_walk(ctx, result);
      z_szig_get_sibling(ctx, result, result, sizeof(result));
      
    }
}

int
z_process_szig_walk_instance(ZInstance *inst, void *user_data)
{
  ZSzigContext *ctx;
  char *root = (char *) user_data;
  
  printf("Instance %s: ", inst->name);
  ctx = z_szig_context_new(inst->name);
  if (!ctx)
    {
      printf("not running\n");
      z_szig_context_destroy(ctx);
      return 0;
    }
  else
    {
      printf("walking\n");
    }
  
  z_szig_walk(ctx, root);
  z_szig_context_destroy(ctx);
  
  return 1;
}

static inline int
z_process_log_func(ZInstance *inst, char *cmd, char *param)
{
  ZSzigContext *ctx;						
  int res = 0;							
                                                                
  ctx = z_szig_context_new(inst->name);				
  if (ctx)							
    {								
      if (z_szig_logging(ctx, cmd, param, NULL, 0))		
        res = 1;						
      z_szig_context_destroy(ctx);				
    }
  else
    {
      z_error(1, "Error connecting to Zorp SZIG socket, instance='%s'", inst->name);
    }
    
  return res;
}

int
z_process_log_vinc_instance(ZInstance *inst, void *user_data UNUSED)
{
  return z_process_log_func(inst, "VINC", "1");
}

int
z_process_log_vdec_instance(ZInstance *inst, void *user_data UNUSED)
{
  return z_process_log_func(inst, "VDEC", "1");
}

int
z_process_log_vset_instance(ZInstance *inst, void *user_data)
{
  char buf[16];
  
  snprintf(buf, sizeof(buf), "%d", *(int *) user_data);
  
  return z_process_log_func(inst, "VSET", buf);
}

int
z_process_log_logspec_instance(ZInstance *inst, void *user_data)
{
  return z_process_log_func(inst, "SETSPEC", (char *) user_data);
}

int
z_process_log_status_instance(ZInstance *inst, void *user_data UNUSED)
{
  ZSzigContext *ctx;
  int res = 0;
  char verb_level[16];
  char spec[128];

  ctx = z_szig_context_new(inst->name);
  if (ctx)
    {
      if (z_szig_logging(ctx, "VGET", "", verb_level, sizeof(verb_level)))
        res = 1;
      if (!res || !z_szig_logging(ctx, "GETSPEC", "", spec, sizeof(spec)))
        res = 0;
      if (res)
        {
          printf("Instance: %s: verbose_level='%s', logspec='%s'\n", inst->name, verb_level, spec);
        }
      z_szig_context_destroy(ctx);
    }
  if (!res)
    {
      printf("Instance: %s, error querying log information\n", inst->name);
    }
  return res;
}

static inline int
z_process_reload(ZInstance *inst, void *user_data UNUSED)
{
  ZSzigContext *ctx;						
  int res = 0;							
                                                                
  ctx = z_szig_context_new(inst->name);				
  if (ctx)							
    {								
      if (z_szig_reload(ctx, NULL, NULL, 0) &&
          z_szig_reload(ctx, "RESULT", NULL, 0))
        res = 1;
      z_szig_context_destroy(ctx);				
    }
  else
    {
      z_error(1, "Error connecting to Zorp SZIG socket, instance='%s'", inst->name);
    }
    
  return res;
}

int
z_process_args(char *ident, int argc, char *argv[], int (*func)(ZInstance *inst, void *user_data), void *user_data, int display_instances)
{
  ZInstance *inst;
  int success;
  int res = 0;
  int i;

  if (argc == 0)
    {
      for (inst = instances; inst; inst = inst->next)
        {
          success = func(inst, user_data);
          if (display_instances)
            printf("%s%s", inst->name, success ? " " : "! ");
        }
    }
  else
    {    
      for (i = 0; i < argc; i++)
        {
          inst = z_search_instance(argv[i]);
          if (!inst)
            {
              z_error(1, "%s: No such instance: %s\n", ident, argv[i]);
              success = 0;
            }
          else
            {
              success = func(inst, user_data);
            }
          if (display_instances)
            printf("%s%s", argv[i], success ? " " : "! ");
        }
    }
  /* FIXME: return appropriate exit code */
  return res;
}

/* command implementations */

int
z_cmd_start(int argc, char *argv[])
{
  printf("Starting Zorp Firewall Suite: ");
  z_process_args("start", argc-2, &argv[2], z_process_start_instance, NULL, 1);
  printf("\n");
  return 0;
}

int
z_cmd_force_start(int argc, char *argv[])
{
  printf("Starting Zorp Firewall Suite: ");
  z_process_args("start", argc-2, &argv[2], z_process_force_start_instance, NULL, 1);
  printf("\n");
  return 0;
}

int
z_cmd_stop(int argc, char *argv[])
{
  printf("Stopping Zorp Firewall Suite: ");
  z_process_args("stop", argc-2, &argv[2], z_process_stop_instance, (void *) SIGTERM, 1);
  printf("\n");
  return 0;
}

int
z_cmd_force_stop(int argc, char *argv[])
{
  printf("Stopping Zorp Firewall Suite: ");
  z_process_args("stop", argc-2, &argv[2], z_process_stop_instance, (void *) SIGKILL, 1);
  printf("\n");
  return 0;
}

int
z_cmd_restart(int argc, char *argv[])
{
  printf("Restarting Zorp Firewall Suite: ");
  z_process_args("restart", argc-2, &argv[2], z_process_restart_instance, (void *) SIGTERM, 1);
  printf("\n");
  return 0;
}

int
z_cmd_force_restart(int argc, char *argv[])
{
  printf("Restarting Zorp Firewall Suite: ");
  z_process_args("restart", argc-2, &argv[2], z_process_restart_instance, (void *) SIGKILL, 1);
  printf("\n");
  return 0;
}


int
z_cmd_status(int argc, char *argv[])
{
  z_process_args("status", argc-2, &argv[2], z_process_status_instance, NULL, 0);
  return 0;
}

int
z_cmd_gui_status(int argc, char *argv[])
{
  printf("\"instance\";\"status\";\"pid\";\"running threads\";\"total threads\";\"thread1avg\";\"thread5avg\";\"thread15avg\"\n");
  z_process_args("gui-status", argc-2, &argv[2], z_process_gui_status_instance, NULL, 0);
  return 0;
}

int
z_cmd_version(int argc UNUSED, char *argv[] UNUSED)
{
  execl(ZORP, ZORP, "--version", NULL);
  return 0;
}

int
z_cmd_inclog(int argc, char *argv[])
{
  printf("Raising Zorp loglevel: ");
  z_process_args("inclog", argc-2, &argv[2], z_process_signal_instance, (void *) SIGUSR1, 1);
  printf("\n");
  return 0;
}

int
z_cmd_declog(int argc, char *argv[])
{
  printf("Lowering Zorp loglevel: ");
  z_process_args("declog", argc-2, &argv[2], z_process_signal_instance, (void *) SIGUSR2, 1);
  printf("\n");
  return 0;
}

#define LOG_ACTION_STATUS 0
#define LOG_ACTION_VINC 1
#define LOG_ACTION_VDEC 2
#define LOG_ACTION_VSET 3
#define LOG_ACTION_LOGSPEC 4

int 
z_cmd_log(int argc, char *argv[])
{
  static struct poptOption log_options[] = 
  {
    { "vinc", 'i', POPT_ARG_NONE, NULL, 'i', "Increment verbosity level by one", NULL },
    { "vdec", 'd', POPT_ARG_NONE, NULL, 'd', "Decrement verbosity level by one", NULL },
    { "vset", 's', POPT_ARG_INT, NULL, 's', "Set verbosity level", "<verbosity>" },
    { "log-spec", 'S', POPT_ARG_STRING, NULL, 'S', "Set log specification", "<logspec>" },
    { "help", 'h', POPT_ARG_NONE, NULL, 'h', "Display this help screen", NULL },
    { NULL, 0, 0, NULL, 0, NULL, NULL },
  };
  poptContext ctx;
  const char **instances;
  int action, level, instance_count;
  const char *logspec = NULL;
  int opt;
  int res = 1;
  
  ctx = poptGetContext("zorpctl log", argc - 1, (const char **) &argv[1], log_options, 0);

  action = LOG_ACTION_STATUS;
  while ((opt = poptGetNextOpt(ctx)) > 0)
    {
      switch (opt)
        {
        case 'i':
          action = LOG_ACTION_VINC;
          break;
        case 'd':
          action = LOG_ACTION_VDEC;
          break;
        case 's':
          action = LOG_ACTION_VSET;
          level = atoi(poptGetOptArg(ctx));
          break;
        case 'S':
          action = LOG_ACTION_LOGSPEC;
          logspec = poptGetOptArg(ctx);
          break;
        case 'h':
          poptPrintHelp(ctx, stdout, 0);
          return 0;
        }
    }
  if (opt < -1)
    {
      z_error(0, "zorpctl log: Invalid arguments\n");
      return 0;
    }
  instances = poptGetArgs(ctx);
  if (instances)
    {
      for (instance_count = 0; instances[instance_count]; instance_count++)
        ;
  
    }
  else
    instance_count = 0;

  if (action != LOG_ACTION_STATUS)
    printf("Changing Zorp log settings: ");
    
  switch (action)
    {
    case LOG_ACTION_VINC:
      res = z_process_args("log", instance_count, (char **) instances, z_process_log_vinc_instance, NULL, 1);
      break;
    case LOG_ACTION_VDEC:
      res = z_process_args("log", instance_count, (char **) instances, z_process_log_vdec_instance, NULL, 1);
      break;
    case LOG_ACTION_VSET:
      res = z_process_args("log", instance_count, (char **) instances, z_process_log_vset_instance, &level, 1);
      break;
    case LOG_ACTION_LOGSPEC:
      res = z_process_args("log", instance_count, (char **) instances, z_process_log_logspec_instance, (char *) logspec, 1);
      break;
    case LOG_ACTION_STATUS:
      z_process_args("log", instance_count, (char **) instances, z_process_log_status_instance, NULL, 0);
      break;
    }
  if (action != LOG_ACTION_STATUS)
    printf("\n");
  poptFreeContext(ctx);
  return res;
}

#define SZIG_ACTION_WALK 1

int
z_cmd_szig(int argc, char *argv[])
{
  static struct poptOption szig_options[] = 
  {
    { "walk", 'w', POPT_ARG_NONE, NULL, 'w', "Walk the specified tree", NULL },
    { "root", 'r', POPT_ARG_STRING, NULL, 'r', "Set the root node of the walk operation", "<node>" },
    { "help", 'h', POPT_ARG_NONE, NULL, 'h', "Display this help screen", NULL },
    { NULL, 0, 0, NULL, 0, NULL, NULL },
  };
  poptContext ctx;
  int action = SZIG_ACTION_WALK, res;
  const char **instances;
  int instance_count, opt;
  const char *root = "zorp";
  
  ctx = poptGetContext("zorpctl szig", argc - 1, (const char **) &argv[1], szig_options, 0);

  while ((opt = poptGetNextOpt(ctx)) > 0)
    {
      switch (opt)
        {
        case 'w':
          action = SZIG_ACTION_WALK;
          break;
        case 'r':
          root = poptGetOptArg(ctx);
          break;
        case 'h':
          poptPrintHelp(ctx, stdout, 0);
          return 0;
        }
    }
  if (opt < -1)
    {
      z_error(0, "zorpctl log: Invalid arguments\n");
      return 0;
    }
  instances = poptGetArgs(ctx);
  if (instances)
    {
      for (instance_count = 0; instances[instance_count]; instance_count++)
        ;
  
    }
  else
    instance_count = 0;
  res = z_process_args("szig", instance_count, (char **) instances, z_process_szig_walk_instance, (void *) root, 0);
  poptFreeContext(ctx);
  
  return res;
}

int 
z_cmd_usage(int argc UNUSED, char *argv[] UNUSED)
{
  int i;
  
  printf("Syntax\n"
         "  %s <command>\n\n"
         "The following commands are available:\n\n", argv[0]);
  for (i = 0; commands[i].cmd; i++)
    {
      printf("%s -- %s\n", commands[i].cmd, commands[i].help);
    }
  return 0;
}

ZCommand commands[] = 
{  
  { "start", z_cmd_start,          "Starts the specified Zorp instance(s)" },
  { "force-start", z_cmd_force_start,"Starts the specified Zorp instance(s) even if they are disabled" },
  { "stop",  z_cmd_stop,           "Stops the specified Zorp instance(s)" },
  { "force-stop",  z_cmd_force_stop,"Forces the specified Zorp instance(s) to stop (SIGKILL)" },
  { "restart", z_cmd_restart,      "Restart the specified Zorp instance(s)" },
  { "force-restart", z_cmd_force_restart,      "Forces the specified Zorp instance(s) to restart (SIGKILL)" },
  { "status", z_cmd_status,        "Status of the specified Zorp instance(s)" },
  { "gui-status", z_cmd_gui_status, "Status of the specified Zorp instance(s)" },
  { "version", z_cmd_version,      "Display Zorp version information" },
  { "inclog",  z_cmd_inclog,       "Raise the specified Zorp instance(s) log level by one" },
  { "declog",  z_cmd_declog,       "Lower the specified Zorp instance(s) log level by one" },
  { "log",  z_cmd_log,             "Change and query Zorp log settings" },
  { "szig",  z_cmd_szig,           "Display internal information from the given Zorp instance(s)" },
  { "help",  z_cmd_usage,          "Display this screen" },
  { NULL, NULL, NULL }
};

int 
main(int argc, char *argv[])
{
  int i;
  ZCommand *cmd = NULL;
  int rc = 0;

  setvbuf(stdout, NULL, _IONBF, 0);
  proc_title = argv[0];
  z_setup_signals();
  z_parse_config();
  z_parse_instances();
  
  z_check_pidfile_dir();
  if (!z_check_config_dir())
    return 1;
  
  if (argc < 2)
    {
      z_cmd_usage(argc, argv);
      return 1;
    }
  
  for (i = 0; commands[i].cmd; i++)
    {
      if (strcmp(commands[i].cmd, argv[1]) == 0)
        {
          cmd = &commands[i];
          break;
        }
    }
  if (cmd)
    {
      rc = cmd->func(argc, argv);
    }
  else
    {
      z_error(0, "Invalid command: %s\n", argv[1]);
      rc = 1;
    }
  
  z_dump_errors();
  return rc;
}
