// file      : xsde/xsde.cxx
// author    : Boris Kolpackov <boris@codesynthesis.com>
// copyright : Copyright (c) 2005-2007 Code Synthesis Tools CC
// license   : GNU GPL v2 + exceptions; see accompanying LICENSE file

#include <cult/types.hxx>

#include <cult/trace/log.hxx>

#include <cult/cli/exceptions.hxx>
#include <cult/cli/arguments.hxx>
#include <cult/cli/options.hxx>
#include <cult/cli/options-spec.hxx>
#include <cult/cli/options-parser.hxx>

#include <xsd-frontend/parser.hxx>

#include <xsd-frontend/transformations/restriction.hxx>

#include <cxx/parser/generator.hxx>
#include <cxx/serializer/generator.hxx>

#include <morphing/anonymous/processor.hxx>

#include <processing/inheritance/processor.hxx>

#include <iostream>

#include <xsde.hxx>
#include <usage.hxx>

#include "../libxsde/xsde/cxx/version.hxx"

using namespace Cult::Types;
using namespace XSDFrontend;

using std::wcerr;
using std::endl;

namespace CLI
{
  using namespace Cult::CLI;

  extern Char const help[]    = "help";
  extern Char const version[] = "version";
  extern Char const proprietary_license[] = "proprietary-license";

  typedef Cult::CLI::Options
  <
    help, Boolean,
    version, Boolean,
    proprietary_license, Boolean
  >
  HelpOptions;

  struct HelpOptionsSpec: Cult::CLI::OptionsSpec<HelpOptions> {};


  extern Char const sloc_limit[] = "sloc-limit";

  typedef Cult::CLI::Options
  <
    sloc_limit, UnsignedLong
  >
  CommonOptions;

  struct CommonOptionsSpec: Cult::CLI::OptionsSpec<CommonOptions> {};
}


Int
main (Int argc, Char* argv[])
{
  std::wostream& e (wcerr);

  Cult::Trace::Log::instance ().level (0);

  try
  {
    CLI::HelpOptions help_options (
      CLI::parse (CLI::HelpOptionsSpec (),
                  argc,
                  argv,
                  CLI::UnknownMode::stop));

    CLI::Arguments args (argc, argv);

    String cmd;

    if (args.size () > 1)
    {
      cmd = args[1];
      args.erase (1);
    }

    if (help_options.value<CLI::version> () || cmd == "version")
    {
      e << "CodeSynthesis XSD/e XML Schema to C++ compiler " <<
        "for embedded systems " << XSDE_STR_VERSION << endl
        << "Copyright (C) 2005-2007 Code Synthesis Tools CC" << endl;

      if (!help_options.value<CLI::proprietary_license> () &&
          cmd == "version")
      {
        // Parse the options after the command to detect trailing
        // --proprietary-license.
        //
        help_options = CLI::parse (
          CLI::HelpOptionsSpec (), argc, argv, CLI::UnknownMode::stop);
      }

      if (help_options.value<CLI::proprietary_license> ())
      {
        e << "The compiler was invoked in the Proprietary License mode. You "
          << "should have\nreceived a proprietary license from Code Synthesis "
          << "Tools CC that entitles\nyou to use it in this mode." << endl;
      }
      else
      {
        e << "This is free software; see the source for copying conditions. "
          << "There is NO\nwarranty; not even for MERCHANTABILITY or FITNESS "
          << "FOR A PARTICULAR PURPOSE." << endl;
      }

      return 0;
    }

    if (help_options.value<CLI::help> () || cmd == "help")
    {
      if (cmd == "help" && args.size () > 1)
      {
        String arg (args[1]);

        if (arg == "cxx-parser")
        {
          e << "Usage: " << argv[0] << " " << arg.c_str () <<
            " [options] file [file ...]" << endl
            << "Options:" << endl;

          CXX::Parser::Generator::usage ();
          Morphing::Anonymous::Processor::usage ();
        }
        else if (arg == "cxx-serializer")
        {
          e << "Usage: " << argv[0] << " " << arg.c_str () <<
            " [options] file [file ...]" << endl
            << "Options:" << endl;

          CXX::Serializer::Generator::usage ();
          Morphing::Anonymous::Processor::usage ();
        }
        else
        {
          e << "error: unknown command '" << arg.c_str () << "'" << endl
            << "info: try '" << argv[0] << " help' for the list of commands"
            << endl;

          return 1;
        }
      }
      else
      {
        e << "Usage: " << argv[0] << " <cmd> ..." << endl
          << "Commands:" << endl;

        e << "  help            Print usage information and exit. Use\n"
          << "                  'help <cmd>' for command-specific options."
          << endl;


        e << "  version         Print version and exit."
          << endl;

        e << "  cxx-parser      Generate Embedded C++/Parser mapping."
          << endl;

        e << "  cxx-serializer  Generate Embedded C++/Serializer mapping."
          << endl;
      }

      return 0;
    }

    if (cmd.empty ())
    {
      e << "error: no command specified" << endl
        << "info: try '" << argv[0] << " help' for usage information" << endl;

      return 1;
    }

    if (cmd != "cxx-parser" && cmd != "cxx-serializer")
    {
      e << "error: unknown command '" << cmd.c_str () << "'" << endl
        << "info: try '" << argv[0] << " help' for the list of commands"
        << endl;

      return 1;
    }

    // We need to parse command line options before we can get to
    // the arguments.
    //
    CLI::CommonOptions common_options (
      CLI::parse (CLI::CommonOptionsSpec (),
                  argc,
                  argv,
                  CLI::UnknownMode::skip,
                  CLI::UnknownMode::skip));

    Morphing::Anonymous::CLI::Options ma_options (
      CLI::parse (Morphing::Anonymous::Processor::options_spec (),
                  argc,
                  argv,
                  CLI::UnknownMode::skip,
                  CLI::UnknownMode::skip));

    Evptr<CXX::Parser::CLI::Options> cxx_parser_options;
    Evptr<CXX::Serializer::CLI::Options> cxx_serializer_options;

    Boolean show_sloc (false);

    if (cmd == "cxx-parser")
    {
      cxx_parser_options = new CXX::Parser::CLI::Options (
        CLI::parse (CXX::Parser::Generator::options_spec (), argc, argv));

      show_sloc = cxx_parser_options->value<CXX::Parser::CLI::show_sloc> ();
    }
    else if (cmd == "cxx-serializer")
    {
      cxx_serializer_options = new CXX::Serializer::CLI::Options (
        CLI::parse (CXX::Serializer::Generator::options_spec (), argc, argv));

      show_sloc =
        cxx_serializer_options->value<CXX::Serializer::CLI::show_sloc> ();
    }

    if (argc < 2)
    {
      e << "error: no input file specified" << endl;
      return 1;
    }

    //
    //
    AutoUnlinks unlinks;
    UnsignedLong sloc (0);

    for (Int i (1); i < argc; ++i)
    {
      // Parse schema.
      //
      SemanticGraph::Path tu;

      try
      {
        tu = SemanticGraph::Path (argv[i], boost::filesystem::native);
      }
      catch (SemanticGraph::InvalidPath const&)
      {
        e << "error: '" << argv[i] << "' is not a valid "
          << "filesystem path" << endl;

        return 1;
      }

      Parser parser (true);

      Evptr<SemanticGraph::Schema> schema;

      schema = parser.parse (tu);


      // Morph anonymous types.
      //
      if (!ma_options.value<Morphing::Anonymous::CLI::preserve_anonymous> ())
      {
        try
        {
          Morphing::Anonymous::Processor proc;
          proc.morph (ma_options, *schema, tu);
        }
        catch (Morphing::Anonymous::Processor::Failed const&)
        {
          return 1; // Diagnostic has already been issued.
        }
      }

      // Try to rearrange definitions so that there is no forward
      // inheritance.
      //
      try
      {
        Processing::Inheritance::Processor proc;
        proc.process (*schema, tu);
      }
      catch (Processing::Inheritance::Processor::Failed const&)
      {
        return 1; // Diagnostic has already been issued.
      }

      // Normalize and annotate complex content restrictions.
      //
      if (cmd == "cxx-parser" || cmd == "cxx-serializer")
      {
        try
        {
          Transformations::Restriction trans;
          trans.transform (*schema, tu);
        }
        catch (Transformations::Restriction::Failed const&)
        {
          return 1; // Diagnostic has already been issued.
        }
      }

      // Generate mapping.
      //
      if (cmd == "cxx-parser")
      {
        try
        {
          CXX::Parser::Generator generator;

          sloc += generator.generate (
            *cxx_parser_options, *schema, tu, unlinks);
        }
        catch (CXX::Parser::Generator::Failed const&)
        {
          // Diagnostics has already been issued.
          //
          return 1;
        }
      }
      else if (cmd == "cxx-serializer")
      {
        try
        {
          CXX::Serializer::Generator generator;

          sloc += generator.generate (
            *cxx_serializer_options, *schema, tu, unlinks);
        }
        catch (CXX::Serializer::Generator::Failed const&)
        {
          // Diagnostics has already been issued.
          //
          return 1;
        }
      }
    }

    if (show_sloc)
      e << "total: " << sloc << endl;

    if (UnsignedLong sloc_limit = common_options.value<CLI::sloc_limit> ())
    {
      if (sloc_limit < sloc)
      {
        e << "error: SLOC limit of " << sloc_limit
          << " lines has been exceeded" << endl;

        return 1;
      }
    }

    unlinks.cancel ();

    return 0;
  }
  catch (InvalidSchema const&)
  {
    // Diagnostic has already been issued.
  }
  catch (CLI::UnexpectedOption const& e)
  {
    wcerr << "error: unknown option '" << e.option ().c_str () << "'" << endl
          << "info: try '" << argv[0] << " help' for usage information"
          << endl;
  }
  catch (CLI::OptionFormat const& e)
  {
    wcerr << "error: value for option '" << e.option ().c_str ()
          << "' is invalid" << endl
          << "info: try '" << argv[0] << " help' for usage information"
          << endl;
  }

  return 1;
}
