# XMMS2tray - GNU/Linux systray integration for xmms2
# Copyright (C) 2006 Thomas Jollans
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

# system libraries
import sys
import os
import time
import gettext
from StringIO import StringIO
import random
import numbers

# extra libraries
import xmmsclient
import xmmsclient.glib
import gtk
import gtk.gdk

# xmms2tray configuration
#import menu_conf

# optional extra libraries
HAVE_NOTIFY = True
try:
    import pynotify
except ImportError:
    HAVE_NOTIFY = False
    sys.stderr.write("pynotify not found. Notifications will not work.\n")

HAVE_IMAGING = True
try:
    from PIL import Image
except ImportError:
    HAVE_IMAGING = False
    sys.stderr.write("Python Imaging Library PIL not found. Cover art disabled.\n")

GETTEXT_DOMAIN = "xmms2tray"

class XMMS2tray(object):
    _curinfo = {'id': 0}

    def get_curinfo(self):
        return self._curinfo

    def set_curinfo(self, info):
        self._curinfo = info

    curinfo = property(get_curinfo, set_curinfo)

    def menu_called(self, what, button, time):
        self.menu.popup(None, None, gtk.status_icon_position_menu,
                                                    button, time, what)

    def menuclicked(self, widget, item):
        if 'action' in item:
            #print item['action']
            if item['action'] == 'quit':
                self.quit()
            elif item['action'] == 'playpause':
                if 'check' in item and item['check']:
                    if widget.get_active(): self.xmms.playback_start()
                    else: self.xmms.playback_pause()
                else:
                    self.xmms.playback_status(self.playpause)
            elif item['action'] == 'kill':
                self.xmms.quit()
            elif item['action'] == 'play':
                self.xmms.playback_start()
            elif item['action'] == 'pause':
                self.xmms.playback_pause()
            elif item['action'] == 'stop':
                self.xmms.playback_stop()
            elif item['action'] == 'next':
                self.xmms.playlist_set_next_rel(1)
                self.xmms.playback_tickle()
            elif item['action'] == 'prev':
                self.xmms.playlist_set_next_rel(-1)
                self.xmms.playback_tickle()
            #elif item['action'] == 'toggle_notifications':
                #self.config['notify'] = item.get('check', self.config['notify'])
                #self.save_conf()
        elif 'command' in item:
            ret = os.spawnv(os.P_NOWAIT, item['command'], [item['command']] + item.get('args', []))
            print "PID: ", ret
        elif 'config' in item:
            if item.get('check'):
                self.config[item['config']] = bool(widget.get_active())
            self.save_conf()

    def playpause(self, res):
        if res.value() == xmmsclient.PLAYBACK_STATUS_PLAY:
            self.xmms.playback_pause()
        else:
            self.xmms.playback_start()

    def set_tip(self):
        if not len(self.curinfo) > 1:
            tip = self.pbstatus
        else:
            if 'artist' in self.curinfo and 'title' in self.curinfo:
                nowpl = '%(artist)s - %(title)s' % self.curinfo
            elif 'title' in self.curinfo: #ex. curl
                nowpl = self.curinfo['title']
            else:
                nowpl = self.curinfo['url'].split('/')[-1]
            tip = "%s: %s" % (self.pbstatus, nowpl)
        #self.tips.set_tip(self.eb, tip)
        self.icon.set_tooltip(tip)

    def newsong(self,res):
        v = res.value()
        
        if isinstance(v, numbers.Integral):
	    # this is the ID. I want the whole info.
	    self.xmms.medialib_get_info(res.value(),self.newsong)
        else:
            if isinstance(v, basestring): # coverart
                info = self.curinfo
                coverimg = StringIO(v)
            elif v is None: # what?
                return
            else:
                info = v
                coverimg = None
                self.curinfo = info

            img = os.getcwd() + '/data/xmms2_64.png'
            if HAVE_IMAGING and 'picture_front' in info:
                if info['picture_front'] == self.lastimg[0]:
                    #same image, just use last.
                    img = self.lastimg[1]
                else:
                    if self.lastimg[1]:
                        #there is an old image to delete.
                        os.remove(self.lastimg[1])
                        self.lastimg = ('', '')
                    if coverimg:
                        orig_pic = Image.open(coverimg)
                        if orig_pic.size[1] <= 64:
                            sm_pic = orig_pic
                        else:
                            sm_pic = orig_pic.resize((64,64), Image.BICUBIC)
                        fname = '/tmp/xmms2tray_cover%08X.png' \
                                    % random.randint(0, 2**32-1)
                        sm_pic.save(fname)
                        self.lastimg = (info['picture_front'], fname)
                        img = fname
                    else:
                        self.xmms.bindata_retrieve(info['picture_front'],
                                                    self.newsong)
                        return


            if not info: return #for "not playing"
            if 'artist' in info and 'title' in info:
                info['artist'] = info['artist']
                nowpl = '%s - <i>%s</i>\n' % \
                    ( info['artist'].replace('&', '&amp;').replace('<', '&lt;'),
                       info['title'].replace('&', '&amp;').replace('<', '&lt;'))

            elif 'title' in info: #ex. curl
                nowpl = info['title'].replace('&', '&amp').replace('<', '&lt;')
            else:
                nowpl = info['url'].split('/')[-1].replace('+', ' ')
            self.set_tip()
            if self.config['notify'] and HAVE_NOTIFY:
                n = pynotify.Notification(_('Now Playing'), nowpl, img)
                if self.config['att_note']:
                    n.set_property('status-icon', self.icon)
                n.show()

    def songchange(self, res):
        if self.curinfo['id'] == res.value():
            self.xmms.medialib_get_info(res.value(),self.newsong)

    def quit(self, gtk_q=True):
        if self.lastimg[1]:
            os.remove(self.lastimg[1])

        if gtk_q: gtk.main_quit()

    def pbstatus_cb(self, res):
        if res.value() is None: return
        status = res.value()
        if status == xmmsclient.PLAYBACK_STATUS_STOP:
            self.pbstatus = _("Stopped")
        elif status == xmmsclient.PLAYBACK_STATUS_PAUSE:
            self.pbstatus = _("Paused")
        else:
            self.pbstatus = _("Playing")
        self.set_tip()

    def __init__(self):
        icon = gtk.StatusIcon()
        icon.set_from_file('data/xmms2_24.png')
        icon.connect('popup-menu', self.menu_called)
        icon.set_visible(True)
        self.icon = icon
        self.pbstatus = _("Stopped")
        self.curinfo = {'id': 0}
        self.lastimg = ('', '')

        self.load_conf()
        if 'notify' not in self.config: self.config['notify'] = True
        if 'att_note' not in self.config: self.config['att_note'] = True

        self.xmms = xmmsclient.XMMS("XMMS2tray")
        try:
            self.connect()
        except IOError:
            mbox = gtk.MessageDialog(
                None, 0, gtk.MESSAGE_ERROR, gtk.BUTTONS_YES_NO,
                _("I could not connect to the XMMS2 daemon. It may not be running. Attempt to start it ?"))
            mbox.set_title(_("Start daemon?"))
            resp = mbox.run()
            mbox.destroy()
            if resp == gtk.RESPONSE_YES:
                os.system('xmms2-launcher')
                self.connect()
            else: sys.exit(1)
        xmmsclient.glib.GLibConnector(self.xmms)

        menucfg = menu_conf.TOP
        menucfg.append(None)
        menucfg.extend(menu_conf.OPTIONS)
        menucfg.append(None)
        for cl in menu_conf.CLIENTS:
            if cl['command'][0] == '$':
                if os.access('/usr'+cl['command'][1:], os.R_OK | os.X_OK):
                    cl['command'] = '/usr'+cl['command'][1:]
                    if cl['iconfile'][0] == '$':
                        cl['iconfile'] = '/usr'+cl['iconfile'][1:]
                    menucfg.append(cl)
                elif os.access('/usr/local'+cl['command'][1:], os.R_OK|os.X_OK):
                    cl['command'] = '/usr/local'+cl['command'][1:]
                    if cl['iconfile'][0] == '$':
                        cl['iconfile'] = '/usr/local'+cl['iconfile'][1:]
                    menucfg.append(cl)
            elif os.access(cl['command'], os.R_OK | os.X_OK):
                menucfg.append(cl)
        menucfg.append(None)
        menucfg += menu_conf.BOTTOM

        self.menu = gtk.Menu()
        mi = None
        for item in menucfg:
            if item in ('separator','sep',None):
                mi = gtk.SeparatorMenuItem()
                self.menu.append(mi)
                mi.show()
                continue
            elif 'stockicon' in item:
                mi = gtk.ImageMenuItem(item['title'])
                img = gtk.Image()
                img.set_from_stock(getattr(gtk,'STOCK_'
                    +item['stockicon'].upper()),gtk.ICON_SIZE_MENU)
                mi.set_image(img)
            elif 'iconfile' in item:
                mi = gtk.ImageMenuItem(item['title'])
                if os.path.isfile(item['iconfile']):
                    img = gtk.Image()
                    pb = gtk.gdk.pixbuf_new_from_file(item['iconfile'])
                    # FIXME: hard-coded icon size !
                    img.set_from_pixbuf(pb.scale_simple(16,16,
                        gtk.gdk.INTERP_BILINEAR))
                    mi.set_image(img)
            elif item.get('check'):
                mi = gtk.CheckMenuItem(item['title'])
                if item.get('action') == 'playpause':
                    CheckWaiter(self.xmms, mi,item)
                elif 'config' in item:
                    mi.set_active(bool(self.config.get(item['config'])))
            else:
                mi = gtk.MenuItem(item['title'])
            self.menu.append(mi)
            mi.connect('activate', self.menuclicked, item)
            mi.show()
        self.menu.show()

        if HAVE_NOTIFY:
            pynotify.init("XMMS2tray")
        self.xmms.playback_status(self.pbstatus_cb)
        self.xmms.broadcast_playback_status(self.pbstatus_cb)
        self.xmms.playback_current_id(self.newsong)
        self.xmms.broadcast_playback_current_id(self.newsong)
        self.xmms.broadcast_medialib_entry_changed(self.songchange)

    def connect(self):
        self.xmms.connect(os.getenv('XMMS_PATH'), self.quit)

    def load_conf(self):
        self.config = {}

        bool_vals = {'True': True, 'False': False, '': None}
        try:
            cfile = file(os.path.join(xmmsclient.userconfdir_get(),
                         'clients', 'xmms2tray.conf'), 'rU')
        except (IOError, OSError):
            sys.stderr.write('XMMS2tray configuration file cannot be read.\n')
            return

        try:
            for line in cfile:
                if '=' not in line:
                    continue
                key, valstr = line.strip().split('=', 1)
                if valstr[0] == '"':
                    self.config[key] = valstr[1:]
                elif valstr.isdigit():
                    self.config[key] = int(valstr)
                elif valstr[0].isdigit():
                    self.config[key] = float(valstr)
                elif valstr in bool_vals:
                    self.config[key] = bool_vals[valstr]
        except ValueError:
            sys.stderr.write('XMMS2tray configuration file is invalid.\n')

        cfile.close()

    def save_conf(self):
        try:
            cfile = file(os.path.join(xmmsclient.userconfdir_get(),
                         'clients', 'xmms2tray.conf'), 'w')
        except (IOError, OSError):
            sys.stderr.write('Cannot open XMMS2tray config file for writing.\n')
            return

        for k,v in self.config.iteritems():
            cfile.write('%s=' % k)
            if isinstance(v, basestring):
                cfile.write('"%s\n' % v)
            elif isinstance(v, bool):
                cfile.write('%s\n' % v)
            elif isinstance(v, (int, long)):
                cfile.write('%d\n' % v)
            elif isinstance(v, float):
                cfile.write('%f\n' % v)
            else:
                cfile.write('\n')

        cfile.close()
            

class CheckWaiter:
    def __init__(self, xmms, mi, item):
        self.item = item
        self.mi = mi
        if item['action'] == 'playpause':
            xmms.broadcast_playback_status(self.cb)
            xmms.playback_status(self.cb)
        self.ok = False
    def cb(self, res):
        if self.item['action'] == 'playpause':
            self.mi.set_active(res.value() == xmmsclient.PLAYBACK_STATUS_PLAY)
        self.ok = True


def main():
    if not os.path.exists('data/xmms2_24.png'): # go somewhere else
        if os.path.exists('%s/share/xmms2tray/data/xmms2_24.png' % sys.prefix):
            os.chdir('%s/share/xmms2tray' % sys.prefix)
        else:
            raise RuntimeError('XMMS2tray has been installed in a non-standard way. Please run from the directory including data/')

    translations = gettext.translation(GETTEXT_DOMAIN, "data/po", fallback=True)

    translations.install()
    global menu_conf
    import menu_conf

    t = XMMS2tray()
    try:
        gtk.main()
    except KeyboardInterrupt:
        t.quit(False)


