/*
 * Copyright (C) 2008 Emweb bvba, Kessel-Lo, Belgium.
 *
 * See the LICENSE file for terms of use.
 */
#include <string>
#include <boost/lexical_cast.hpp>

#include "Wt/WResource"
#include "Wt/WApplication"
#include "Wt/WEnvironment"
#include "Wt/Http/Request"
#include "Wt/Http/Response"

#include "WebController.h"
#include "WebRequest.h"
#include "WebSession.h"
#include "WtException.h"
#include "Utils.h"

#ifdef WT_THREADED
#include <boost/thread/recursive_mutex.hpp>
#endif // WT_THREADED

namespace Wt {

WResource::WResource(WObject* parent)
  : WObject(parent),
    dataChanged_(this),
    beingDeleted_(false),
    trackUploadProgress_(false),
    dispositionType_(NoDisposition)
{ 
#ifdef WT_THREADED
  mutex_.reset(new boost::recursive_mutex());
#endif // WT_THREADED
}

void WResource::beingDeleted()
{
#ifdef WT_THREADED
  boost::recursive_mutex::scoped_lock lock(*mutex_);
  beingDeleted_ = true;
#endif // WT_THREADED
}

WResource::~WResource()
{
  beingDeleted();

  for (unsigned i = 0; i < continuations_.size(); ++i) {
    continuations_[i]->stop();
    delete continuations_[i];
  }

  if (wApp) {
    wApp->removeExposedResource(this);
    if (trackUploadProgress_)
      WebSession::instance()->controller()->removeUploadProgressUrl(url());
  }
}

void WResource::setUploadProgress(bool enabled)
{
  if (trackUploadProgress_ != enabled) {
    trackUploadProgress_ = enabled;

    WebController *c = WebSession::instance()->controller();
    if (enabled)
      c->addUploadProgressUrl(url());
    else
      c->removeUploadProgressUrl(url());
  }
}

void WResource::doContinue(Http::ResponseContinuation *continuation)
{
  WebResponse *webResponse = continuation->response();
  WebRequest *webRequest = webResponse;

  try {
    handle(webRequest, webResponse, continuation);
  } catch (std::exception& e) {
    std::cerr << "Exception while handling resource continuation: "
	      << e.what() << std::endl;
  } catch (...) {
    std::cerr << "Exception while handling resource continuation." << std::endl;
  }
}

void WResource::handle(WebRequest *webRequest, WebResponse *webResponse,
		       Http::ResponseContinuation *continuation)
{
#ifdef WT_THREADED
  boost::shared_ptr<boost::recursive_mutex> mutex = mutex_;
  boost::recursive_mutex::scoped_lock lock(*mutex);

  if (beingDeleted_)
    return;

  // when we are handling a continuation, we do not have the session
  // lock
  if (!continuation) {
    WebSession::Handler *h = WebSession::Handler::instance();
    if (h && h->lock().owns_lock())
      h->lock().unlock();
  }
#endif // WT_THREADED

  if (continuation)
    continuation->resource_ = 0;

  Http::Request  request(*webRequest, continuation);
  Http::Response response(this, webResponse, continuation);

  if (!continuation &&
      (dispositionType_ != NoDisposition ||
       !suggestedFileName_.empty())) {
    std::string theDisposition;
    switch (dispositionType_) {
      default:
      case Inline:
        theDisposition = "inline";
        break;
      case Attachment:
        theDisposition = "attachment";
        break;
    }
    if (!suggestedFileName_.empty()) {
      if (dispositionType_ == NoDisposition) {
        // backward compatibility-ish with older Wt versions
        theDisposition = "attachment";
      }
      // Browser incompatibility hell: internatianalized filename suggestions
      // First filename is for browsers that don't support RFC 5987
      // Second filename is for browsers that do support RFC 5987
      std::string fileField;
      // We cannot query wApp here, because wApp doesn't exist for
      // static resources.
      bool isIE = webRequest->userAgent().find("MSIE") != std::string::npos;
      bool isChrome = webRequest->userAgent().find("Chrome") != std::string::npos;
      if (isIE || isChrome) {
        // filename="foo-%c3%a4-%e2%82%ac.html"
        // Note: IE never converts %20 back to space, so avoid escaping
        // IE wil also not url decode the filename if the file has no ASCII
        // extension (e.g. .txt)
        fileField = "filename=\"" +
          Utils::urlEncode(suggestedFileName_.toUTF8(), " ") + "\";";
      } else {
        // Binary UTF-8 sequence: for FF3, Safari, Chrome, Chrome9
        fileField = "filename=\"" + suggestedFileName_.toUTF8() + "\";";
      }
      // Next will be picked by RFC 5987 in favour of the
      // one without specified encoding (Chrome9, 
      fileField+= Utils::EncodeHttpHeaderField("filename", suggestedFileName_);
      response.addHeader("Content-Disposition",
        theDisposition + ";" + fileField);
    } else {
      response.addHeader("Content-Disposition", theDisposition);
    }
  }

  handleRequest(request, response);

  if (!response.continuation_ || !response.continuation_->resource_) {
    if (response.continuation_) {
      Utils::erase(continuations_, response.continuation_);
      delete response.continuation_;
    }
    webResponse->flush(WebResponse::ResponseDone);
  } else
    webResponse->flush(WebResponse::ResponseFlush,
		       boost::bind(&Http::ResponseContinuation::doContinue,
				   response.continuation_));
}

void WResource::suggestFileName(const WString& name,
                                DispositionType dispositionType)
{
  suggestedFileName_ = name;
  dispositionType_ = dispositionType;

  generateUrl();
}

void WResource::setInternalPath(const std::string& path)
{
  internalPath_ = path;

  generateUrl();
}

void WResource::setDispositionType(DispositionType dispositionType)
{
  dispositionType_ = dispositionType;
}

void WResource::setChanged()
{
  generateUrl();

  dataChanged_.emit();
}

const std::string& WResource::url() const
{
  if (currentUrl_.empty())
    (const_cast<WResource *>(this))->generateUrl();

  return currentUrl_;
}

const std::string& WResource::generateUrl()
{
  WApplication *app = WApplication::instance();

  if (app) {
    WebController *c = 0;
    if (trackUploadProgress_)
      c = WebSession::instance()->controller();

    if (c && !currentUrl_.empty())
      c->removeUploadProgressUrl(currentUrl_);
    currentUrl_ = app->addExposedResource(this, internalPath_);
    if (c)
      c->addUploadProgressUrl(currentUrl_);    
  } else
    currentUrl_ = internalPath_;

  return currentUrl_;
}

void WResource::write(WT_BOSTREAM& out,
		      const Http::ParameterMap& parameters,
		      const Http::UploadedFileMap& files)
{
  Http::Request  request(parameters, files);
  Http::Response response(this, out);

  handleRequest(request, response);

  // While the resource indicates more data to be sent, get it too.
  while (response.continuation_	&& response.continuation_->resource_) {
    response.continuation_->resource_ = 0;
    request.continuation_ = response.continuation_;

    handleRequest(request, response);
  }

  delete response.continuation_;
}

}
