/*
 * SFVec3f.cpp
 *
 * Copyright (C) 1999 Stephen F. White
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file "COPYING" for details); if 
 * not, write to the Free Software Foundation, Inc., 675 Mass Ave, 
 * Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include "stdafx.h"
#include "DuneApp.h"

#include "SFVec3f.h"
#include "SFFloat.h"

SFVec3f::SFVec3f(float x, float y, float z)
{
    _value[0] = x; _value[1] = y; _value[2] = z;
}

SFVec3f::SFVec3f(const Vec3f &v)
{
    _value[0] = v.x; _value[1] = v.y; _value[2] = v.z;
}

SFVec3f::SFVec3f(void)
{
    _value[0] = 0.0f; _value[1] = 0.0f; _value[2] = 0.0f;
}

void
SFVec3f::setValue(float v1, float v2, float v3)
{
    _value[0] = v1;
    _value[1] = v2;
    _value[2] = v3;
}

bool
SFVec3f::equals(const FieldValue *value) const
{
    if (value->getType() == SFVEC3F) {
	SFVec3f		*v = (SFVec3f *) value;
	return v->getValue()[0] == _value[0]
	    && v->getValue()[1] == _value[1]
	    && v->getValue()[2] == _value[2];
    } else {
	return false;
    }
}

int SFVec3f::write(int f, int /* indent */) const
{
    RET_ONERROR( mywritef(f, "%g %g %g\n", _value[0], _value[1], _value[2]) )
    TheApp->incSelectionLinenumber();
    return 0;
}

void SFVec3f::setValue(int index, float value)
{
    assert(index >= 0 && index < 3);

    _value[index] = value;
}

void
SFVec3f::clamp(const FieldValue *min, const FieldValue *max)
{
    if (min) {
	float	fmin = ((SFFloat *) min)->getValue();
	for (int i = 0; i < 3; i++) {
	    if (_value[i] < fmin) _value[i] = fmin;
	}
    }

    if (max) {
	float	fmax = ((SFFloat *) max)->getValue();
	for (int i = 0; i < 3; i++) {
	    if (_value[i] > fmax) _value[i] = fmax;
	}
    }
}

void
SFVec3f::scale(float* mult)
{
		
		_value[0]*=mult[0];
		_value[1]*=mult[1];
		_value[2]*=mult[2];

	
}

void
SFVec3f::scale(float scale1, float scale2, float scale3)
{
		
		_value[0]*=scale1;
		_value[1]*=scale2;
		_value[2]*=scale3;

	
}
void
SFVec3f::translate(float* v)
{
	_value[0] += v[0];
	_value[1] += v[1];
	_value[2] += v[2];
}

void
SFVec3f::rotate(float* rot)
{
	Vec3f rotationAxis(rot[0],rot[1],rot[2]);
	Quaternion quat(rotationAxis,rot[3]);
	quat.normalize();
	Vec3f vector(_value[0], _value[1], _value[2]);
	Vec3f rotatedVector=quat*vector;
	_value[0]=rotatedVector.x;
	_value[1]=rotatedVector.y;
	_value[2]=rotatedVector.z;
}

MyString
SFVec3f::getEcmaScriptComment(MyString name, int flags) const
{
    const char *indent = ((FieldValue *)this)->getEcmaScriptIndent(flags);
    MyString ret;
    ret = "";
    if (TheApp->GetEcmaScriptAddAllowedValues()) {
        ret += indent;
        ret += "// allowed values:\n";

        ret += indent;
        ret += "   // 3D Vector, 3 floating point numbers\n";
    }
    if (TheApp->GetEcmaScriptAddAllowedComponents()) {
        ret += indent;
        ret += "// allowed components:\n";

        ret += indent;
        ret += "   // x: ";
        ret += name;
        ret += ".x or ";
        ret += name;
        ret += "[0]\n";

        ret += indent;
        ret += "   // y: ";
        ret += name;
        ret += ".y or ";
        ret += name;
        ret += "[1]\n";

        ret += indent;
        ret += "   // z: ";
        ret += name;
        ret += ".z or ";
        ret += name;
        ret += "[2]\n";
    }
    if (TheApp->GetEcmaScriptAddAvailableFunctions()) {
        ret += indent;
        ret += "// available functions:\n";
        if (flags != EL_EVENT_IN) {
            ret += indent;
            ret += "   // ";
            ret += name;
            ret += " = new SFVec3f(float_x, float_y, float_z);\n";

            ret += indent;
            ret += "   // ";
            ret += name;
            ret += " = sfvec3f_vec1.negate();\n";

            ret += indent;
            ret += "   // ";
            ret += name;
            ret += " = sfvec3f_vec1.normalize();\n";

            ret += indent;
            ret += "   // ";
            ret += name;
            ret += " = sfvec3f_vec1.add(sfvec3f_vec2);\n";

            ret += indent;
            ret += "   // ";
            ret += name;
            ret += " = sfvec3f_vec1.subtract(sfvec3f_vec2);\n";

            ret += indent;
            ret += "   // ";
            ret += name;
            ret += " = sfvec3f_vec1.cross(sfvec3f_vec2);\n";

            ret += indent;
            ret += "   // ";
            ret += name;
            ret += " = sfvec3f_vec.multiply(float_f);\n";

            ret += indent;
            ret += "   // ";
            ret += name;
            ret += " = sfvec3f_vec.divide(float_f);\n";

            ret += indent;
            ret += "   // ";
            ret += name;
            ret += " = sfrotaton_rot.getAxis();\n";

            ret += indent;
            ret += "   // ";
            ret += name;
            ret += " = sfrotaton_rot.multVec(sfvec3f_vec);\n";
       }
        if (flags != EL_EVENT_OUT) {
            ret += indent;
            ret += "   // float_f = ";
            ret += name;
            ret += ".length();\n";

            ret += indent;
            ret += "   // float_f = ";
            ret += name;
            ret += ".dot(sfvec3f_vec);\n";

            ret += indent;
            ret += "   // string_str = ";
            ret += name;
            ret += ".toString();\n";
       }
    }
    if (TheApp->GetEcmaScriptAddExampleUsage()) {
        ret += indent;
        ret += "// example usage:\n";
        if (flags != EL_EVENT_IN) {
             ret += indent;
             ret += "   // ";
             ret += name;
             ret += " = new SFVec3f(3, 5.2, 1);\n";

             ret += indent;
             ret += "   // ";
             ret += name;
             ret += ".z = 0.5;\n";
        } 
        if (flags != EL_EVENT_OUT) {
             ret += indent;
             ret += "   // float_z =";
             ret += name;
             ret += ".z;\n";
        }
        if (flags == EL_FIELD_DEF) {
             ret += indent;
             ret += "   // ";
             ret += name;
             ret += " = ";
             ret += name;
             ret += ".normalize();\n";
        } 
    }
    return ret;
}
