/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "logindialog.h"

#include <QLayout>
#include <QLabel>
#include <QCheckBox>
#include <QLineEdit>
#include <QSettings>

#include "commands/commandmanager.h"
#include "data/datamanager.h"
#include "data/connectioninfo.h"
#include "data/loginbatch.h"
#include "data/bookmark.h"
#include "connectionmanager.h"
#include "configdata.h"
#include "iconloader.h"

#if defined( HAVE_OPENSSL )
#include "dialogs/sslerrorsdialog.h"
#endif

LoginDialog::LoginDialog( QWidget* parent ) : CommandDialog( parent ),
    m_successful( false )
{
    connectionManager->createCommandManager();

#if defined( HAVE_OPENSSL )
    connect( commandManager, SIGNAL( sslErrors( const QList<QSslError>& ) ),
        this, SLOT( sslErrors( const QList<QSslError>& ) ) );
#endif

    setWindowTitle( tr( "Login" ) );
    setPrompt( tr( "Enter server address and login information:" ) );
    setPromptPixmap( IconLoader::pixmap( "connection-open", 22 ) );

    showInfo( tr( "Enter connection information" ) );

    QGridLayout* layout = new QGridLayout();

    QLabel* urlLabel = new QLabel( tr( "&Server URL:" ), this );
    layout->addWidget( urlLabel, 0, 0 );

    m_urlEdit = new QLineEdit( "http://", this );
    layout->addWidget( m_urlEdit, 0, 1 );

    urlLabel->setBuddy( m_urlEdit );

    QLabel* loginLabel = new QLabel( tr( "&Login:" ), this );
    layout->addWidget( loginLabel, 1, 0 );

    m_loginEdit = new QLineEdit( this );
    m_loginEdit->setMaxLength( 40 );
    layout->addWidget( m_loginEdit, 1, 1 );

    loginLabel->setBuddy( m_loginEdit );

    QLabel* passwordLabel = new QLabel( tr( "&Password:" ), this );
    layout->addWidget( passwordLabel, 2, 0 );

    m_passwordEdit = new QLineEdit( this );
    m_passwordEdit->setEchoMode( QLineEdit::Password );
    m_passwordEdit->setMaxLength( 40 );
    layout->addWidget( m_passwordEdit, 2, 1 );

    passwordLabel->setBuddy( m_passwordEdit );

    m_rememberCheck = new QCheckBox( tr( "&Remember password" ), this );
    layout->addWidget( m_rememberCheck, 3, 1 );

    setContentLayout( layout, true );

    m_urlEdit->setFocus();
}

LoginDialog::~LoginDialog()
{
    if ( !m_successful )
        connectionManager->closeConnection();
}

void LoginDialog::loadBookmark( const Bookmark& bookmark, bool editable )
{
    m_urlEdit->setText( bookmark.url() );
    m_loginEdit->setText( bookmark.login() );

    m_urlEdit->setEnabled( editable );
    m_loginEdit->setEnabled( editable );

    m_passwordEdit->setText( bookmark.password() );

    m_rememberCheck->setChecked( !bookmark.password().isEmpty() );

    if ( !editable )
        m_passwordEdit->setFocus();
}

Bookmark LoginDialog::createBookmark()
{
    ConnectionInfo* info = connectionManager->connectionInfo();

    QString password;
    if ( m_rememberCheck->isChecked() )
        password = m_passwordEdit->text();

    return Bookmark( info->serverName(), info->serverUuid(), m_urlEdit->text(), m_loginEdit->text(), password );
}

void LoginDialog::accept()
{
    QString url = m_urlEdit->text();
    QString login = m_loginEdit->text();
    QString password = m_passwordEdit->text();

    if ( url.isEmpty() ) {
        showWarning( tr( "No server URL entered" ) );
        return;
    }

    if ( login.isEmpty() ) {
        showWarning( tr( "No login entered" ) );
        return;
    }

    if ( password.isEmpty() ) {
        showWarning( tr( "No password entered" ) );
        return;
    }

    commandManager->setServerUrl( url );

    LoginBatch* batch = new LoginBatch( login, password );

    executeBatch( batch );
}

bool LoginDialog::batchSuccessful( AbstractBatch* batch )
{
    ConnectionInfo* info = ( (LoginBatch*)batch )->connectionInfo();
    connectionManager->openConnection( info );

    m_successful = true;

    return true;
}

#if defined( HAVE_OPENSSL )

void LoginDialog::sslErrors( const QList<QSslError>& errors )
{
    SslErrorsDialog dialog( this );
    dialog.setErrors( errors );
    dialog.setCertificates( commandManager->certificateChain() );

    if ( dialog.exec() == QDialog::Accepted )
        commandManager->ignoreSslErrors();
}

#endif // defined( HAVE_OPENSSL )
