
// vbmaskinfo.cpp
// mask/lesion summary utility, mostly for counting unique voxels
// Copyright (c) 2005-2009 by The VoxBo Development Team

// VoxBo is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// VoxBo is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with VoxBo.  If not, see <http://www.gnu.org/licenses/>.
// 
// For general information on VoxBo, including the latest complete
// source code and binary distributions, manual, and associated files,
// see the VoxBo home page at: http://www.voxbo.org/
//
// original version written by Dan Kimberg

using namespace std;

#include <math.h>
#include <map>
#include "vbutil.h"
#include "vbio.h"
#include "vbversion.h"
#include "vbmaskinfo.hlp.h"

class pdata {
public:
  pdata();
  int voxels;
};

void vbmaskinfo_help();
void vbmaskinfo_version();

class img {
public:
  Tes ts;
  Cube cb;
  int pos;
};

int
main(int argc,char *argv[])
{
  if (argc==1) {
    vbmaskinfo_help();
    exit(0);
  }

  int minpatients=1;

  tokenlist args,filelist;
  args.Transfer(argc-1,argv+1);
  for (int i=0; i<args.size(); i++) {
    if (args[i]=="-h") {
      vbmaskinfo_help();
      exit(0);
    }
    else if (args[i]=="-v") {
      vbmaskinfo_version();
      exit(0);
    }
    else if (args[i]=="-m" && i<args.size()-1) {
      minpatients=strtol(args[++i]);
    }
    else
      filelist.Add(args[i]);
  }

  Tes bigtes,tes;
  Cube cube;
  int dimx=0,dimy=0,dimz=0,dimt=0;
  for (int i=0; i<filelist.size(); i++) {
    if (tes.ReadHeader(filelist[i])==0) {
      if (dimx==0) {dimx=tes.dimx; dimy=tes.dimy; dimz=tes.dimz;}
      if (tes.dimx!=dimx || tes.dimy!=dimy || tes.dimz!=dimz) {
        printf("[E] vbmaskinfo: inconsistent image dimensions\n");
        exit(109);
      }
      dimt+=tes.dimt;
    }
    else if (cube.ReadHeader(filelist[i])==0) {
      if (dimx==0) {dimx=cube.dimx; dimy=cube.dimy; dimz=cube.dimz;}
      if (cube.dimx!=dimx || cube.dimy!=dimy || cube.dimz!=dimz) {
        printf("[E] vbmaskinfo: inconsistent image dimensions\n");
        exit(109);
      }
      dimt++;      
    }
    else {
      printf("[E] vbmaskinfo: couldn't read %s\n",filelist[i].c_str());
      exit(109);
    }
  }
  if (dimt==0) {
    printf("[E] vbmaskinfo: no valid volumes read\n");
    exit(129);
  }
  
  img myimages[filelist.size()];

  int pos=0;
  for (int i=0; i<filelist.size(); i++) {
    if (myimages[i].ts.ReadFile(filelist[i])==0) {
      myimages[i].pos=pos;
      pos+=myimages[i].ts.dimt;
    }
    else if (myimages[i].cb.ReadFile(filelist[i])==0) { 
      myimages[i].pos=pos;
      pos++;
    }
    else {
      printf("[E] vbmaskinfo: couldn't read %s\n",filelist[i].c_str());
      exit(109);
    }
  }

  printf("[I] vbmaskinfo: counting patterns (min patients %d)\n",minpatients);
  map<bitmask,pdata> patterns;

  int emptyvoxels=dimx*dimy*dimz;
  VB_Vector myvec(dimt);

  int i,j,k,v,t;
  bitmask bm;
  bm.resize(dimt);
  for (i=0; i<dimx; i++) {
    for (j=0; j<dimy; j++) {
      for (k=0; k<dimz; k++) {
        bm.clear();

        for (v=0; v<filelist.size(); v++) {
          if (myimages[v].ts.data) {
            for (t=0; t<myimages[v].ts.dimt; t++)
              if (myimages[v].ts.GetValueUnsafe(i,j,k,t)>FLT_MIN)
                bm.set(myimages[v].pos+t);
          }
          else {
            // if (myimages[v].cb.GetValue(i,j,k)>FLT_MIN)
            if (myimages[v].cb.testValue(i,j,k))
              bm.set(myimages[v].pos);
          }
        }
        if (bm.count()<minpatients)
          continue;
        // use the cool [] operator for map containers
        patterns[bm].voxels++;
        emptyvoxels--;
      }
    }
  }
  int voxeltotal[dimt+1];   // number of voxels with n lesions
  int uniquetotal[dimt+1];  // number of unique voxels with n lesions
  for (int i=0; i<dimt+1; i++)
    voxeltotal[i]=uniquetotal[i]=0;
  for (map<bitmask,pdata>::iterator pp=patterns.begin(); pp!=patterns.end(); pp++) {
    int cnt=pp->first.count();
    voxeltotal[cnt]+=pp->second.voxels;
    uniquetotal[cnt]++;
  }
  printf("[I] total patients: %d\n",dimt);
  printf("[I] total voxels in volume: %d\n",dimx*dimy*dimz);
  printf("[I] voxels with data: %d\n",dimx*dimy*dimz-emptyvoxels);
  printf("[I] uncounted voxels: %d\n",emptyvoxels);
  printf("[I] total distinct voxels: %d\n",(int)patterns.size());
  for (int i=minpatients; i<dimt+1; i++) {
    if (uniquetotal[i]>0)
      printf("[I] voxels with %d lesions: %d voxels, %d distinct voxels\n",
             i,voxeltotal[i],uniquetotal[i]);
  }
  printf("[I] Distinct voxels are voxels with different patterns of lesioned\n");
  printf("[I] patients.  Sets of voxels in which exactly the same patients are\n");
  printf("[I] lesioned are counted as a single distinct voxel.\n");
  exit(0);
}


pdata::pdata()
{
  voxels=0;
}

void
vbmaskinfo_help()
{
  cout << boost::format(myhelp) % vbversion;
}

void
vbmaskinfo_version()
{
  printf("VoxBo vbmaskinfo (v%s)\n",vbversion.c_str());
}
