# Copyright (C) 2007  Lars Wirzenius <liw@iki.fi>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.


"""Unit tests for Unperish publish plugin."""


import os
import shutil
import tempfile
import unittest

import unperishlib


class PublishTests(unittest.TestCase):

    def dirpath(self, name):
        return os.path.join(self.dirname, name)
        
    def buildpath(self, name):
        return os.path.join(self.build_area, name)

    def setUp(self):
        self.copy_args = []
        self.tempdir = tempfile.mkdtemp()
        self.dirname = os.path.join(self.tempdir, "directory")
        self.build_area = os.path.join(self.tempdir, "build_area")
        self.resultname = os.path.join(self.tempdir, "result")

        os.mkdir(self.dirname)
        os.mkdir(self.build_area)

        for name in ["NEWS", "README"]:
            file(self.dirpath(name), "w").close()

        tarball = self.buildpath("pink-1.2.tar.gz")
        file(tarball, "w").close()

        changelog = self.buildpath("pink.changelog")
        file(changelog, "w").close()
        
        app = unperishlib.Application(plugins=["plugins"])
        self.options, _ = app.parse_args(["--directory", self.dirname,
                                          "--upstream-tarball", tarball,
                                          "--changelog", changelog,
                                          "--publish-to", self.resultname])
        self.oper = app.get_plugin_manager().get_operation("publish")

    def tearDown(self):
        shutil.rmtree(self.tempdir)

    def testParsesLocalLocationCorrectly(self):
        path = self.dirpath("README")
        x = self.oper.parse_url(path)
        self.failUnlessEqual(x.scheme, "")
        self.failUnlessEqual(x.netloc, "")
        self.failUnlessEqual(x.path, path)

    def testParsesAbsoluteSshLocationCorrectly(self):
        x = self.oper.parse_url("sftp://foo/bar/foobar")
        self.failUnlessEqual(x.scheme, "sftp")
        self.failUnlessEqual(x.netloc, "foo")
        self.failUnlessEqual(x.path, "/bar/foobar")

    def testParsesRelativeSshLocationCorrectly(self):
        x = self.oper.parse_url("sftp://foo/~/bar/foobar")
        self.failUnlessEqual(x.scheme, "sftp")
        self.failUnlessEqual(x.netloc, "foo")
        self.failUnlessEqual(x.path, "bar/foobar")

    def testCopiesFileToLocation(self):
        self.oper.copy_to_location(self.dirpath("README"), self.build_area)
        self.failUnless(os.path.exists(self.buildpath("README")))

    def testSilentlyDoesNotCopyNonExistentFile(self):
        self.oper.copy_to_location(self.dirpath("notexist"), self.build_area)
        self.failIf(os.path.exists(self.buildpath("notexist")))

    def mock_network_copy(self, filename, location):
        self.did_network_copy = True

    def testCopiesToNetworkLocationWhenRequested(self):
        self.oper.network_copy = self.mock_network_copy
        self.oper.copy_to_location(self.dirpath("README"), "sftp://foo/bar")
        self.failUnless(self.did_network_copy)

    def mock_copy(self, filename, location):
        self.copy_args.append((filename, location))

    def testPublishesNEWS(self):
        self.oper.copy_to_location = self.mock_copy
        self.oper.do_it(self.options)
        self.failUnless((self.dirpath("NEWS"), self.resultname) 
                        in self.copy_args)

    def testPublishesREADME(self):
        self.oper.copy_to_location = self.mock_copy
        self.oper.do_it(self.options)
        self.failUnless((self.dirpath("README"), self.resultname) 
                        in self.copy_args)

    def testPublishesTarball(self):
        self.oper.copy_to_location = self.mock_copy
        self.oper.do_it(self.options)
        self.failUnless((self.buildpath("pink-1.2.tar.gz"), self.resultname) 
                            in self.copy_args)

    def testPublishesChangelog(self):
        self.oper.copy_to_location = self.mock_copy
        self.oper.do_it(self.options)
        self.failUnless((self.buildpath("pink.changelog"), self.resultname) 
                        in self.copy_args)
