# encoding: utf-8
"""Test the TurboMail Message class."""

import calendar
from datetime import datetime, timedelta
import email
import logging
import re
import time
import unittest

from turbomail.compat import Header, MIMEText, formatdate, parsedate_tz
from turbomail.control import interface
from turbomail.message import Message

logging.disable(logging.WARNING)


class TestBasicMessage(unittest.TestCase):
    """Test the basic output of the Message class."""
    
    def build_message(self):
        return Message(
                       author=('Author', 'author@example.com'),
                       to=('Recipient', 'recipient@example.com'),
                       subject='Test message subject.',
                       plain='This is a test message plain text body.'
                      )
    
    def setUp(self):
        self._old_config = interface.config.copy()
        self.message = self.build_message()
    
    def tearDown(self):
        interface.config = self._old_config
    
    def test_message_properties(self):
        self.assertEqual(self.message.author, [("Author", "author@example.com")])
        self.assertEqual(str(self.message.author), "Author <author@example.com>")
        self.failUnless(isinstance(self.message.mime, MIMEText))
    
    def test_message_string_with_basic(self):
        message_string = str(self.message)
        msg = email.message_from_string(message_string)
        self.assertEqual('Author <author@example.com>', msg['From'])
        self.assertEqual('Recipient <recipient@example.com>', msg['To'])
        self.assertEqual('Test message subject.', msg['Subject'])
        self.assertEqual('This is a test message plain text body.', msg.get_payload())
    
    def test_message_recipients_and_addresses(self):
        self.message.cc = 'cc@example.com'
        self.message.bcc = 'bcc@example.com'
        self.message.sender = 'sender@example.com'
        self.message.reply_to = 'replyto@example.com'
        self.message.disposition = 'disposition@example.com'
        
        message_string = str(self.message)
        msg = email.message_from_string(message_string)
        
        self.assertEqual('cc@example.com', msg['cc'])
        # BCC must not be encoded in the message itself, it is only sent in the
        # smtp envelope
        self.assertEqual(None, msg['bcc'])
        self.assertEqual('sender@example.com', msg['sender'])
        self.assertEqual('replyto@example.com', msg['reply-to'])
        self.assertEqual('disposition@example.com', msg['disposition-notification-to'])
    
    def test_mime_generation(self):
        mime = self.message.mime
        self.failUnless(self.message.mime is mime)
        self.message.subject = "Test message subject."
        self.failIf(self.message.mime is mime)
    
    def test_recipients_collection(self):
        self.message.cc.append("copied@example.com")
        self.assertEqual(["recipient@example.com", "copied@example.com"],
                         self.message.recipients.addresses)
    
    def test_smtp_from_as_envelope(self):
        self.message.smtp_from = 'devnull@example.com'
        self.assertEqual('devnull@example.com', str(self.message.envelope_sender))
    
    def test_subject_with_umlaut(self):
        subject_string = u"Test with äöü"
        self.message.subject = subject_string
        self.message.encoding = "UTF-8"
        
        msg = email.message_from_string(str(self.message))
        encoded_subject = str(Header(subject_string, "UTF-8"))
        self.assertEqual(encoded_subject, msg['Subject'])
    
    def test_from_with_umlaut(self):
        from_name = u"Karl Müller"
        from_email = u"karl.mueller@example.com"
        
        self.message.author = [(from_name, from_email)]
        self.message.encoding = "ISO-8859-1"
        
        msg = email.message_from_string(str(self.message))
        encoded_name = "%s <%s>" % (str(Header(from_name, "ISO-8859-1")), from_email)
        self.assertEqual(encoded_name, msg['From'])
    
    def test_multiple_authors(self):
        """For better readability, provide a property 'authors' (alias for 
        author)."""
        self.message.authors = 'authors@example.com'
        self.assertEqual(self.message.authors, self.message.author)
        
        self.message.authors = ['bar@example.com', 'baz@example.com']
        self.message.sender = 'foo@example.com'
        msg = email.message_from_string(str(self.message))
        from_addresses = re.split(r",\n?\s+", msg['From'])
        self.assertEqual(['bar@example.com', 'baz@example.com'], from_addresses)
    
    def test_multiple_authors_require_sender(self):
        self.message.authors = ['bar@example.com', 'baz@example.com']
        self.assertRaises(ValueError, str, self.message)
        
        self.message.sender = 'bar@example.com'
        str(self.message)
    
    def test_permit_one_sender_at_most(self):
        self.message.sender = ['bar@example.com', 'baz@example.com']
        self.assertRaises(ValueError, str, self.message)
        
        self.message.sender = 'bar@example.com'
        str(self.message)
    
    def test_raise_error_for_unknown_kwargs_at_class_instantiation(self):
        self.assertRaises(TypeError, Message, invalid_argument=True)

    def test_add_custom_headers_dict(self):
        "Test that custom headers (dict type) can be attached."
        self.message.headers = {'Precedence': 'bulk', 'X-User': 'Alice'}
        msg = email.message_from_string(str(self.message))
        self.assertEqual('bulk', msg['Precedence'])
        self.assertEqual('Alice', msg['X-User'])

    def test_add_custom_headers_tuple(self):
        "Test that a custom header (tuple type) can be attached."
        self.message.headers = (('Precedence', 'bulk'), ('X-User', 'Alice'))
        
        msg = email.message_from_string(str(self.message))
        self.assertEqual('bulk', msg['Precedence'])
        self.assertEqual('Alice', msg['X-User'])

    def test_add_custom_headers_list(self):
        "Test that a custom header (list type) can be attached."
        self.message.headers = [('Precedence', 'bulk'), ('X-User', 'Alice')]
        
        msg = email.message_from_string(str(self.message))
        self.assertEqual('bulk', msg['Precedence'])
        self.assertEqual('Alice', msg['X-User'])
    
    def test_no_sender_header_if_no_sender_required(self):
        msg = email.message_from_string(str(self.message))
        self.assertEqual(None, msg['Sender'])
    
    def _date_header_to_utc_datetime(self, date_string):
        """Converts a date_string from the Date header into a naive datetime
        object in UTC."""
        # There is pytz which could solve whole isssue but it is not in Fedora
        # EPEL 4 currently so I don't want to depend on out-of-distro modules - 
        # hopefully I'll get it right anyway...
        assert date_string != None
        tztime_struct = parsedate_tz(date_string)
        time_tuple, tz_offset = (tztime_struct[:9], tztime_struct[9])
        epoch_utc_seconds = calendar.timegm(time_tuple)
        if tz_offset is not None:
            epoch_utc_seconds -= tz_offset
        datetime_obj = datetime.utcfromtimestamp(epoch_utc_seconds)
        return datetime_obj
    
    def _almost_now(self, date_string):
        """Returns True if the date_string represents a time which is 'almost 
        now'."""
        utc_date = self._date_header_to_utc_datetime(date_string)
        delta = abs(datetime.utcnow() - utc_date)
        return (delta < timedelta(seconds=1))
    
    def test_date_header_added_even_if_date_not_set_explicitely(self):
        msg = email.message_from_string(str(self.message))
        self.failUnless(self._almost_now(msg['Date']))
    
    def test_date_can_be_set_as_string(self):
        date_string = 'Fri, 26 Dec 2008 11:19:42 +0530'
        self.message.date = date_string
        msg = email.message_from_string(str(self.message))
        self.assertEqual(date_string, msg['Date'])
    
    def test_date_can_be_set_as_float(self):
        expected_date = datetime(2008, 12, 26, 12, 55)
        expected_time = time.mktime(expected_date.timetuple())
        self.message.date = expected_time
        msg = email.message_from_string(str(self.message))
        header_string = msg['Date']
        header_date = self._date_header_to_utc_datetime(header_string)
        self.assertEqual(self.localdate_to_utc(expected_date), header_date)
        expected_datestring = formatdate(expected_time, localtime=True)
        self.assertEqual(expected_datestring, header_string)
    
    def localdate_to_utc(self, localdate):
        local_epoch_seconds = time.mktime(localdate.timetuple())
        date_string = formatdate(local_epoch_seconds, localtime=True)
        return self._date_header_to_utc_datetime(date_string)
    
    def test_date_can_be_set_as_datetime(self):
        expected_date = datetime(2008, 12, 26, 12, 55)
        self.message.date = expected_date
        msg = email.message_from_string(str(self.message))
        header_date = self._date_header_to_utc_datetime(msg['Date'])
        self.assertEqual(self.localdate_to_utc(expected_date), header_date)
    
    def test_date_header_is_set_even_if_reset_to_none(self):
        self.message.date = None
        msg = email.message_from_string(str(self.message))
        self.failUnless(self._almost_now(msg['Date']))
    
    def test_recipients_property_includes_cc_and_bcc(self):
        self.message.cc = 'cc@example.com'
        self.message.bcc = 'bcc@example.com'
        expected_recipients = ['recipient@example.com', 'cc@example.com', 
                               'bcc@example.com']
        recipients = map(str, list(self.message.recipients.addresses))
        self.assertEqual(expected_recipients, recipients)
    
    def test_can_set_encoding_for_message_explicitely(self):
        self.failIf('iso-8859-1' in str(self.message).lower())
        self.message.encoding = 'ISO-8859-1'
        msg = email.message_from_string(str(self.message))
        self.assertEqual('text/plain; charset="iso-8859-1"', msg['Content-Type'])
        self.assertEqual('quoted-printable', msg['Content-Transfer-Encoding'])
    
    def test_message_encoding_can_be_set_in_config_file(self):
        interface.config['mail.message.encoding'] = 'ISO-8859-1'
        message = self.build_message()
        msg = email.message_from_string(str(message))
        self.assertEqual('text/plain; charset="iso-8859-1"', msg['Content-Type'])
        self.assertEqual('quoted-printable', msg['Content-Transfer-Encoding'])
    
    def test_plain_utf8_encoding_uses_base64(self):
        self.failIf('utf-8' in str(self.message).lower())
        self.message.encoding = 'UTF-8'
        msg = email.message_from_string(str(self.message))
        self.assertEqual('text/plain; charset="utf-8"', msg['Content-Type'])
        self.assertEqual('base64', msg['Content-Transfer-Encoding'])
    
    # def test_can_use_utf8qp_encoding_without_turbogears_extension(self):
    #     self.failIf('utf-8' in str(self.message).lower())
    #     for encoding in ('utf8qp', 'utf-8-qp', 'UTF-8-QP', 'UtF-8-qP'):
    #         self.message.encoding = encoding
    #         msg = email.message_from_string(str(self.message))
    #         self.assertEqual('text/plain; charset="utf-8"', msg['Content-Type'])
    #         self.assertEqual('quoted-printable', msg['Content-Transfer-Encoding'])


