#ifndef INDII_TINT_IMAGERESOURCE_HPP
#define INDII_TINT_IMAGERESOURCE_HPP

#include "wx/image.h"

namespace indii {
  namespace tint {

/**
 * Abstract image resource manager.
 *
 * @author Lawrence Murray <lawrence@indii.org>
 * @version $Rev: 116 $
 * @date $Date: 2009-09-19 17:32:46 +0800 (Sat, 19 Sep 2009) $
 */
class ImageResource {
public:
  /**
   * Constructor.
   */
  ImageResource();

  /**
   * Destructor.
   */
  virtual ~ImageResource();
  
  /**
   * Fit width and height around lower bounds.
   *
   * @param widthBound Width bound.
   * @param heightBound Height bound.
   * @param width Holds width on output.
   * @param height Holds height on output.
   *
   * Calculates the smallest size of the image such that @p width is not
   * smaller than @p widthBound and @p height is not smaller than
   * @p heightBound, while maintaining the aspect ratio.
   */
  void fitOutside(const int widthBound, const int heightBound, int* width,
      int* height);

  /**
   * Fit width and height within upper bounds.
   *
   * @param widthBound Width bound.
   * @param heightBound Height bound.
   * @param width Holds width on output.
   * @param height Holds height on output.
   *
   * Calculates the largest size of the image such that @p width does not
   * exceed @p widthBound and @p height does not exceed @p heightBound,
   * while maintaining the aspect ratio.
   */
  void fitInside(const int widthBound, const int heightBound, int* width,
      int* height);

  /**
   * Fit scale within upper bounds.
   *
   * @param widthBound Width bound.
   * @param heightBound Height bound.
   *
   * @return Scale.
   *
   * Calculates the largest scale of the image such that its width does not
   * exceed @p widthBound and its height does not exceed @p heightBound,
   * while maintaining the aspect ratio.
   */
  unsigned int fitScale(const int widthBound, const int heightBound);

  /**
   * Get width of the image.
   *
   * @return Width of image.
   */
  virtual unsigned int getWidth() = 0;
  
  /**
   * Get height of the image.
   *
   * @return Height of the image.
   */
  virtual unsigned int getHeight() = 0;
  
  /**
   * Get image.
   *
   * @param width Width of the image.
   * @param height Height of the image.
   * @param high Should high quality resizing be used?
   *
   * @return A version of the image of the specified size.
   *
   * If either @p width or @p height is <= 0 the original version of the
   * image is returned. Resized versions are cached internally and reused
   * where possible.
   */
  virtual wxImage* get(const int width = 0, const int height = 0,
      const bool high = false) = 0;

  /**
   * Calculate image region.
   *
   * @param rect Rectangular region.
   * @param width Width of the image.
   * @param height Height of the image.
   * @param high Should high quality resizing be used?
   *
   * @return The given region of the image of the specified size.
   */
  virtual wxImage calc(const wxRect& rect, const unsigned int width = 0,
      const unsigned int height = 0, const bool high = false) = 0;

};

  }
}

#endif
