package tijmp.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.Arrays;
import javax.swing.BorderFactory;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextArea;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableColumn;
import tijmp.ProfilerHandler;
import tijmp.ui.ClassRenderer;
import tijmp.ui.ReadableSizeRenderer;
import tijmp.actions.ChildObjectsSummary;

/** A class that show a table with instances of a certain class.
 */
class InstanceTable {    
    private ITM m;
    private ProfilerHandler ph;

    public InstanceTable (ProfilerHandler ph, Class<?> clz, 
			  Object[] objects, long[] sizes, int[] lengths) {
	m = new ITM (clz, objects, sizes, lengths);
	this.ph = ph;
    }

    public void showFrame () {
	JTable table = new JTable (m); 
	JScrollPane scrollPane = new JScrollPane (table);
	table.setAutoResizeMode (JTable.AUTO_RESIZE_SUBSEQUENT_COLUMNS);
	table.setSelectionMode (ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
	table.setAutoCreateRowSorter (true);

	TableColumn column;
	for (int i = 0; i < m.getColumnCount (); i++) {
	    column = table.getColumnModel ().getColumn (i);
	    if (i == ITM.COL_CLASS)
		column.setPreferredWidth (350);
	    else
		column.setPreferredWidth (75);
	    if (i == ITM.COL_SIZE) 
		column.setCellRenderer (new ReadableSizeRenderer ());
	}
	table.setDefaultRenderer (Class.class, new ClassRenderer ());
	table.getRowSorter ().toggleSortOrder (ITM.COL_SIZE);
	table.getRowSorter ().toggleSortOrder (ITM.COL_SIZE);
	table.addMouseListener (new InstanceMouseHandler (table, ph));
	
	if (!m.getInstanceClass ().isArray ())
	    table.removeColumn (table.getColumnModel ().getColumn (ITM.COL_LENGTH));
	table.removeColumn (table.getColumnModel ().getColumn (ITM.COL_OBJECT));
        table.setPreferredScrollableViewportSize (new Dimension (500, 200));
	JFrame f = new JFrame ("Instances of " + m.getInstanceClass ().getName ());
	f.setDefaultCloseOperation (JFrame.DISPOSE_ON_CLOSE);
	
	JTextArea ta = new JTextArea (5, 80);
	ta.setEnabled (false);
	JScrollPane spTa = new JScrollPane (ta);

	JLabel equals = new JLabel ("-");
	equals.setBorder (BorderFactory.createLoweredBevelBorder ());
	JLabel hashCodeEquals = new JLabel ("-");
	hashCodeEquals.setBorder (BorderFactory.createLoweredBevelBorder ());
	JPanel status = new JPanel ();
	status.add (equals);
	status.add (hashCodeEquals);
	status.setBorder (BorderFactory.createLineBorder (Color.BLACK));
	status.setLayout (new FlowLayout (FlowLayout.LEFT, 0, 0));

	ILSM lsm = new ILSM (table, equals, hashCodeEquals, ta);
	table.getSelectionModel ().addListSelectionListener (lsm);
	
	GridBagLayout gb = new GridBagLayout ();
	f.setLayout (gb);

	GridBagConstraints c = new GridBagConstraints ();
	c.insets = new Insets (2, 2, 2, 2);
	c.gridx = 0; 
	c.gridy = 0;
	c.weightx = 1;
	c.weighty = 9;
	c.fill = GridBagConstraints.BOTH;

	f.add (scrollPane, c);
	c.weighty = 1;
	c.gridy++;
	f.add (spTa, c);

	c.weighty = 0;
	c.gridy++;
	f.add (status, c);

	f.pack ();
	f.setVisible (true);	
    }    
}

class InstanceMouseHandler extends MouseAdapter {
    private JTable table;
    private ProfilerHandler ph;
    
    public InstanceMouseHandler (JTable table, ProfilerHandler ph) {
	this.table = table;
	this.ph = ph;
    }
    
    @Override public void mousePressed (MouseEvent e) {
	if (!SwingUtilities.isRightMouseButton (e))
	    return;
	int viewRow = table.rowAtPoint (e.getPoint ());
	int modelRow = table.convertRowIndexToModel (viewRow);
	table.getSelectionModel ().setSelectionInterval (viewRow, viewRow);
	Object o = table.getModel ().getValueAt (modelRow, ITM.COL_OBJECT);
	Class<?> c = o.getClass ();
	if (c.isPrimitive () || 
	    (c.isArray () && c.getComponentType ().isPrimitive ()))
	    return;
	JPopupMenu m = new JPopupMenu ();
	m.add (new JMenuItem (new ChildObjectsSummary (ph, o)));
	m.show (e.getComponent (), e.getX (), e.getY ());
    }
}

class ILSM implements ListSelectionListener {
    private JTable table;
    private JLabel equals;
    private JLabel hashCodeEquals;
    private JTextArea ta;

    public ILSM (JTable table, JLabel equals, 
		 JLabel hashCodeEquals, JTextArea ta) {
	this.table = table;
	this.equals = equals;
	this.hashCodeEquals = hashCodeEquals;
	this.ta = ta;
    }

    public void valueChanged (ListSelectionEvent e) {
	int[] rows = table.getSelectedRows ();
	if (rows == null || rows.length < 1)
	    ta.setText ("");
	StringBuilder sb = new StringBuilder ();
	for (int r : rows) {
	    int modelRow = table.convertRowIndexToModel (r);
	    Object o = table.getModel ().getValueAt (modelRow, ITM.COL_OBJECT);
	    if (o != null) {
		Class<?> c = o.getClass ();
		if (c.isArray ()) {
		    Class<?> ac = c.getComponentType ();
		    if (ac.isPrimitive ()) {
			String s = Arrays.deepToString (new Object[] {o});
			s = s.substring (1, s.length () - 1);
			sb.append (s);
		    } else {
			sb.append (Arrays.deepToString ((Object[])o));
		    }
		} else {
		    sb.append (o.toString ());
		}
	    } else {
		sb.append ("<null>");
	    }
	    sb.append ("\n");
	}
	sb.setLength (sb.length () - 1);
	ta.setText (sb.toString ());
	ta.setCaretPosition (0);

	if (rows == null || rows.length < 2) {
	    equals.setText ("-");
	    hashCodeEquals.setText ("-");
	    return;
	}

	Object o1 = table.getModel ().getValueAt (rows[0], ITM.COL_OBJECT);
	if (o1 == null) {
	    equals.setText ("-");
	    hashCodeEquals.setText ("-");
	    return;
	}
	
	int hashCode = o1.hashCode ();
	boolean be = true;
	boolean bh = true;
	for (int i = 1; i < rows.length; i++) {
	    Object o2 = table.getModel ().getValueAt (rows[i], ITM.COL_OBJECT);
	    if (o2 == null) {
		be = false;
		bh = false;
		break;
	    }
	    if (!(be && o1.equals (o2))) 
		be = false;
	    if (!(bh && o2.hashCode () == hashCode))
		bh = false;
	}
	equals.setText (be ? "equals" : "not equals");
	hashCodeEquals.setText (bh ? "same hashCode" : "not same hashCode");
    }
}

class ITM extends AbstractTableModel {
    private Class<?> clz;
    private Object[] objects;
    private long[] sizes;
    private int[] lengths;

    private String[] columnNames = {"OBJECT", "Class", "Size", "Length"};

    public static final int COL_OBJECT = 0;
    public static final int COL_CLASS = COL_OBJECT + 1;
    public static final int COL_SIZE = COL_CLASS + 1;
    public static final int COL_LENGTH = COL_SIZE + 1;
    
    public ITM (Class<?> clz, Object[] objects, long[] sizes, int[] lengths) {
	this.clz = clz; 
	this.objects = objects;
	this.sizes = sizes;
	this.lengths = lengths;
    }

    @Override public String getColumnName (int col) {
        return columnNames[col];
    }

    public int getRowCount () { 
	return sizes.length;
    }

    public int getColumnCount () { 
	return columnNames.length; 
    }

    public Class<?> getInstanceClass () {
	return clz;
    }
    
    @Override public Object getValueAt (int row, int col) {
	switch (col) {
	case COL_OBJECT: 
	    return objects[row];
	case COL_CLASS:
	    return clz;
	case COL_SIZE:
	    return sizes[row];
	case COL_LENGTH:
	    return lengths[row];
	default:
	    throw new IllegalArgumentException ("do not know how to handle col: " + 
						col);
	}
    }

    @Override public Class<?> getColumnClass (int col) {
	switch (col) {
	case COL_OBJECT: 
	    return Object.class;
	case COL_CLASS:
	    return Class.class;
	case COL_SIZE:
	    return Long.class;
	case COL_LENGTH:
	    return Long.class;
	default:
	    throw new IllegalArgumentException ("do not know how to handle col: " + 
						col);
	}	
    }

    @Override public boolean isCellEditable (int row, int col) { 
	return false; 
    }

    @Override public void setValueAt (Object value, int row, int col) {
	throw new IllegalStateException ("non editable table");
    }
}
