/*
 *	TICKR - GTK-based Feed Reader - Copyright (C) Emmanuel Thomas-Maurin 2009-2011
 *	<manutm007@gmail.com>
 *
 * 	This program is free software: you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation, either version 3 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "news.h"

#if USE_GUI
#define URL_ENTRY_LENGTH	100

/* predefined GTK_RESPONSE_ are < 0, those ones are app level defined and >= 0
*/
enum {
	GTK_RESPONSE_SELECT_ALL, GTK_RESPONSE_UNSELECT_ALL, GTK_RESPONSE_REMOVE,
	GTK_RESPONSE_ADD, GTK_RESPONSE_HOME, GTK_RESPONSE_SINGLE, GTK_RESPONSE_SELECTION
};

enum {COLUMN_INT, COLUMN_BOOLEAN_CHECKED, COLUMN_STRING_TITLE, COLUMN_STRING_URL, N_COLUMNS};

static GtkWidget	*dialog, *entry_url;
static TickerEnv	*env;
static char		**p_url;
static int		rank;
static gchar		home_feed[FILE_NAME_MAXLEN + 1];

static gint esc_key_pressed(GtkWidget *dialog2, GdkEventKey *event, gpointer unused)
{
	unused = unused;
	if (event->keyval == GDK_Escape) {
		gtk_dialog_response(GTK_DIALOG(dialog2), GTK_RESPONSE_CLOSE);
		return TRUE;
	} else
		return FALSE;
}

static void force_quit_dialog(GtkWidget *dialog2)
{
	gtk_dialog_response(GTK_DIALOG(dialog2), GTK_RESPONSE_CLOSE);
}

/* tree selection callback - copy selected url to entry_url (override) */
static gint tree_selection_changed(GtkTreeSelection *selection, GdkEventAny *event, gpointer unused)
{
	GtkTreeModel	*tree_model = NULL;
	GtkTreeIter	iter;
	gchar		*str_url;

	event = event;
	unused = unused;
	if (gtk_tree_selection_get_selected(selection, &tree_model, &iter)) {
		gtk_tree_model_get(tree_model, &iter,
			COLUMN_INT, &rank,
			COLUMN_STRING_URL, &str_url,
			-1);
		gtk_entry_set_text(GTK_ENTRY(entry_url), str_url);
		g_free(str_url);
	}
	gtk_dialog_response(GTK_DIALOG(dialog), GTK_RESPONSE_NONE);
	return TRUE;
}

/* catch double-click on tree view */
static gint double_click_on_tree_view(GtkTreeView *tree_view, GtkTreePath *tree_path, gpointer unused)
{
	GtkTreeModel	*tree_model = NULL;
	GtkTreeIter	iter;
	gchar		*str_url;

	unused = unused;
	tree_model = gtk_tree_view_get_model(tree_view);
	if (gtk_tree_model_get_iter(tree_model, &iter, tree_path)) {
		gtk_tree_model_get(tree_model, &iter,
			COLUMN_INT, &rank,
			COLUMN_STRING_URL, &str_url,
			-1);
		gtk_entry_set_text(GTK_ENTRY(entry_url), str_url);
		g_free(str_url);
	}
	gtk_dialog_response(GTK_DIALOG(dialog), GTK_RESPONSE_SINGLE);
	return TRUE;
}

static gint enter_key_pressed_in_entry(GtkWidget *widget, GdkEvent *event, gpointer unused)
{
	widget = widget;
	event = event;
	unused = unused;
	gtk_dialog_response(GTK_DIALOG(dialog), GTK_RESPONSE_SINGLE);
	return TRUE;
}

static gint checkbox_toggled(GtkCellRendererToggle *renderer, gchar *path_str, gpointer tree_view)
{
	GtkTreeModel	*tree_model;
	GtkTreeIter	iter;
	gboolean	checked;

	renderer = renderer;
	tree_model = gtk_tree_view_get_model(GTK_TREE_VIEW(tree_view));
	if (gtk_tree_model_get_iter_from_string(tree_model, &iter, path_str)) {
		gtk_tree_model_get(tree_model, &iter,
			COLUMN_INT, &rank,
			COLUMN_BOOLEAN_CHECKED, &checked,
			-1);
			checked = !checked;
			gtk_list_store_set(GTK_LIST_STORE(tree_model), &iter,
				COLUMN_BOOLEAN_CHECKED, checked,
				-1);
	}
	if (checked)
		p_url[rank][0] = SELECTED_URL_CHAR;
	else
		p_url[rank][0] = UNSELECTED_URL_CHAR;
	gtk_dialog_response(GTK_DIALOG(dialog), GTK_RESPONSE_NONE);
	return TRUE;
}

static void select_all(GtkTreeView *tree_view)
{
	GtkTreeModel	*tree_model;
	GtkTreeIter	iter;
	int		i = 0;

	tree_model = gtk_tree_view_get_model(GTK_TREE_VIEW(tree_view));
	if (gtk_tree_model_get_iter_first(tree_model, &iter)) {
		do
			gtk_list_store_set(GTK_LIST_STORE(tree_model), &iter,
				COLUMN_BOOLEAN_CHECKED, TRUE,
				-1);
		while (gtk_tree_model_iter_next(tree_model, &iter));
		while (p_url[i] != NULL)
			p_url[i++][0] = SELECTED_URL_CHAR;
	}
}

static void unselect_all(GtkTreeView *tree_view)
{
	GtkTreeModel	*tree_model;
	GtkTreeIter	iter;
	int		i = 0;

	tree_model = gtk_tree_view_get_model(GTK_TREE_VIEW(tree_view));
	if (gtk_tree_model_get_iter_first(tree_model, &iter)) {
		do
			gtk_list_store_set(GTK_LIST_STORE(tree_model), &iter,
				COLUMN_BOOLEAN_CHECKED, FALSE,
				-1);
		while (gtk_tree_model_iter_next(tree_model, &iter));
		while (p_url[i] != NULL)
			p_url[i++][0] = UNSELECTED_URL_CHAR;
	}
}

/* clear the list before filling it */
static void fill_list_store_from_array(GtkListStore *list_store, char **url)
{
	GtkTreeIter		iter;
	gboolean		rss_feed_checked;
	char			rss_feed_url[FILE_NAME_MAXLEN + 1];
	char			rss_feed_title[FEED_TITLE_MAXLEN + 1];
	int			i, j;

	gtk_list_store_clear(list_store);
	for (i = 0; i < NURLMAX && url[i] != NULL; i++) {
		gtk_list_store_append(list_store, &iter);
		/* store feed checked status, url and title (if any) in list_store */
		if (url[i][0] == SELECTED_URL_CHAR)
			rss_feed_checked = TRUE;
		else
			rss_feed_checked = FALSE;
		str_n_cpy(rss_feed_url, url[i] + 1, FILE_NAME_MAXLEN);
		for (j = 0; j < FILE_NAME_MAXLEN; j++) {
			if (rss_feed_url[j] == '\0' || rss_feed_url[j] == '\n') {
				rss_feed_title[0] = '\0';
				break;
			} else if (rss_feed_url[j] == TITLE_TAG_CHAR) {
				str_n_cpy(rss_feed_title, rss_feed_url + j + 1,
					FEED_TITLE_MAXLEN);
				rss_feed_url[j] = '\0';
				break;
			}
		}
		gtk_list_store_set(list_store, &iter,
			COLUMN_INT, i,
			COLUMN_BOOLEAN_CHECKED, rss_feed_checked,
			COLUMN_STRING_TITLE, rss_feed_title,
			COLUMN_STRING_URL, rss_feed_url,
			-1);
	}
}

static void remove_feed_from_array_and_list_store(GtkListStore *list_store, GtkTreeIter *iter)
{
	remove_url_from_list(p_url, rank);
	sort_url_list(p_url, 1);
	gtk_list_store_remove(list_store, iter);
	rank = -1;
}

static int add_feed_to_array_and_list_store(Resource *resrc, GtkListStore *list_store)
{
	int		rss_status;
	char		rss_feed_title[FEED_TITLE_MAXLEN + 1];
	char		rss_feed_link[FILE_NAME_MAXLEN + 1];
	char		rss_feed_ttl[32];
	char		tmp[TMPSTR_SIZE + 1];
	int		i;

	str_n_cpy(resrc->id, (char *)gtk_entry_get_text(GTK_ENTRY(entry_url)), FILE_NAME_MAXLEN);
	if ((rss_status = get_feed(resrc, get_params())) == OK) {
		/* locate last array entry */
		for (i = 0; i < NURLMAX  && p_url[i] != NULL; i++);
		/* we need to re-init p_url because list is NULL terminated */
		p_url = init_url_list();
		if (i < NURLMAX) {
			if (get_feed_info(resrc->id, rss_feed_title, rss_feed_link, rss_feed_ttl) == OK)
				snprintf(p_url[i], FILE_NAME_MAXLEN + 1, "%c%s%c%s",
					UNSELECTED_URL_CHAR, resrc->id, TITLE_TAG_CHAR, rss_feed_title);
			else
				snprintf(p_url[i], FILE_NAME_MAXLEN + 1, "%c%s",
					UNSELECTED_URL_CHAR, resrc->id);
			p_url[i + 1] = NULL;
			sort_url_list(p_url, 1);
			fill_list_store_from_array(list_store, p_url);
		} else {
			snprintf(tmp, TMPSTR_SIZE + 1, "You can't save more than %d URLs.\n"
				"(You may set NURLMAX to a higher value in news.h and recompile.)",
				NURLMAX);
			warning(tmp, "", "", "", FALSE);
		}
	}
	return rss_status;
}

/*
 * open a dialog with a list of urls to choose from and an entry_url
 * (with feed titles)
 *
 * entry format in url list file:
 *	["*" (selected) or "-" (unselected) + url [+ ">" + title] + '\n']
 *
 * entry max length = FILE_NAME_MAXLEN
 * see also: (UN)SELECTED_URL_CHAR in news.h
 */
void get_new_url(Resource *resrc)
{
	GtkWidget		*vbox, *hbox, *label[2], *sc_win, *tree_view = NULL;
	GtkWidget		*cancel_but, *remove_but, *add_but, *home_but;
	GtkWidget		*single_but, *selection_but;
	GtkWidget		*selectall_but, *unselectall_but;
	GtkListStore		*list_store = NULL;
	GtkTreeModel		*tree_model = NULL;
	GtkTreeIter		iter;
	GtkCellRenderer		*renderer1 = NULL, *renderer2 = NULL, *renderer3 = NULL;
	GtkTreeViewColumn	*column1 = NULL, *column2 = NULL, *column3 = NULL;
	GtkTreeSelection	*selection = NULL;
	GtkRequisition		req;
	gint			list_height;
	gint			response;
	Resource		resrc_bak;
	int			i;

	env = get_ticker_env();
	env->suspend_rq = TRUE;
	gtk_window_set_keep_above(GTK_WINDOW(env->win), FALSE);

	dialog = gtk_dialog_new_with_buttons(
			"RSS Feed Picker", GTK_WINDOW(env->win),
			GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
			NULL);

	cancel_but = gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE);
	selectall_but = gtk_dialog_add_button(GTK_DIALOG(dialog), "Select All", GTK_RESPONSE_SELECT_ALL);
	unselectall_but = gtk_dialog_add_button(GTK_DIALOG(dialog), "Unselect All", GTK_RESPONSE_UNSELECT_ALL);
	home_but = gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_HOME, GTK_RESPONSE_HOME);
	remove_but = gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_REMOVE, GTK_RESPONSE_REMOVE);
	add_but = gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_ADD, GTK_RESPONSE_ADD);
	single_but = gtk_dialog_add_button(GTK_DIALOG(dialog), "OK (Single)", GTK_RESPONSE_SINGLE);
	selection_but = gtk_dialog_add_button(GTK_DIALOG(dialog), "OK (Selec)", GTK_RESPONSE_SELECTION);

	set_news_icon_to_dialog(GTK_WINDOW(dialog));
	gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), vbox, FALSE, TRUE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)), 15);

	str_n_cpy(home_feed, get_params()->homefeed, FILE_NAME_MAXLEN);
	p_url = init_url_list();
	list_store = gtk_list_store_new(N_COLUMNS, G_TYPE_INT, G_TYPE_BOOLEAN, G_TYPE_STRING, G_TYPE_STRING);

	/* init pointers then load url list from file into array */
	if (load_url_list(p_url) == OK) {
		sort_url_list(p_url, 1);
		fill_list_store_from_array(list_store, p_url);

		tree_view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(list_store));

		renderer1 = gtk_cell_renderer_toggle_new();
		gtk_cell_renderer_toggle_set_radio(GTK_CELL_RENDERER_TOGGLE(renderer1), FALSE);
		column1 = gtk_tree_view_column_new_with_attributes(NULL, renderer1,
			"active", COLUMN_BOOLEAN_CHECKED, NULL);
		gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view), column1);

		renderer2 = gtk_cell_renderer_text_new();
		column2 = gtk_tree_view_column_new_with_attributes("RSS Feed Title", renderer2,
			"text", COLUMN_STRING_TITLE, NULL);
		gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view), column2);

		renderer3 = gtk_cell_renderer_text_new();
		column3 = gtk_tree_view_column_new_with_attributes("RSS Feed URL", renderer3,
			"text", COLUMN_STRING_URL, NULL);
		gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view), column3);

		sc_win = gtk_scrolled_window_new(NULL, NULL);
		gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sc_win),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
		gtk_container_add(GTK_CONTAINER(vbox), sc_win);
		gtk_container_add(GTK_CONTAINER(sc_win), tree_view);
		label[0] = gtk_label_new("");
		gtk_container_add(GTK_CONTAINER(vbox), label[0]);

		selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(tree_view));
		gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);
		gtk_widget_size_request(tree_view, &req);
		list_height = (req.height > 380) ? 380 : req.height;
		gtk_widget_set_size_request(sc_win, -1, list_height);

		g_signal_connect(G_OBJECT(selection), "changed", G_CALLBACK(tree_selection_changed), NULL);
		g_signal_connect(G_OBJECT(tree_view), "row_activated", G_CALLBACK(double_click_on_tree_view), NULL);
		g_signal_connect(G_OBJECT(renderer1), "toggled", G_CALLBACK(checkbox_toggled), tree_view);
	} else
		p_url[0] = NULL;

	hbox = gtk_hbox_new(FALSE, 0);
	label[1] = gtk_label_new("Enter new URL:  ");
	gtk_box_pack_start(GTK_BOX(hbox), label[1], FALSE, FALSE, 0);
	entry_url = gtk_entry_new();
	gtk_entry_set_max_length(GTK_ENTRY(entry_url), FILE_NAME_MAXLEN);
	gtk_entry_set_width_chars(GTK_ENTRY(entry_url), URL_ENTRY_LENGTH);
	gtk_box_pack_start(GTK_BOX(hbox), entry_url, FALSE, FALSE, 0);
	gtk_container_add(GTK_CONTAINER(vbox), hbox);

	g_signal_connect(G_OBJECT(entry_url), "activate", G_CALLBACK(enter_key_pressed_in_entry), NULL);
	g_signal_connect(G_OBJECT(dialog), "key-press-event", G_CALLBACK(esc_key_pressed), NULL);
	g_signal_connect(G_OBJECT(dialog), "delete_event", G_CALLBACK(force_quit_dialog), NULL);

	gtk_widget_show_all(dialog);
	gtk_widget_grab_focus(tree_view);

	/* backup last valid opened resource (if any) */
	str_n_cpy(resrc_bak.id, resrc->id, FILE_NAME_MAXLEN);
	gtk_entry_set_text(GTK_ENTRY(entry_url), "");
	rank = -1;

	while ((response = gtk_dialog_run(GTK_DIALOG(dialog))) != GTK_RESPONSE_CLOSE) {
		if (response == GTK_RESPONSE_SELECT_ALL) {
			if (question_win("Select all URLs ?") == YES) {
				select_all(GTK_TREE_VIEW(tree_view));
			}
		} else if (response == GTK_RESPONSE_UNSELECT_ALL) {
			if (question_win("Unselect all URLs ?") == YES) {
				unselect_all(GTK_TREE_VIEW(tree_view));
			}
		} else if (response == GTK_RESPONSE_HOME)
			gtk_entry_set_text(GTK_ENTRY(entry_url), (const gchar *)home_feed);
		else if (response == GTK_RESPONSE_REMOVE) {
			if (rank > -1) {
				if (gtk_tree_selection_get_selected(selection, &tree_model, &iter) &&
						question_win("Remove selected URL from list ?") == YES) {
					remove_feed_from_array_and_list_store(list_store, &iter);
					save_url_list(p_url);
					gtk_entry_set_text(GTK_ENTRY(entry_url), "");
				}
			} else
				warning("You must select an URL first",
					"", "", "", FALSE);
		} else if (response == GTK_RESPONSE_ADD) {
			if (gtk_entry_get_text(GTK_ENTRY(entry_url))[0] != '\0') {
				if (add_feed_to_array_and_list_store(resrc, list_store) == OK) {
					save_url_list(p_url);
					gtk_entry_set_text(GTK_ENTRY(entry_url), "");
				} else {
					/* restore last valid opened resource (if any) */
					str_n_cpy(resrc->id, resrc_bak.id, FILE_NAME_MAXLEN);
					/* re-init pointers then re-load url list from file into array */
					if (load_url_list(p_url) == OK) {
						sort_url_list(p_url, 1);
						/* also re-fill list_store */
						fill_list_store_from_array(list_store, p_url);
					} else
						p_url[0] = NULL;
				}
			} else
				warning("You must enter an URL first",
					"", "", "", FALSE);
		} else if (response == GTK_RESPONSE_SINGLE) {
			env->selection_mode = SINGLE;
			if (gtk_entry_get_text(GTK_ENTRY(entry_url))[0] != '\0' || rank != -1) {
				if (add_feed_to_array_and_list_store(resrc, list_store) == OK) {
					save_url_list(p_url);
					env->reload_rq = TRUE;
					break;
				} else {
					/* restore last valid opened resource (if any) */
					str_n_cpy(resrc->id, resrc_bak.id, FILE_NAME_MAXLEN);
					/* re-init pointers then re-load url list from file into array */
					if (load_url_list(p_url) == OK) {
						sort_url_list(p_url, 1);
						current_feed();
					} else
						p_url[0] = NULL;
				}
			} else
				warning("You must enter or select an URL first",
					"", "", "", FALSE);
		} else if (response == GTK_RESPONSE_SELECTION) {
			env->selection_mode = MULTIPLE;
			save_url_list(p_url);
			if ((i = load_selected_url_list(init_selected_url_list())) == OK) {
				first_feed();
				env->reload_rq = TRUE;
				break;
			} else if (i == SELECTION_EMPTY)
				warning("Selection is empty", "", "", "", FALSE);
			else
				warning("Selection error", "", "", "", FALSE);
		}
	}
	/* update url selection */
	load_selected_url_list(init_selected_url_list());
	if (list_store != NULL)
		g_object_unref(list_store);
	gtk_widget_destroy(dialog);
	check_main_win_always_on_top();
	env->suspend_rq = FALSE;
}
#endif
