/* ps.c
 * by Hirotsugu Kakugawa
 */
/*
 * Copyright (C) 1998  Hirotsugu Kakugawa. 
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

#include "../config.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#if STDC_HEADERS
#  include <string.h>
#else
#  include <strings.h>
#endif
#if HAVE_UNISTD_H
#  include <unistd.h>
#endif

#include "libdvi29.h"
#include "defs.h"

extern int           param_verbose;

static int           pcol;

static void    ps_prologue(DVI,DVI_DEVICE, int);
static void    ps_epilogue(DVI,DVI_DEVICE);
static void    ps_begin_page(DVI,DVI_DEVICE, int);
static void    ps_end_page(DVI,DVI_DEVICE);
static void    ps_frame_shipout(void);
static void    ps_shipout_range(long,long,long,long);


void
dvi2bdps(char *dvi_file, long w, long h) 
{
  int  page, ppp;  

  pcol = 0;

  ps_prologue(dvi, dev, dvi->pages);

  page = (param_page_rev == 0) ? param_page_from : param_page_to;
  do {
    if (interrupted == 1)
      break;
    if (param_verbose > 0){
      if (pcol > 70){
	fprintf(stderr, "\n");
	pcol = 0;
      }
      fprintf(stderr, "[%d] ", page);
      fflush(stderr);
      pcol += 4;
      ppp = page;
      while (ppp >= 10){
	pcol++;
	ppp /= 10;
      }
    }
    ps_begin_page(dvi, dev, page);
    if (DVI_DRAW_PAGE(dvi, dev, page, 1.0) < 0)
      break;
    ps_end_page(dvi, dev);
    page = page + ((param_page_rev == 0) ? 1 : -1);
  } while ((param_page_from <= page) && (page <= param_page_to));

  ps_epilogue(dvi, dev);

  if (param_verbose > 0)
    fprintf(stderr, "\n");
}


static void  
ps_prologue(DVI dvi, DVI_DEVICE dev, int pages)
{
  fprintf(output_fp, "%%!PS-Adobe-2.0\n");
  fprintf(output_fp, "%%%%Creator: %s-%s (Brain Damaged DVI to PostScript)\n",
	  program_name, VERSION);
  fprintf(output_fp, "%%%%Title: %s\n", 
	  (param_dvi_file!=NULL)?param_dvi_file:"(stdin)");
  fprintf(output_fp, "%%%%BoundingBox: 0 0 %d %d\n", 
	  (int)ceil(paper_list[param_paper_id].w * 72.27), 
	  (int)ceil(paper_list[param_paper_id].h * 72.27));
  fprintf(output_fp, "%%%%Pages: %d\n", param_page_to - param_page_from + 1);
  fprintf(output_fp, "%%%%DocumentPaperSizes: %s\n", 
	  paper_list[param_paper_id].name);
  fprintf(output_fp, "%%%%Orientation: %s\n",
	  (param_landscape == TRUE) ? "Landscape" : "Portrait");
  fprintf(output_fp, "%%%%EndComments\n");
  fprintf(output_fp, "%%%%BeginProlog\n");
  fprintf(output_fp,
    "\n"
    "/BP {\n"
    "  save\n"
    "  OffsetX OffsetY translate\n"
    "  MarginLeft MarginTop\n"
    "  Landscape { exch translate 90 rotate }\n"
    "            { PaperHeight exch sub translate}\n"
    "    ifelse\n"
    "  72.0 Resolution div dup neg scale\n"
    "  0 0 moveto\n"
    "} bind def\n"
    "/EP {\n"
    "  showpage\n"
    "  restore\n"
    "} bind def\n"
    "/START {\n"
    "  /Resolution exch def\n"
    "} bind def\n"
    "/END {} bind def\n");
  fprintf(output_fp, "%%%%EndProlog\n");
  fprintf(output_fp, "%%%%BeginSetup\n");
  fprintf(output_fp, "%%%%PaperSize: %s\n",
	  paper_list[param_paper_id].name);
  fprintf(output_fp, "/PaperWidth  %d def\n", 
	  (int)ceil(paper_list[param_paper_id].w * 72.27));
  fprintf(output_fp, "/PaperHeight %d def\n",  
	  (int)ceil(paper_list[param_paper_id].h * 72.27));
  fprintf(output_fp, "/OffsetX %f def\n", 0.0);
  fprintf(output_fp, "/OffsetY %f def\n", 0.0);
  fprintf(output_fp, "/MarginLeft  %f def\n", 0.0);
  fprintf(output_fp, "/MarginTop %f def\n",   0.0);
  fprintf(output_fp, "/Landscape %s def\n", 
	  param_landscape ? "true" : "false");
  fprintf(output_fp, "/Resolution %d def\n", param_dpi);
  fprintf(output_fp, "%%%%EndSetup\n");
  fprintf(output_fp, "%d START\n", param_dpi);
}

static void  
ps_epilogue(DVI dvi,DVI_DEVICE dev)
{
  fprintf(output_fp, "%%%%Trailer\n");
  fprintf(output_fp, "END\n");
  fprintf(output_fp, "%%%%EOF\n");
  fflush(output_fp);
}

static void  
ps_begin_page(DVI dvi, DVI_DEVICE dev, int seq_page)
{
  DVI_fb_clear(framebuff);

  fprintf(output_fp, "%%%%Page: %ld %d\n", 
	  dvi->page_no[seq_page-1].c[0], seq_page);
  fprintf(output_fp, "%%%%BeginPageSetup\n");
  fprintf(output_fp, "BP\n");
  fprintf(output_fp, "%%%%EndPageSetup\n");
  fflush(output_fp);
}

static void  
ps_end_page(DVI dvi, DVI_DEVICE dev)
{
  ps_frame_shipout();

  fprintf(output_fp, "%%%%PageTrailer\n");
  fprintf(output_fp, "EP\n");
  fprintf(output_fp, "\n");
  fflush(output_fp);
}

static void
ps_frame_shipout(void)
{
  int      br;
  long     y, y0 = 0, y1, xw0, xw1;
  unsigned char  *p;

  br = 0;
  for (xw0 = 0; xw0 < framebuff->w_bytes; xw0++){
    p = &framebuff->data[xw0];
    for (y = 0; y < framebuff->height; y++){
      if (*p != 0x00){
	br = 1;
	break;
      }
      p = p + framebuff->raster;
    }
    if (br == 1)
      break;
  }
  if (xw0 >= framebuff->w_bytes)
    return;

  br = 0;
  for (xw1 = framebuff->w_bytes - 1; xw1 >= 0; --xw1){
    p = &framebuff->data[xw1];
    for (y = 0; y < framebuff->height; y++){
      if (*p != 0x00){
	br = 1;
	break;
      }
      p = p + framebuff->raster;
    }
    if (br == 1)
      break;
  }
  if (xw1 < 0)
    return;

  y1 = -1;
  for (;;){
    for (y = y1+1; y < framebuff->height; y++){
      if (DVI_fb_raster_nonzero(framebuff, y) == 1){
	y0 = y;
	break;
      }
    }
    if (y >= framebuff->height)
      break;
    for (y = y0+1; y < framebuff->height; y++){
      if (DVI_fb_raster_zero(framebuff, y) == 1){
	y1 = y-1;
	break;
      }
    }
    if (y >= framebuff->height)
      y1 = framebuff->height-1;

    ps_shipout_range(y0, y1, xw0, xw1);
  }
}

static void
ps_shipout_range(long y0, long y1, long xw0, long xw1)
{
  int    n;
  long   x, y, w, h;
  unsigned char  *fp;

  w = (xw1 - xw0 + 1) * 8;
  h = y1 - y0 + 1; 

  fprintf(output_fp, "%% Range: (%ld,%ld)-(%ld,%ld) in [%ld,%ld] \n", 
	  xw0*8, y0, xw1*8, y1, 
	  framebuff->width, framebuff->height);
  fprintf(output_fp, "gsave\n");
  fprintf(output_fp, "%ld %ld translate\n", xw0*8, y0);
  fprintf(output_fp, "%ld %ld scale\n", w, h);
  fprintf(output_fp, "/pictstr %ld string def\n", w/8L);
  fprintf(output_fp, "%ld %ld 1 [ %ld 0 0 %ld 0 0 ]\n", w, h, w, h);
  fprintf(output_fp, "{ currentfile pictstr readhexstring pop } image\n");

  n = 0;
  for (y = y0; y <= y1; y++){
    fp = DVI_fb_raster_pointer(framebuff, y);
    fp = &fp[xw0];
    for (x = xw0; x <= xw1; x++){
      fprintf(output_fp, "%02x", 0xff - (*fp));
      fp++;
      if (++n >= 30){
	fprintf(output_fp, "\n");
	n = 0;
      }
    }
    if (n != 0){
      fprintf(output_fp, "\n");
      n = 0;
    }
  }
  if (n != 0)
    fprintf(output_fp, "\n");

  fprintf(output_fp, "grestore\n");
  fprintf(output_fp, "\n");
  fflush(output_fp);
}

/*EOF*/
