/*
 * Telepathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 *
 * ti-wnd-main.c:
 * Main Window - Main application window.
 *
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Copyright (C) 2007-2008 Collabora Ltd. <http://www.collabora.co.uk/>
 * Copyright (C) 2007-2008 Nokia Corporation
 *
 * Originally by Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include <stdlib.h>
#include <string.h>

#include <gdk/gdkkeysyms.h>
#include <glade/glade.h>
#include <gtk/gtk.h>
#include <telepathy-glib/connection-manager.h>
#include <telepathy-glib/util.h>

#include "wnd-main.h"
#include "dlg-about.h"
#include "dlg-preferences.h"
#include "dlg-request-conn.h"
#include "wnd-connection.h"
#include "page-active-conns.h"
#include "config.h"
#include "util.h"

struct _TIWndMainClass {
    GObjectClass parent;
};

G_DEFINE_TYPE (TIWndMain, ti_wnd_main, G_TYPE_OBJECT);

/* Function prototypes */
static void _ti_wnd_main_setup_conn_man_page (TIWndMain *wnd_main);
static void _ti_wnd_main_refresh (TIWndMain *wnd_main);
static void _ti_wnd_main_build_protocols_treeview (TIWndMain *wnd_main);
static void _ti_wnd_main_fill_protocols_list (TIWndMain *wnd_main);
static void _ti_wnd_main_build_prot_params_treeview (TIWndMain *wnd_main);
static void _ti_wnd_main_fill_prot_params_list (TIWndMain *wnd_main);
static void _ti_wnd_main_conn_manager_combo_changed (TIWndMain *wnd_main);
static void _ti_wnd_main_request_connection (TIWndMain *wnd_main);
static gchar *_ti_wnd_main_get_selected_protocol_name (TIWndMain *wnd_main);
static void _ti_wnd_main_on_conn_window_closed (TIWndMain *wnd_main,
    TIWndConnection *wnd_connection);
static void _ti_wnd_main_add_prot_param (TIWndMain *wnd_main,
    const gchar *name, guint flags, const gchar *type_signature,
    const GValue *default_value);
static void _ti_wnd_main_protocol_selection_changed (TIWndMain *wnd_main);
static void _ti_wnd_main_show_about_dialog (TIWndMain *wnd_main);
static void _ti_wnd_main_show_preferences_dialog (TIWndMain *wnd_main);


/**
 * Instance private data.
 */
struct _TIWndMainPrivate {
    gboolean disposed;

    TpDBusDaemon *dbus_daemon;
    TpConnectionManager *conn_manager;

    GladeXML *glade_xml;

    GtkWidget *window;
    GtkNotebook *notebook;

    TIPageActiveConns *page_active_conns;

    TIDlgAbout *dlg_about;
    TIDlgRequestConn *dlg_request_conn;
    TIDlgPreferences *dlg_preferences;

    GtkListStore *protocols_list;
    GtkListStore *prot_params_list;
    GtkListStore *conn_managers_list;

    GtkWidget *combo_conn_man;
    GtkWidget *button_refresh;
    GtkWidget *button_request_conn;

    GtkTreeSelection *protocol_selection;

    /* An array of TIWndConnection *
       containing all connection windows currently open. */
    GPtrArray *connection_wnds;

};
typedef struct _TIWndMainPrivate TIWndMainPrivate;

enum {
    CM_LIST_COLUMN_NAME = 0,
    CM_LIST_COLUMN_PROXY,
    CM_LIST_N_COLUMNS
};

#define TI_WND_MAIN_GET_PRIVATE(object) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((object), TI_TYPE_WND_MAIN, TIWndMainPrivate))

/**
 * Drop all references to other objects.
 */
static void
ti_wnd_main_dispose (GObject *object)
{
  TIWndMain *wnd_main = TI_WND_MAIN (object);
  TIWndMainPrivate *priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);
  guint i;
  GObject *obj = NULL;

  if (priv->disposed)
    {
      return;
    }

  priv->disposed = TRUE;

  if (priv->dbus_daemon != NULL)
    {
      g_object_unref (priv->dbus_daemon);
      priv->dbus_daemon = NULL;
    }

  if (priv->conn_manager != NULL)
    {
      g_object_unref (priv->conn_manager);
      priv->conn_manager = NULL;
    }

  if (priv->glade_xml != NULL)
    {
      g_object_unref (priv->glade_xml);
      priv->glade_xml = NULL;
    }

  if (priv->page_active_conns != NULL)
    {
      g_object_unref (priv->page_active_conns);
      priv->page_active_conns = NULL;
    }

  if (priv->dlg_about != NULL)
    {
      g_object_unref (priv->dlg_about);
      priv->dlg_about = NULL;
    }

  if (priv->dlg_request_conn != NULL)
    {
      g_object_unref (priv->dlg_request_conn);
      priv->dlg_request_conn = NULL;
    }

  if (priv->dlg_preferences != NULL)
    {
      g_object_unref (priv->dlg_preferences);
      priv->dlg_preferences = NULL;
    }

  if (priv->connection_wnds != NULL)
    {

      for (i = 0; i < priv->connection_wnds->len; i++)
        {
          obj = G_OBJECT (g_ptr_array_index (priv->connection_wnds, i));
          g_object_unref (obj);
        }

      g_ptr_array_free (priv->connection_wnds, TRUE);
      priv->connection_wnds = NULL;
    }

  G_OBJECT_CLASS (ti_wnd_main_parent_class)->dispose (object);
}

/**
 * Class initialization.
 */
static void
ti_wnd_main_class_init (TIWndMainClass *ti_wnd_main_class)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (ti_wnd_main_class);

  /* override base object methods */
  gobject_class->dispose = ti_wnd_main_dispose;

  /* Add private */
  g_type_class_add_private (ti_wnd_main_class, sizeof (TIWndMainPrivate));
}

/**
 * Instance initialization.
 */
static void
ti_wnd_main_init (TIWndMain *ti_wnd_main)
{
  TIWndMainPrivate *priv = TI_WND_MAIN_GET_PRIVATE (ti_wnd_main);

  priv->disposed = FALSE;
  priv->glade_xml = NULL;
  priv->conn_manager = NULL;
  priv->dlg_about = NULL;
  priv->dlg_request_conn = NULL;
  priv->dlg_preferences = NULL;
}

/**
 * Returns a new instance.
 */
TIWndMain *
ti_wnd_main_new ()
{
  TIWndMain *wnd_main = NULL;
  TIWndMainPrivate *priv = NULL;
  gchar *glade_file_path = NULL;

  wnd_main = g_object_new (TI_TYPE_WND_MAIN, NULL);
  priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);

  priv->dbus_daemon = tp_dbus_daemon_new (tp_get_bus ());

  /* Set up windows, dialogs, etc. */

  glade_file_path = ti_get_glade_path ("wnd-main.xml");
  priv->glade_xml = glade_xml_new (glade_file_path, NULL, NULL);
  if (priv->glade_xml == NULL)
    {
      g_critical ("Error loading glade file \"%s\".", glade_file_path);
      g_object_unref (wnd_main);
      wnd_main = NULL;
      goto CLEAN_UP;
    }

  priv->window = glade_xml_get_widget (priv->glade_xml, "main_window");
  g_assert (GTK_IS_WINDOW (priv->window));

  priv->notebook = GTK_NOTEBOOK (glade_xml_get_widget (priv->glade_xml,
        "notebook"));
  g_assert (GTK_IS_NOTEBOOK (priv->notebook));

  priv->page_active_conns = ti_page_active_conns_new
      (GTK_WINDOW (priv->window), priv->notebook, priv->dbus_daemon);

  priv->dlg_about = ti_dlg_about_new (GTK_WINDOW (priv->window));
  priv->dlg_request_conn = ti_dlg_request_conn_new (GTK_WINDOW (priv->window));
  priv->dlg_preferences = ti_dlg_preferences_new (GTK_WINDOW (priv->window));

  priv->connection_wnds = g_ptr_array_sized_new (5);

  _ti_wnd_main_setup_conn_man_page (wnd_main);

CLEAN_UP:
  g_free (glade_file_path);

  return wnd_main;
}

/**
 * Show
 */
void
ti_wnd_main_show (TIWndMain *wnd_main)
{
  TIWndMainPrivate *priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);

  gtk_widget_show_all (priv->window);
}

static void
got_cm_info_cb (TpConnectionManager *cm,
                guint source,
                TIWndMain *self)
{
  TIWndMainPrivate *priv = TI_WND_MAIN_GET_PRIVATE (self);

  if (priv->conn_manager == cm && source > TP_CM_INFO_SOURCE_NONE)
    {
      _ti_wnd_main_fill_protocols_list (self);
    }
}

static void
got_connection_managers (TpConnectionManager * const *cms,
                         gsize n,
                         const GError *error,
                         gpointer unused,
                         GObject *weak_object)
{
  TIWndMain *wnd_main = TI_WND_MAIN (weak_object);
  TIWndMainPrivate *priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);

  gtk_widget_set_sensitive (priv->button_refresh, TRUE);

  if (error != NULL)
    {
      g_warning ("%s", error->message);
    }
  else
    {
      TpConnectionManager * const *iter = cms;

      gtk_list_store_clear (priv->conn_managers_list);

      for (iter = cms; *iter != NULL; iter++)
        {
          gchar *name;
          GtkTreeIter tree_iter;

          g_object_get (*iter,
              "connection-manager", &name,
              NULL);

          gtk_list_store_append (priv->conn_managers_list, &tree_iter);
          gtk_list_store_set (priv->conn_managers_list, &tree_iter,
              CM_LIST_COLUMN_NAME, name,
              CM_LIST_COLUMN_PROXY, g_object_ref (*iter),
              -1);

          g_free (name);

          g_signal_connect (*iter, "got-info", G_CALLBACK (got_cm_info_cb),
              wnd_main);
        }
    }
}

/**
 * Set up Connection Managers Page
 */
static void
_ti_wnd_main_setup_conn_man_page (TIWndMain *wnd_main)
{
    TIWndMainPrivate *priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);
    GtkWidget *widget = NULL;

    /* Setup page label */
    widget = gtk_notebook_get_nth_page (priv->notebook, 0);
    gtk_notebook_set_tab_label_text (priv->notebook, widget,
        "Connection Managers");

    /* "Refresh" button. */
    priv->button_refresh = glade_xml_get_widget (priv->glade_xml,
        "button_refresh");
    g_assert (GTK_IS_BUTTON (priv->button_refresh));
    g_signal_connect_swapped (priv->button_refresh, "clicked",
        G_CALLBACK (_ti_wnd_main_refresh), wnd_main);

    /* ComboEntry "Connection Manager" */
    priv->combo_conn_man = glade_xml_get_widget (priv->glade_xml,
        "combo_conn_man");
    g_assert (GTK_IS_COMBO_BOX (priv->combo_conn_man));
    g_signal_connect_swapped (GTK_COMBO_BOX (priv->combo_conn_man),
        "changed", G_CALLBACK (_ti_wnd_main_conn_manager_combo_changed),
        wnd_main);

    priv->conn_managers_list = gtk_list_store_new (2, G_TYPE_STRING,
        TP_TYPE_CONNECTION_MANAGER);
    gtk_combo_box_set_model (GTK_COMBO_BOX (priv->combo_conn_man),
        GTK_TREE_MODEL (priv->conn_managers_list) );

    _ti_wnd_main_refresh (wnd_main);

    /* Main Window */
    glade_xml_signal_connect (priv->glade_xml, "delete_main_window",
        G_CALLBACK(gtk_main_quit));

    /* Protocols' list */
    _ti_wnd_main_build_protocols_treeview (wnd_main);

    /* Protocol parameters' list */
    _ti_wnd_main_build_prot_params_treeview (wnd_main);

    /* "Request Connection" button */
    priv->button_request_conn = glade_xml_get_widget (priv->glade_xml,
        "button_request_conn");
    g_assert (GTK_IS_BUTTON (priv->button_request_conn));
    g_signal_connect_swapped (priv->button_request_conn, "clicked",
        G_CALLBACK (_ti_wnd_main_request_connection), wnd_main);

    /* "About" menu item */
    widget = glade_xml_get_widget (priv->glade_xml, "menuitem_about");
    g_assert (GTK_IS_MENU_ITEM (widget));
    g_signal_connect_swapped (widget, "activate",
        G_CALLBACK (_ti_wnd_main_show_about_dialog), wnd_main);

    /* "Preferences" menu item */
    widget = glade_xml_get_widget (priv->glade_xml, "menuitem_preferences");
    g_assert (GTK_IS_MENU_ITEM (widget));
    g_signal_connect_swapped (widget, "activate",
        G_CALLBACK (_ti_wnd_main_show_preferences_dialog), wnd_main);
}

static void
_ti_wnd_main_refresh (TIWndMain *wnd_main)
{
  TIWndMainPrivate *priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);

  gtk_widget_set_sensitive (priv->button_refresh, FALSE);

  tp_list_connection_managers (priv->dbus_daemon,
      got_connection_managers, NULL, NULL, (GObject *) wnd_main);
}

/**
 * Fill Protocols' List
 */
static void
_ti_wnd_main_fill_protocols_list (TIWndMain *wnd_main)
{
  TIWndMainPrivate *priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);
  const TpConnectionManagerProtocol * const *iter;
  GtkTreeIter tree_iter;

  gtk_list_store_clear (priv->protocols_list);

  if (priv->conn_manager == NULL)
    return;

  for (iter = priv->conn_manager->protocols; *iter != NULL; iter++)
    {
      gtk_list_store_append (priv->protocols_list, &tree_iter);
      gtk_list_store_set (priv->protocols_list, &tree_iter,
          0, (*iter)->name,
          -1);
    }
}

/**
 * Build Protocols Treeview
 */
static void
_ti_wnd_main_build_protocols_treeview (TIWndMain *wnd_main)
{
  TIWndMainPrivate *priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);
  GtkWidget *treeview_protocols;
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;

  treeview_protocols = glade_xml_get_widget (priv->glade_xml,
      "treeview_protocols");
  g_assert (GTK_IS_TREE_VIEW (treeview_protocols));

  priv->protocols_list = gtk_list_store_new (1, G_TYPE_STRING);
  gtk_tree_view_set_model (GTK_TREE_VIEW (treeview_protocols),
      GTK_TREE_MODEL (priv->protocols_list));

  renderer = gtk_cell_renderer_text_new ();
  column = gtk_tree_view_column_new_with_attributes ("Protocol", renderer,
      "text", 0,
      NULL);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview_protocols), column);

  priv->protocol_selection = gtk_tree_view_get_selection
      (GTK_TREE_VIEW (treeview_protocols));
  gtk_tree_selection_set_mode (priv->protocol_selection, GTK_SELECTION_SINGLE);

  g_signal_connect_swapped (priv->protocol_selection, "changed",
      G_CALLBACK (_ti_wnd_main_protocol_selection_changed), wnd_main);
}

/**
 * Build the GtkTreeView for the parameters of the selected protocol.
 */
static void
_ti_wnd_main_build_prot_params_treeview (TIWndMain *wnd_main)
{
  TIWndMainPrivate *priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);
  GtkWidget *treeview_prot_params;
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;
  GtkTreeSelection *tree_selection;

  treeview_prot_params = glade_xml_get_widget (priv->glade_xml,
      "treeview_prot_params");

  priv->prot_params_list = gtk_list_store_new (4, G_TYPE_STRING,
      G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING);
  gtk_tree_view_set_model (GTK_TREE_VIEW (treeview_prot_params),
      GTK_TREE_MODEL (priv->prot_params_list));

  renderer = gtk_cell_renderer_text_new ();

  column = gtk_tree_view_column_new_with_attributes ("Name", renderer,
      "text", 0,
      NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview_prot_params), column);

  column = gtk_tree_view_column_new_with_attributes ("Flags", renderer,
      "text", 1,
      NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview_prot_params), column);

  column = gtk_tree_view_column_new_with_attributes ("D-Bus Type", renderer,
      "text", 2,
      NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview_prot_params), column);

  column = gtk_tree_view_column_new_with_attributes ("Default Value", renderer,
      "text", 3,
      NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview_prot_params), column);

  tree_selection = gtk_tree_view_get_selection
      (GTK_TREE_VIEW (treeview_prot_params));
  gtk_tree_selection_set_mode (tree_selection, GTK_SELECTION_NONE);
}

/**
 * Fill protocol parameters list for the given protocol name.
 */
static void
_ti_wnd_main_fill_prot_params_list (TIWndMain *wnd_main)
{
  TIWndMainPrivate *priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);
  const TpConnectionManagerProtocol * const *proto_iter;
  gchar *protocol_name;

  gtk_list_store_clear (priv->prot_params_list);
  gtk_widget_set_sensitive (priv->button_request_conn, FALSE);

  if (priv->conn_manager == NULL)
    return;

  protocol_name = _ti_wnd_main_get_selected_protocol_name (wnd_main);

  if (protocol_name == NULL)
    return;

  for (proto_iter = priv->conn_manager->protocols;
       proto_iter != NULL && *proto_iter != NULL;
       proto_iter++)
    {
      if (!tp_strdiff ((*proto_iter)->name, protocol_name))
        {
          const TpConnectionManagerParam *param = (*proto_iter)->params;

          for (param = (*proto_iter)->params; param->name != NULL; param++)
            {
              _ti_wnd_main_add_prot_param (wnd_main, param->name,
                  param->flags, param->dbus_signature,
                  &(param->default_value));
            }

          gtk_widget_set_sensitive (priv->button_request_conn, TRUE);
        }
    }

  g_free (protocol_name);
}

static void
_ti_wnd_main_conn_manager_combo_changed (TIWndMain *wnd_main)
{
  TIWndMainPrivate *priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);
  GtkTreeIter iter;

  if (gtk_combo_box_get_active_iter (GTK_COMBO_BOX (priv->combo_conn_man),
        &iter))
    {
      GValue val = { 0 };

      gtk_tree_model_get_value (GTK_TREE_MODEL (priv->conn_managers_list),
          &iter, CM_LIST_COLUMN_PROXY, &val);

      if (priv->conn_manager != g_value_get_object (&val))
        {
          if (priv->conn_manager != NULL)
            g_object_unref (priv->conn_manager);

          priv->conn_manager = g_value_dup_object (&val);

          _ti_wnd_main_fill_protocols_list (wnd_main);
        }
    }
  else
    {
      /* clear the protocols list */
      if (priv->conn_manager != NULL)
        g_object_unref (priv->conn_manager);

      priv->conn_manager = NULL;

      _ti_wnd_main_fill_protocols_list (wnd_main);
    }
}

/**
 * Request Connection
 */
static void
_ti_wnd_main_request_connection (TIWndMain *wnd_main)
{
  TIWndMainPrivate *priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);
  gchar *protocol_name = NULL;
  GHashTable *param_values = NULL;
  TpProxy *cm_as_proxy = TP_PROXY (priv->conn_manager);
  TpConnection *connection = NULL;
  TIWndConnection *connection_window = NULL;
  GError *error = NULL;
  gchar *service_name;
  gchar *obj_path;

  protocol_name = _ti_wnd_main_get_selected_protocol_name (wnd_main);

  if (!ti_dlg_request_conn_run (priv->dlg_request_conn, protocol_name,
        GTK_TREE_MODEL (priv->prot_params_list), &param_values))
    goto CLEAN_UP;

  if (!tp_cli_connection_manager_run_request_connection
        (priv->conn_manager, -1, protocol_name, param_values,
        &service_name, &obj_path, &error, NULL))
    {
      g_printerr ("Error: %s\n", error->message);
      g_error_free (error);
      goto CLEAN_UP;
    }

  connection = tp_connection_new (cm_as_proxy->dbus_daemon,
      service_name, obj_path, &error);

  g_free (service_name);
  g_free (obj_path);

  if (connection == NULL)
    {
      g_printerr ("Error: %s\n", error->message);
      g_error_free (error);
      goto CLEAN_UP;
    }

  connection_window = ti_wnd_connection_new (GTK_WINDOW (priv->window),
      connection);

  g_object_unref (connection);

  if (connection_window == NULL)
    goto CLEAN_UP;

  g_ptr_array_add (priv->connection_wnds, connection_window);
  g_signal_connect_swapped (connection_window, "closed",
      G_CALLBACK (_ti_wnd_main_on_conn_window_closed), wnd_main);
  ti_wnd_connection_show (connection_window);

CLEAN_UP:

  if (param_values != NULL)
    {
      g_hash_table_destroy (param_values);
    }

  g_free (protocol_name);
}

/**
 * Returns the name of the currently selected protocol or NULL if no protocol
 * is being selected.
 *
 * The returned string must be freed after use.
 */
static gchar*
_ti_wnd_main_get_selected_protocol_name (TIWndMain *wnd_main)
{
  TIWndMainPrivate *priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);
  gboolean ok;
  GtkTreeModel *tree_model;
  GtkTreeIter iter;
  GValue value = {0, };
  const gchar *protocol_name;
  gchar *ret_val = NULL;

  ok = gtk_tree_selection_get_selected (priv->protocol_selection, &tree_model,
      &iter);
  if (!ok)
    goto CLEAN_UP;

  gtk_tree_model_get_value (tree_model, &iter, 0, &value);
  g_assert (G_VALUE_HOLDS (&value, G_TYPE_STRING));
  if (!G_VALUE_HOLDS (&value, G_TYPE_STRING))
      goto CLEAN_UP;

  protocol_name = g_value_get_string (&value);
  ret_val = g_strdup (protocol_name);

CLEAN_UP:
  g_value_unset (&value);

  return ret_val;
}

/**
 * On Connection Window Closed
 * Called when a connection window is closed, i.e., when a TIWndConnection
 * emits a "closed" signal.
 */
static void
_ti_wnd_main_on_conn_window_closed (TIWndMain *wnd_main,
                                    TIWndConnection *wnd_connection)
{
  TIWndMainPrivate *priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);
  gboolean ok;

  ok = g_ptr_array_remove (priv->connection_wnds, wnd_connection);
  g_assert (ok);

  g_object_unref (wnd_connection);
}

/**
 * Add Protocol Parameter
 * Adds a protocol parameter to protocol parameters' list.
 */
static void
_ti_wnd_main_add_prot_param (TIWndMain *wnd_main,
                             const gchar *name,
                             guint flags,
                             const gchar *type_signature,
                             const GValue *default_value)
{
  TIWndMainPrivate *priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);
  GtkTreeIter iter;
  gchar *flags_str;
  gchar *def_val_str;

  flags_str = g_strdup_printf ("%u", flags);

  def_val_str = ti_value_to_string (default_value);

  gtk_list_store_append (priv->prot_params_list, &iter);
  gtk_list_store_set (priv->prot_params_list, &iter,
      0, name,
      1, flags_str,
      2, type_signature,
      3, def_val_str,
      -1);

  g_free (flags_str);
  g_free (def_val_str);
}

/**
 * Protocol Selection Changed
 */
static void
_ti_wnd_main_protocol_selection_changed (TIWndMain *wnd_main)
{
  _ti_wnd_main_fill_prot_params_list (wnd_main);
}

/**
 * Show Preferences Dialog
 */
static void
_ti_wnd_main_show_preferences_dialog (TIWndMain *wnd_main)
{
  TIWndMainPrivate *priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);

  ti_dlg_preferences_run (priv->dlg_preferences);
}

/**
 * Show About Dialog
 */
static void
_ti_wnd_main_show_about_dialog (TIWndMain *wnd_main)
{
    TIWndMainPrivate *priv = TI_WND_MAIN_GET_PRIVATE (wnd_main);

    ti_dlg_about_run (priv->dlg_about);
}
