/*
 * Telepathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 *
 * ti-wnd-channel.c:
 * Channel Window - UI to expose org.freedesktop.Telepathy.Channel functionality
 *
 * Copyright (C) 2008 Collabora Ltd. <http://www.collabora.co.uk/>
 * Copyright (C) 2008 Nokia Corporation
 * Copyright (C) 2006-2007 INdT - Instituto Nokia de Tecnologia
 * Originally by Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "wnd-channel.h"

#include <string.h>

#include <glade/glade.h>
#include <telepathy-glib/enums.h>
#include <telepathy-glib/interfaces.h>

#include "constants.h"
#include "extensions/extensions.h"
#include "page-channel-group.h"
#include "page-channel-text.h"
#include "page-hold.h"
#include "page-media-signalling.h"
#include "page-properties.h"
#include "page-room-list.h"
#include "page-streamed-media.h"
#include "preferences.h"
#include "util.h"

struct _TIWndChannelClass {
    GObjectClass parent;
    /* class members */
    guint close_wnd_id;
};

G_DEFINE_TYPE (TIWndChannel, ti_wnd_channel, G_TYPE_OBJECT);

static gboolean on_delete_window (TIWndChannel *, GdkEvent *);
static void _ti_wnd_channel_setup_first_page (TIWndChannel *);
static void _ti_wnd_channel_build_interfaces_treeview (TIWndChannel *);
static void _ti_wnd_channel_close_channel (TIWndChannel *);
static void _ti_wnd_channel_add_interface_pages (TIWndChannel *);
static void _ti_wnd_channel_add_interface_page (TIWndChannel *,
    const gchar *interface_name);
static gchar *_get_handle_type_str (guint handle_type);
static void _ti_wnd_channel_close_window (TIWndChannel *);
static void _ti_wnd_channel_setup_type_interface_page (TIWndChannel *);
static void _ti_wnd_channel_handle_display_mode_changed (TIWndChannel *,
    guint handle_display_mode);

/**
 * Instance private data.
 */
struct _TIWndChannelPrivate {
    TIHandleMapper *handle_mapper;
    TIPreferences *preferences;

    GtkWindow *parent;

    GladeXML *glade_xml;

    GtkWidget *window;
    GtkNotebook *notebook;

    guint self_handle_type;
    guint self_handle_number;

    GtkLabel *label_handle_number;
    GtkLabel *label_handle_number_value;
    GtkWidget *button_close;

    gchar *channel_type;
    gchar **interfaces;
    GtkListStore *interfaces_list;

    TpChannel *channel;

    TIPageChannelGroup *page_group;
    TIPageRoomList *page_room_list;
    TIPageMediaSignalling *page_media_signalling;
    TIPageChannelText *page_text;
    TIPageStreamedMedia *page_streamed_media;
    TIPageProperties *page_properties;
#ifdef ENABLE_HOLD
    TIPageHold *page_hold;
#endif

    gboolean should_close_window;
    gboolean emitted_close;
};
typedef struct _TIWndChannelPrivate TIWndChannelPrivate;

#define TI_WND_CHANNEL_GET_PRIVATE(object) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((object), TI_TYPE_WND_CHANNEL, \
                                TIWndChannelPrivate))

static void
ti_wnd_channel_dispose (GObject *object)
{
  TIWndChannel *wnd_channel = TI_WND_CHANNEL (object);
  TIWndChannelPrivate *priv = TI_WND_CHANNEL_GET_PRIVATE (wnd_channel);

  TI_OBJ_UNREF (priv->handle_mapper);

  if (priv->preferences != NULL)
    {
      g_signal_handlers_disconnect_by_func (priv->preferences,
          G_CALLBACK (_ti_wnd_channel_handle_display_mode_changed),
          wnd_channel);

      g_object_unref (priv->preferences);
      priv->preferences = NULL;
    }

  if (priv->window != NULL)
    {
      gtk_widget_hide_all (priv->window);
    }

  TI_OBJ_UNREF (priv->glade_xml);

  if (priv->channel != NULL)
    {
      g_signal_handlers_disconnect_by_func (priv->channel,
          _ti_wnd_channel_close_window, wnd_channel);

      TI_OBJ_UNREF (priv->channel);
    }

  TI_OBJ_UNREF (priv->page_group);
  TI_OBJ_UNREF (priv->page_room_list);
  TI_OBJ_UNREF (priv->page_text);
  TI_OBJ_UNREF (priv->page_properties);

  G_OBJECT_CLASS (ti_wnd_channel_parent_class)->dispose (object);
}

static void
ti_wnd_channel_finalize (GObject *object)
{
  TIWndChannel *self = TI_WND_CHANNEL (object);
  TIWndChannelPrivate *priv = TI_WND_CHANNEL_GET_PRIVATE (self);

  g_free (priv->channel_type);
  g_strfreev (priv->interfaces);

  G_OBJECT_CLASS (ti_wnd_channel_parent_class)->finalize (object);
}

static void
ti_wnd_channel_class_init (TIWndChannelClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

  gobject_class->dispose = ti_wnd_channel_dispose;
  gobject_class->finalize = ti_wnd_channel_finalize;

  g_type_class_add_private (klass, sizeof (TIWndChannelPrivate));

  klass->close_wnd_id = g_signal_newv ("close-wnd",
      G_TYPE_FROM_CLASS (klass),
      G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
      NULL /* class closure */,
      NULL /* accumulator */,
      NULL /* accu_data */,
      g_cclosure_marshal_VOID__VOID,
      G_TYPE_NONE /* return_type */,
      0     /* n_params */,
      NULL  /* param_types */);
}

static void
ti_wnd_channel_init (TIWndChannel *ti_wnd_channel)
{
  TIWndChannelPrivate *priv = TI_WND_CHANNEL_GET_PRIVATE (ti_wnd_channel);

  priv->self_handle_type = 0;
  priv->self_handle_number = 0;
  priv->handle_mapper = NULL;
  priv->preferences = ti_preferences_new ();
  priv->glade_xml = NULL;
  priv->window = NULL;
  priv->notebook = NULL;
  priv->button_close = NULL;
  priv->channel = NULL;
  priv->page_group = NULL;
  priv->page_room_list = NULL;
  priv->page_text = NULL;
  priv->page_properties = NULL;
  priv->should_close_window = FALSE;
}

/**
 * Returns a new instance.
 */
TIWndChannel *
ti_wnd_channel_new (GtkWindow *parent,
                    TpChannel *channel,
                    TIHandleMapper *handle_mapper)
{
  TIWndChannel *wnd_channel = NULL;
  TIWndChannelPrivate *priv = NULL;
  gchar *glade_file_path = NULL;

  g_assert (channel != NULL);
  g_assert (parent != NULL);

  wnd_channel = g_object_new (TI_TYPE_WND_CHANNEL, NULL);

  priv = TI_WND_CHANNEL_GET_PRIVATE (wnd_channel);
  priv->parent = parent;

  priv->channel = g_object_ref (channel);

  priv->handle_mapper = handle_mapper;
  g_object_ref (handle_mapper);

  glade_file_path = ti_get_glade_path ("wnd-channel.xml");

  priv->glade_xml = glade_xml_new (glade_file_path, NULL, NULL);

  if (priv->glade_xml == NULL)
    {
      g_critical ("Error loading glade file \"%s\".", glade_file_path);
      g_object_unref (wnd_channel);
      wnd_channel = NULL;
      goto CLEAN_UP;
    }

  priv->window = glade_xml_get_widget (priv->glade_xml, "window_channel");
  g_assert (GTK_IS_WINDOW (priv->window));

  priv->notebook = GTK_NOTEBOOK (glade_xml_get_widget (priv->glade_xml,
        "notebook"));
  g_assert (GTK_IS_NOTEBOOK (priv->notebook));

  /* FIXME: connect this to a wrapper function that will log the closure
   * reason? */
  g_signal_connect_swapped (priv->channel, "invalidated",
      G_CALLBACK (_ti_wnd_channel_close_window), wnd_channel);

  g_signal_connect_swapped (priv->window, "delete-event",
      G_CALLBACK (on_delete_window), wnd_channel);

  gtk_window_set_transient_for (GTK_WINDOW (priv->window),
      GTK_WINDOW (priv->parent));

  g_signal_connect_swapped (priv->preferences, "handle-display-mode-changed",
      G_CALLBACK (_ti_wnd_channel_handle_display_mode_changed), wnd_channel);

  _ti_wnd_channel_setup_first_page (wnd_channel);
  _ti_wnd_channel_setup_type_interface_page (wnd_channel);
  _ti_wnd_channel_add_interface_pages (wnd_channel);

CLEAN_UP:
  g_free (glade_file_path);

  return wnd_channel;
}

static void
_ti_wnd_channel_setup_first_page (TIWndChannel *wnd_channel)
{
  TIWndChannelPrivate *priv = TI_WND_CHANNEL_GET_PRIVATE (wnd_channel);
  GtkWidget *first_page;
  GtkLabel *label;
  gchar *channel_type;
  gchar *str;
  gchar **interfaces, **iter;
  GtkTreeIter tree_iter;

  g_object_get (priv->channel,
      "channel-type", &channel_type,
      "handle-type", &priv->self_handle_type,
      "handle", &priv->self_handle_number,
      "interfaces", &interfaces,
      NULL);

  first_page = gtk_notebook_get_nth_page (priv->notebook, 0);
  gtk_notebook_set_tab_label_text (priv->notebook, first_page, "Channel");

  label = GTK_LABEL (glade_xml_get_widget (priv->glade_xml,
        "label_channel_type_value"));
  gtk_label_set_text (label, channel_type);
  g_free (priv->channel_type);
  priv->channel_type = channel_type;

  label = GTK_LABEL (glade_xml_get_widget (priv->glade_xml,
        "label_handle_type_value"));
  str = _get_handle_type_str (priv->self_handle_type);
  gtk_label_set_text (label, str);
  g_free (str);

  priv->label_handle_number = GTK_LABEL (glade_xml_get_widget (priv->glade_xml,
        "label_handle_number"));
  g_assert (priv->label_handle_number != NULL &&
      GTK_IS_LABEL (priv->label_handle_number));

  priv->label_handle_number_value = GTK_LABEL (glade_xml_get_widget (
        priv->glade_xml, "label_handle_number_value"));
  g_assert (priv->label_handle_number_value != NULL &&
      GTK_IS_LABEL (priv->label_handle_number_value));

  if (priv->self_handle_type != TP_HANDLE_TYPE_CONTACT ||
      ti_preferences_get_handle_display_mode (priv->preferences) ==
          TI_PREFERENCES_HANDLE_DISPLAY_HANDLE)
    {
      str = g_strdup_printf ("%u", priv->self_handle_number);
      gtk_label_set_text (priv->label_handle_number_value, str);
    }
  else
    {
      gtk_label_set_text (priv->label_handle_number, "Name:");
      str = ti_handle_mapper_get_handle_name (priv->handle_mapper,
          priv->self_handle_type, priv->self_handle_number);
      gtk_label_set_text (priv->label_handle_number_value, str);
    }
  g_free (str);

  priv->interfaces = interfaces;
  priv->interfaces_list = gtk_list_store_new (1, G_TYPE_STRING);
  _ti_wnd_channel_build_interfaces_treeview (wnd_channel);

  gtk_list_store_clear (priv->interfaces_list);
  for (iter = interfaces; iter != NULL && *iter != NULL; iter++)
    {
      gtk_list_store_append (priv->interfaces_list, &tree_iter);
      gtk_list_store_set (priv->interfaces_list, &tree_iter,
          0, *iter,
          -1);
    }

  priv->button_close = glade_xml_get_widget (priv->glade_xml, "button_close");
  g_assert (GTK_IS_BUTTON (priv->button_close));
  g_signal_connect_swapped (priv->button_close, "clicked",
      G_CALLBACK (_ti_wnd_channel_close_channel), wnd_channel);
}

static void
_ti_wnd_channel_close_channel (TIWndChannel *self)
{
  TIWndChannelPrivate *priv = TI_WND_CHANNEL_GET_PRIVATE (self);
  GError *error = NULL;

  if (!tp_cli_channel_run_close (priv->channel, -1, &error, NULL))
    {
      g_printerr ("Close(): %s\n", error->message);
      g_error_free (error);
    }

  _ti_wnd_channel_close_window (self);
}

/**
 * Build the GtkTreeView for the interfaces list.
 */
static void
_ti_wnd_channel_build_interfaces_treeview (TIWndChannel *wnd_channel)
{
  TIWndChannelPrivate *priv = TI_WND_CHANNEL_GET_PRIVATE (wnd_channel);
  GtkWidget *treeview_interfaces;
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;
  GtkTreeSelection *tree_selection;

  treeview_interfaces = glade_xml_get_widget (priv->glade_xml,
      "treeview_interfaces");
  gtk_tree_view_set_model (GTK_TREE_VIEW (treeview_interfaces),
      GTK_TREE_MODEL (priv->interfaces_list));

  renderer = gtk_cell_renderer_text_new ();
  column = gtk_tree_view_column_new_with_attributes ("Interface", renderer,
      "text", 0,
      NULL);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview_interfaces), column);

  tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW
      (treeview_interfaces));
  gtk_tree_selection_set_mode (tree_selection, GTK_SELECTION_NONE);
}

void
ti_wnd_channel_show (TIWndChannel *wnd_channel)
{
  TIWndChannelPrivate *priv = TI_WND_CHANNEL_GET_PRIVATE (wnd_channel);

  gtk_widget_show_all (priv->window);
}

static gboolean
on_delete_window (TIWndChannel *self,
                  GdkEvent *event)
{
  /* Just close the window. If the user really wants to close the
   * *telepathy channel* he should click the "Close" *button* instead */
  _ti_wnd_channel_close_window (self);
  return TRUE;
}

static void
_ti_wnd_channel_add_interface_pages (TIWndChannel *self)
{
  TIWndChannelPrivate *priv = TI_WND_CHANNEL_GET_PRIVATE (self);
  gchar **iter;

  for (iter = priv->interfaces; iter != NULL && *iter != NULL; iter++)
    {
      _ti_wnd_channel_add_interface_page (self, *iter);
    }
}

static void
_ti_wnd_channel_add_interface_page (TIWndChannel *self,
                                    const gchar *interface_name)
{
  TIWndChannelPrivate *priv = TI_WND_CHANNEL_GET_PRIVATE (self);

  if (g_str_equal (interface_name, TP_IFACE_CHANNEL_INTERFACE_GROUP))
    {
      priv->page_group = ti_page_channel_group_new (priv->notebook,
          GTK_WINDOW (priv->window), priv->channel, priv->handle_mapper);
    }
  else if (g_str_equal (interface_name, TP_IFACE_PROPERTIES_INTERFACE))
    {
      priv->page_properties = ti_page_properties_new (priv->notebook,
          TP_PROXY (priv->channel));
    }
  else if (g_str_equal (interface_name,
        TP_IFACE_CHANNEL_INTERFACE_MEDIA_SIGNALLING))
    {
      priv->page_media_signalling = ti_page_media_signalling_new (
          priv->notebook, priv->channel);
    }
#ifdef ENABLE_HOLD
  else if (g_str_equal (interface_name, TI_IFACE_CHANNEL_INTERFACE_HOLD))
    {
      priv->page_hold = ti_page_hold_new (priv->notebook,
          priv->channel);
    }
#endif
  else
    {
      g_debug ("Unrecognized interface: \"%s\"", interface_name);
    }
}

static gchar *
_get_handle_type_str (guint handle_type)
{
  switch (handle_type)
    {
    case TP_HANDLE_TYPE_NONE:
      return g_strdup ("0 - HANDLE_TYPE_NONE");
      break;

    case TP_HANDLE_TYPE_CONTACT:
      return g_strdup ("1 - HANDLE_TYPE_CONTACT");
      break;

    case TP_HANDLE_TYPE_ROOM:
      return g_strdup ("2 - HANDLE_TYPE_ROOM");
      break;

    case TP_HANDLE_TYPE_LIST:
      return g_strdup ("3 - HANDLE_TYPE_LIST");
      break;

    default:
      return g_strdup_printf ("%u - Unknown type", handle_type);
    }
}

static void
_ti_wnd_channel_close_window (TIWndChannel *self)
{
  TIWndChannelPrivate *priv = TI_WND_CHANNEL_GET_PRIVATE (self);

  /* Make the connection window destroy us. */
  if (!priv->emitted_close)
    {
      priv->emitted_close = TRUE;
      gtk_widget_hide_all (priv->window);
      g_signal_emit (self, TI_WND_CHANNEL_GET_CLASS (self)->close_wnd_id, 0,
          NULL);
    }
}

/**
 * If the channel has an interface for its type (e.g. [...].Type.Text
 * interface for text channels), add it.
 */
static void
_ti_wnd_channel_setup_type_interface_page (TIWndChannel *wnd_channel)
{
  TIWndChannelPrivate *priv = TI_WND_CHANNEL_GET_PRIVATE (wnd_channel);

  if (g_str_equal (priv->channel_type, TP_IFACE_CHANNEL_TYPE_TEXT))
    {
      priv->page_text = ti_page_channel_text_new (priv->notebook,
          priv->channel);
    }
  else if (g_str_equal (priv->channel_type, TP_IFACE_CHANNEL_TYPE_ROOM_LIST))
    {
      priv->page_room_list = ti_page_room_list_new (priv->notebook,
          priv->channel, priv->handle_mapper);
    }
  else if (g_str_equal (priv->channel_type,
        TP_IFACE_CHANNEL_TYPE_STREAMED_MEDIA))
    {
      priv->page_streamed_media = ti_page_streamed_media_new (priv->notebook,
          priv->channel);
    }
}

static void
_ti_wnd_channel_handle_display_mode_changed (TIWndChannel *wnd_channel,
                                             guint handle_display_mode)
{
  TIWndChannelPrivate *priv = TI_WND_CHANNEL_GET_PRIVATE (wnd_channel);
  gchar *str = NULL;

  if (priv->self_handle_type != TP_HANDLE_TYPE_CONTACT)
    return;

  if (handle_display_mode == TI_PREFERENCES_HANDLE_DISPLAY_HANDLE)
    {
      gtk_label_set_text (priv->label_handle_number, "Number:");
      str = g_strdup_printf ("%u", priv->self_handle_number);
    }
  else
    {
      gtk_label_set_text (priv->label_handle_number, "Name:");
      str = ti_handle_mapper_get_handle_name (priv->handle_mapper,
          priv->self_handle_type, priv->self_handle_number);
    }

  gtk_label_set_text (priv->label_handle_number_value, str);
  g_free (str);
}
