/*
 * MediaSignalling page for channel notebook.
 *
 * Copyright (C) 2008 Collabora Ltd. <http://www.collabora.co.uk/>
 * Copyright (C) 2008 Nokia Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "config.h"

#include "page-media-signalling.h"

#include <glade/glade.h>

#include "page-priv.h"


#define STREAM_ENGINE_BUS_NAME "org.freedesktop.Telepathy.StreamEngine"
#define STREAM_ENGINE_OBJECT_PATH "/org/freedesktop/Telepathy/StreamEngine"
#define STREAM_ENGINE_INTERFACE STREAM_ENGINE_BUS_NAME
#define CHANNEL_HANDLER_INTERFACE "org.freedesktop.Telepathy.ChannelHandler"


struct _TIPageMediaSignallingClass {
    TIPageClass parent;
    gpointer priv;
};


struct _TIPageMediaSignallingPrivate {
    TpChannel *channel;
    GtkButton *se_button;
};


G_DEFINE_TYPE (TIPageMediaSignalling, ti_page_media_signalling, TI_TYPE_PAGE);

enum {
    PROP_CHANNEL = 1
};


static void
ti_page_media_signalling_init (TIPageMediaSignalling *self)
{
  self->priv = G_TYPE_INSTANCE_GET_PRIVATE (self,
      TI_TYPE_PAGE_MEDIA_SIGNALLING, TIPageMediaSignallingPrivate);
}


static void
se_button_clicked (TIPageMediaSignalling *self)
{
  /* The ChannelHandler interface is trivial and this is a one-shot request,
   * so we won't bother using a TpProxy for this just now. */
  DBusGProxy *proxy;
  DBusGConnection *dc;
  TpProxy *connection;
  gchar *channel_path;
  gchar *channel_type;
  guint handle_type;
  guint handle;

  g_object_get (self->priv->channel,
      "dbus-connection", &dc,
      "connection", &connection,
      "object-path", &channel_path,
      "channel-type", &channel_type,
      "handle-type", &handle_type,
      "handle", &handle,
      NULL);

  proxy = dbus_g_proxy_new_for_name (dc, STREAM_ENGINE_BUS_NAME,
      STREAM_ENGINE_OBJECT_PATH, CHANNEL_HANDLER_INTERFACE);
  dbus_g_proxy_call_no_reply (proxy, "HandleChannel",
      G_TYPE_STRING, connection->bus_name,
      DBUS_TYPE_G_OBJECT_PATH, connection->object_path,
      G_TYPE_STRING, channel_type,
      DBUS_TYPE_G_OBJECT_PATH, channel_path,
      G_TYPE_UINT, handle_type,
      G_TYPE_UINT, handle,
      G_TYPE_INVALID);

  g_free (channel_path);
  g_free (channel_type);
  g_object_unref (connection);
  g_object_unref (proxy);
  dbus_g_connection_unref (dc);
}


static void
ti_page_media_signalling_dispose (GObject *object)
{
  TIPageMediaSignalling *self = TI_PAGE_MEDIA_SIGNALLING (object);

  if (self->priv->channel != NULL)
    {
      g_object_unref (self->priv->channel);
      self->priv->channel = NULL;
    }

  if (self->priv->se_button != NULL)
    {
      g_signal_handlers_disconnect_by_func (self->priv->se_button,
          G_CALLBACK (se_button_clicked), self);
    }

  G_OBJECT_CLASS (ti_page_media_signalling_parent_class)->dispose (object);
}


static void
ti_page_media_signalling_get_property (GObject *object,
                                     guint property_id,
                                     GValue *value,
                                     GParamSpec *pspec)
{
  TIPageMediaSignalling *self = TI_PAGE_MEDIA_SIGNALLING (object);

  switch (property_id)
    {
    case PROP_CHANNEL:
      g_value_set_object (value, self->priv->channel);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}


static void
ti_page_media_signalling_set_property (GObject *object,
                                     guint property_id,
                                     const GValue *value,
                                     GParamSpec *pspec)
{
  TIPageMediaSignalling *self = TI_PAGE_MEDIA_SIGNALLING (object);

  switch (property_id)
    {
    case PROP_CHANNEL:
      g_return_if_fail (self->priv->channel == NULL);
      self->priv->channel = TP_CHANNEL (g_value_dup_object (value));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}


static void
ti_page_media_signalling_setup_page (TIPage *page,
                                   GladeXML *glade_xml)
{
  TIPageMediaSignalling *self = TI_PAGE_MEDIA_SIGNALLING (page);

  self->priv->se_button = GTK_BUTTON (glade_xml_get_widget (glade_xml,
        "se_button"));
  g_assert (self->priv->se_button != NULL);
  g_signal_connect_swapped (self->priv->se_button, "clicked",
      G_CALLBACK (se_button_clicked), self);
}


static void
ti_page_media_signalling_class_init (TIPageMediaSignallingClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  TIPageClass *page_class = TI_PAGE_CLASS (klass);
  GParamSpec *param_spec;

  object_class->dispose = ti_page_media_signalling_dispose;
  object_class->get_property = ti_page_media_signalling_get_property;
  object_class->set_property = ti_page_media_signalling_set_property;

  page_class->setup_page = ti_page_media_signalling_setup_page;

  g_type_class_add_private (klass, sizeof (TIPageMediaSignallingPrivate));

  param_spec = g_param_spec_object ("channel", "TpChannel",
      "This page's associated channel", TP_TYPE_CHANNEL,
      G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE |
      G_PARAM_STATIC_NAME | G_PARAM_STATIC_BLURB | G_PARAM_STATIC_NICK);
  g_object_class_install_property (object_class, PROP_CHANNEL, param_spec);
}


TIPageMediaSignalling *
ti_page_media_signalling_new (GtkNotebook *parent_notebook,
                            TpChannel *channel)
{
  TIPage *page = NULL;

  g_return_val_if_fail (channel != NULL, NULL);

  page = g_object_new (TI_TYPE_PAGE_MEDIA_SIGNALLING,
      "channel", channel,
      NULL);

  _ti_page_new (&page, parent_notebook, "page-media-signalling.xml");

  return TI_PAGE_MEDIA_SIGNALLING (page);
}
