# -*- coding: utf-8 -*-
"""
This module contains the representation class for cell integrals.
"""

# Copyright (C) 2008-2009 Martin Sandve Alnes and Simula Resarch Laboratory
#
# This file is part of SyFi.
#
# SyFi is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 2 of the License, or
# (at your option) any later version.
#
# SyFi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with SyFi. If not, see <http://www.gnu.org/licenses/>.
#
# First added:  2008-08-13
# Last changed: 2009-03-17

from sfc.common import sfc_assert, sfc_debug
from sfc.representation.integralrepresentation import IntegralRepresentation

class CellIntegralRepresentation(IntegralRepresentation):
    def __init__(self, integrals, formrep):
        IntegralRepresentation.__init__(self, integrals, formrep, False)
    
    def compute_A(self, data, iota, facet=None):
        "Compute expression for A[iota]. Overload in subclasses!"
        
        if data.integration_method == "quadrature":
            
            if self.options.safemode:
                integrand = data.integral.integrand()
                data.evaluator.update(iota)
                integrand = data.evaluator.visit(integrand)
            else:
                n = len(data.G.V())
                integrand = data.vertex_data_set[iota][n-1]
            
            D = self.formrep.D_sym
            A = integrand * D
            
            if self.formrep.options.output.enable_debug_prints:
                sfc_debug("In compute_A(", iota, "):")
                sfc_debug("  data.integral.integrand() = ", data.integral.integrand())
                sfc_debug("  integrand = ", integrand)
                sfc_debug("  A = ", A)
        
        elif data.integration_method == "symbolic":
            integrand = data.integral.integrand()
            data.evaluator.update(iota)
            integrand = data.evaluator.visit(integrand)
            detG = self.formrep.detG_sym
            polygon = self.formrep.cell.polygon
            A = polygon.integrate(integrand) * detG
            
            if self.formrep.options.output.enable_debug_prints:
                sfc_debug("In compute_A(", iota, "):")
                sfc_debug("  data.integral.integrand() = ", data.integral.integrand())
                sfc_debug("  integrand = ", integrand)
                sfc_debug("  A = ", A)
        
        return A
