// Copyright (C) 2009-2009 Martin Sandve Alnæs
// Licensed under the GNU LGPL Version 3.0.
//
// First added:  2009-01-01
// Last changed: 2009-04-01
//
// This demo program solves a hyperelasticity problem
// in 3D using the Mooney-Rivlin model.

#include <iostream>
#include <vector>
#include <dolfin.h>
#include "generated_code/HyperElasticityMooneyRivlin.h"

using dolfin::message;
using dolfin::Function;
using dolfin::Constant;
using dolfin::MeshFunction;
using dolfin::FunctionSpace;
using dolfin::SubDomain;
using dolfin::uint;
using dolfin::UnitCube;
using dolfin::DirichletBC;
using dolfin::BoundaryCondition;
using dolfin::VariationalProblem;
using dolfin::Vector;
using dolfin::Matrix;
using dolfin::File;

using namespace HyperElasticityMooneyRivlin;

using std::cout;
using std::endl;

// ---------------------------------------- Functions

class Displacement: public Function
{
public:
  Displacement(const FunctionSpace & V): Function(V) {}

  void eval(double* values, const double* x) const
  {
    const double a = -0.1;
    const double b = 0.05;
    double dy = x[1]-0.5;
    double dz = x[2]-0.5;
    values[0] = x[0]*a;
    values[1] = dy*b;
    values[2] = dz*b;
  }
};

// ---------------------------------------- Subdomains

class DirichletBoundary: public SubDomain
{
  bool inside(const double* x, bool on_boundary) const
  {
    return on_boundary && (x[0] < DOLFIN_EPS || x[0] > 1.0-DOLFIN_EPS);
    //return on_boundary;
  }
};

// ---------------------------------------- Main program

int main(int argc, char**argv)
{
    // Geometry
    message("Mesh");
    unsigned n = 5;
    UnitCube mesh(n, n, n);

    // Boundaries
    message("Boundaries");
    unsigned dirichlet_boundary_marker = 1;

    MeshFunction<unsigned> boundaries(mesh, mesh.geometry().dim()-1);

    DirichletBoundary dirichlet_boundary;
    dirichlet_boundary.mark(boundaries, dirichlet_boundary_marker);

    // Function spaces
    message("Function spaces");
    Form_a::TestSpace V(mesh);
    CoefficientSpace_c1 C(mesh);
    
    // Coefficient functions
    message("Functions");
    
    // Displacement
    Function u(V);
    u.vector().zero();

    // Material parameters
    Constant c1(3.0);
    Constant c2(3.0);
    Constant c3(100.0);

    // Dirichlet BC function
    Displacement ubc(V);

    // Collect coefficients
    CoefficientSet coeffs;
    coeffs.u  = u;
    coeffs.c1 = c1;
    coeffs.c2 = c2;
    coeffs.c3 = c3;

    coeffs.disp();

    // Forms
    message("Forms");
    Form_M M(coeffs);
    Form_L L(V, coeffs);
    Form_a a(V, V, coeffs);

    // Setup boundary conditions
    message("Boundary conditions");

    // Dirichlet boundary conditions
    DirichletBC bc(V, ubc, boundaries, dirichlet_boundary_marker);
    
    std::vector<BoundaryCondition*> bcs;
    bcs.push_back(&bc);

    // Create variational problem (a, L, bcs, cell_domains, exterior_facet_domains, interior_facet_domains, nonlinear)
    message("Variational problem");
    VariationalProblem problem(a, L, bcs, 0, &boundaries, 0, true);

    // Compute integral of strain energy over domain, should be zero
    double energy = 0.0;
    energy = assemble(M);
    cout << "Energy before solve: " << energy << endl;

    // Solve!
    message("Solving");
    problem.set("linear solver", "direct");
    problem.solve(u);

    // Compute integral of strain energy over domain
    energy = assemble(M);
    cout << "Energy after solve: " << energy << endl;

    // Write functions to file
    message("Writing to file");

    File ufile("u.pvd");
    ufile << u;

    //plot(u);
    message("Done!");

    return 0;
}

