/* Copyright (C) 2000-2004  Thomas Bopp, Thorsten Hampel, Ludger Merkens
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * $Id: ldap.pike,v 1.24 2006/10/14 20:18:21 exodusd Exp $
 */

constant cvs_version="$Id: ldap.pike,v 1.24 2006/10/14 20:18:21 exodusd Exp $";

inherit "/kernel/module";

#include <configure.h>
#include <macros.h>
#include <config.h>
#include <attributes.h>
#include <classes.h>
#include <events.h>
#include <database.h>

//#define LDAP_DEBUG 1

#ifdef LDAP_DEBUG
#define LDAP_LOG(s, args...) werror("ldap: "+s+"\n", args)
#else
#define LDAP_LOG(s, args...)
#endif

#define DEPENDENCIES cache

//! This module is a ldap client inside sTeam. It reads configuration
//! parameters of sTeam to contact a ldap server.
//! All the user management can be done with ldap this way. Some
//! special scenario might require to modify the modules code to 
//! make it work.
//!
//! The configuration variables used are:
//! server - the server name to connect to (if none is specified, then ldap will not be used)
//! cacheTime - how long (in seconds) shall ldap entries be cached? (to reduce server requests)
//! reconnectTime - how long (in seconds) between reconnection attempts in case the LDAP server cannot be connected (if not set, or set to 0, then no reconnection attempts will be made)
//! user   - ldap user for logging in
//! password - the password for the user
//! base_dc - ldap base dc, consult ldap documentation
//! userdn - the dn path where new users are stored
//! groupdn - the dn path where new groups are stored
//! objectName - the ldap object name to be used for the search
//! userAttr - the attribute containing the user's login name
//! passwordAttr - the attribute containing the password
//! emailAttr - the attribute containing the user's email address
//! iconAttr - the attribute containing the user's icon
//! fullnameAttr - the attribute containing the user's surname
//! nameAttr - the attribute containing the user's first name
//! userClass - an attribute value that identifies users
//! userId - an attribute that contains a value that can be used to match users to groups
//! groupAttr - the attribute containing the group's name
//! groupClass - an attribute value that identifies groups
//! groupId - an attribute that contains a value that can be used to match users to groups
//! memberAttr - an attribute that contains a value that can be used to match users to groups
//! descriptionAttr - the attribute that contains a user or group description
//! notfound - defines what should be done if a user or group could not be found in LDAP:
//!            "create" : create and insert a new user in LDAP (at userdn)
//!            "ignore" : do nothing
//! sync - "true"/"false" : sync user or group data (false: read-only LDAP access)
//! bindUser - "true" : when authorizing users, try ldap bind first (you will need this if the ldap lookup doesn't return a password for the user)
//! bindUserWithoutDN - "false" : when binding users, only use the user name instead of the dn
//! requiredAttr : required attributes for creating new users in LDAP
//! adminAccount - "root" : sTeam account that will receive administrative mails about ldap
//! charset - "utf-8" : charset used in ldap server

static string sServerURL;
static mapping    config = ([ ]);

static object charset_decoder;
static object charset_encoder;

static object user_cache;
static object group_cache;
static object authorize_cache;
static int cache_time = 0;

static object admin_account = 0;
static array ldap_conflicts = ({ });


bool ldap_activated ()
{
  if ( !stringp(config->server) || sizeof(config->server) < 1 )
    return false;
  else
    return true;
}


object connect ( void|string user, void|string password ) {
  if ( !ldap_activated() ) return UNDEFINED;

  string bind_dn = "";
  if ( stringp(user) ) {
    if ( Config.bool_value(config->bindUserWithoutDN) )
      bind_dn = user;
    else {
      bind_dn = config->userAttr + "=" + user + ",";
    if ( stringp(config["userdn"]) && sizeof(config["userdn"]) > 0 )
      bind_dn += config["userdn"];
    if ( stringp(config["base_dc"]) && sizeof(config["base_dc"]) > 0 )
      bind_dn += config["base_dc"];
    }
  }
  else {  // bind with default/root user
    if ( !stringp(config["user"]) || sizeof(config["user"]) < 1 )
      return UNDEFINED;
    bind_dn = "cn=" + config["user"];
    if ( stringp(config["root_dn"]) && sizeof(config["root_dn"]) > 0 )
      bind_dn += "," + config["root_dn"];
    password = config["password"];
  }

  object ldap;
  mixed err = catch( ldap = Protocols.LDAP.client( config["server"] ) );
  if ( err ) werror( "ldap: error while connecting: %O\n", err[0] );
  if ( !objectp(ldap) ) return UNDEFINED;

  err = catch {
    if ( !ldap->bind( bind_dn, password ) )
      throw( "bind failed on " + bind_dn );
    ldap->set_scope( 2 );
    if ( stringp(config["base_dc"]) && sizeof(config["base_dc"]) )
      ldap->set_basedn( config["base_dc"] );
  };
  if ( err != 0 ) {
    string error_msg = "";
    if ( ldap->error_number() > 0 )
      error_msg = "\n" + ldap->error_string() +
	sprintf( " (#%d)", ldap->error_number() );
    LDAP_LOG( "Failed to bind " + bind_dn + " on ldap" + error_msg );
    disconnect( ldap );
    return UNDEFINED;
  }
  LDAP_LOG( "Connected to LDAP: %s", config["server"] );
  return ldap;
}


void disconnect ( object ldap ) {
  if ( !objectp(ldap) ) return;
  destruct( ldap );
  gc();  // force garbage collection to free ldap file descriptors
}


mapping get_config ()
{
  return config;
}


static void init_module()
{
    config = Config.read_config_file( _Server.get_config_dir()+"/modules/ldap.cfg", "ldap" );
    if ( !mappingp(config) ) {
        config = ([ ]);
	MESSAGE("LDAP Service not started - missing configuration !");
	return; // ldap not started !
    }
    if ( !ldap_activated() ) {
      MESSAGE("LDAP deactivated.");
      return;  // ldap deactivated
    }

    LDAP_LOG("configuration is %O", config);
    if ( stringp(config["charset"]) )
      charset_decoder = Locale.Charset.decoder( config["charset"] );
    else charset_decoder = 0;
    charset_encoder = Locale.Charset.encoder( "utf-8" );

    if ( intp(config->cacheTime) )
	cache_time = config->cacheTime;
    else if ( !stringp(config->cacheTime) || (sscanf(config->cacheTime, "%d", cache_time) < 1) )
	cache_time = 0;
    object cache_module = get_module("cache");
    if ( objectp(cache_module) ) {
      user_cache = get_module("cache")->create_cache( "ldap:users", cache_time );
      group_cache = get_module("cache")->create_cache("ldap:groups", cache_time );
      authorize_cache = get_module("cache")->create_cache("ldap:auth", cache_time );
    }

    if ( stringp(config->notfound) && lower_case(config->notfound) == "create"
         && !config->objectName )
      steam_error("objectName configuration missing !");

    if ( Config.bool_value( config->sync ) ) {
      // if our main dc does not exist - create it
      object ldap = connect();
      mixed err = catch {
        ldap->add( config->base_dc, ([
          "objectclass": ({ "dcObject", "organization" }),
       	  "o": "sTeam Authorization Directory",
          "dc": "steam" ]) );
      };
      if ( err ) werror( "ldap: failed to create main dc: %O\n", err[0] );
      disconnect( ldap );
    }
    else {
      // check whether the connection is working
      object ldap;
      mixed err = catch( ldap = Protocols.LDAP.client( config["server"] ) );
      if ( err ) werror( "ldap: error while connecting: %O\n", err[0] );
      if ( objectp(ldap) ) {
	MESSAGE( "LDAP: connected to %s", config["server"] );
	disconnect( ldap );
      }
      else {
	MESSAGE( "LDAP: failed to connect to %s", config["server"] );
	werror( "LDAP: failed to connect to %s\n", config["server"] );
      }
    }
}


void load_module()
{
  add_global_event(EVENT_USER_CHANGE_PW, sync_password, PHASE_NOTIFY);
  add_global_event(EVENT_USER_NEW_TICKET, sync_ticket, PHASE_NOTIFY);
}


private static bool notify_admin ( string msg ) {
  if ( zero_type( config["adminAccount"] ) ) return false;
  object admin = USER( config["adminAccount"] );
  if ( !objectp(admin) ) admin = GROUP( config["adminAccount"] );
  if ( !objectp(admin) ) return false;
  string msg_text = "The following LDAP situation occured on the server "
    + _Server->get_server_name() + " at " + ctime(time()) + " :\n" + msg;
  admin->mail( msg_text, "LDAP on " + _Server->get_server_name(), 0, "text/plain" );
  return true;
}


static mixed map_results(object results)
{
  array result = ({ });
  
  for ( int i = 1; i <= results->num_entries(); i++ ) {
    mapping            data = ([ ]);
    mapping res = results->fetch(i);
    
    foreach(indices(res), string attr) {
      if ( arrayp(res[attr]) ) {
	if ( sizeof(res[attr]) == 1 )
	  data[attr] = res[attr][0];
	else
	  data[attr] = res[attr];
      }
    }
    if ( results->num_entries() == 1 )
      return data;
    result += ({ data });
  }
  return result;
}


mapping search_user ( string search_str, void|string user, void|string pass )
{
  if ( !ldap_activated() )
    return UNDEFINED;

  if ( !stringp(search_str) || sizeof(search_str)<1 ) {
    werror( "LDAP: search_user: invalid search_str: %O\n", search_str );
    return UNDEFINED;
  }

  mapping udata = ([ ]);
  object results;
    
  object ldap;
  if ( Config.bool_value(config->bindUser) && stringp(user) && stringp(pass) )
    ldap = connect(user, pass);
  else
    ldap = connect();

  if ( !objectp(ldap) )
    return UNDEFINED;

  LDAP_LOG("looking up user in LDAP: %s\n", search_str);
  if ( config->userdn ) {
    ldap->set_basedn(config->userdn+","+config->base_dc);
    if ( ldap->error_number() )
      werror( "ldap: error searching user: %s (#%d)\n",
	      ldap->error_string(), ldap->error_number() );
  }

  mixed err = catch( results = ldap->search( search_str ) );
  if ( err ) {
    werror( "ldap: error searching user: %s\n", err[0] );
    disconnect( ldap );
    return UNDEFINED;
  }

  if ( objectp(ldap) && ldap->error_number() ) {
    werror( "ldap: Error while searching user: %s (#%d)\n",
	    ldap->error_string(), ldap->error_number() );
    disconnect( ldap );
    return UNDEFINED;
  }

  if ( !objectp(results) ) {
    werror("ldap: Invalid results while searching user.\n");
    disconnect( ldap );
    return UNDEFINED;
  }
    
  if ( results->num_entries() == 0 ) {
    LDAP_LOG("User not found in LDAP directory: %s", search_str);
    disconnect( ldap );
    return UNDEFINED;
  }

  udata = map_results( results );
  LDAP_LOG( "user %s has dn: %O", search_str, udata["dn"] );

  disconnect( ldap );

  return udata;
}


mapping search_group ( string search_str )
{
  if ( !ldap_activated() )
    return UNDEFINED;

  if ( !stringp(search_str) || sizeof(search_str)<1 ) {
    werror( "LDAP: search_group: invalid search_str: %O\n", search_str );
    return UNDEFINED;
  }
  
  mapping gdata = ([ ]);
  object results;

  if ( !stringp(config->groupAttr) || sizeof(config->groupAttr)<1 )
    return UNDEFINED;

  object ldap = connect();

  if ( !objectp(ldap) )
    return UNDEFINED;

  LDAP_LOG("looking up group in LDAP: %s\n", search_str);
  if ( config->groupdn ) {
    ldap->set_basedn(config->groupdn+"," + config->base_dc);
    if ( ldap->error_number() )
      werror( "ldap: error searching group: %s (#%d)\n",
	      ldap->error_string(), ldap->error_number() );
  }

  mixed err = catch( results = ldap->search( search_str ) );
  if ( err ) {
    werror( "ldap: error searching group: %s\n", err[0] );
    disconnect( ldap );
    return UNDEFINED;
  }

  if ( objectp(ldap) && ldap->error_number() ) {
    werror( "ldap: Error while searching group: %s (#%d)\n",
	    ldap->error_string(), ldap->error_number() );
    disconnect( ldap );
    return UNDEFINED;
  }

  if ( !objectp(results) ) {
    werror("ldap: Invalid results while searching group.\n");
    disconnect( ldap );
    return UNDEFINED;
  }
    
  if ( results->num_entries() == 0 ) {
    LDAP_LOG("Group not found in LDAP directory: %s", search_str);
    disconnect( ldap );
    return UNDEFINED;
  }

  gdata = map_results(results);
  LDAP_LOG( "group %s has dn: %O", search_str, gdata["dn"] );

  disconnect( ldap );

  return gdata;
}


mapping fetch_user ( string identifier, void|string pass )
{
  if ( !stringp(identifier) || !ldap_activated() )
    return UNDEFINED;

  if ( !objectp(user_cache) )
    user_cache = get_module("cache")->create_cache( "ldap:users", cache_time );

  LDAP_LOG("fetch_user(%s)", identifier);
  string search_str = "("+config->userAttr+"="+identifier+")";
  if ( stringp(config->userClass) )
    search_str = "(&"+search_str+"(objectclass="+config->userClass+"))";
  return user_cache->get( identifier, lambda(){ return fix_charset( search_user( search_str, identifier, pass ) ); } );
}


mapping fetch_group ( string identifier )
{
  if ( !stringp(identifier) || !ldap_activated() )
    return UNDEFINED;

  if ( !stringp(config->groupAttr) || sizeof(config->groupAttr)<1 )
    return UNDEFINED;

  if ( !objectp(group_cache) )
    group_cache = get_module("cache")->create_cache("ldap:groups", cache_time );

  string search_str = "("+config->groupAttr+"="+identifier+")";
  if ( stringp(config->groupClass) )
    search_str = "(&"+search_str+"(objectclass="+config->groupClass+"))";

  return group_cache->get( identifier, lambda(){ return fix_charset( search_group( search_str ) ); } );
}


mapping fetch ( string dn, string pattern )
{
  if ( !stringp(dn) || !stringp(pattern) || !ldap_activated() )
    return UNDEFINED;

  // caller must be module...
  if ( !_Server->is_module( CALLER ) )
    steam_error( "Access for non-module denied !" );

  object results;

  object ldap = connect();

  if ( !objectp(ldap) )
    return UNDEFINED;

  if ( stringp(dn) && sizeof(dn)>0 )
    ldap->set_basedn(dn+"," + config->base_dc);
  else
    ldap->set_basedn(config->base_dc);
  if ( ldap->error_number() )
    werror( "ldap: error fetching: %s (#%d)\n",
	    ldap->error_string(), ldap->error_number() );

  mixed err = catch( results = ldap->search( pattern ) );
  if ( err ) {
    werror( "ldap: error fetching: %s\n", err[0] );
    disconnect( ldap );
    return UNDEFINED;
  }
  
  if ( objectp(ldap) && ldap->error_number() ) {
    werror( "ldap: Error while fetching: %s (#%d)\n",
	    ldap->error_string(), ldap->error_number() );
    disconnect( ldap );
    return UNDEFINED;
  }

  if ( !objectp(results) ) {
    werror("ldap: Invalid results while fetching.\n");
    disconnect( ldap );
    return UNDEFINED;
  }
    
  if ( results->num_entries() == 0 ) {
    LDAP_LOG("No results when fetching: %s", pattern);
    disconnect( ldap );
    return UNDEFINED;
  }

  disconnect( ldap );

  return map_results( results );
}


mixed fix_charset ( string|mapping|array v )
{
  if ( zero_type(v) ) return UNDEFINED;
  if ( !objectp(charset_encoder) || !objectp(charset_decoder) ) return v;
  if ( stringp(v) ) {
    if ( xml.utf8_check(v) ) return v;  // already utf-8
    string tmp = charset_decoder->feed(v)->drain();
    tmp = charset_encoder->feed(tmp)->drain();
    //LDAP_LOG( "charset conversion: from \"%s\" to \"%s\".", v, tmp );
    return tmp;
  }
  else if ( arrayp(v) ) {
    array tmp = ({ });
    foreach ( v, mixed i )
      tmp += ({ fix_charset(i) });
    return tmp;
  }
  else if ( mappingp(v) ) {
    mapping tmp = ([ ]);
    foreach ( indices(v), mixed i )
      tmp += ([ fix_charset(i) : fix_charset(v[i]) ]);
    return tmp;
  }
  else return UNDEFINED;
}


static bool check_password(string pass, string user_pw)
{
  if ( !stringp(pass) || !stringp(user_pw) )
    return false;

  LDAP_LOG("check_password()");
  if ( user_pw[0..4] == "{SHA}" )
    return user_pw[5..] == MIME.encode_base64( sha_hash(pass) );
  if ( user_pw[0..6] == "{crypt}" )
    return crypt(pass, user_pw[7..]);
  return verify_crypt_md5(pass, user_pw);
}


bool authorize_ldap ( object user, string pass )
{
  if ( !ldap_activated() )
    return false;

  if ( !objectp(user) )
    steam_error("User object expected for authorization !");

  if ( !stringp(pass) || sizeof(pass)<1 )
    return false;

  string uname = user->get_user_name();

  // don't authorize restricted users:
  if ( _Persistence->user_restricted( uname ) ) return false;

  if ( !objectp(authorize_cache) )
    authorize_cache = get_module("cache")->create_cache("ldap:auth", cache_time );

  string cached = authorize_cache->get( uname );
  if ( stringp(cached) ) {
      if ( check_password( pass, cached ) ) 
      {
        LDAP_LOG("user %s LDAP cache authorized", uname);
        return true;
      }
      else 
      {
        LDAP_LOG("user %s found in LDAP cache - password failed: %O", uname, cached);
        return false;
      }
  }
  
  // try authorizing via bind:
  if ( Config.bool_value(config->bindUser) && stringp(uname) && stringp(pass) ) {
    object ldap = connect( uname, pass );
    if ( objectp(ldap) ) {
      authorize_cache->put( uname, make_crypt_md5( pass ) );
      LDAP_LOG("authorized %s via bind", uname);
      disconnect( ldap );
      return true;
    }
  }

  // fetch user data and authorize via password:
  mapping udata = fetch_user(uname, pass);
  LDAP_LOG("trying to authorize user %s via password", uname);

  if ( mappingp(udata) ) {
    string dn = udata["dn"];
    if ( !stringp(dn) ) dn = "";

    // check for conflicts (different user in sTeam than in LDAP):
    if (config->checkConflicts && !stringp(user->query_attribute("ldap:dn")) ) {
      if ( search(ldap_conflicts,uname)<0 ) {
	ldap_conflicts += ({ uname });
        if ( notify_admin(
	    "Dear LDAP administrator at "+_Server->get_server_name()
	    +",\n\nthere has been a conflict between LDAP and sTeam:\n"
	    +"User \""+uname+"\" already exists in sTeam, but now "
	    +"there is also an LDAP user with the same name/id.\nYou "
	    +"will need to remove/rename one of them or, if they are "
	    +"the same user, you can overwrite the sTeam data from LDAP "
	    +"by adding a \"dn\" attribute to the sTeam user." ) );
	else
	  werror( "ldap: user conflict: %s in sTeam vs. %s in LDAP\n", uname, dn );
	return false;
      }
    }
    else if ( search(ldap_conflicts,uname) >= 0 )
      ldap_conflicts -= ({ uname });

    if ( check_password( pass, udata[config->passwordAttr] ) ) {
      // need to synchronize passwords from ldap if ldap is down ?!
      // this is only done when the ldap password is received
      if ( udata[config->passwordAttr] != user->get_user_password()) {
	user->set_user_password(udata[config->passwordAttr], 1);
      }
      authorize_cache->put( uname, udata[config->passwordAttr] );

      LDAP_LOG("user %s LDAP authorized !", uname);
      return true;
    }
    else {
      LDAP_LOG("user %s found in LDAP directory - password failed!", uname);
      return false;
    }
  }
  LDAP_LOG("user " + uname + " was not found in LDAP directory.");
  // if notfound configuration is set to create, then we should create
  // a user.
  if ( config->notfound == "create" )
    add_user( uname, pass, user );
  return false;
}

object sync_user(string name)
{
  if ( !ldap_activated() )
    return UNDEFINED;

  // don't sync restricted users:
  if ( _Persistence->user_restricted( name ) )
    return UNDEFINED;

  mapping udata = fetch_user( name );
  if ( !mappingp(udata) )
    return UNDEFINED;

  LDAP_LOG("sync of ldap user \"%s\": %O", name, udata);
  object user = get_module("users")->get_value(name);
  if ( objectp(user) ) {
    // update user date from LDAP
    if ( ! user->set_attributes( ([
             "pw" : udata[config->passwordAttr],
	     "email" : udata[config->emailAttr],
	     "fullname" : udata[config->fullnameAttr],
	     "firstname" : udata[config->nameAttr],
	     "OBJ_DESC" : udata[config->descriptionAttr],
	   ]) ) )
      werror( "LDAP: Could not sync user attributes with ldap for \"%s\".\n", name );
  } else {
    // create new user to match LDAP user
    object factory = get_factory(CLASS_USER);
    user = factory->execute( ([
	     "name" : name,
	     "pw" : udata[config->passwordAttr],
	     "email" : udata[config->emailAttr],
	     "fullname" : udata[config->fullnameAttr],
	     "firstname" : udata[config->nameAttr],
	     "OBJ_DESC" : udata[config->descriptionAttr],
	   ]) );
    user->set_user_password( udata[config->passwordAttr], 1 );
    user->activate_user( factory->get_activation() );
  }
  // sync group membership:
  if ( objectp( user ) ) {
    string primaryGroupId = udata[config->groupId];
    if ( stringp( primaryGroupId ) ) {
      mapping group = search_group("("+config->groupId+"="+primaryGroupId+")");
    }
  }

  return user;
}

object sync_group(string name)
{
  if ( !ldap_activated() )
    return UNDEFINED;

  // don't syncronize restricted groups:
  if ( _Persistence->group_restricted( name ) )
    return UNDEFINED;

  mapping gdata = fetch_group( name );
  if ( !mappingp(gdata) )
    return UNDEFINED;

  LDAP_LOG("sync of ldap group: %O", gdata);
  //object group = get_module("groups")->lookup(name);
  object group = get_module("groups")->get_value( name );
  if ( objectp(group) ) {
    // group memberships are handled by the persistence manager
    // update group date from LDAP
    group->set_attributes( ([
	     "OBJ_DESC" : gdata[config->descriptionAttr],
	   ]) );
  } else {
    // create new group to match LDAP group
    object factory = get_factory(CLASS_GROUP);
    group = factory->execute( ([
	      "name": name,
	      "OBJ_DESC" : gdata[config->descriptionAttr],
	    ]) );
  }
  return group;
}

static void sync_password(int event, object user, object caller)
{
  if ( !Config.bool_value(config->sync) )
    return;
  string oldpw = user->get_old_password();
  string crypted = user->get_user_password();
  string name = user->get_user_name();
  // don't sync password for restricted users:
  if ( _Persistence->group_restricted( name ) ) return;
  LDAP_LOG("password sync for " + user->get_user_name());

  object ldap;
  string dn;

  if ( Config.bool_value(config->bindUser) && oldpw &&
       objectp(ldap = connect( name, oldpw )) ) {
    if ( config->userdn )
      dn = sprintf("%s=%s,%s,%s", config->userAttr, name, config->userdn, config->base_dc);
    else
      dn = sprintf("%s=%s,%s", config->userAttr, name, config->base_dc);
  }
  else if ( ldap = connect() ) {
    dn = config->base_dc + " , " + config->userAttr + "=" + name;
  }

  if ( !stringp(dn) ) {
    LDAP_LOG("sync_password(): no dn");
    disconnect( ldap );
    return;
  }

  mixed err;
  if ( crypted[..2] == "$1$" )
    crypted = "{crypt}" + crypted;
  err = catch( ldap->modify(dn, ([ config->passwordAttr: ({ 2,crypted }),])) );
  authorize_cache->remove( name );
  user_cache->remove( name );
  LDAP_LOG("sync_password(): %s - %s - %O\n", crypted, dn, ldap->error_string());
  disconnect( ldap );
}

static void sync_ticket(int event, object user, object caller, string ticket)
{
  mixed err;
  if ( !ldap_activated() || !Config.bool_value(config->sync) )
    return;
  string name = user->get_user_name();
  string dn = config->base_dc + " , " + config->userAttr + "=" + name;
  object ldap = connect();
  err = catch( ldap->modify(dn, ([ "userCertificate": ({ 2,ticket }),])) );
  disconnect( ldap );
}

bool is_user(string user)
{
  object ldap = connect();
  if ( !objectp(ldap) ) return false;
  object results = ldap->search( "("+config["userAttr"]+"="+user+")" );
  bool result = (objectp(results) && results->num_entries() > 0);
  disconnect( ldap );
  return result;
}

static bool add_user(string name, string password, object user)
{
  if ( !ldap_activated() || !Config.bool_value(config->sync) ) return false;
  if ( !stringp(config->notfound) || lower_case(config->notfound) != "create" )
    return false;

  // don't add restricted users:
  if ( _Persistence->user_restricted( name ) ) return false;
  
  string fullname = user->get_name();
  string firstname = user->query_attribute(USER_FIRSTNAME);
  string email = user->query_attribute(USER_EMAIL);
  
  mapping attributes = ([
    config["userAttr"]: ({ name }),
    config["fullnameAttr"]: ({ fullname }),
    "objectClass": ({ config["objectName"] }),
    config["passwordAttr"]: ({ make_crypt_md5(password) }),
  ]);
  if ( stringp(firstname) && strlen(firstname) > 0 )
    config["nameAttr"] = ({ firstname });
  if ( stringp(email) && strlen(email) > 0 )
    config["emailAttr"] = ({ email });

  array(string) requiredAttributes =  config["requiredAttr"];
  
  if ( arrayp(requiredAttributes) && sizeof(requiredAttributes) > 0 ) {
    foreach(requiredAttributes, string attr) {
      if ( zero_type(attributes[attr]) )
	attributes[attr] = ({ "-" });
    }
  }
  
  object ldap = connect();
  if ( !objectp(ldap) ) return false;
  ldap->add( config["userAttr"]+"="+name+","+config["base_dc"], attributes );
  int err = ldap->error_number();
  if ( err != 0 )
    FATAL( "Failed to add user , error is " + ldap->error_string() );
  bool result = ldap->error_number() == 0;
  disconnect( ldap );
  return result;
}

string get_identifier() { return "ldap"; }
