/*===========================================================================
*
*                            PUBLIC DOMAIN NOTICE
*               National Center for Biotechnology Information
*
*  This software/database is a "United States Government Work" under the
*  terms of the United States Copyright Act.  It was written as part of
*  the author's official duties as a United States Government employee and
*  thus cannot be copyrighted.  This software/database is freely available
*  to the public for use. The National Library of Medicine and the U.S.
*  Government have not placed any restriction on its use or reproduction.
*
*  Although all reasonable efforts have been taken to ensure the accuracy
*  and reliability of the software and data, the NLM and the U.S.
*  Government do not and cannot warrant the performance or results that
*  may be obtained by using this software or data. The NLM and the U.S.
*  Government disclaim all warranties, express or implied, including
*  warranties of performance, merchantability or fitness for any particular
*  purpose.
*
*  Please cite the author in any work or product based on this material.
*
* ===========================================================================
*
*/

#include "vdb-copy-includes.h"
#include "vdb-copy.vers.h"
#include "definitions.h"
#include <kdb/meta.h>
#include <kdb/namelist.h>
#include <sysalloc.h>
#include <stdlib.h>
#include <time.h>


static rc_t copy_metadata_data ( const KMDataNode *snode, KMDataNode *dnode )
{
    char buffer [ 1024 ];
    size_t total, bytes, remaining;
    /* copy node data unless already set */
    rc_t rc = KMDataNodeRead ( dnode, 0, buffer, 0, & bytes, & total );
    DISP_RC( rc, "copy_metadata_child:KMDataNodeRead(dst) failed" );
    if ( rc == 0 && total == 0 )
    do
    {
        rc = KMDataNodeRead ( snode, total, buffer, sizeof buffer, & bytes, & remaining );
        DISP_RC( rc, "copy_metadata_child:KMDataNodeRead(src) failed" );
        if ( rc == 0 )
        {
            rc = KMDataNodeAppend ( dnode, buffer, bytes );
            DISP_RC( rc, "copy_metadata_child:KMDataNodeAppend(dst) failed" );
            if ( rc != 0 ) break;
        }
        total += bytes;
     } while ( remaining != 0 );
    return rc;
}


static rc_t copy_metadata_attribs ( const KMDataNode *snode, KMDataNode *dnode,
                                    const bool show_meta )
{
    KNamelist *attrs;
    uint32_t i, count;
    rc_t rc = KMDataNodeListAttr ( snode, & attrs );
    DISP_RC( rc, "copy_metadata_child:KMDataNodeListAttr(src) failed" );
    if ( rc != 0 ) return rc;
    rc = KNamelistCount ( attrs, & count );
    for ( i = 0; rc == 0 && i < count; ++ i )
    {
        const char *attr;
        rc = KNamelistGet ( attrs, i, & attr );
        if ( rc == 0 )
        {
            char buffer [ 1024 ];
            size_t bytes;
            /* test for attr existence */
            rc = KMDataNodeReadAttr ( dnode, attr, buffer, sizeof buffer, & bytes );
            if ( rc != 0 )
            {
                rc = KMDataNodeReadAttr ( snode, attr, buffer, sizeof buffer, & bytes );
                if ( rc == 0 )
                {
                    if ( show_meta )
                        OUTMSG (( "copy atr: %s\n", attr ));
                    rc = KMDataNodeWriteAttr ( dnode, attr, buffer );
                }
            }
        }
        DISP_RC( rc, "copy_metadata_child:failed to copy attribute" );
    }
    KNamelistRelease ( attrs );
    return rc;
}

/* forward decl for copy_metadata_childs() - recursion */
static rc_t copy_metadata_child ( const KMDataNode *src, KMDataNode *dst,
                                  const char *name, const char *parent,
                                  const bool show_meta );

static rc_t copy_metadata_childs ( const KMDataNode *snode, KMDataNode *dnode,
                                   const char *parent, const bool show_meta )
{
    KNamelist *names;
    uint32_t i, count;
    rc_t rc = KMDataNodeListChild ( snode, & names );
    DISP_RC( rc, "copy_metadata_child:KMDataNodeListChild(src) failed" );
    if ( rc != 0 ) return rc;
    rc = KNamelistCount ( names, & count );
    for ( i = 0; rc == 0 && i < count; ++ i )
    {
        const char *child;
        rc = KNamelistGet ( names, i, & child );
        if ( rc == 0 )
            rc = copy_metadata_child ( snode, dnode, child, parent, show_meta );
    }
    KNamelistRelease ( names );
    return rc;
}

static rc_t copy_metadata_child ( const KMDataNode *src, KMDataNode *dst,
                                  const char *name, const char *parent,
                                  const bool show_meta )
{
    const KMDataNode *snode;
    KMDataNode *dnode;
    rc_t rc = KMDataNodeOpenNodeRead ( src, & snode, name );
    DISP_RC( rc, "copy_metadata_child:KMDataNodeOpenNodeRead(src) failed" );
    if ( rc != 0 ) return rc;

    rc = KMDataNodeOpenNodeUpdate ( dst, & dnode, name );
    DISP_RC( rc, "copy_metadata_child:KMDataNodeOpenNodeUpdate(dst) failed" );
    if ( rc == 0 )
    {
        if ( show_meta )
        {
            if ( parent != NULL )
                OUTMSG (( "copy meta: %s.%s\n", parent, name ));
            else
                OUTMSG (( "copy meta: %s\n", name ));
        }

        rc = copy_metadata_data ( snode, dnode );
        if ( rc == 0 )
            rc = copy_metadata_attribs ( snode, dnode, show_meta );

        if ( rc == 0 )
            rc = copy_metadata_childs ( snode, dnode, name, show_meta );
        KMDataNodeRelease ( dnode );
    }
    else
    {
        if ( parent != NULL )
            OUTMSG (( "cannot open meta(dst): %s.%s\n", parent, name ));
        else
            OUTMSG (( "cannot open meta(dst): %s\n", name ));
    }
    KMDataNodeRelease ( snode );
    return rc;
}


static rc_t copy_metadata_root ( const KMDataNode *src_node, KMDataNode *dst_node,
                                 const char * excluded_nodes,
                                 const bool show_meta )
{
    KNamelist *names;
    const KNamelist *excluded_names = NULL;
    uint32_t i, count;

    rc_t rc = KMDataNodeListChild ( src_node, & names );
    DISP_RC( rc, "copy_metadata_root:KMDataNodeListChild() failed" );
    if ( rc != 0 ) return rc;
    
    if ( excluded_nodes != NULL )
    {
        rc = nlt_make_namelist_from_string( &excluded_names, excluded_nodes );
        DISP_RC( rc, "copy_metadata_root:nlt_make_namelist_from_string() failed" );
        if ( rc != 0 ) return rc;
    }
    
    rc = KNamelistCount ( names, & count );
    for ( i = 0; rc == 0 && i < count; ++ i )
    {
        const char *name;
        rc = KNamelistGet ( names, i, & name );
        DISP_RC( rc, "copy_metadata_root:KNamelistGet() failed" );
        if ( rc == 0 )
        {
            bool is_excluded = false;
            if ( excluded_names != NULL )
                is_excluded = nlt_is_name_in_namelist( excluded_names, name );
            if ( !is_excluded )
                rc = copy_metadata_child ( src_node, dst_node, name, NULL, show_meta );
        }
    }

    if ( excluded_names != NULL )
        KNamelistRelease( excluded_names );
    KNamelistRelease ( names );
    return rc;
}


static rc_t copy_stray_metadata ( const KMetadata *src_meta, KMetadata *dst_meta,
                                  const char * excluded_nodes,
                                  const bool show_meta )
{
    /* open root node */
    const KMDataNode *src_node;
    rc_t rc = KMetadataOpenNodeRead ( src_meta, & src_node, NULL );
    DISP_RC( rc, "copy_stray_metadata:KMetadataOpenNodeRead() failed" );
    if ( rc == 0 )
    {
        KMDataNode *dst_node;
        rc = KMetadataOpenNodeUpdate ( dst_meta, & dst_node, NULL );
        DISP_RC( rc, "copy_stray_metadata:KMetadataOpenNodeUpdate() failed" );
        if ( rc == 0 )
        {
            /* treat the root node in a special way */
            rc = copy_metadata_root ( src_node, dst_node, excluded_nodes, 
                                      show_meta );
            KMDataNodeRelease ( dst_node );
        }
        KMDataNodeRelease ( src_node );
    }
    return rc;
}


static rc_t drop_all( KMetadata *dst_meta )
{
    KMDataNode *dst_node;
    rc_t rc = KMetadataOpenNodeUpdate ( dst_meta, & dst_node, NULL );
    DISP_RC( rc, "drop_all:KMetadataOpenNodeUpdate() failed" );
    if ( rc == 0 )
    {
        rc = KMDataNodeDropAll ( dst_node );
        DISP_RC( rc, "drop_all:KMetadataDropAll() failed" );
        KMDataNodeRelease ( dst_node );
    }
    return rc;
}


static rc_t copy_back_revisions ( const KMetadata *src_meta, KMetadata *dst_meta,
                                  const bool show_meta )
{
    uint32_t max_revision, revision;

    rc_t rc = KMetadataMaxRevision ( src_meta, &max_revision );
    DISP_RC( rc, "copy_back_revisions:KMetadataMaxRevision() failed" );
    if ( rc != 0 ) return rc;
    if ( max_revision == 0 ) return rc;
    for ( revision = 1; revision <= max_revision && rc == 0; ++revision )
    {
        const KMetadata *src_rev_meta;

        if ( show_meta )
            OUTMSG (( "+++copy metadata rev. #%u:\n", revision ));
        rc = KMetadataOpenRevision ( src_meta, &src_rev_meta, revision );
        DISP_RC( rc, "copy_back_revisions:KMetadataOpenRevision() failed" );
        if ( rc == 0 )
        {
            rc = copy_stray_metadata ( src_rev_meta, dst_meta, /*"col"*/ NULL, 
                                       show_meta );
            if ( rc == 0 )
            {
                rc = KMetadataCommit ( dst_meta );
                DISP_RC( rc, "copy_back_revisions:KMetadataCommit() failed" );
                if ( rc == 0 )
                {
                    rc = KMetadataFreeze ( dst_meta );
                    DISP_RC( rc, "copy_back_revisions:KMetadataFreeze() failed" );
                }
            }
            KMetadataRelease ( src_rev_meta );
            rc = drop_all ( dst_meta );
        }
    }
    return rc;
}


static 
rc_t enter_time( KMDataNode *node, const char * key )
{
    time_t rawtime;
    struct tm * timeinfo;
    size_t size;
    char * s;
    rc_t rc;

    time( &rawtime );
    timeinfo = localtime( &rawtime );
    s = string_dup_measure ( asctime( timeinfo ), &size );
    if ( s == NULL )
        return RC( rcExe, rcNoTarg, rcCopying, rcMemory, rcExhausted );
    if ( size < 2 )
    {
        free( s );
        return RC( rcExe, rcNoTarg, rcCopying, rcBuffer, rcInvalid );
    }
    s[ size - 1 ] = 0; /* remove the newline */
    rc = KMDataNodeWriteAttr ( node, key, s );
    DISP_RC( rc, "enter_time:KMDataNodeWriteAttr() failed" );
    free( s );
    return rc;
}


static 
rc_t enter_version( KMDataNode *node, const char * key )
{
    char buff[32];
    rc_t rc;

    knprintf ( buff, sizeof( buff ), "%V", VDB_COPY_VERS );
    rc = KMDataNodeWriteAttr ( node, key, buff );
    DISP_RC( rc, "enter_version:KMDataNodeWriteAttr() failed" );
    return rc;
}

static
rc_t enter_schema_update( KMetadata *dst_meta, const bool show_meta )
{
    rc_t rc;
    KMDataNode *sw_node;

    if ( show_meta )
        OUTMSG(( "--- entering schema-update\n" ));

    rc = KMetadataOpenNodeUpdate ( dst_meta, &sw_node, "SOFTWARE" );
    DISP_RC( rc, "enter_schema_update:KMetadataOpenNodeUpdate() failed" );
    if ( rc == 0 )
    {
        KMDataNode *update_node;
        rc = KMDataNodeOpenNodeUpdate ( sw_node, &update_node, "update" );
        DISP_RC( rc, "enter_schema_update:KMDataNodeOpenNodeUpdate() failed" );
        if ( rc == 0 )
        {
            rc = enter_time( update_node, "date" );
            rc = KMDataNodeWriteAttr ( update_node, "name", "vdb-copy" );
            DISP_RC( rc, "enter_schema_update:KMDataNodeWriteAttr( name=vdb-copy) failed" );
            rc = enter_version ( update_node, "vers" );
            KMDataNodeRelease ( update_node );
        }
    }
    KMDataNodeRelease ( sw_node );
    return rc;
}

rc_t copy_table_meta ( const VTable *src_table, VTable *dst_table,
                       const char * excluded_nodes,
                       const bool show_meta, const bool schema_updated )
{
    const KMetadata *src_meta;
    rc_t rc;

    if ( src_table == NULL || dst_table == NULL )
        return RC( rcExe, rcNoTarg, rcCopying, rcParam, rcNull );
    /* it is OK if excluded_nodes is NULL */
    
    rc = VTableOpenMetadataRead ( src_table, & src_meta );
    DISP_RC( rc, "copy_table_meta:VTableOpenMetadataRead() failed" );
    if ( rc == 0 )
    {
        KMetadata *dst_meta;
        rc = VTableOpenMetadataUpdate ( dst_table, & dst_meta );
        DISP_RC( rc, "copy_table_meta:VTableOpenMetadataUpdate() failed" );
        if ( rc == 0 )
        {
            rc = copy_back_revisions ( src_meta, dst_meta, show_meta );
            if ( rc == 0 )
            {
                if ( show_meta )
                    OUTMSG (( "+++copy current metadata\n" ));
                rc = copy_stray_metadata ( src_meta, dst_meta, excluded_nodes,
                                           show_meta );
                if ( show_meta )
                    OUTMSG (( "+++end of copy current metadata\n" ));
                if ( rc == 0 && schema_updated )
                    rc = enter_schema_update( dst_meta, show_meta );
            }
            KMetadataRelease ( dst_meta );
        }
        KMetadataRelease ( src_meta );
    }
    return rc;
}
