/*===========================================================================
 *
 *                            PUBLIC DOMAIN NOTICE
 *               National Center for Biotechnology Information
 *
 *  This software/database is a "United States Government Work" under the
 *  terms of the United States Copyright Act.  It was written as part of
 *  the author's official duties as a United States Government employee and
 *  thus cannot be copyrighted.  This software/database is freely available
 *  to the public for use. The National Library of Medicine and the U.S.
 *  Government have not placed any restriction on its use or reproduction.
 *
 *  Although all reasonable efforts have been taken to ensure the accuracy
 *  and reliability of the software and data, the NLM and the U.S.
 *  Government do not and cannot warrant the performance or results that
 *  may be obtained by using this software or data. The NLM and the U.S.
 *  Government disclaim all warranties, express or implied, including
 *  warranties of performance, merchantability or fitness for any particular
 *  purpose.
 *
 *  Please cite the author in any work or product based on this material.
 *
 * ===========================================================================
 *
 */

#define TRACK_REFERENCES 0

#include <vdb/extern.h>
#include <klib/rc.h>
#include <atomic.h>

#include <bitstr.h>

#include <klib/pack.h>
#include <klib/vlen-encode.h>
#include <klib/refcount.h>
#include <klib/data-buffer.h>
#include <sysalloc.h>
#include "page-map.h"

#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <stdio.h>

#include <zlib.h>

/* Page maps describe the layout of rows within a blob.
 * The data within a page map is run-length encoded.
 * I.e. if two identical rows are stored sequencially 
 */

typedef uint32_t pm_size_t;

struct PageMap {
    /* memory allocation object for length[], leng_run[], data_run[] */
    KDataBuffer cstorage;
    /* memory allocation object for explength[], exp_offset[] */
    KDataBuffer estorage;
    
    /* array of row lengths
     * has leng_recs elements
     * is sized to reserve_leng elements
     * == storage.base
     */
    elem_count_t *length;
    
    /* array of run lengths of row lengths
     * has leng_recs elements
     * is sized to reserve_leng elements
     * == length + reserve_leng
     */
    row_count_t *leng_run;
    
    /* array of repeat counts of data
     * has data_recs elements
     * is sized to reserve_data elements
     * == leng_run + reserve_leng
     */
    row_count_t *data_run;

    elem_count_t *exp_length;
    elem_count_t *exp_offset;
    row_count_t  *exp_repeat;
    pm_size_t    exp_length_last; /* index of last length expanded */
    pm_size_t    exp_offset_last; /* index of last offset expanded */
    elem_count_t exp_offset_next;
    
    pm_size_t leng_recs;     /* number of valid elements in length[] and leng_run[] */
    pm_size_t data_recs;     /* number of valid elements in data_run[] */
    pm_size_t reserve_leng;  /* number of allocated elements in length[] and leng_run[] */
    pm_size_t reserve_data;  /* number of allocated elements in data_run[] */
    pm_size_t start_valid;   /* the expanded array contains valid data upto start_valid */
    row_count_t row_count;     /* total number of rows in page map */
    KRefcount refcount;
};

#define VALIDATE_PAGEMAPS 0

elem_count_t PageMapLastLength(const PageMap *cself) {
    return cself->leng_recs > 0 ? cself->length[cself->leng_recs - 1] : 0;
}

static rc_t PageMapExpand( const PageMap *cself, row_count_t upto ) {
    if (cself->start_valid < upto) {
        PageMap *self = (PageMap *)cself;
        elem_count_t *offset;
        row_count_t *repeat;
        pm_size_t i;
        pm_size_t j;
        pm_size_t m;
        
        if (cself->estorage.elem_count < cself->row_count) {
            rc_t rc;
            
            self->exp_length_last = self->exp_offset_last = self->exp_offset_next = 0;
            self->start_valid = 0;
            
            self->estorage.elem_bits = 32 * 3;
            rc = KDataBufferResize(&self->estorage, cself->row_count * 2);
            if (rc) return rc;
            
            self->exp_offset = self->estorage.base;
            self->exp_repeat = &self->exp_offset[self->estorage.elem_count];
            self->exp_length = &self->exp_repeat[self->estorage.elem_count];
        }
        offset = self->exp_offset;
        repeat = self->exp_repeat;
        
        if (cself->leng_recs > 1) {
            elem_count_t *length = self->exp_length;
            elem_count_t n = cself->exp_offset_next;
            
            j = cself->start_valid;
            for (i = cself->exp_length_last; i != cself->leng_recs; ++i) {
                for (m = 0; m != cself->leng_run[i]; ++m, ++j) {
                    length[j] = cself->length[i];
                }
            }
            
            j = cself->start_valid;
            for (i = cself->exp_offset_last; i != cself->data_recs; ++i) {
                elem_count_t temp = n;
                
                n += length[j];
                for (m = 0; m != cself->data_run[i]; ++m, ++j) {
                    offset[j] = temp;
                    repeat[j] = cself->data_run[i] - m;
                }
            }
            self->exp_offset_next = n; /* where to start next time */
        }
        else {
            j = cself->start_valid;
            for (i = cself->exp_offset_last; i != cself->data_recs; ++i) {
                for (m = 0; m != cself->data_run[i]; ++m, ++j) {
                    offset[j] = i * cself->length[0];
                    repeat[j] = cself->data_run[i] - m;
                }
            }
        }
        self->exp_length_last = cself->leng_recs; /* where to start next time */
        self->exp_offset_last = cself->data_recs; /* where to start next time */
        
        self->start_valid = j;
    }
    return 0;
}

rc_t PageMapNewIterator(const PageMap *self, PageMapIterator *lhs, uint64_t first_row, uint64_t num_rows)
{
    rc_t rc;

    if (num_rows > self->row_count)
        num_rows = self->row_count;

#if _DEBUGGING
    if (self->data_recs > 1) {
        assert(first_row == (row_count_t)first_row);
        assert(num_rows == (row_count_t)num_rows);
        assert(first_row + num_rows == (row_count_t)(first_row + num_rows));
    }
#endif
    
    memset(lhs, 0, sizeof(*lhs));

    lhs->end_row = first_row + num_rows;
    lhs->cur_row = first_row;
    
    lhs->length = self->length;
    
    if (self->data_recs == 1) {
        lhs->variant = pmivConstant;
        
        return 0;
    }
    if (self->leng_recs == 1 && self->data_recs == self->row_count) {
        lhs->variant = pmivVersion1;
        
        return 0;
    }

    rc = PageMapExpand(self, lhs->end_row);
    if (rc == 0) {
        lhs->repeat = self->exp_repeat;
        lhs->offset = self->exp_offset;
        if (self->leng_recs == 1) {
            lhs->variant = pmivFixedRowLength;
        }
        else {
            lhs->variant = pmivFull;
            lhs->length = self->exp_length;
        }
    }
    return rc;
}

static
rc_t PageMapGrow( PageMap *self, uint32_t new_reserve_leng, uint32_t new_reserve_data ) {
    uint32_t sz;
    PageMap temp = *self;
    uint32_t reserve_data = self->reserve_data;
    uint32_t reserve_leng = self->reserve_leng;
    rc_t rc;
    
    if (new_reserve_leng > (1UL << 31) || new_reserve_data > (1UL << 31))
        return RC(rcVDB, rcPagemap, rcAllocating, rcParam, rcExcessive);
    
    if (reserve_leng == 0)
        reserve_leng = 1;
    if (reserve_data == 0)
        reserve_data = 1;
#define MIN_KBUF_RESERVE_SIZE 256  
    while (reserve_leng < new_reserve_leng)
        reserve_leng <<= 1;
    if ( reserve_leng < MIN_KBUF_RESERVE_SIZE) reserve_leng = MIN_KBUF_RESERVE_SIZE;
    while (reserve_data < new_reserve_data)
        reserve_data <<= 1;
    if ( reserve_data < MIN_KBUF_RESERVE_SIZE) reserve_data = MIN_KBUF_RESERVE_SIZE;

    sz = reserve_leng * 2 + reserve_data;
    KDataBufferWhack(&self->estorage);
    {
        KDataBuffer new_buffer;
        
        rc = KDataBufferMake(&new_buffer, 8 * sizeof(uint32_t), sz);
        if (rc)
            return rc;
        self->cstorage = new_buffer;
    }
    self->reserve_leng = reserve_leng;
    self->reserve_data = reserve_data;
    self->length = self->cstorage.base;
    self->leng_run = self->length + reserve_leng;
    self->data_run = self->leng_run + reserve_leng;
    self->exp_length = NULL;
    self->exp_offset = NULL;
    self->start_valid = 0;
    
    if (self->leng_recs > 0 && temp.length != NULL) {
        memcpy(self->length  , temp.length  , self->leng_recs * sizeof(uint32_t));
        memcpy(self->leng_run, temp.leng_run, self->leng_recs * sizeof(uint32_t));
    }
    if (self->data_recs > 0 && temp.data_run != NULL)
        memcpy(self->data_run, temp.data_run, self->data_recs * sizeof(uint32_t));

    KDataBufferWhack(&temp.cstorage);

    return 0;
}

rc_t PageMapNew ( PageMap **lhs, uint32_t reserve ) {
    PageMap *y;

    y = calloc(1, sizeof(*y));
    if (y == NULL)
        return RC(rcVDB, rcPagemap, rcConstructing, rcMemory, rcExhausted);

    KRefcountInit(&y->refcount, 1, "PageMap", "new", "");
    if (reserve > 0) {
        rc_t rc = PageMapGrow(y, reserve, reserve);
        if (rc) {
            free(y);
            return rc;
        }
    }
    *lhs = y;
    return 0;
}

rc_t PageMapNewFixedRowLength ( PageMap **lhs, uint64_t row_count, uint64_t row_len ) {
    PageMap *y;
    rc_t rc;
    unsigned i;

    if (row_count >> 32 != 0 || row_len >> 32 != 0)
        return RC(rcVDB, rcPagemap, rcConstructing, rcParam, rcTooBig);
    
    rc = PageMapNew(&y, 0);
    if (rc)
        return rc;
    rc = PageMapGrow(y, 1, (uint32_t)row_count);
    if (rc == 0) {
        *lhs = y;
        y->length[0] = (uint32_t)row_len;
        y->leng_run[0] = (uint32_t)row_count;
        y->leng_recs = 1;
        y->data_recs = (uint32_t)row_count;
        y->row_count = (uint32_t)row_count;
        for (i = 0; i != (uint32_t)row_count; ++i) {
            y->data_run[i] = 1;
        }
    }
    if (rc)
        PageMapRelease(y);
    return rc;
}

rc_t PageMapNewSingle ( PageMap **lhs, uint64_t row_count, uint64_t row_len ) {
    PageMap *y;
    rc_t rc;
    
    if (row_count >> 32 != 0)
        return RC(rcVDB, rcPagemap, rcConstructing, rcParam, rcTooBig);
    
    rc = PageMapNewFixedRowLength(&y, 1, row_len);
    if (rc == 0) {
        y->row_count = y->data_run[0] = y->leng_run[0] = (uint32_t)row_count;
        *lhs = y;
    }
    return rc;
}

bool PageMapHasRows(const PageMap *self) {
    return self->data_recs > 0;
}

uint32_t PageMapFastRowCount ( const PageMap *self ) {
    uint32_t rslt = 0;

    if (self->data_recs == 1)
        rslt = self->data_run[0];
    return rslt;
}

uint32_t PageMapFixedRowLength ( const PageMap *self ) {
    uint32_t rslt = 0;
    
    if (self->leng_recs == 1)
        rslt = self->length[0];
    return rslt;
}

uint32_t PageMapHasSimpleStructure( const PageMap *self ) {
    uint32_t rslt = PageMapFixedRowLength(self);
    unsigned i;
    
    if (rslt == 0)
        return rslt;

    for (i = 0; i != self->data_recs; ++i) {
        if (self->data_run[i] != 1)
            return 0;
    }
    return rslt;
}

#if 0
static bool PageMapValidate( const PageMap *cself ) {
    unsigned i;
    uint32_t n;
    uint32_t m;
        
    for (m = 0, i = 0; i != cself->leng_recs; ++i) m += cself->leng_run[i];
    for (n = 0, i = 0; i != cself->data_recs; ++i) n += cself->data_run[i];

    return m == n ? true : false;
}

static
rc_t PageMapAppendRun( PageMap *self, uint32_t run_length, uint32_t row_length ) {
    return -1;
}
#endif

rc_t PageMapAppendRows(PageMap *self, uint64_t row_length, uint64_t run_length, bool same_data) {
    rc_t rc;
    uint32_t leng_cur = self->leng_recs - 1;
    uint32_t data_cur = self->data_recs - 1;
    
    if ((uint32_t)row_length != row_length)
        return RC(rcVDB, rcPagemap, rcConstructing, rcParam, rcTooBig);
    
    if ((uint32_t)run_length != run_length)
        return RC(rcVDB, rcPagemap, rcConstructing, rcParam, rcTooBig);
    
    if (self->leng_recs && row_length == self->length[leng_cur])
        self->leng_run[leng_cur] += run_length;
    else {
        same_data = false;
        leng_cur = self->leng_recs;
        ++self->leng_recs;
        if (self->leng_recs >= self->reserve_leng) {
            rc = PageMapGrow(self, self->leng_recs, 0);
            if (rc)
                return rc;
        }
        self->leng_run[leng_cur] = run_length;
        self->length[leng_cur] = (uint32_t)row_length;
    }
    if (self->data_recs && same_data)
        self->data_run[data_cur] += run_length;
    else {
        data_cur = self->data_recs;
        ++self->data_recs;
        if (self->data_recs >= self->reserve_data) {
            rc = PageMapGrow(self, 0, self->data_recs);
            if (rc)
                return rc;
        }
        self->data_run[data_cur] = run_length;
    }
    self->row_count += run_length;
    return 0;
}

static rc_t serialize_lengths(
                              KDataBuffer *Dst,
                              uint64_t offset, 
                              const uint32_t run[],
                              unsigned runs,
                              uint64_t *consumed
) {
    rc_t rc;
    unsigned i;
    uint64_t j;
    uint64_t n;
    uint8_t *dst = Dst->base;
    
    dst += offset;
    *consumed = 0;
    for (i = 0, j = 0, n = 0, rc = 0; rc == 0 && i != runs; ++i, j += n)
        rc = vlen_encodeU1(dst + j, 5, &n, run[i]);
    
    if (rc == 0)
        *consumed = j;
    
    return rc;
}

static rc_t deserialize_lengths(
                                uint32_t run[],
                                unsigned runs,
                                const uint8_t *src,
                                uint64_t ssize,
                                uint64_t *consumed
) {
    rc_t rc = 0;
    unsigned i;
    uint64_t j;
    uint64_t n;
    
    *consumed = 0;
    for (i = 0, j = 0; i != runs; ++i, j += n) {
        uint64_t val;
        
        rc = vlen_decodeU1(&val, src + j, ssize - j, &n);
        if (rc)
            break;
        run[i] = (uint32_t)val;
    }
    if (rc == 0)
        *consumed = j;
    return rc;
}

static
rc_t serialize( const PageMap *self, KDataBuffer *buffer, uint64_t *size ) {
/*
 variant 0: fixed row length, data_run[..] == 1
    vlencode(row_length)

 variant 1: fixed row length, variable data_run
    vlencode(row_length)
    vlencode(data_recs)
    serialize(data_run)

 variant 2: variable row length, data_run[..] == 1
    vlencode(leng_recs)
    serialize(length)
    serialize(leng_run)
 
 variant 3: variable row length, variable data_run
    vlencode(leng_recs)
    vlencode(data_recs)
    serialize(length)
    serialize(leng_run)
    serialize(data_run)

 */
    uint8_t version = 1;
    uint8_t variant = 0;
    uint8_t header;
    rc_t rc = 0;
    bool simple_data = true;
    unsigned i;
    uint64_t sz;
    KDataBuffer compress;

    memset(&compress, 0, sizeof(compress));
    
    for (i = 0; i != self->data_recs; ++i) {
        if (self->data_run[i] != 1) {
            simple_data = false;
            break;
        }
    }
    
    variant = (simple_data ? 0 : 1) | ((self->leng_recs == 1 ? 0 : 1) << 1);
    header = (version << 2) | variant;
    switch (variant) {
    case 0:
        rc = KDataBufferResize(buffer, 6);
        if (rc == 0) {
            ((uint8_t *)buffer->base)[0] = header;
            vlen_encodeU1(((uint8_t *)buffer->base) + 1, 5, &sz, self->length[0]);
            buffer->elem_count = sz + 1;
        }
        break;
    case 1:
        rc = KDataBufferResize(buffer, 11);
        if (rc == 0) {
            rc = KDataBufferMakeBytes(&compress, 5 * self->data_recs);
            if (rc == 0) {
                ((uint8_t *)buffer->base)[0] = header;
                vlen_encodeU1(((uint8_t *)buffer->base) + 1, 5, &sz, self->length[0]);
                buffer->elem_count = sz + 1;
                vlen_encodeU1(((uint8_t *)buffer->base) + 1 + sz, 5, &sz, self->data_recs);
                buffer->elem_count += sz;
                
                rc = serialize_lengths(&compress, 0, self->data_run, self->data_recs, &sz);
                compress.elem_count = sz;
            }
        }
        break;
    case 2:
        rc = KDataBufferResize(buffer, 6);
        if (rc == 0) {
            rc = KDataBufferMakeBytes(&compress, 10 * self->leng_recs);
            if (rc == 0) {
                ((uint8_t *)buffer->base)[0] = header;
                vlen_encodeU1(((uint8_t *)buffer->base) + 1, 5, &sz, self->leng_recs);
                buffer->elem_count = sz + 1;

                rc = serialize_lengths(&compress, 0, self->length, self->leng_recs, &sz);
                compress.elem_count = sz;
                if (rc == 0) {
                    rc = serialize_lengths(&compress, sz, self->leng_run, self->leng_recs, &sz);
                    compress.elem_count += sz;
                }
            }
        }
        break;
    case 3:
        rc = KDataBufferResize(buffer, 11);
        if (rc == 0) {
            rc = KDataBufferMakeBytes(&compress, 10 * self->leng_recs + 5 * self->data_recs);
            if (rc == 0) {
                ((uint8_t *)buffer->base)[0] = header;
                vlen_encodeU1(((uint8_t *)buffer->base) + 1, 5, &sz, self->leng_recs);
                buffer->elem_count = sz + 1;
                vlen_encodeU1(((uint8_t *)buffer->base) + 1 + sz, 5, &sz, self->data_recs);
                buffer->elem_count += sz;
                
                rc = serialize_lengths(&compress, 0, self->length, self->leng_recs, &sz);
                compress.elem_count = sz;
                if (rc == 0) {
                    rc = serialize_lengths(&compress, compress.elem_count, self->leng_run, self->leng_recs, &sz);
                    compress.elem_count += sz;
                    if (rc == 0) {
                        rc = serialize_lengths(&compress, compress.elem_count, self->data_run, self->data_recs, &sz);
                        compress.elem_count += sz;
                    }
                }
            }
        }
        break;
    }
    if (rc == 0 && compress.base) {
        uint64_t hsize = buffer->elem_count;
        
        rc = KDataBufferResize(buffer, hsize + compress.elem_count);
        if (rc == 0) {
            if (version == 0)
                memcpy(((uint8_t *)buffer->base) + hsize, compress.base, compress.elem_count);
            else {
                z_stream zs;
                int zr;
                
                memset(&zs, 0, sizeof(zs));
                
                assert(compress.elem_count >> 32 == 0);
                
                zs.next_out = ((Bytef *)buffer->base) + hsize;
                zs.avail_out = (uInt)compress.elem_count;
                
                zs.next_in = compress.base;
                zs.avail_in = (uInt)compress.elem_count;
                
                zr = deflateInit2(&zs, Z_BEST_SPEED, Z_DEFLATED, -15, 9, Z_DEFAULT_STRATEGY);
                switch (zr) {
                case Z_OK:
                    break;
                case Z_MEM_ERROR:
                    rc = RC(rcVDB, rcPagemap, rcWriting, rcMemory, rcExhausted);
                    break;
                default:
                    rc = RC(rcVDB, rcPagemap, rcWriting, rcParam, rcInvalid);
                }
                if (rc == 0) {
                    for ( ; ; ) {
                        zr = deflate(&zs, Z_FINISH);
                        if (zr == Z_OK) {
                            unsigned offset = (unsigned int)( zs.next_out - (Bytef *)buffer->base );
                            rc = KDataBufferResize(buffer, buffer->elem_count * 2);
                            if (rc)
                                break;
                            zs.next_out = ((Bytef *)buffer->base) + offset;
                            zs.avail_out = (uInt)(buffer->elem_count - offset);
                        }
                        else if (zr == Z_STREAM_END) {
                            KDataBufferResize(buffer, zs.total_out + hsize);
                            break;
                        }
                        else {
                            rc = RC(rcVDB, rcPagemap, rcWriting, rcParam, rcInvalid);
                            break;
                        }
                    }
                    deflateEnd(&zs);
                }
            }
        }
    }
    KDataBufferWhack(&compress);
    if (rc == 0)
        *size = buffer->elem_count;
    
    return rc;
}

rc_t PageMapSerialize ( const PageMap *self, KDataBuffer *buffer, uint64_t offset, uint64_t *size ) {
    rc_t rc;
    KDataBuffer temp;

    assert(buffer->elem_bits == 8);
    assert(buffer->bit_offset == 0);
    
    rc = KDataBufferMakeBytes(&temp, 0);
    if (rc == 0) {
        uint64_t sz;
        
        rc = serialize(self, &temp, &sz);
        if (rc == 0) {
            rc = KDataBufferResize(buffer, offset + sz);
            if (rc == 0)
                memcpy(&((char *)buffer->base)[offset], temp.base, sz);
            *size = sz;
        }
        KDataBufferWhack(&temp);
    }
    return rc;
}

static
rc_t PageMapDeserialize_v0( PageMap **lhs, const uint8_t *src, uint64_t ssize, uint32_t row_count ) {
    uint8_t variant = *src & 3;
    const uint8_t * const endp = src + ssize;
    uint64_t val;
    uint64_t sz;
    rc_t rc = 0;
    
    ++src;
    switch (variant) {
    case 0:
        rc = vlen_decodeU1(&val, src, endp - src, &sz);
        if (rc == 0)
            return PageMapNewFixedRowLength( lhs, row_count, (uint32_t)val );
        break;
    case 1:
        {
            uint32_t row_length;
            
            rc = vlen_decodeU1( &val, src, endp - src, &sz );
            if (rc == 0) {
                src += sz;
                row_length = (uint32_t)val;
                rc = vlen_decodeU1( &val, src, endp - src, &sz );
                if (rc == 0) {
                    src += sz;
                    /* i'm lying about the row_count, this will create val entries
                     * in the data_run array and that is how many i need
                     * i'll then decompress into the data_run array.
                     */
                    rc = PageMapNewFixedRowLength( lhs, (uint32_t)val, row_length );
                    if (rc == 0) {
                        (*lhs)->leng_run[0] = row_count;
                        rc = deserialize_lengths( (*lhs)->data_run, (unsigned int)val, src, endp - src, &sz );
                        if (rc == 0)
                        {
                            uint32_t ix;
                            uint32_t sum;
                            uint32_t *pdr = (**lhs).data_run;
                                
                            for (sum = ix = 0; ix < (**lhs).data_recs; ++ix)
                                sum += *pdr++;
                            (**lhs).leng_run[0] = sum;
                            return 0;
                        }
                        PageMapRelease(*lhs);
                    }
                }
            }
        }
        break;
    case 2:
        rc = vlen_decodeU1( &val, src, endp - src, &sz );
        if (rc == 0) {
            src += sz;
            rc = PageMapNewFixedRowLength( lhs, row_count, 1 );
            if (rc == 0) {
                do {
                    rc = PageMapGrow( *lhs, (uint32_t)val, 0 );
                    if (rc)
                        break;
                    rc = deserialize_lengths( (*lhs)->length, (unsigned)val, src, endp - src, &sz );
                    if (rc)
                        break;
                    src += sz;
                    rc = deserialize_lengths( (*lhs)->leng_run, (unsigned)val, src, endp - src, &sz );
                    if (rc == 0) {
                        (*lhs)->leng_recs = (uint32_t)val;
                        (*lhs)->data_recs = row_count;
                        return 0;
                    }
                } while (0);
                PageMapRelease(*lhs);
                *lhs = 0;
                break;
            }
        }
        break;
    case 3:
        do {
            uint32_t leng_recs;
            uint32_t data_recs;
            
            rc = vlen_decodeU1( &val, src, endp - src, &sz );
            if (rc)
                break;
            src += sz;
            leng_recs = (uint32_t)val;
            
            rc = vlen_decodeU1( &val, src, endp - src, &sz );
            if (rc)
                break;
            src += sz;
            data_recs = (uint32_t)val;
            
            rc = PageMapNew(lhs, 0);
            if (rc == 0) {
                do {
                    rc = PageMapGrow(*lhs, leng_recs, data_recs);
                    if (rc)
                        break;
                    rc = deserialize_lengths( (*lhs)->length, leng_recs, src, endp - src, &sz );
                    if (rc)
                        break;
                    src += sz;
                    rc = deserialize_lengths( (*lhs)->leng_run, leng_recs, src, endp - src, &sz );
                    if (rc)
                        break;
                    src += sz;
                    rc = deserialize_lengths( (*lhs)->data_run, data_recs, src, endp - src, &sz );
                    if (rc == 0) {
                        (*lhs)->leng_recs = leng_recs;
                        (*lhs)->data_recs = data_recs;
                        return 0;
                    }
                } while (0);
                PageMapRelease(*lhs);
                *lhs = 0;
                break;
            }
        } while (0);
        break;
    }
    return rc;
}

static
rc_t PageMapDeserialize_v1( PageMap **lhs, const uint8_t *Src, uint64_t ssize, uint32_t row_count ) {
    const uint8_t *src = Src;
    const uint8_t * const endp = src + ssize;
    uint8_t variant = *src & 3;
    uint64_t bsize;
    uint64_t hsize;
    uint64_t val;
    uint64_t sz;
    rc_t rc = 0;
    KDataBuffer decompress;
    z_stream zs;
    int zr;
    
    switch (variant) {
    case 0:
        return PageMapDeserialize_v0(lhs, src, ssize, row_count);
    case 1:
        ++src;
        rc = vlen_decodeU1(&val, src, endp - src, &sz);
        if (rc == 0) {
            src += sz;
            rc = vlen_decodeU1(&val, src, endp - src, &sz);
            if (rc == 0) {
                src += sz;
                hsize = src - Src;
                bsize = 5 * val;
            }
        }
        break;
    case 2:
        ++src;
        rc = vlen_decodeU1(&val, src, endp - src, &sz);
        if (rc == 0) {
            src += sz;
            hsize = src - Src;
            bsize = 10 * val;
        }
        break;
    case 3:
        ++src;
        rc = vlen_decodeU1(&val, src, endp - src, &sz);
        if (rc == 0) {
            src += sz;
            bsize = 10 * val;
            rc = vlen_decodeU1(&val, src, endp - src, &sz);
            if (rc == 0) {
                src += sz;
                hsize = src - Src;
                bsize += 5 * val;
            }
        }
        break;
    default:
        return RC(rcVDB, rcPagemap, rcConstructing, rcData, rcInvalid);
    }
    if (rc)
        return rc;
    
    rc = KDataBufferMakeBytes(&decompress, hsize + bsize);
    if (rc)
        return rc;

    memcpy(decompress.base, Src, hsize);
    memset(&zs, 0, sizeof(zs));
    
    zs.next_in = (Bytef *)src;
    assert((endp - src) == (uInt)(endp - src));
    zs.avail_in = (uInt)(endp - src);
    
    zs.next_out = ((Bytef *)decompress.base) + hsize;
    assert(bsize == (uInt)(bsize));
    zs.avail_out = (uInt)bsize;
    
    zr = inflateInit2(&zs, -15);
    if (zr == Z_OK) {
        zr = inflate(&zs, Z_FINISH);
        if (zr != Z_STREAM_END)
            rc = RC(rcVDB, rcPagemap, rcConstructing, rcData, rcInvalid);
        inflateEnd(&zs);
    }
    else
        rc = RC(rcVDB, rcPagemap, rcConstructing, rcMemory, rcExhausted);
    
    if (rc == 0)
        rc = PageMapDeserialize_v0(lhs, decompress.base, hsize + zs.total_out, row_count);
    KDataBufferWhack(&decompress);
    return rc;
}

rc_t PageMapDeserialize ( PageMap **lhs, const void *src, uint64_t ssize, uint64_t row_count ) {
    rc_t rc;

    if ((uint32_t)row_count != row_count)
        return RC(rcVDB, rcPagemap, rcConstructing, rcParam, rcTooBig);
        
    if (lhs == NULL)
        return RC(rcVDB, rcPagemap, rcConstructing, rcParam, rcNull);
    
    *lhs = NULL;
    if (src == NULL || ssize == 0)
        return 0;
    switch (*(const uint8_t *)src >> 2) {
    case 0:
        rc = PageMapDeserialize_v0(lhs, src, ssize, (uint32_t)row_count);
        break;
    case 1:
        rc = PageMapDeserialize_v1(lhs, src, ssize, (uint32_t)row_count);
        break;
    default:
        return RC(rcVDB, rcPagemap, rcConstructing, rcData, rcBadVersion);
    }
    if (rc == 0)
        (**lhs).row_count = (uint32_t)row_count;
    return rc;
}

static
rc_t PageMapDestroy( PageMap *that )
{
    KDataBufferWhack( &that->estorage );
    KDataBufferWhack( &that->cstorage );
    free( that );
    return 0;
}

rc_t PageMapRelease ( const PageMap *self ) {
    if (self) {
        if (KRefcountDrop(&self->refcount, "PageMap") == krefWhack)
          return PageMapDestroy((PageMap *)self);
    }
    return 0;
}

rc_t PageMapAddRef ( const PageMap *self ) {
    if (self)
        KRefcountAdd(&self->refcount, "PageMap");
    return 0;
}

uint32_t PageMapGetIdxRowInfo ( const PageMap *cself, uint32_t idx, uint32_t *starting_element ) {
    rc_t rc;

	if (idx == 0 || cself->data_recs == 1) {
        if (starting_element)
            *starting_element = 0;
        if (cself->leng_recs == 0)
            return 0;
        return cself->length[0];
    }
    if (cself->leng_recs == 1) {
        if (starting_element == NULL)
            return cself->length[0];
        
        if (cself->data_recs == cself->row_count) {
        /* fixed row length and no rle on data */
            if (starting_element)
                *starting_element = idx * cself->length[0];
            return cself->length[0];
        }
    }
    
    rc = PageMapExpand(cself, idx);
    if (rc)
        return 0;

    if (starting_element)
        *starting_element = cself->exp_offset[idx];
    return cself->leng_recs > 1 ? cself->exp_length[idx] : cself->length[0];
}

int PageMapCompare( const PageMap *a, const PageMap *b ) {
    unsigned i;
    unsigned j;
    unsigned i_rl;
    unsigned j_rl;
    
    if (
        a == b || (
        a->leng_recs == b->leng_recs &&
        a->data_recs == b->data_recs &&
        memcmp(a->length, b->length, sizeof(a->length[0]) * a->leng_recs) == 0 &&
        memcmp(a->leng_run, b->leng_run, sizeof(a->leng_run[0]) * a->leng_recs) == 0 &&
        memcmp(a->data_run, b->data_run, sizeof(a->data_run[0]) * a->data_recs) == 0
    ))
        return 2;
    
    i = j = 0;
    i_rl = a->leng_run[0];
    j_rl = b->leng_run[0];
    
    while (i < a->leng_recs && j < b->leng_recs) {
        if (a->length[i] != b->length[j])
            return 0;
        if (i_rl < j_rl) {
            j_rl -= i_rl;
            i_rl = 0;
        }
        else {
            i_rl -= j_rl;
            j_rl = 0;
        }
        if (i_rl == 0)
            i_rl = a->leng_run[++i];
        if (j_rl == 0)
            j_rl = b->leng_run[++j];
    }
    if (i != a->leng_recs || j != b->leng_recs)
        return -1;
    else
        return 1;
}

int PageMapCompareStatic ( const PageMap *a, const PageMap *b ) {
    if (
	  a->data_recs != 1 ||
	  a->leng_recs != 1 ||
	  a->leng_run[0] != a->data_run[0]
    )
        return -1; /* *a is not static */
    
    if (a == b)
        return 2; /* static and identical */

    if (
	  b->data_recs != 1 ||
	  b->leng_recs != 1 ||
	  b->leng_run[0] != b->data_run[0]
    )
        return -1; /* *b is not static */
    
    if (a->length[0] == b->length[0] && a->leng_run[0] <= b->leng_run[0])
        /* static and compatible, but we want to prevent deeper inspection
         * so we're lying and saying that they are identical */
        return 2;

    return 0;  /* not same */
}

rc_t PageMapAppend( PageMap *self, const PageMap *other ) {
    KDataBuffer cstorage;
    rc_t rc;
    
    rc = KDataBufferMake(&cstorage, sizeof(self->length[0]) * 8, (self->leng_recs + other->leng_recs) * 2 + self->data_recs + other->data_recs);
    if (rc == 0) {
        uint32_t *length = cstorage.base;
        uint32_t *leng_run = length + self->leng_recs + other->leng_recs;
        uint32_t *data_run = leng_run + self->leng_recs + other->leng_recs;
        
        memcpy(length                  , self->length , self->leng_recs  * sizeof(length[0]));
        memcpy(length + self->leng_recs, other->length, other->leng_recs * sizeof(length[0]));

        memcpy(leng_run                  , self->leng_run , self->leng_recs  * sizeof(leng_run[0]));
        memcpy(leng_run + self->leng_recs, other->leng_run, other->leng_recs * sizeof(leng_run[0]));
        
        memcpy(data_run                  , self->data_run , self->data_recs  * sizeof(data_run[0]));
        memcpy(data_run + self->data_recs, other->data_run, other->data_recs * sizeof(data_run[0]));
        
        KDataBufferWhack(&self->cstorage);
        self->cstorage = cstorage;

        self->length = length;
        self->leng_run = leng_run;
        self->data_run = data_run;
        
        self->leng_recs += other->leng_recs;
        self->data_recs += other->data_recs;
        self->row_count += other->row_count;
        self->reserve_leng = self->leng_recs;
        self->reserve_data = self->data_recs;
        
        return 0;
    }
    return rc;
}
