# This module is a more generalzed debugging facility. It can be configured
# to log information to serveral different facilities, stdout, stderr,
# syslog and/or a file.
#
# The logger object is initialized with following key-value pairs.
#    stdout     Log to Stdout
#    stderr     Log to Stderr
#    filename   File name to log entries into
#    syslog     Log to syslog
#    ident      Identity argument for openlog()
#    facility   Syslog facility to log to             Defaults to 'user'
#    priority   Default syslog priority to log        Defaults to 'notice'
#    logopt     Log Options parameter for openlog()   
#
# Functions
#  log(msg [, pri])  Log an entry
#                   Where
#                      msg   is message to log
#                      pri   is priority to use in syslog

package Spong::Logger;
use Sys::Syslog qw(:DEFAULT setlogsock);
use FileHandle;
use Carp;

# Constructor.  A simple constructor that just sets all of the instance vars
# according to what was passed in. 

sub new {
   my( $proto, %arg ) = @_;
   my( $class ) = ref($proto) || $proto;
   my $self = {};

   $self->{'stdout'} = defined $arg{stdout} ? $arg{stdout} : 0;
   $self->{'stderr'} = defined $arg{stderr} ? $arg{stderr} : 0;
   $self->{'syslog'} = defined $arg{syslog} ? $arg{syslog} : 0;
   $self->{'ident'} = defined $arg{ident} ? $arg{ident} : "";
   $self->{'facility'} = defined $arg{facility} ? $arg{facility} : 'user';
   $self->{'logopt'} = defined $arg{logopt} ? $arg{logopt} : "";
   $self->{'priority'} = defined $arg{priority} ? $arg{priority} : 'info';
   $self->{'filename'} = $arg{filename} if defined $arg{filename};

   # If we are logging to syslog, open a proper channel
   if ($self->{'syslog'}) {
      setlogsock 'unix';
   }

   # If we are logging to a file, open the file 
   if ( defined $self->{'filename'}  && $self->{'filename'} ne "" ) {
      $self->{'fh'} = new FileHandle ">> " . $arg{filename};
      if (! defined $self->{'fh'} ) {
          warn "Could not open file $arg{filename}";
      } else {
         $self->{'fh'}->autoflush(1);
      }
   }
      
   bless ($self,$class);
   return $self;
}

sub DESTROY {
  my( $self ) = @_;

  # If were are logging to a file, close it
  if (defined $self->{'fh'}) { $self->{'fh'}->close();}
  # And if using syslog, closelog for completion
  if ( $self->{'syslog'} ) { closelog(); }

}

sub log {
   my( $self, $message, $pri ) = @_;
   $pri = $self->{priority} if ! defined $pri;

   if ($self->{stdout}) { print scalar localtime() , " ", $message,"\n"; }

   if ($self->{stderr}) { print STDERR scalar localtime(), " ",$message,"\n"; }

   if ($self->{syslog}) {
      openlog $self->{'ident'},$self->{'logopt'},$self->{'facility'};
      syslog $pri,$message,'';
      closelog;
   }

   if ($self->{'fh'} ) {
      my $fh = $self->{'fh'};
      print $fh scalar(localtime())," ",$message,"\n";
   }

}

1;  # I'm included perl code, I need this line

