/* specter_BASE.c
 *
 * specter input plugin for 
 * 	o MAC addresses
 * 	o NFMARK field
 * 	o TIME
 * 	o Interface names
 * 	o IP header
 * 	o TCP header
 * 	o UDP header
 * 	o ICMP header
 * 	o AH/ESP header
 *
 * (C) 2000-2001 by Harald Welte <laforge@gnumonks.org>
 * (C) 2004 by Michal Kwiatkowski <ruby@joker.linuxstuff.pl>
 */

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 
 *  as published by the Free Software Foundation
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/ip.h>
#include <netinet/in.h>
#include <linux/tcp.h> /* we need tcp->ece and tcp->cwr */
#include <netinet/ip_icmp.h>
#include <netinet/udp.h>
#include <specter/specter.h>

#define BASE_RETS_NUM 54
static specter_iret_t base_rets[BASE_RETS_NUM] = {
/*  0 */{ .type = SPECTER_IRET_STRING, .name = "raw.mac" },
	{ .type = SPECTER_IRET_RAW, .name = "raw.pkt" },
	{ .type = SPECTER_IRET_UINT32, .name = "raw.pktlen" },

/*  3 */{ .type = SPECTER_IRET_STRING, .name = "oob.prefix" },
	{ .type = SPECTER_IRET_UINT32, .name = "oob.time_sec" },
	{ .type = SPECTER_IRET_UINT32, .name = "oob.time_usec" },
	{ .type = SPECTER_IRET_UINT32, .name = "oob.mark" },
	{ .type = SPECTER_IRET_STRING, .name = "oob.in" },
	{ .type = SPECTER_IRET_STRING, .name = "oob.out" },

/*  9 */{ .type = SPECTER_IRET_IPADDR, .name = "ip.saddr" },
	{ .type = SPECTER_IRET_IPADDR, .name = "ip.daddr" },
	{ .type = SPECTER_IRET_UINT8, .name = "ip.protocol" },
	{ .type = SPECTER_IRET_UINT8, .name = "ip.tos" },
	{ .type = SPECTER_IRET_UINT8, .name = "ip.ttl" },
	{ .type = SPECTER_IRET_UINT16, .name = "ip.totlen" },
	{ .type = SPECTER_IRET_UINT8, .name = "ip.ihl" },
	{ .type = SPECTER_IRET_UINT8, .name = "ip.version" },
	{ .type = SPECTER_IRET_UINT16, .name = "ip.csum" },
	{ .type = SPECTER_IRET_UINT16, .name = "ip.id" },
	{ .type = SPECTER_IRET_UINT16, .name = "ip.fragoff" },
	{ .type = SPECTER_IRET_BOOL, .name = "ip.rf" },
	{ .type = SPECTER_IRET_BOOL, .name = "ip.df" },
	{ .type = SPECTER_IRET_BOOL, .name = "ip.mf" },
	{ .type = SPECTER_IRET_STRING, .name = "ip.opt" },

/* 24 */{ .type = SPECTER_IRET_UINT16, .name = "tcp.sport" },
	{ .type = SPECTER_IRET_UINT16, .name = "tcp.dport" },
	{ .type = SPECTER_IRET_UINT32, .name = "tcp.seq" },
	{ .type = SPECTER_IRET_UINT32, .name = "tcp.ackseq" },
	{ .type = SPECTER_IRET_UINT16, .name = "tcp.window" },
	{ .type = SPECTER_IRET_UINT16, .name = "tcp.csum" },
	{ .type = SPECTER_IRET_BOOL, .name = "tcp.urg" },
	{ .type = SPECTER_IRET_UINT16, .name = "tcp.urgp" },
	{ .type = SPECTER_IRET_BOOL, .name = "tcp.ack" },
	{ .type = SPECTER_IRET_BOOL, .name = "tcp.psh" },
	{ .type = SPECTER_IRET_BOOL, .name = "tcp.rst" },
	{ .type = SPECTER_IRET_BOOL, .name = "tcp.syn" },
	{ .type = SPECTER_IRET_BOOL, .name = "tcp.fin" },
	{ .type = SPECTER_IRET_UINT8, .name = "tcp.res" },
	{ .type = SPECTER_IRET_BOOL, .name = "tcp.cwr" },
	{ .type = SPECTER_IRET_BOOL, .name = "tcp.ece" },
	{ .type = SPECTER_IRET_UINT8, .name = "tcp.doff" },
	{ .type = SPECTER_IRET_STRING, .name = "tcp.opt" },

/* 42 */{ .type = SPECTER_IRET_UINT16, .name = "udp.sport" },
	{ .type = SPECTER_IRET_UINT16, .name = "udp.dport" },
	{ .type = SPECTER_IRET_UINT16, .name = "udp.len" },
	{ .type = SPECTER_IRET_UINT16, .name = "udp.csum" },

/* 46 */{ .type = SPECTER_IRET_UINT8, .name = "icmp.type" },
	{ .type = SPECTER_IRET_UINT8, .name = "icmp.code" },
	{ .type = SPECTER_IRET_UINT16, .name = "icmp.csum" },
	{ .type = SPECTER_IRET_UINT16, .name = "icmp.echoid" },
	{ .type = SPECTER_IRET_UINT16, .name = "icmp.echoseq" },
	{ .type = SPECTER_IRET_IPADDR, .name = "icmp.gateway" },
	{ .type = SPECTER_IRET_UINT16, .name = "icmp.fragmtu" },

/* 53 */{ .type = SPECTER_IRET_UINT32, .name = "ahesp.spi" },
};


/***********************************************************************
 * 			RAW HEADER
 ***********************************************************************/
static int interp_raw(ulog_packet_msg_t *pkt)
{
	unsigned char *p;
	int i;
	char *buf, *curr_buf;
	specter_iret_t *ret = &base_rets[0];

	if (pkt->mac_len) {
		buf = (char *) malloc(3 * pkt->mac_len + 1);
		if (!buf) {
			specter_log(SPECTER_ERROR, "Couldn't allocate buffer: %s.\n",
					strerror(errno));
			return -1;
		}
		*buf = '\0';

		curr_buf = buf;
		p = pkt->mac;
		for (i = 0; i < pkt->mac_len; i++, p++)
			curr_buf += sprintf(curr_buf, "%02x%c", *p, i==pkt->mac_len-1 ? '\0':':');
		ret[0].value.ptr = buf;
		ret[0].flags |= SPECTER_RETF_FREE | SPECTER_RETF_VALID;
	}

	/* include pointer to raw ipv4 packet */
	ret[1].value.ptr = pkt->payload;
	ret[1].flags |= SPECTER_RETF_VALID;
	ret[2].value.ui32 = pkt->data_len;
	ret[2].flags |= SPECTER_RETF_VALID;

	return 0;
}

/***********************************************************************
 * 			OUT OF BAND
 ***********************************************************************/
static int interp_oob(ulog_packet_msg_t *pkt)
{
	specter_iret_t *ret = &base_rets[3];

	ret[0].value.ptr = pkt->prefix;
	ret[0].flags |= SPECTER_RETF_VALID;

	/* god knows why timestamp_usec contains crap if timestamp_sec == 0
	 * if (pkt->timestamp_sec || pkt->timestamp_usec) { */
	if (pkt->timestamp_sec) {
		ret[1].value.ui32 = pkt->timestamp_sec;
		ret[1].flags |= SPECTER_RETF_VALID;
		ret[2].value.ui32 = pkt->timestamp_usec;
		ret[2].flags |= SPECTER_RETF_VALID;
	}

	ret[3].value.ui32 = pkt->mark;
	ret[3].flags |= SPECTER_RETF_VALID;
	ret[4].value.ptr = pkt->indev_name;
	ret[4].flags |= SPECTER_RETF_VALID;
	ret[5].value.ptr = pkt->outdev_name;
	ret[5].flags |= SPECTER_RETF_VALID;
	
	return 0;
}

/***********************************************************************
 * 			IP HEADER
 ***********************************************************************/
static int interp_iphdr(ulog_packet_msg_t *pkt)
{
	specter_iret_t *ret = &base_rets[9];
	struct iphdr *iph = (struct iphdr *) pkt->payload;
	int optsize = 4 * iph->ihl - sizeof(struct iphdr);

	ret[0].value.ui32 = ntohl(iph->saddr);
	ret[0].flags |= SPECTER_RETF_VALID;
	ret[1].value.ui32 = ntohl(iph->daddr);
	ret[1].flags |= SPECTER_RETF_VALID;
	ret[2].value.ui8 = iph->protocol;
	ret[2].flags |= SPECTER_RETF_VALID;
	ret[3].value.ui8 = iph->tos;
	ret[3].flags |= SPECTER_RETF_VALID;
	ret[4].value.ui8 = iph->ttl;
	ret[4].flags |= SPECTER_RETF_VALID;
	ret[5].value.ui16 = ntohs(iph->tot_len);
	ret[5].flags |= SPECTER_RETF_VALID;
	ret[6].value.ui8 = iph->ihl;
	ret[6].flags |= SPECTER_RETF_VALID;
	ret[7].value.ui8 = iph->version;
	ret[7].flags |= SPECTER_RETF_VALID;
	ret[8].value.ui16 = ntohs(iph->check);
	ret[8].flags |= SPECTER_RETF_VALID;
	ret[9].value.ui16 = ntohs(iph->id);
	ret[9].flags |= SPECTER_RETF_VALID;
	ret[10].value.ui16 = ntohs(iph->frag_off) & IP_OFFMASK;
	ret[10].flags |= SPECTER_RETF_VALID;
	if (ntohs(iph->frag_off) & IP_RF)
		ret[11].value.b = 1;
	else
		ret[11].value.b = 0;
	ret[11].flags |= SPECTER_RETF_VALID;
	if (ntohs(iph->frag_off) & IP_DF)
		ret[12].value.b = 1;
	else
		ret[12].value.b = 0;
	ret[12].flags |= SPECTER_RETF_VALID;
	if (ntohs(iph->frag_off) & IP_MF)
		ret[13].value.b = 1;
	else
		ret[13].value.b = 0;
	ret[13].flags |= SPECTER_RETF_VALID;

	if (optsize > 0) {
		int i;
		unsigned char *opt = (char *)iph + sizeof(struct iphdr);
		char *buf, *curr_buf;

		buf = (char *) malloc(2 * optsize + 1);
		if (!buf) {
			specter_log(SPECTER_ERROR, "Couldn't allocate buffer: %s.\n",
					strerror(errno));
			return -1;
		}

		for (curr_buf = buf, i = 0; i < optsize; i++)
			curr_buf += sprintf(curr_buf, "%02X", opt[i]);

		curr_buf = '\0';
		ret[14].value.ptr = buf;
		ret[14].flags |= SPECTER_RETF_FREE | SPECTER_RETF_VALID;
	}

	return 0;
}

/***********************************************************************
 * 			TCP HEADER
 ***********************************************************************/
static int interp_tcphdr(ulog_packet_msg_t *pkt)
{
	struct iphdr *iph = (struct iphdr *) pkt->payload;
	void *protoh = (u_int32_t *)iph + iph->ihl;
	struct tcphdr *tcph = (struct tcphdr *) protoh;
	specter_iret_t *ret = &base_rets[24];
	int optsize = 4 * tcph->doff - sizeof(struct tcphdr);

	ret[0].value.ui16 = ntohs(tcph->source);
	ret[0].flags |= SPECTER_RETF_VALID;
	ret[1].value.ui16 = ntohs(tcph->dest);
	ret[1].flags |= SPECTER_RETF_VALID;
	ret[2].value.ui32 = ntohl(tcph->seq);
	ret[2].flags |= SPECTER_RETF_VALID;
	ret[3].value.ui32 = ntohl(tcph->ack_seq);
	ret[3].flags |= SPECTER_RETF_VALID;
	ret[4].value.ui16 = ntohs(tcph->window);
	ret[4].flags |= SPECTER_RETF_VALID;
	ret[5].value.ui16 = ntohs(tcph->check);
	ret[5].flags |= SPECTER_RETF_VALID;

	ret[6].value.b = tcph->urg;
	ret[6].flags |= SPECTER_RETF_VALID;
	if (tcph->urg) {
		ret[7].value.ui16 = ntohs(tcph->urg_ptr);
		ret[7].flags |= SPECTER_RETF_VALID;
	}
	ret[8].value.b = tcph->ack;
	ret[8].flags |= SPECTER_RETF_VALID;
	ret[9].value.b = tcph->psh;
	ret[9].flags |= SPECTER_RETF_VALID;
	ret[10].value.b = tcph->rst;
	ret[10].flags |= SPECTER_RETF_VALID;
	ret[11].value.b = tcph->syn;
	ret[11].flags |= SPECTER_RETF_VALID;
	ret[12].value.b = tcph->fin;
	ret[12].flags |= SPECTER_RETF_VALID;

	ret[13].value.ui8 = tcph->res1;
	ret[13].flags |= SPECTER_RETF_VALID;
	ret[14].value.b = tcph->cwr;
	ret[14].flags |= SPECTER_RETF_VALID;
	ret[15].value.b = tcph->ece;
	ret[15].flags |= SPECTER_RETF_VALID;
	ret[16].value.ui8 = tcph->doff;
	ret[16].flags |= SPECTER_RETF_VALID;

	if (optsize > 0) {
		int i;
		unsigned char *opt = (char *)tcph + sizeof(struct tcphdr);
		char *buf, *curr_buf;

		buf = (char *) malloc(2 * optsize + 1);
		if (!buf) {
			specter_log(SPECTER_ERROR, "Couldn't allocate buffer: %s.\n",
					strerror(errno));
			return -1;
		}

		for (curr_buf = buf, i = 0; i < optsize; i++)
			curr_buf += sprintf(curr_buf, "%02X", opt[i]);

		curr_buf = '\0';
		ret[17].value.ptr = buf;
		ret[17].flags |= SPECTER_RETF_FREE | SPECTER_RETF_VALID;
	}

	return 0;
}

/***********************************************************************
 * 			UDP HEADER
 ***********************************************************************/
static int interp_udp(ulog_packet_msg_t *pkt)
{
	struct iphdr *iph = (struct iphdr *) pkt->payload;
	void *protoh = (u_int32_t *)iph + iph->ihl;
	struct udphdr *udph = protoh;
	specter_iret_t *ret = &base_rets[42];

	ret[0].value.ui16 = ntohs(udph->source);
	ret[0].flags |= SPECTER_RETF_VALID;
	ret[1].value.ui16 = ntohs(udph->dest);
	ret[1].flags |= SPECTER_RETF_VALID;
	ret[2].value.ui16 = ntohs(udph->len);
	ret[2].flags |= SPECTER_RETF_VALID;
	ret[3].value.ui16 = ntohs(udph->check);
	ret[3].flags |= SPECTER_RETF_VALID;

	return 0;
}

/***********************************************************************
 * 			ICMP HEADER
 ***********************************************************************/
static int interp_icmp(ulog_packet_msg_t *pkt)
{
	struct iphdr *iph = (struct iphdr *) pkt->payload;
	void *protoh = (u_int32_t *)iph + iph->ihl;
	struct icmphdr *icmph = protoh;
	specter_iret_t *ret = &base_rets[46];

	ret[0].value.ui8 = icmph->type;
	ret[0].flags |= SPECTER_RETF_VALID;
	ret[1].value.ui8 = icmph->code;
	ret[1].flags |= SPECTER_RETF_VALID;
	ret[2].value.ui16 = ntohs(icmph->checksum);
	ret[2].flags |= SPECTER_RETF_VALID;

	switch(icmph->type) {
		case ICMP_ECHO:
		case ICMP_ECHOREPLY:
			ret[3].value.ui16 = ntohs(icmph->un.echo.id);
			ret[3].flags |= SPECTER_RETF_VALID;
			ret[4].value.ui16 = ntohs(icmph->un.echo.sequence);
			ret[4].flags |= SPECTER_RETF_VALID;
			break;
		case ICMP_REDIRECT:
		case ICMP_PARAMETERPROB:
			ret[5].value.ui32 = ntohl(icmph->un.gateway);
			ret[5].flags |= SPECTER_RETF_VALID;
			break;
		case ICMP_DEST_UNREACH:
			if (icmph->code == ICMP_FRAG_NEEDED) {
				ret[6].value.ui16 = ntohs(icmph->un.frag.mtu);
				ret[6].flags |= SPECTER_RETF_VALID;
			}
			break;
	}
	return 0;
}

/***********************************************************************
 * 			IPSEC HEADER 
 ***********************************************************************/
static int interp_ahesp(ulog_packet_msg_t *pkt)
{

#if 0
	specter_iret_t *ret = &base_rets[53];
	struct iphdr *iph = (struct iphdr *) pkt->payload;
	void *protoh = (u_int32_t *) (iph + iph->ihl);
	struct esphdr *esph = protoh;

	ret[0].value.ui32 = ntohl(esph->spi);
	ret[0].flags |= SPECTER_RETF_VALID;
#endif

	return 0;
}


static int interp_base(ulog_packet_msg_t *pkt)
{
	struct iphdr *iph = (struct iphdr *) pkt->payload;

	if (interp_raw(pkt) == -1 || interp_oob(pkt) == -1)
		return -1;

	if (interp_iphdr(pkt) == -1)
		return -1;

	switch (iph->protocol) {
		case IPPROTO_TCP:
			if (interp_tcphdr(pkt) == -1)
				return -1;
			break;
		case IPPROTO_UDP:
			if (interp_udp(pkt) == -1)
				return -1;
			break;
		case IPPROTO_ICMP:
			if (interp_icmp(pkt) == -1)
				return -1;
			break;
		case IPPROTO_ESP:
			if (interp_ahesp(pkt) == -1)
				return -1;
			break;
	}

	return 0;
}


static specter_input_t base_ip = {
	.name = "base",
	.input = &interp_base
};

void _init(void)
{
	if (register_input(&base_ip, base_rets, BASE_RETS_NUM, 0) == -1) {
		specter_log(SPECTER_FATAL, "Couldn't register.\n");
		exit(EXIT_FAILURE);
	}
}

