# GNU Solfege - free ear training software
# Copyright (C) 2007, 2008 Tom Cato Amundsen
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import time
import subprocess
import os

import gobject
import gtk

import lessonfile
import gu
import configureoutput
import filesystem
from multipleintervalconfigwidget import IntervalCheckBox
import ElementTree as et


class NameIt(object):
    def __init__(self, name, lilycode):
        self.m_name = name
        self.m_lilycode = lilycode

class Section(list):
    def __init__(self, section_title, line_len):
        self.m_title = section_title
        self.m_line_len = line_len

class BaseSheetWriter(object):
    lilyout = 'lilyout'
    def __init__(self, title):
        self.m_title = title
        self.m_data = []
    def create_outdir(self, directory):
        if not os.path.exists(directory):
            os.mkdir(directory)
        if not os.path.exists(os.path.join(directory, self.lilyout)):
            os.mkdir(os.path.join(directory, self.lilyout))
    def new_section(self, section_title, line_len):
        """
        Return a list where we can append questions.
        """
        v = Section(section_title, line_len)
        self.m_data.append(v)
        return v
    def set_title(self, s):
        self.m_title = s

class HtmlSheetWriter(BaseSheetWriter):
    def _write(self, k, directory, filename):
        assert k in ('question', 'answer')
        f = open(os.path.join(directory, filename), 'w')
        print >> f, "<html>"
        print >> f, "<!--\n    Generated by GNU Solfege %s\n-->" % configureoutput.VERSION_STRING
        print >> f, "<link rel='stylesheet' href='style.css' type='text/css'/>"
        print >> f, "<body>"
        print >> f, "<h1>%s</h1>" % self.m_title
        for section in self.m_data:
            print >> f, "<h2>%s</h2>" % section.m_title
            print >> f, "<table>"
            first_row = True
            for idx, question_dict in enumerate(section):
                if idx % section.m_line_len == 0:
                    if not first_row:
                        print >> f, "</tr>"
                        first_row = False
                    print >> f, "<tr>"
                print >> f, "<td class='lilycode'>"
                # each question is in a table too
                print >> f, "<table><tr><td>"
                if r"\score" in question_dict[k]['music']:
                    print >> f, "<lilypond relative>"
                else:
                    print >> f, "<lilypond fragment relative>"
                print >> f, question_dict[k]['music']
                print >> f, "</lilypond>"
                print >> f, "</td></tr><tr><td class='answer'>"
                print >> f, question_dict[k]['name']
                print >> f, "</td></tr></table>"
                print >> f, "</td>"
            print >> f, "</table>"
        print >> f, "</body>"
        f.close()
        p = subprocess.Popen("lilypond-book --psfonts --out %s %s" % (
                            self.lilyout, filename),
                             shell=True, cwd=directory)
        sts = os.waitpid(p.pid, 0)
    def write_to(self, directory):
        self.create_outdir(directory)
        f = open(os.path.join(directory, self.lilyout, 'style.css'), 'w')
        print >> f, "table { border: none }"
        print >> f, ".answer { text-align: center }"
        f.close()
        self._write('question', directory, 'questions.html')
        self._write('answer', directory, 'answers.html')

class LatexSheetWriter(BaseSheetWriter):
    def write_to(self, directory):
        self.create_outdir(directory)
        self._write('question', directory, 'questions.tex')
        self._write('answer', directory, 'answers.tex')
    def _write(self, k, directory, filename):
        assert k in ('question', 'answer')
        f = open(os.path.join(directory, filename), 'w')
        print >> f, r"\documentclass{article}"
        print >> f, "%%\n%% Created by GNU Solfege %s\n%%" % configureoutput.VERSION_STRING
        print >> f, r"\title{%s}" % self.m_title
        print >> f, r"\usepackage[margin=1.0cm]{geometry}"
        print >> f, r"\begin{document}"
        print >> f, r"\maketitle"
        def finish_table(scores, answers):
            for idx, score in enumerate(scores):
                print >> f, score
                if idx != len(scores) - 1:
                    print >> f, "&"
            print >> f, r"\\"
            for idx, answer in enumerate(answers):
                print >> f, answer
                if idx != len(scores) - 1:
                    print >> f, "&"
            print >> f, r"\\"
        for section in self.m_data:
            print >> f, r"\section{%s}" % section.m_title
            first_row = True
            for idx, question_dict in enumerate(section):
                if idx % section.m_line_len == 0:
                    if first_row:
                        print >> f, r"\begin{tabular}{%s}" % ('c' * section.m_line_len)
                        first_row = False
                    else:
                        finish_table(scores, answers)
                    scores = []
                    answers = []
                if r"\score" in question_dict[k]['music']:
                    scores.append("\n".join((r"\begin{lilypond}",
                                question_dict[k]['music'], r"\end{lilypond}")))
                else:
                    scores.append("\n".join((r"\begin[fragment]{lilypond}",
                                question_dict[k]['music'], r"\end{lilypond}")))
                answers.append(question_dict[k]['name'].replace("#", r"\#"))
            finish_table(scores, answers)
            print >> f, r"\end{tabular}"
        print >> f, r"\end{document}"
        f.close()
        p = subprocess.Popen("lilypond-book --out %s %s" % (
                             self.lilyout, filename),
                             shell=True, cwd=directory)
        sts = os.waitpid(p.pid, 0)
        p = subprocess.Popen("latex %s" % filename, shell=True, 
            cwd=os.path.join(directory, self.lilyout))
        sts = os.waitpid(p.pid, 0)

class PractiseSheetDialog(gu.EditorDialogBase):
    """
    The definition of which lesson files to create questions from are
    stored in the list PractiseSheetDialog.m_sections. Each item is a
    dict filled with data. See PratiseSheet._add_common for details.

    When an exercise is selected from the menu, on_select_exercise(...)
    is called, and submethods from this will create a set of questions,
    and store the generated questions in m_sections[-1]['questions]
    and the definition (lesson_id, number of questions to generate etc) in
    other items in the dict in PractiseSheetDialog.m_sections.

    If we change some of the parameters for an exercise, for example how
    many questions, or which intervals, then the 'questions' variable
    for that lesson is emptied, and it will be regenerated the first
    time we randomize this lesson, or when we generate the whole sheet.

    on_create_sheet is called to create the .tex or .html files. Calling
    is several times in a row will generate the same test.
    """
    STORE_TITLE = 0
    ok_music_types = (lessonfile.Chord, lessonfile.Voice, lessonfile.Rvoice)
    ok_modules = ('idbyname', 'harmonicinterval', 'melodicinterval')
    def __init__(self, app):
        gu.EditorDialogBase.__init__(self)
        self.savedir = os.path.join(filesystem.user_data(), "eartrainingtests")
        self.m_changed = False
        self.m_savetime = time.time()
        self.set_title(self._get_a_filename())
        self.m_exported_to = None
        self.m_app = app
        self.set_default_size(800, 300)
        # self.vbox contains the toolbar and the vbox with the rest of the
        # window contents
        self.vbox = gtk.VBox()
        self.add(self.vbox)
        self.setup_toolbar()
        vbox = gtk.VBox()
        vbox.set_spacing(6)
        self.vbox.pack_start(vbox)
        vbox.set_border_width(8)
        hbox = gtk.HBox()
        vbox.pack_start(hbox, False)
        hbox.pack_start(gtk.Label(_("Output format:")), False)
        self.g_latex_radio = gtk.RadioButton(None, "LaTeX")
        hbox.pack_start(self.g_latex_radio, False)
        self.g_html_radio = gtk.RadioButton(self.g_latex_radio, "HTML")
        hbox.pack_start(self.g_html_radio, False)
        #
        hbox = gtk.HBox()
        hbox.set_spacing(6)
        vbox.pack_start(hbox, False)
        hbox.pack_start(gtk.Label(_("Title:")), False)
        self.g_title = gtk.Entry()
        hbox.pack_start(self.g_title, False)
        #
        self.m_sections = []
        self.g_liststore = gtk.ListStore(
            gobject.TYPE_STRING, # lesson-file title
        )
        self.g_treeview = gtk.TreeView(self.g_liststore)
        self.g_treeview.set_size_request(400, 100)
        self.g_treeview.set_headers_visible(False)
        self.g_treeview.connect('cursor-changed', self.on_tv_cursor_changed)
        self.g_treeview.connect('unselect-all', self.on_tv_unselect_all)
        scrolled_window = gtk.ScrolledWindow()
        scrolled_window.add(self.g_treeview)
        scrolled_window.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        vbox.pack_start(scrolled_window)
        #
        renderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn(None, renderer, text=self.STORE_TITLE)
        self.g_treeview.append_column(column)
        #
        sizegroup = gtk.SizeGroup(gtk.SIZE_GROUP_HORIZONTAL)
        self.g_lbox = gtk.VBox()
        self.g_lbox.set_sensitive(False)
        vbox.pack_start(self.g_lbox)
        self.g_lesson_title = gtk.Entry()
        self.g_lbox.pack_start(gu.hig_label_widget(_("Section title:"),
            self.g_lesson_title, sizegroup), False)
        self.g_lesson_title_event_handle =\
            self.g_lesson_title.connect('changed', self.on_lesson_title_changed)
        #
        self.g_qtype = gtk.combo_box_new_text()
        # We should not change the order of music types, as the index
        # of the different types are used in SolfegeApp.on_create_sheet
        self.g_qtype.append_text(_("Name the music"))
        self.g_qtype.append_text(_("Write the music, first tone given"))
        self.g_qtype_event_handler = \
            self.g_qtype.connect('changed', self.on_qtype_changed)
        self.g_lbox.pack_start(gu.hig_label_widget(_("Type of question:"),
            self.g_qtype, sizegroup), False)
        #
        self.g_intervals = IntervalCheckBox()
        self.g_intervals_box = gu.hig_label_widget(_("Intervals:"),
            self.g_intervals, sizegroup)
        self.g_intervals_event_handler = \
            self.g_intervals.connect('value-changed', self.on_intervals_changed)
        self.g_lbox.pack_start(self.g_intervals_box, False)
        #
        self.g_line_len = gtk.SpinButton(gtk.Adjustment(0, 1, 10, 1, 10))
        self.g_line_len_event_handler = \
            self.g_line_len.connect('value-changed', self.on_spin_changed, 'line_len')
        self.g_lbox.pack_start(gu.hig_label_widget(_("Questions per line:"),
            self.g_line_len, sizegroup), False)
        #
        self.g_count = gtk.SpinButton(gtk.Adjustment(0, 1, 1000, 1, 10))
        self.g_count_event_handler = \
            self.g_count.connect('value-changed', self.on_spin_changed, 'count')
        self.g_lbox.pack_start(gu.hig_label_widget(_("Number of questions:"),
            self.g_count, sizegroup), False)
    def create_learning_tree_menu(self):
        """
        Create and return a gtk.Menu object that has submenus that
        let us select all lessons on the learning tree.
        """
        def create_submenu(menudata, parent_name):
            """
            Menudata is a dict. Key we will use:
            'name'
            'children': list of dict or string (or maybe both?).
            """
            if isinstance(menudata, list):
                menu = gtk.Menu()
                for lesson_id in menudata:
                    if self.m_app.lessonfile_manager.get(lesson_id, 'module') not in self.ok_modules:
                        continue
                    if lesson_id in (
                            # melodic-interval-self-config
                            "f62929dc-7122-4173-aad1-4d4eef8779af",
                            # harmonic-interval-self-config
                            "466409e7-9086-4623-aff0-7c27f7dfd13b",
                            # the csound-fifth-* files:
                            "b465c807-d7bf-4e3a-a6da-54c78d5b59a1",
                            "aa5c3b18-664b-4e3d-b42d-2f06582f4135",
                            "5098fb96-c362-45b9-bbb3-703db149a079",
                            "3b1f57e8-2983-4a74-96da-468aa5414e5e",
                            "a06b5531-7422-4ea3-8711-ec57e2a4ce22",
                            "e67c5bd2-a275-4d9a-96a8-52e43a1e8987",
                            "1cadef8c-859e-4482-a6c4-31bd715b4787",
                            ):
                        continue
                    i = gtk.MenuItem(
                        self.m_app.lessonfile_manager.get(lesson_id, 'title'))
                    i.set_data('lesson_id', lesson_id)
                    i.set_data('menus', parent_name)
                    i.connect('activate', self.on_select_exercise)
                    menu.append(i)
                return menu
            item = gtk.MenuItem(_(menudata['name']))
            menu = gtk.Menu()
            for m in menudata['children']:
                i = gtk.MenuItem(_(m['name']))
                menu.append(i)
                i.set_submenu(create_submenu(m['children'], parent_name + [_(m['name'])]))
            item.set_submenu(menu)
            return item
        menu = gtk.Menu()
        for m in self.m_app.m_ui.m_tree.m_menus:
            menu.append(create_submenu(m, [_(m['name'])]))

        menu.show_all()
        self._menu_hide_stuff(menu)
        return menu
    def _menu_hide_stuff(self, menu):
        """
        Hide the menu if it has no menu items, or all menu items are hidden.
        """
        for sub in menu.get_children():
            assert isinstance(sub, gtk.MenuItem)
            if sub.get_submenu():
                self._menu_hide_stuff(sub.get_submenu())
                if not [c for c in sub.get_submenu().get_children() if c.props.visible]:
                    sub.hide()
    def view_lesson(self, idx):
        self.g_qtype.handler_block(self.g_qtype_event_handler)
        self.g_intervals.handler_block(self.g_intervals_event_handler)
        self.g_line_len.handler_block(self.g_line_len_event_handler)
        self.g_count.handler_block(self.g_count_event_handler)
        self.g_lesson_title.handler_block(self.g_lesson_title_event_handle)

        self.g_lbox.set_sensitive(idx is not None)
        if idx is None:
            self.g_lesson_title.set_text("")
            self.g_count.set_value(1)
            self.g_line_len.set_value(1)
        else:
            if self.m_app.lessonfile_manager.get(
                    self.m_sections[idx]['lesson_id'], 'module') \
                        == 'harmonicinterval':
                self.g_intervals_box.show()
                self.g_intervals.set_value(self.m_sections[idx]['intervals'])
            else:
                self.g_intervals_box.hide()
            self.g_lesson_title.set_text(self.m_sections[idx]['title'])
            self.g_count.set_value(self.m_sections[idx]['count'])
            self.g_line_len.set_value(self.m_sections[idx]['line_len'])
            self.g_qtype.set_active(self.m_sections[idx]['qtype'])
        self.g_qtype.handler_unblock(self.g_qtype_event_handler)
        self.g_intervals.handler_unblock(self.g_intervals_event_handler)
        self.g_line_len.handler_unblock(self.g_line_len_event_handler)
        self.g_count.handler_unblock(self.g_count_event_handler)
        self.g_lesson_title.handler_unblock(self.g_lesson_title_event_handle)
    def on_add_lesson_clicked(self, button):
        menu = self.create_learning_tree_menu()
        menu.popup(None, None, None, 1, 0)
    def on_remove_lesson_clicked(self, button):
        path, column = self.g_treeview.get_cursor()
        if path is not None:
            iter = self.g_liststore.get_iter(path)
            self.g_liststore.remove(iter)
            del self.m_sections[path[0]]
    def on_create_sheet(self, widget):
        if not self.m_exported_to:
            self.m_exported_to = self.select_empty_directory(_("Select where to export the files"))
        if not self.m_exported_to:
            return
        if self.g_html_radio.get_active():
            writer = HtmlSheetWriter(self.g_title.get_text())
        else:
            writer = LatexSheetWriter(self.g_title.get_text())
        iter = self.g_liststore.get_iter_first()
        for idx, sect in enumerate(self.m_sections):
            new_section = writer.new_section(sect['title'], sect['line_len'])
            self._generate_questions(idx)
            for question_dict in sect['questions']:
                new_section.append(question_dict)
        writer.write_to(self.m_exported_to)
    def on_intervals_changed(self, widget, value):
        self.m_changed = True
        idx = self.g_treeview.get_cursor()[0][0]
        self._delete_questions(idx)
        self.m_sections[idx]['intervals'] = value
    def on_lesson_title_changed(self, widget):
        self.m_changed = True
        path = self.g_treeview.get_cursor()[0]
        iter = self.g_liststore.get_iter(path)
        self.m_sections[path[0]]['title'] = widget.get_text()
        self.g_liststore.set(iter, self.STORE_TITLE, widget.get_text())
    def on_spin_changed(self, widget, varname):
        self.m_changed = True
        idx = self.g_treeview.get_cursor()[0][0]
        if varname == 'count':
            if len(self.m_sections[idx]['questions']) > widget.get_value_as_int():
                self.m_sections[idx]['questions'] = \
                    self.m_sections[idx]['questions'][:widget.get_value_as_int()]
        self.m_sections[idx][varname] = widget.get_value_as_int()
    def on_select_exercise(self, menuitem):
        lesson_id = menuitem.get_data('lesson_id')
        module = self.m_app.lessonfile_manager.get(lesson_id, 'module')
        if module not in self.ok_modules:
            print "only some modules work:", self.ok_modules
            return
        fn = self.m_app.lessonfile_manager.get(lesson_id, 'filename')
        p = lessonfile.LessonfileCommon()
        p.parse_file(fn)
        if module == 'idbyname':
            self._add_idbyname_lesson(p)
        elif module == 'harmonicinterval':
            self._add_harmonicinterval_lesson(p)
        elif module == 'melodicinterval':
            self._add_melodicinterval_lesson(p)
        self.g_treeview.set_cursor((len(self.m_sections)-1,))
    def _add_idbyname_lesson(self, p):
        """
        p is a lessonfile.LessonfileCommon parser that has parsed the file.
        """
        not_ok = len([q.music for q in p.m_questions if not isinstance(q.music, self.ok_music_types)])
        ok = len([q.music for q in p.m_questions if isinstance(q.music, self.ok_music_types)])
        if not_ok > 0:
            if ok > 0:
                do_add = gu.dialog_yesno(_("Not all music types are supported. This file contain %(ok)i supported questions and %(not_ok)i that are not supported. The unsupported questions will be ignored. Add any way?" % locals()))
            else:
                gu.dialog_ok(_("Could not add the lesson file. It has no questions with a music object with supported music type."))
                do_add = False
        else:
            do_add = True
        if do_add:
            self.m_changed = True
            self._add_common(p)
            self._generate_questions(-1)
    def _add_melodicinterval_lesson(self, p):
        self._add_common(p)
        self.m_sections[-1]['intervals'] = p.header.ask_for_intervals_0
        self._generate_questions(-1)
    def _add_harmonicinterval_lesson(self, p):
        self._add_common(p)
        self.m_sections[-1]['intervals'] = p.header.intervals
        self._generate_questions(-1)
    def _add_common(self, p):
        self.m_changed = True
        self.m_sections.append({
            'lesson_id': p.header.lesson_id,
            'title': self.m_app.lessonfile_manager.get(p.header.lesson_id, 'title'),
            'count': 6,
            'qtype': 0,
            'line_len': 3,
            'questions': [],
        })
        self.g_liststore.append((self.m_sections[-1]['title'],))
    def _generate_questions(self, idx):
        """
        Generate random questions for the exercise idx in self.m_sections
        if it does not have enough questions in the 'questions' variable
        """
        count = self.m_sections[idx]['count'] - len(self.m_sections[idx]['questions'])
        assert count >= 0
        if count:
            for question_dict in self.m_app.sheet_gen_questions(
                    count, self.m_sections[idx]):
                self.m_sections[idx]['questions'].append(question_dict)
    def _delete_questions(self, idx):
        """
        Delete the generated questions for exercise idx in self.m_sections.
        """
        self.m_sections[idx]['questions'] = []
    def on_show_help(self, widget):
        self.m_app.handle_href("ear-training-test-printout-editor.html")
    def on_tv_cursor_changed(self, treeview, *v):
        if self.g_treeview.get_cursor()[0] is None:
            return
        self.view_lesson(self.g_treeview.get_cursor()[0][0])
    def on_tv_unselect_all(self, treeview, *v):
        self.view_lesson(None)
    def on_qtype_changed(self, widget):
        self.m_changed = True
        idx = self.g_treeview.get_cursor()[0][0]
        self._delete_questions(idx)
        self.m_sections[idx]['qtype'] = widget.get_active()
    def load_file(self, filename):
        tree = et.ElementTree()
        tree.parse(filename)
        # section.find('text').text will be none if it was not set
        # when saved, and we need a string.
        if tree.find("title").text:
            self.g_title.set_text(tree.find("title").text)
        else:
            self.g_title.set_text("")
        if tree.find('output_format').text == 'latex':
            self.g_latex_radio.set_active(True)
        else:
            self.g_latex_radio.set_active(False)
        self.g_liststore.clear()
        self.m_sections = []
        for section in tree.findall("section"):
            d = {}
            d['lesson_id'] = section.find('lesson_id').text
            # section.find('text').text will be none if it was not set
            # when saved, and d['title'] need to be a string
            if section.find('title').text:
                d['title'] = section.find('title').text
            else:
                d['title'] = ""
            d['count'] = int(section.find('count').text)
            d['line_len'] = int(section.find('line_len').text)
            d['qtype'] = int(section.find('qtype').text)
            d['questions'] = []
            for question in section.findall("question"):
                q = {'question': {}, 'answer': {}}
                q['question']['music'] = question.find("students").find("music").text
                q['question']['name'] = question.find("students").find("name").text
                q['answer']['music'] = question.find("teachers").find("music").text
                q['answer']['name'] = question.find("teachers").find("name").text
                d['questions'].append(q)
            self.m_sections.append(d)
            self.g_liststore.append((d['title'],))
            self.g_treeview.set_cursor((0,))
        self.m_filename = filename
    def save(self):
        assert self.m_filename
        if self.g_latex_radio.get_active():
            format = "latex"
        else:
            format = "html"
        doc = et.Element("sheet", fileformat_version="1.0",
                         creator="GNU Solfege",
                         app_version=configureoutput.VERSION_STRING)
        doc.append(et.Comment(""))
        et.SubElement(doc, "title").text = self.g_title.get_text()
        et.SubElement(doc, "output_format").text = format
        for sect in self.m_sections:
            s = et.Element("section")
            et.SubElement(s, "title").text = sect['title']
            et.SubElement(s, "lesson_id").text = sect['lesson_id']
            # We do save 'count' because this variable say how many questions
            # we want to generate, and len(questions) will just say how many
            # are generated now.
            et.SubElement(s, "count").text = "%i" % sect['count']
            et.SubElement(s, "line_len").text = "%i" % sect['line_len']
            et.SubElement(s, "qtype").text = "%i" % sect['qtype']
            for qdict in sect['questions']:
                q = et.SubElement(s, "question")
                t = et.SubElement(q, "teachers")
                et.SubElement(t, "name").text = qdict['answer']['name']
                et.SubElement(t, "music").text = qdict['answer']['music']
                t = et.SubElement(q, "students")
                et.SubElement(t, "name").text = qdict['question']['name']
                et.SubElement(t, "music").text = qdict['question']['music']
            doc.append(s)
        tree = et.ElementTree(doc)
        f = open(self.m_filename, 'w')
        print >> f, '<?xml version="1.0" encoding="utf-8"?>'
        tree.write(f, encoding="utf-8")
        f.close()
        self.m_changed = False
        self.m_savetime = time.time()
    def setup_toolbar(self):
        self.g_toolbar = gtk.Toolbar()
        self.g_actiongroup.add_actions([
         ('Add', gtk.STOCK_ADD, None, None, None, self.on_add_lesson_clicked),
         ('Remove', gtk.STOCK_REMOVE, None, None, None, self.on_remove_lesson_clicked),
         ('Create', gtk.STOCK_EXECUTE, _("Create sheet"), None, None, self.on_create_sheet),
        ])
        self.g_ui_manager.insert_action_group(self.g_actiongroup, 0)
        uixml = """
        <ui>
         <toolbar name='ExportToolbar'>
          <toolitem action='Add'/>
          <toolitem action='Remove'/>
          <toolitem action='New'/>
          <toolitem action='Open'/>
          <toolitem action='Save'/>
          <toolitem action='SaveAs'/>
          <toolitem action='Create'/>
          <toolitem action='Close'/>
          <toolitem action='Help'/>
         </toolbar>
        </ui>
        """
        self.g_ui_manager.add_ui_from_string(uixml)
        self.vbox.pack_start(self.g_ui_manager.get_widget("/ExportToolbar"), False)
        self.g_ui_manager.get_widget("/ExportToolbar").set_style(gtk.TOOLBAR_BOTH)
