/***************************************************************************
    smb4ksharesview_part  -  This Part includes the shares view of Smb4K.
                             -------------------
    begin                : Sa Jun 30 2007
    copyright            : (C) 2007-2008 by Alexander Reinholdt
    email                : dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

// Qt includes
#include <QHeaderView>
#include <QDropEvent>

// KDE includes
#include <klocale.h>
#include <kaboutdata.h>
#include <klocale.h>
#include <kaction.h>
#include <kdebug.h>
#include <kicon.h>
#include <kactioncollection.h>
#include <kmenu.h>
#include <kapplication.h>
#include <kurl.h>
#include <kio/copyjob.h>
#include <kjobuidelegate.h>

// application specific includes
#include <smb4ksharesview_part.h>
#include <smb4kshareslistviewitem.h>
#include <smb4ksharesiconviewitem.h>
#include <../dialogs/smb4ksynchronizationdialog.h>
#include <core/smb4kshare.h>
#include <core/smb4kcore.h>
#include <core/smb4ksettings.h>
#include <core/smb4kglobal.h>

using namespace Smb4KGlobal;

typedef KParts::GenericFactory<Smb4KSharesViewPart> Smb4KSharesViewPartFactory;
K_EXPORT_COMPONENT_FACTORY( libsmb4ksharesview, Smb4KSharesViewPartFactory )


Smb4KSharesViewPart::Smb4KSharesViewPart( QWidget *parentWidget, QObject *parent, const QStringList &args )
: KParts::Part( parent ), m_mode( IconMode )
{
  // Parse the arguments.
  for ( int i = 0; i < args.size(); ++i )
  {
    if ( args.at( i ).startsWith( "viewmode" ) )
    {
      if ( QString::compare( args.at( i ).section( "=", 1, 1 ).trimmed(), "list" ) == 0 )
      {
        m_mode = ListMode;
      }
      else
      {
        // Do nothing
      }

      continue;
    }
    else
    {
      continue;
    }
  }

  // Set the XML file:
  setXMLFile( "smb4ksharesview_part.rc" );

  // Set the container widget and its layout.
  m_container = new QWidget( parentWidget );

  m_layout = new QGridLayout( m_container );
  m_layout->setMargin( 0 );
  m_layout->setSpacing( 0 );

  setWidget( m_container );

  // Set up the actual view.
  m_icon_view = NULL;
  m_list_view = NULL;

  setupView();

  // Set up the actions.
  // Do not put this before setWidget() or the shortcuts
  // will not be shown.
  setupActions();

  // Import the shares:
  slotMountedShares();

  // Load settings:
  loadSettings();

  // Add some connections:
  connect( Smb4KCore::mounter(),      SIGNAL( updated() ),
           this,                      SLOT( slotMountedShares() ) );

  connect( Smb4KCore::synchronizer(), SIGNAL( state( int ) ),
           this,                      SLOT( slotSynchronizationState( int ) ) );

  connect( kapp,                      SIGNAL( aboutToQuit() ),
           this,                      SLOT( slotAboutToQuit() ) );
}


Smb4KSharesViewPart::~Smb4KSharesViewPart()
{
}


void Smb4KSharesViewPart::setupView()
{
  // First disconnect the signals, so that we do not
  // get multiple connections.
  if ( m_icon_view )
  {
    disconnect( m_icon_view, SIGNAL( customContextMenuRequested( const QPoint & ) ),
                this,        SLOT( slotContextMenuRequested( const QPoint & ) ) );

    disconnect( m_icon_view, SIGNAL( itemSelectionChanged() ),
                this,        SLOT( slotItemSelectionChanged() ) );

    disconnect( m_icon_view, SIGNAL( itemPressed( QListWidgetItem * ) ),
               this,        SLOT( slotItemPressed( QListWidgetItem * ) ) );

    disconnect( m_icon_view, SIGNAL( itemExecuted( QListWidgetItem * ) ),
                this,        SLOT( slotItemExecuted( QListWidgetItem * ) ) );

    disconnect( m_icon_view, SIGNAL( acceptedDropEvent( Smb4KSharesIconViewItem *, QDropEvent * ) ),
                this,        SLOT( slotIconViewDropEvent( Smb4KSharesIconViewItem *, QDropEvent * ) ) );
  }
  else
  {
    // Do nothing
  }

  if ( m_list_view )
  {
    disconnect( m_list_view, SIGNAL( customContextMenuRequested( const QPoint & ) ),
                this,        SLOT( slotContextMenuRequested( const QPoint & ) ) );

    disconnect( m_list_view, SIGNAL( itemSelectionChanged() ),
                this,        SLOT( slotItemSelectionChanged() ) );

    disconnect( m_list_view, SIGNAL( itemPressed( QTreeWidgetItem *, int ) ),
                this,        SLOT( slotItemPressed( QTreeWidgetItem *, int ) ) );

    disconnect( m_list_view, SIGNAL( itemExecuted( QTreeWidgetItem *, int ) ),
                this,        SLOT( slotItemExecuted( QTreeWidgetItem *, int ) ) );

    disconnect( m_list_view, SIGNAL( acceptedDropEvent( Smb4KSharesListViewItem *, QDropEvent * ) ),
                this,        SLOT( slotListViewDropEvent( Smb4KSharesListViewItem *, QDropEvent * ) ) );
  }
  else
  {
    // Do nothing
  }

  // Set the widget of this part:
  switch ( m_mode )
  {
    case IconMode:
    {
      if ( m_list_view )
      {
        delete m_list_view;
        m_list_view = NULL;
      }
      else
      {
        // Do nothing
      }

      if ( !m_icon_view )
      {
        m_icon_view = new Smb4KSharesIconView( m_container );

        m_layout->addWidget( m_icon_view, 0, 0, 0 );

        m_icon_view->setVisible( true );
      }
      else
      {
        // Do nothing
      }

      connect( m_icon_view, SIGNAL( customContextMenuRequested( const QPoint & ) ),
               this,        SLOT( slotContextMenuRequested( const QPoint & ) ) );

      connect( m_icon_view, SIGNAL( itemSelectionChanged() ),
               this,        SLOT( slotItemSelectionChanged() ) );

      connect( m_icon_view, SIGNAL( itemPressed( QListWidgetItem * ) ),
               this,        SLOT( slotItemPressed( QListWidgetItem * ) ) );

      connect( m_icon_view, SIGNAL( itemExecuted( QListWidgetItem * ) ),
               this,        SLOT( slotItemExecuted( QListWidgetItem * ) ) );

      connect( m_icon_view, SIGNAL( acceptedDropEvent( Smb4KSharesIconViewItem *, QDropEvent * ) ),
               this,        SLOT( slotIconViewDropEvent( Smb4KSharesIconViewItem *, QDropEvent * ) ) );

      break;
    }
    case ListMode:
    {
      if ( m_icon_view )
      {
        delete m_icon_view;
        m_icon_view = NULL;
      }
      else
      {
        // Do nothing
      }

      if ( !m_list_view )
      {
        m_list_view = new Smb4KSharesListView( m_container );

        m_layout->addWidget( m_list_view, 0, 0, 0 );

        m_list_view->setVisible( true );
      }
      else
      {
        // Do nothing
      }

      connect( m_list_view, SIGNAL( customContextMenuRequested( const QPoint & ) ),
               this,        SLOT( slotContextMenuRequested( const QPoint & ) ) );

      connect( m_list_view, SIGNAL( itemSelectionChanged() ),
               this,        SLOT( slotItemSelectionChanged() ) );

      connect( m_list_view, SIGNAL( itemPressed( QTreeWidgetItem *, int ) ),
               this,        SLOT( slotItemPressed( QTreeWidgetItem *, int ) ) );

      connect( m_list_view, SIGNAL( itemExecuted( QTreeWidgetItem *, int ) ),
               this,        SLOT( slotItemExecuted( QTreeWidgetItem *, int ) ) );

      connect( m_list_view, SIGNAL( acceptedDropEvent( Smb4KSharesListViewItem *, QDropEvent * ) ),
               this,        SLOT( slotListViewDropEvent( Smb4KSharesListViewItem *, QDropEvent * ) ) );

      break;
    }
    default:
    {
      break;
    }
  }
}


void Smb4KSharesViewPart::setupActions()
{
  KAction *unmount_action     = new KAction( KIcon( "media-eject" ), i18n( "&Unmount" ),
                                actionCollection() );
  unmount_action->setShortcut( QKeySequence( Qt::CTRL+Qt::Key_U ) );
  connect( unmount_action, SIGNAL( triggered( bool ) ), this, SLOT( slotUnmountShare( bool ) ) );

#ifdef __linux__
  KAction *force_action       = new KAction( KIcon( "media-eject" ), i18n( "&Force Unmounting" ),
                                actionCollection() );
  force_action->setShortcut( QKeySequence( Qt::CTRL+Qt::Key_F ) );
  connect( force_action, SIGNAL( triggered( bool ) ), this, SLOT( slotForceUnmountShare( bool ) ) );
#endif

  KAction *unmount_all_action = new KAction( KIcon( "system-run" ), i18n( "U&nmount All" ),
                                actionCollection() );
  unmount_all_action->setShortcut( QKeySequence( Qt::CTRL+Qt::Key_N ) );
  connect( unmount_all_action, SIGNAL( triggered( bool ) ), this, SLOT( slotUnmountAllShares( bool ) ) );

  KAction *synchronize_action = new KAction( KIcon( "go-bottom" ), i18n( "S&ynchronize" ),
                                actionCollection() );
  synchronize_action->setShortcut( QKeySequence( Qt::CTRL+Qt::Key_Y ) );
  connect( synchronize_action, SIGNAL( triggered( bool ) ), this, SLOT( slotSynchronize( bool ) ) );

  KAction *konsole_action     = new KAction( KIcon( "utilities-terminal" ), i18n( "Open with Konso&le" ),
                                actionCollection() );
  konsole_action->setShortcut( QKeySequence( Qt::CTRL+Qt::Key_L ) );
  connect( konsole_action, SIGNAL( triggered( bool ) ), this, SLOT( slotKonsole( bool ) ) );

  KAction *filemanager_action = new KAction( KIcon( "system-file-manager" ), i18n( "Open with F&ile Manager" ),
                                actionCollection() );
  QList<QKeySequence> shortcuts;
  shortcuts.append( QKeySequence( Qt::CTRL+Qt::Key_I ) );
  shortcuts.append( QKeySequence( Qt::CTRL+Qt::Key_K ) );  // legacy shortcut
  filemanager_action->setShortcuts( shortcuts );
  connect( filemanager_action, SIGNAL( triggered( bool ) ), this, SLOT( slotFileManager( bool ) ) );

  actionCollection()->addAction( "unmount_action", unmount_action );
#ifdef __linux__
  actionCollection()->addAction( "force_unmount_action", force_action );
#endif
  actionCollection()->addAction( "unmount_all_action", unmount_all_action );
  actionCollection()->addAction( "synchronize_action", synchronize_action );
  actionCollection()->addAction( "konsole_action", konsole_action );
  actionCollection()->addAction( "filemanager_action", filemanager_action );

  // Disable all actions for now:
  unmount_action->setEnabled( false );
#ifdef __linux__
  force_action->setEnabled( false );
#endif
  unmount_all_action->setEnabled( false );
  synchronize_action->setEnabled( false );
  konsole_action->setEnabled( false );
  filemanager_action->setEnabled( false );

  // Insert the actions into the menu:
  m_menu = new KActionMenu( this );
  m_menu_title = m_menu->menu()->addTitle( KIcon( "folder-remote" ), i18n( "Shares" ) );
  m_menu->addAction( unmount_action );
#ifdef __linux__
  m_menu->addAction( force_action );
#endif
  m_menu->addAction( unmount_all_action );
  m_menu->addSeparator();
  m_menu->addAction( synchronize_action );
  m_menu->addSeparator();
  m_menu->addAction( konsole_action );
  m_menu->addAction( filemanager_action );
}


void Smb4KSharesViewPart::loadSettings()
{
  if ( Smb4KSettings::sharesIconView() )
  {
    m_mode = IconMode;

    setupView();

    // Change the text of the share (first column):
    Smb4KSharesIconViewItem *item = NULL;

    for ( int i = 0; i < m_icon_view->count(); ++i )
    {
      item = static_cast<Smb4KSharesIconViewItem *>( m_icon_view->item( i ) );

      item->setShowMountPoint( Smb4KSettings::showMountPoint() );
    }
  }
  else
  {
    m_mode = ListMode;

    setupView();

    m_list_view->setColumnHidden( Smb4KSharesListView::Owner, !Smb4KSettings::showOwner() );
#ifndef __FreeBSD__
    m_list_view->setColumnHidden( Smb4KSharesListView::Login, !Smb4KSettings::showLogin() );
#endif
    m_list_view->setColumnHidden( Smb4KSharesListView::FileSystem, !Smb4KSettings::showFileSystem() );
    m_list_view->setColumnHidden( Smb4KSharesListView::Free, !Smb4KSettings::showFreeDiskSpace() );
    m_list_view->setColumnHidden( Smb4KSharesListView::Used, !Smb4KSettings::showUsedDiskSpace() );
    m_list_view->setColumnHidden( Smb4KSharesListView::Total, !Smb4KSettings::showTotalDiskSpace() );
    m_list_view->setColumnHidden( Smb4KSharesListView::Usage, !Smb4KSettings::showDiskUsage() );

    // Change the text of the share (first column):
    Smb4KSharesListViewItem *item = NULL;

    for ( int i = 0; i < m_list_view->topLevelItemCount(); ++i )
    {
      item = static_cast<Smb4KSharesListViewItem *>( m_list_view->topLevelItem( i ) );

      item->setShowMountPoint( Smb4KSettings::showMountPoint() );
    }

    // Load and apply the positions of the columns.
    KConfigGroup configGroup( Smb4KSettings::self()->config(), "SharesViewPart" );

    QMap<int, int> map;
    map.insert( configGroup.readEntry( "ColumnPositionItem", (int)Smb4KSharesListView::Item ), Smb4KSharesListView::Item );
#ifndef __FreeBSD__
    map.insert( configGroup.readEntry( "ColumnPositionLogin", (int)Smb4KSharesListView::Login ), Smb4KSharesListView::Login );
#endif
    map.insert( configGroup.readEntry( "ColumnPositionFileSystem", (int)Smb4KSharesListView::FileSystem ), Smb4KSharesListView::FileSystem );
    map.insert( configGroup.readEntry( "ColumnPositionOwner", (int)Smb4KSharesListView::Owner ), Smb4KSharesListView::Owner );
    map.insert( configGroup.readEntry( "ColumnPositionFree", (int)Smb4KSharesListView::Free ), Smb4KSharesListView::Free );
    map.insert( configGroup.readEntry( "ColumnPositionUsed", (int)Smb4KSharesListView::Used ), Smb4KSharesListView::Used );
    map.insert( configGroup.readEntry( "ColumnPositionTotal", (int)Smb4KSharesListView::Total ), Smb4KSharesListView::Total );
    map.insert( configGroup.readEntry( "ColumnPositionUsage", (int)Smb4KSharesListView::Usage ), Smb4KSharesListView::Usage );

    QMap<int, int>::const_iterator it = map.constBegin();

    while ( it != map.constEnd() )
    {
      if ( it.key() != m_list_view->header()->visualIndex( it.value() ) )
      {
        m_list_view->header()->moveSection( m_list_view->header()->visualIndex( it.value() ), it.key() );
      }
      else
      {
        // Do nothing
      }

      ++it;
    }
  }

#ifdef __linux__
  if ( Smb4KSettings::sharesIconView() )
  {
    Smb4KSharesIconViewItem *item = static_cast<Smb4KSharesIconViewItem *>( m_icon_view->currentItem() );

    actionCollection()->action( "force_unmount_action" )->setEnabled( Smb4KSettings::useForceUnmount() &&
    (item && (!item->itemData()->share()->isForeign() || Smb4KSettings::unmountForeignShares())) );
  }
  else
  {
    Smb4KSharesListViewItem *item = static_cast<Smb4KSharesListViewItem *>( m_list_view->currentItem() );

    actionCollection()->action( "force_unmount_action" )->setEnabled( Smb4KSettings::useForceUnmount() &&
    (item && (!item->itemData()->share()->isForeign() || Smb4KSettings::unmountForeignShares())) );
  }
#endif


  // The rest of the settings will be applied on the fly.
}


void Smb4KSharesViewPart::saveSettings()
{
  switch ( m_mode )
  {
    case IconMode:
    {
      break;
    }
    case ListMode:
    {
      // Save the position of the columns.
      KConfigGroup configGroup( Smb4KSettings::self()->config(), "SharesViewPart" );

      configGroup.writeEntry( "ColumnPositionItem", m_list_view->header()->visualIndex( Smb4KSharesListView::Item ) );
#ifndef __FreeBSD__
      configGroup.writeEntry( "ColumnPositionLogin", m_list_view->header()->visualIndex( Smb4KSharesListView::Login ) );
#endif
      configGroup.writeEntry( "ColumnPositionFileSystem", m_list_view->header()->visualIndex( Smb4KSharesListView::FileSystem ) );
      configGroup.writeEntry( "ColumnPositionOwner", m_list_view->header()->visualIndex( Smb4KSharesListView::Owner ) );
      configGroup.writeEntry( "ColumnPositionFree", m_list_view->header()->visualIndex( Smb4KSharesListView::Free ) );
      configGroup.writeEntry( "ColumnPositionUsed", m_list_view->header()->visualIndex( Smb4KSharesListView::Used ) );
      configGroup.writeEntry( "ColumnPositionTotal", m_list_view->header()->visualIndex( Smb4KSharesListView::Total ) );
      configGroup.writeEntry( "ColumnPositionUsage", m_list_view->header()->visualIndex( Smb4KSharesListView::Usage ) );

      configGroup.sync();

      break;
    }
    default:
    {
      break;
    }
  }
}


KAboutData *Smb4KSharesViewPart::createAboutData()
{
  KAboutData *aboutData = new KAboutData( "smb4ksharesviewpart",
                          "smb4k",
                          ki18n( "Smb4KSharesViewPart" ),
                          "2.0",
                          ki18n( "The shares view KPart of Smb4K" ),
                          KAboutData::License_GPL_V2,
                          ki18n( "(c) 2007-2008, Alexander Reinholdt" ),
                          KLocalizedString(),
                          "http://smb4k.berlios.de",
                          "smb4k-bugs@lists.berlios.de" );

  return aboutData;
}


void Smb4KSharesViewPart::customEvent( QEvent *e )
{
  switch ( e->type() )
  {
    case EVENT_LOAD_SETTINGS:
    {
      // Before we reread the settings, let's save
      // widget specific things.
      saveSettings();

      // Load settings.
      loadSettings();

      // (Re-)load the list of shares.
      slotMountedShares();

      break;
    }
    case EVENT_SET_FOCUS:
    {
      switch ( m_mode )
      {
        case IconMode:
        {
          if ( m_icon_view->count() != 0 )
          {
            kDebug() << "Do we need to port the selection stuff?" << endl;
          }
          else
          {
            // Do nothing
          }

          m_icon_view->setFocus( Qt::OtherFocusReason );

          break;
        }
        case ListMode:
        {
          if ( m_list_view->topLevelItemCount() != 0 )
          {
            kDebug() << "Do we need to port the selection stuff?" << endl;
          }
          else
          {
            // Do nothing
          }

          m_list_view->setFocus( Qt::OtherFocusReason );

          break;
        }
        default:
        {
          break;
        }
      }

//       KListView *view = static_cast<KListView *>( m_widget );
//
//       if ( view->childCount() != 0 )
//       {
//         view->setSelected( !view->currentItem() ?
//                            view->firstChild() :
//                            view->currentItem(), true );
//       }
//
//       view->setFocus();

      break;
    }
    default:
    {
      break;
    }
  }

  KParts::Part::customEvent( e );
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS (Smb4KSharesViewPart)
/////////////////////////////////////////////////////////////////////////////

void Smb4KSharesViewPart::slotContextMenuRequested( const QPoint &pos )
{
  m_menu->removeAction( m_menu_title );
  delete m_menu_title;

  switch ( m_mode )
  {
    case IconMode:
    {
      QListWidgetItem *item = m_icon_view->itemAt( pos );

      if ( item )
      {
        m_menu_title = m_menu->menu()->addTitle( item->icon(),
                                                 item->text(),
                                                 actionCollection()->action( "unmount_action" ) );
      }
      else
      {
        m_menu_title = m_menu->menu()->addTitle( KIcon( "folder-remote" ),
                                                 i18n( "Shares" ),
                                                 actionCollection()->action( "unmount_action" ) );
      }

      m_menu->menu()->popup( m_icon_view->viewport()->mapToGlobal( pos ) );

      break;
    }
    case ListMode:
    {
      QTreeWidgetItem *item = m_list_view->itemAt( pos );

      if ( item )
      {
        m_menu_title = m_menu->menu()->addTitle( item->icon( Smb4KSharesListViewItem::Item ),
                                                 item->text( Smb4KSharesListViewItem::Item ),
                                                 actionCollection()->action( "unmount_action" ) );
      }
      else
      {
        m_menu_title = m_menu->menu()->addTitle( KIcon( "folder-remote" ),
                                                 i18n( "Shares" ),
                                                 actionCollection()->action( "unmount_action" ) );
      }

      m_menu->menu()->popup( m_list_view->viewport()->mapToGlobal( pos ) );

      break;
    }
    default:
    {
      break;
    }
  }
}


void Smb4KSharesViewPart::slotItemSelectionChanged()
{
  switch ( m_mode )
  {
    case IconMode:
    {
      QList<QListWidgetItem *> items = m_icon_view->selectedItems();

      if ( !items.isEmpty() )
      {
        Smb4KSharesIconViewItem *item = static_cast<Smb4KSharesIconViewItem *>( items.first() );

        actionCollection()->action( "unmount_action" )->setEnabled( (!item->itemData()->share()->isForeign() ||
                                                                    Smb4KSettings::unmountForeignShares()) );
#ifdef __linux__
        actionCollection()->action( "force_unmount_action" )->setEnabled( Smb4KSettings::useForceUnmount() &&
                                                                          (!item->itemData()->share()->isForeign() ||
                                                                          Smb4KSettings::unmountForeignShares()) );
#endif
        Smb4KShare *share = item->itemData()->share();

        if ( !share->isInaccessible() )
        {
          actionCollection()->action( "synchronize_action" )->setEnabled( !Smb4KSettings::rsync().isEmpty() &&
                                                                          !Smb4KCore::synchronizer()->isRunning() );
          actionCollection()->action( "konsole_action" )->setEnabled( !Smb4KSettings::konsole().isEmpty() );
          actionCollection()->action( "filemanager_action" )->setEnabled( true );
        }
        else
        {
          actionCollection()->action( "synchronize_action" )->setEnabled( false );
          actionCollection()->action( "konsole_action" )->setEnabled( false );
          actionCollection()->action( "filemanager_action" )->setEnabled( false );
        }
      }
      else
      {
        actionCollection()->action( "unmount_action" )->setEnabled( false );
#ifdef __linux__
        actionCollection()->action( "force_unmount_action" )->setEnabled( false );
#endif
        actionCollection()->action( "synchronize_action" )->setEnabled( false );
        actionCollection()->action( "konsole_action" )->setEnabled( false );
        actionCollection()->action( "filemanager_action" )->setEnabled( false );
      }

      break;
    }
    case ListMode:
    {
      // Get the selected item.
      QList<QTreeWidgetItem *> items = m_list_view->selectedItems();

      if ( !items.isEmpty() )
      {
        Smb4KSharesListViewItem *item = static_cast<Smb4KSharesListViewItem *>( items.first() );

        actionCollection()->action( "unmount_action" )->setEnabled( (!item->itemData()->share()->isForeign() ||
                                                                    Smb4KSettings::unmountForeignShares()) );
    #ifdef __linux__
        actionCollection()->action( "force_unmount_action" )->setEnabled( Smb4KSettings::useForceUnmount() &&
                                                                          (!item->itemData()->share()->isForeign() ||
                                                                          Smb4KSettings::unmountForeignShares()) );
    #endif
        Smb4KShare *share = item->itemData()->share();

        if ( !share->isInaccessible() )
        {
          actionCollection()->action( "synchronize_action" )->setEnabled( !Smb4KSettings::rsync().isEmpty() &&
                                                                          !Smb4KCore::synchronizer()->isRunning() );
          actionCollection()->action( "konsole_action" )->setEnabled( !Smb4KSettings::konsole().isEmpty() );
          actionCollection()->action( "filemanager_action" )->setEnabled( true );
        }
        else
        {
          actionCollection()->action( "synchronize_action" )->setEnabled( false );
          actionCollection()->action( "konsole_action" )->setEnabled( false );
          actionCollection()->action( "filemanager_action" )->setEnabled( false );
        }
      }
      else
      {
        actionCollection()->action( "unmount_action" )->setEnabled( false );
#ifdef __linux__
        actionCollection()->action( "force_unmount_action" )->setEnabled( false );
#endif
        actionCollection()->action( "synchronize_action" )->setEnabled( false );
        actionCollection()->action( "konsole_action" )->setEnabled( false );
        actionCollection()->action( "filemanager_action" )->setEnabled( false );
      }

      break;
    }
    default:
    {
      break;
    }
  }
}


void Smb4KSharesViewPart::slotItemPressed( QTreeWidgetItem *item, int /*column*/ )
{
  Smb4KSharesListViewItem *shareItem = static_cast<Smb4KSharesListViewItem *>( item );

  if ( !shareItem && m_list_view->selectedItems().isEmpty() )
  {
    actionCollection()->action( "unmount_action" )->setEnabled( false );
#ifndef __linux__
    actionCollection()->action( "force_unmount_action" )->setEnabled( false );
#endif
    actionCollection()->action( "synchronize_action" )->setEnabled( false );
    actionCollection()->action( "konsole_action" )->setEnabled( false );
    actionCollection()->action( "filemanager_action" )->setEnabled( false );
  }
  else
  {
    // Do nothing. This will be done elsewhere.
  }
}


void Smb4KSharesViewPart::slotItemPressed( QListWidgetItem *item )
{
  Smb4KSharesIconViewItem *shareItem = static_cast<Smb4KSharesIconViewItem *>( item );

  if ( !shareItem && m_icon_view->selectedItems().isEmpty() )
  {
    actionCollection()->action( "unmount_action" )->setEnabled( false );
#ifndef __linux__
    actionCollection()->action( "force_unmount_action" )->setEnabled( false );
#endif
    actionCollection()->action( "synchronize_action" )->setEnabled( false );
    actionCollection()->action( "konsole_action" )->setEnabled( false );
    actionCollection()->action( "filemanager_action" )->setEnabled( false );
  }
  else
  {
    // Do nothing. This will be done elsewhere.
  }
}


void Smb4KSharesViewPart::slotItemExecuted( QTreeWidgetItem *item, int /*column*/ )
{
  // This is a precaution.
  if ( item != m_list_view->currentItem() )
  {
    m_list_view->setCurrentItem( item );
  }
  else
  {
    // Do nothing
  }

  slotFileManager( false );
}


void Smb4KSharesViewPart::slotItemExecuted( QListWidgetItem *item )
{
  // This is a precaution.
  if ( item != m_icon_view->currentItem() )
  {
    m_icon_view->setCurrentItem( item );
  }
  else
  {
    // Do nothing
  }

  slotFileManager( false );
}


void Smb4KSharesViewPart::slotListViewDropEvent( Smb4KSharesListViewItem *item, QDropEvent *e )
{
  if ( item && e )
  {
    switch ( e->proposedAction() )
    {
      case Qt::CopyAction:
      {
        if ( KUrl::List::canDecode( e->mimeData() ) )
        {
          KUrl::List urlList = KUrl::List::fromMimeData( e->mimeData() );

          KUrl dest;
          dest.setPath( item->itemData()->share()->path() );

          KIO::CopyJob *job = KIO::copy( urlList, dest, KIO::DefaultFlags );

          job->uiDelegate()->setAutoErrorHandlingEnabled( true );
          job->uiDelegate()->setAutoWarningHandlingEnabled( true );
        }
        else
        {
          // Do nothing
        }

        break;
      }
      case Qt::MoveAction:
      {
        if ( KUrl::List::canDecode( e->mimeData() ) )
        {
          KUrl::List urlList = KUrl::List::fromMimeData( e->mimeData() );

          KUrl dest;
          dest.setPath( item->itemData()->share()->path() );

          KIO::CopyJob *job = KIO::move( urlList, dest, KIO::DefaultFlags );

          job->uiDelegate()->setAutoErrorHandlingEnabled( true );
          job->uiDelegate()->setAutoWarningHandlingEnabled( true );
        }
        else
        {
          // Do nothing
        }

        break;
      }
      default:
      {
        break;
      }
    }
  }
  else
  {
    // Do nothing
  }
}


void Smb4KSharesViewPart::slotIconViewDropEvent( Smb4KSharesIconViewItem *item, QDropEvent *e )
{
  if ( item && e )
  {
    switch ( e->proposedAction() )
    {
      case Qt::CopyAction:
      {
        if ( KUrl::List::canDecode( e->mimeData() ) )
        {
          KUrl::List urlList = KUrl::List::fromMimeData( e->mimeData() );

          KUrl dest;
          dest.setPath( item->itemData()->share()->path() );

          KIO::CopyJob *job = KIO::copy( urlList, dest, KIO::DefaultFlags );

          job->uiDelegate()->setAutoErrorHandlingEnabled( true );
          job->uiDelegate()->setAutoWarningHandlingEnabled( true );
        }
        else
        {
          // Do nothing
        }

        break;
      }
      case Qt::MoveAction:
      {
        if ( KUrl::List::canDecode( e->mimeData() ) )
        {
          KUrl::List urlList = KUrl::List::fromMimeData( e->mimeData() );

          KUrl dest;
          dest.setPath( item->itemData()->share()->path() );

          KIO::CopyJob *job = KIO::move( urlList, dest, KIO::DefaultFlags );

          job->uiDelegate()->setAutoErrorHandlingEnabled( true );
          job->uiDelegate()->setAutoWarningHandlingEnabled( true );
        }
        else
        {
          // Do nothing
        }

        break;
      }
      default:
      {
        break;
      }
    }
  }
  else
  {
    // Do nothing
  }
}


void Smb4KSharesViewPart::slotMountedShares()
{
  // Get the list of shares:
  QList<Smb4KShare *> *list = mountedSharesList();

  switch ( m_mode )
  {
    case IconMode:
    {
      if ( !list->isEmpty() )
      {
        // Remove obsolete items and unset the current item
        // if we remove it so that we won't get an unwanted
        // selected item.
        Smb4KSharesIconViewItem *item = NULL;

        for ( int i = 0; i < m_icon_view->count(); ++i )
        {
          item = static_cast<Smb4KSharesIconViewItem *>( m_icon_view->item( i ) );

          Smb4KShare *share = findShareByPath( item->itemData()->share()->path() );

          if ( !share || (share->isForeign() && !Smb4KSettings::showAllShares()) )
          {
            if ( item == m_icon_view->currentItem() )
            {
              m_icon_view->setCurrentItem( NULL );
            }
            else
            {
              // Do nothing
            }

            delete item;
            item = NULL;
          }
          else
          {
            // Do nothing
          }
        }

        // Insert the new items and update the existing ones.
        Smb4KSharesIconViewItem *shareItem = NULL;

        for ( int i = 0; i < list->size(); ++i )
        {
          // Search for the item in the tree widget. Use the
          // path for this.
          Smb4KSharesIconViewItem *tmp = NULL;

          for ( int j = 0; j < m_icon_view->count(); ++j )
          {
            tmp = static_cast<Smb4KSharesIconViewItem *>( m_icon_view->item( j ) );

            if ( QString::compare( tmp->itemData()->share()->path(), list->at( i )->path() ) == 0 ||
                 QString::compare( tmp->itemData()->share()->canonicalPath(), list->at( i )->canonicalPath() ) == 0 )
            {
              shareItem = tmp;
            }
            else
            {
              // Do nothing
            }
          }

          if ( shareItem )
          {
            if ( !shareItem->sameShareObject( list->at( i ) ) )
            {
              shareItem->replaceShareObject( list->at( i ) );
            }
            else
            {
              // Do nothing
            }

            m_icon_view->updateToolTip();
          }
          else
          {
            if ( !list->at( i )->isForeign() || Smb4KSettings::showAllShares() )
            {
              shareItem = new Smb4KSharesIconViewItem( list->at( i ), m_icon_view );
              shareItem->setShowMountPoint( Smb4KSettings::showMountPoint() );
            }
            else
            {
              // Do nothing
            }
          }

          shareItem = NULL;
        }
      }
      else
      {
        m_icon_view->clear();
      }

      // Sort the items:
      m_icon_view->sortItems( Qt::AscendingOrder );

      break;
    }
    case ListMode:
    {
      if ( !list->isEmpty() )
      {
        // Remove obsolete items.
        QTreeWidgetItemIterator it( m_list_view );
        Smb4KSharesListViewItem *item = NULL;

        while ( *it )
        {
          item = static_cast<Smb4KSharesListViewItem *>( *it );

          Smb4KShare *share = findShareByPath( item->itemData()->share()->path() );

          if ( !share || (share->isForeign() && !Smb4KSettings::showAllShares()) )
          {
            if ( item == m_list_view->currentItem() )
            {
              m_list_view->setCurrentItem( NULL );
            }
            else
            {
              // Do nothing
            }

            delete item;
            item = NULL;
          }
          else
          {
            // Do nothing
          }

          ++it;
        }

        // Insert the new items and update the existing ones.
        Smb4KSharesListViewItem *shareItem = NULL;

        for ( int i = 0; i < list->size(); ++i )
        {
          // Search for the item in the tree widget. Use the
          // path for this.
          it = QTreeWidgetItemIterator( m_list_view );
          Smb4KSharesListViewItem *tmp = NULL;

          while ( *it )
          {
            tmp = static_cast<Smb4KSharesListViewItem *>( *it );

            if ( QString::compare( tmp->itemData()->share()->path(), list->at( i )->path() ) == 0 ||
                 QString::compare( tmp->itemData()->share()->canonicalPath(), list->at( i )->canonicalPath() ) == 0 )
            {
              shareItem = tmp;
            }
            else
            {
              // Do nothing
            }

            ++it;
          }

          if ( shareItem )
          {
            if ( !shareItem->sameShareObject( list->at( i ) ) )
            {
              shareItem->replaceShareObject( list->at( i ) );
            }
            else
            {
              // Do nothing
            }

            m_list_view->updateToolTip();
          }
          else
          {
            if ( !list->at( i )->isForeign() || Smb4KSettings::showAllShares() )
            {
              shareItem = new Smb4KSharesListViewItem( list->at( i ), m_list_view );
              shareItem->setShowMountPoint( Smb4KSettings::showMountPoint() );
            }
            else
            {
              // Do nothing
            }
          }

          shareItem = NULL;
        }
      }
      else
      {
        m_list_view->clear();
      }

      // Sort the items:
      m_list_view->sortItems( Smb4KSharesListView::Item, Qt::AscendingOrder );

      break;
    }
    default:
    {
      break;
    }
  }

  // Enable the "Unmount All" action.
  actionCollection()->action( "unmount_all_action" )->setEnabled( !list->isEmpty() );
}


void Smb4KSharesViewPart::slotUnmountShare( bool /*checked*/ )
{
  switch ( m_mode )
  {
    case IconMode:
    {
      Smb4KSharesIconViewItem *item = static_cast<Smb4KSharesIconViewItem *>( m_icon_view->currentItem() );

      if ( item )
      {
        Smb4KCore::mounter()->unmountShare( item->itemData()->share(), false );
      }
      else
      {
        // Do nothing
      }

      break;
    }
    case ListMode:
    {
      Smb4KSharesListViewItem *item = static_cast<Smb4KSharesListViewItem *>( m_list_view->currentItem() );

      if ( item )
      {
        Smb4KCore::mounter()->unmountShare( item->itemData()->share(), false );
      }
      else
      {
        // Do nothing
      }

      break;
    }
    default:
    {
      break;
    }
  }
}


void Smb4KSharesViewPart::slotForceUnmountShare( bool /*checked*/ )
{
  switch ( m_mode )
  {
    case IconMode:
    {
      Smb4KSharesIconViewItem *item = static_cast<Smb4KSharesIconViewItem *>( m_icon_view->currentItem() );

      if ( item )
      {
        Smb4KCore::mounter()->unmountShare( item->itemData()->share(), true );
      }
      else
      {
        // Do nothing
      }

      break;
    }
    case ListMode:
    {
      Smb4KSharesListViewItem *item = static_cast<Smb4KSharesListViewItem *>( m_list_view->currentItem() );

      if ( item )
      {
        Smb4KCore::mounter()->unmountShare( item->itemData()->share(), true );
      }
      else
      {
        // Do nothing
      }

      break;
    }
    default:
    {
      break;
    }
  }
}


void Smb4KSharesViewPart::slotUnmountAllShares( bool /*checked*/ )
{
  Smb4KCore::mounter()->unmountAllShares();
}


void Smb4KSharesViewPart::slotSynchronize( bool /*checked*/ )
{
  switch ( m_mode )
  {
    case IconMode:
    {
      Smb4KSynchronizationDialog *dlg = m_icon_view->findChild<Smb4KSynchronizationDialog *>();
      Smb4KSharesIconViewItem *item = static_cast<Smb4KSharesIconViewItem *>( m_icon_view->currentItem() );

      if ( item && !item->itemData()->share()->isInaccessible() && !dlg )
      {
        dlg = new Smb4KSynchronizationDialog( item->itemData()->share(), m_icon_view );

        dlg->setVisible( true );
      }
      else
      {
        // Do nothing
      }

      break;
    }
    case ListMode:
    {
      Smb4KSynchronizationDialog *dlg = m_list_view->findChild<Smb4KSynchronizationDialog *>();
      Smb4KSharesListViewItem *item = static_cast<Smb4KSharesListViewItem *>( m_list_view->currentItem() );

      if ( item && !item->itemData()->share()->isInaccessible() && !dlg )
      {
        dlg = new Smb4KSynchronizationDialog( item->itemData()->share(), m_list_view );

        dlg->setVisible( true );
      }
      else
      {
        // Do nothing
      }

      break;
    }
    default:
    {
      break;
    }
  }
}

void Smb4KSharesViewPart::slotKonsole( bool /*checked*/ )
{
  switch ( m_mode )
  {
    case IconMode:
    {
      Smb4KSharesIconViewItem *item = static_cast<Smb4KSharesIconViewItem *>( m_icon_view->currentItem() );

      if ( item && !item->itemData()->share()->isInaccessible() )
      {
        Smb4KCore::open( item->itemData()->share(), Smb4KCore::Konsole );
      }
      else
      {
        // Do nothing
      }


      break;
    }
    case ListMode:
    {
      Smb4KSharesListViewItem *item = static_cast<Smb4KSharesListViewItem *>( m_list_view->currentItem() );

      if ( item && !item->itemData()->share()->isInaccessible() )
      {
        Smb4KCore::open( item->itemData()->share(), Smb4KCore::Konsole );
      }
      else
      {
        // Do nothing
      }

      break;
    }
    default:
    {
      break;
    }
  }
}


void Smb4KSharesViewPart::slotFileManager( bool /*checked*/ )
{
  switch ( m_mode )
  {
    case IconMode:
    {
      Smb4KSharesIconViewItem *item = static_cast<Smb4KSharesIconViewItem *>( m_icon_view->currentItem() );

      if ( item && !item->itemData()->share()->isInaccessible() )
      {
        Smb4KCore::open( item->itemData()->share(), Smb4KCore::FileManager );
      }
      else
      {
        // Do nothing
      }

      break;
    }
    case ListMode:
    {
      Smb4KSharesListViewItem *item = static_cast<Smb4KSharesListViewItem *>( m_list_view->currentItem() );

      if ( item && !item->itemData()->share()->isInaccessible() )
      {
        Smb4KCore::open( item->itemData()->share(), Smb4KCore::FileManager );
      }
      else
      {
        // Do nothing
      }

      break;
    }
    default:
    {
      break;
    }
  }
}


void Smb4KSharesViewPart::slotSynchronizationState( int state )
{
  switch ( state )
  {
    case SYNCHRONIZER_START:
    {
      actionCollection()->action( "synchronize_action" )->setEnabled( false );

      break;
    }
    case SYNCHRONIZER_STOP:
    {
      actionCollection()->action( "synchronize_action" )->setEnabled( true );

      break;
    }
    default:
    {
      break;
    }
  }
}


void Smb4KSharesViewPart::slotAboutToQuit()
{
  saveSettings();
}


#include "smb4ksharesview_part.moc"
