/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A table with graph with more features.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgtablewithgraph.h"
#include "skgservices.h"
#include "skgtraces.h"
#include "skgmainpanel.h"
#include "skggraphicsscene.h"
#include "skgcolorbutton.h"
#include "skgcombobox.h"

#include <ksavefile.h>
#include <kmenu.h>
#include <kfiledialog.h>
#include <kstringhandler.h>
#include <kstandarddirs.h>
#include <kcolorscheme.h>

#include <QtGui/QHeaderView>
#include <QtGui/QGraphicsLineItem>
#include <QDomDocument>
#include <QRegExp>
#include <QPrinter>
#include <QPainter>
#include <QTableWidgetItem>
#include <QDesktopServices>
#include <QTimer>
#include <QScriptEngine>
#include <QGraphicsDropShadowEffect>
#include <qmath.h>
#include <qtextcodec.h>
#include <qpropertyanimation.h>


/**
  * Data identifier for value
  */
static const int DATA_VALUE = 12;
/**
  * Data identifier for color
  */
static const int DATA_COLOR_H = 11;
/**
  * Data identifier for color
  */
static const int DATA_COLOR_S = 12;
/**
  * Data identifier for color
  */
static const int DATA_COLOR_V = 13;
/**
  * Data identifier for Z value
  */
static const int DATA_Z_VALUE = 14;
/**
  * Alpha value
  */
static const int ALPHA = 200;
/**
  * Box size
  */
static const double BOX_SIZE = 120.0;
/**
  * Box margin
  */
static const double BOX_MARGIN = 10.0;

Qt::SortOrder SKGTableWithGraph::m_sortOrder = Qt::AscendingOrder;
int SKGTableWithGraph::m_sortColumn = 0;

SKGTableWithGraph::SKGTableWithGraph(QWidget *parent)
    : QWidget(parent), m_scene(NULL), m_additionalInformation(NONE), m_nbVirtualColumns(0),
      m_selectable(true), m_toolBarVisible(true), m_graphTypeVisible(true), m_limitVisible(true),
      m_linearRegressionVisible(true), m_legendVisible(false), m_graphVisible(true), m_tableVisible(true), m_textVisible(false), m_zeroVisible(true),
      m_mainMenu(NULL),
      m_indexSum(-1), m_indexAverage(-1), m_indexMin(-1), m_indexLinearRegression(-1)
{
    m_axisColor = Qt::gray;
    m_gridColor = Qt::lightGray;
    m_minColor = Qt::red;
    m_maxColor = Qt::green;
    m_averageColor = Qt::blue;
    m_tendencyColor = Qt::darkYellow;
    m_backgroundColor = Qt::white;
    m_textColor = Qt::black;
    m_NegativeColor = KColorScheme(QPalette::Normal).foreground(KColorScheme::NegativeText);

    ui.setupUi(this);
    ui.kTextEdit->hide();

#ifdef SKGKDE46ENABLED
    ui.kFilterEdit->setPlaceholderText(i18n("Search"));
#endif

    m_displayMode = new SKGComboBox();
    m_displayMode->addItem(KIcon("office-chart-bar-stacked"), i18nc("Noun, a type of graph, with bars stacked upon each other", "Stack of lines"), STACK);
    m_displayMode->addItem(KIcon("office-chart-bar-stacked"), i18nc("Noun, a type of graph, with bars stacked upon each other", "Stack of columns"), STACKCOLUMNS);
    m_displayMode->addItem(KIcon("office-chart-bar"), i18nc("Noun, a type of graph, with bars placed besides each other", "Histogram"), HISTOGRAM);
    m_displayMode->addItem(KIcon("office-chart-scatter"), i18nc("Noun, a type of graph with only points", "Point"), POINT);
    m_displayMode->addItem(KIcon("office-chart-line"), i18nc("Noun, a type of graph with only lines", "Line"), LINE);
    m_displayMode->addItem(KIcon("office-chart-area-stacked"), i18nc("Noun, a type of graph, with lines stacked upon each other", "Stacked area"), STACKAREA);
    m_displayMode->addItem(KIcon("skg-chart-bubble"), i18nc("Noun, a type of graph, with bubbles", "Bubble"), BUBBLE);
    m_displayMode->addItem(KIcon("office-chart-pie"), i18nc("Noun, a type of graph that looks like a sliced pie", "Pie"), PIE);
    m_displayMode->addItem(KIcon("office-chart-ring"), i18nc("Noun, a type of graph that looks like concentric slices of a pie (a la filelight)", "Concentric pie"), CONCENTRICPIE);

    ui.graphicView->addToolbarWidget(m_displayMode);

    ui.kShow->addItem("table", i18n("Table"),               "view-list-details", "", "",            "text",        "graph");
    ui.kShow->addItem("graph", i18n("Graph"),               "office-chart-pie", "", "",                "text",        "table");
    ui.kShow->addItem("text", i18n("Text"),                 "view-list-text", "", "",               "table;graph", "table;graph");

    ui.kShow->setDefaultState("table;graph");

    connect(ui.kShow, SIGNAL(stateChanged()), this, SLOT(onDisplayModeChanged()), Qt::QueuedConnection);

    m_timer.setSingleShot(true);
    connect(&m_timer, SIGNAL(timeout()), this, SLOT(refresh()), Qt::QueuedConnection);

    m_timerRedraw.setSingleShot(true);
    connect(&m_timerRedraw, SIGNAL(timeout()), this, SLOT(redrawGraph()), Qt::QueuedConnection);

    //Build contextual menu
    ui.kTable->setContextMenuPolicy(Qt::CustomContextMenu);
    connect(ui.kTable, SIGNAL(customContextMenuRequested(const QPoint &)), this, SLOT(showMenu(const QPoint&)));

    m_mainMenu = new KMenu(ui.kTable);

    QAction* actExport = m_mainMenu->addAction(KIcon("document-export"), i18nc("Noun, user action", "Export..."));
    connect(actExport, SIGNAL(triggered(bool)), this, SLOT(onExport()));

    //Add graph mode in menu
    getGraphContextualMenu()->addSeparator();
    SKGComboBox* displayModeMenu = new SKGComboBox();
    displayModeMenu->addItem(KIcon("office-chart-bar-stacked"), i18nc("Noun, a type of graph, with bars stacked upon each other", "Stack of lines"), STACK);
    displayModeMenu->addItem(KIcon("office-chart-bar-stacked"), i18nc("Noun, a type of graph, with bars stacked upon each other", "Stack of columns"), STACKCOLUMNS);
    displayModeMenu->addItem(KIcon("office-chart-bar"), i18nc("Noun, a type of graph, with bars placed besides each other", "Histogram"), HISTOGRAM);
    displayModeMenu->addItem(KIcon("office-chart-scatter"), i18nc("Noun, a type of graph with only points", "Point"), POINT);
    displayModeMenu->addItem(KIcon("office-chart-line"), i18nc("Noun, a type of graph with only lines", "Line"), LINE);
    displayModeMenu->addItem(KIcon("office-chart-area-stacked"), i18nc("Noun, a type of graph, with lines stacked upon each other", "Stacked area"), STACKAREA);
    displayModeMenu->addItem(KIcon("skg-chart-bubble"), i18nc("Noun, a type of graph, with bubbles", "Bubble"), BUBBLE);
    displayModeMenu->addItem(KIcon("office-chart-pie"), i18nc("Noun, a type of graph that looks like a sliced pie", "Pie"), PIE);
    displayModeMenu->addItem(KIcon("office-chart-ring"), i18nc("Noun, a type of graph that looks like concentric slices of a pie (a la filelight)", "Concentric pie"), CONCENTRICPIE);

    m_displayModeWidget = new QWidgetAction(this);
    m_displayModeWidget->setDefaultWidget(displayModeMenu);
    getGraphContextualMenu()->addAction(m_displayModeWidget);

    connect(displayModeMenu, SIGNAL(currentIndexChanged(int)), m_displayMode, SLOT(setCurrentIndex(int)));
    connect(m_displayMode, SIGNAL(currentIndexChanged(int)), displayModeMenu, SLOT(setCurrentIndex(int)));

    //Reset Colors
    QAction* resetColors = m_mainMenu->addAction(KIcon("format-fill-color"), i18nc("Noun, user action", "Reset default colors"));
    connect(resetColors, SIGNAL(triggered(bool)), this, SLOT(onResetColors()));
    getGraphContextualMenu()->addAction(resetColors);

    //Add item in menu of the graph
    m_allPositiveMenu = getGraphContextualMenu()->addAction(KIcon("go-up-search"), i18nc("Noun, user action", "All values in positive"));
    if(m_allPositiveMenu)  m_allPositiveMenu->setCheckable(true);
    connect(m_allPositiveMenu, SIGNAL(toggled(bool)), this, SLOT(redrawGraphDelayed()), Qt::QueuedConnection);

    //Add item in menu of the graph
    m_actShowLimits = getGraphContextualMenu()->addAction(i18nc("Noun, user action", "Show limits and average"));
    if(m_actShowLimits) {
        m_actShowLimits->setCheckable(true);
        m_actShowLimits->setChecked(m_limitVisible);
        connect(m_actShowLimits, SIGNAL(triggered(bool)), this, SLOT(onSwitchLimits()));
    }

    //Add item in menu of the graph
    m_actShowLinearRegression = getGraphContextualMenu()->addAction(i18nc("Noun, user action", "Show tendency line"));
    if(m_actShowLinearRegression) {
        m_actShowLinearRegression->setCheckable(true);
        m_actShowLinearRegression->setChecked(m_linearRegressionVisible);
        connect(m_actShowLinearRegression, SIGNAL(triggered(bool)), this, SLOT(onSwitchLinearRegression()));
    }

    //Add item in menu of the graph
    m_actShowLegend = getGraphContextualMenu()->addAction(KIcon("help-contents"), i18nc("Noun, user action", "Show legend"));
    if(m_actShowLegend) {
        m_actShowLegend->setCheckable(true);
        m_actShowLegend->setChecked(m_legendVisible);
        connect(m_actShowLegend, SIGNAL(triggered(bool)), this, SLOT(onSwitchLegend()));
    }

    //Add item in menu of the graph
    m_actShowZero = getGraphContextualMenu()->addAction(KIcon("snap-orto"), i18nc("Noun, user action", "Show origin"));
    if(m_actShowZero) {
        m_actShowZero->setCheckable(true);
        m_actShowZero->setChecked(m_zeroVisible);
        connect(m_actShowZero, SIGNAL(triggered(bool)), this, SLOT(onSwitchZero()));
    }

    //Set headers parameters
    QHeaderView* verticalHeader = ui.kTable->verticalHeader();
    if(verticalHeader) {
        verticalHeader->hide();
        verticalHeader->setDefaultSectionSize(verticalHeader->minimumSectionSize());
        verticalHeader->setResizeMode(QHeaderView::ResizeToContents);
    }

    ui.kTable->setSortingEnabled(false);    //sort must be enable for refresh method
    QHeaderView* horizontalHeader = ui.kTable->horizontalHeader();
    if(horizontalHeader) {
        horizontalHeader->setResizeMode(QHeaderView::ResizeToContents);
        horizontalHeader->show();
        horizontalHeader->setSortIndicatorShown(true);
        horizontalHeader->setSortIndicator(m_sortColumn, m_sortOrder);
        connect(horizontalHeader, SIGNAL(sortIndicatorChanged(int, Qt::SortOrder)), this, SLOT(refresh()));
    }

    connect(m_displayMode, SIGNAL(currentIndexChanged(int)), this, SLOT(redrawGraphDelayed()), Qt::QueuedConnection);
    connect(ui.graphicView, SIGNAL(resized()), this, SLOT(redrawGraphDelayed()), Qt::QueuedConnection);

    /*SKGStringListList data;
    QStringList h, a, b;
    h << "" << "2010-01" << "2010-02" << "2010-03";
    a << "a" << "10" << "20" << "15";
    b << "b" << "-25" << "-5" << "-30";
    data << h << a << b;
    SKGServices::SKGUnitInfo u;
    u.Symbol = "F";
    u.NbDecimal = 2;
    setData(data, u, u);*/
}


SKGTableWithGraph::~SKGTableWithGraph()
{
    delete m_scene;
    m_scene = NULL;

    m_mainMenu = NULL;
    m_actShowLimits = NULL;
    m_actShowLinearRegression = NULL;
    m_displayModeWidget = NULL;
    m_displayMode = NULL;
}

bool SKGTableWithGraph::isGraphVisible() const
{
    return m_graphVisible;
}

bool SKGTableWithGraph::isTableVisible() const
{
    return m_tableVisible;
}

bool SKGTableWithGraph::isTextReportVisible() const
{
    return m_textVisible;
}

SKGShow* SKGTableWithGraph::getShowWidget() const
{
    return ui.kShow;
}
void SKGTableWithGraph::setAverageColor(const QColor& iColor)
{
    m_averageColor = iColor;
}

void SKGTableWithGraph::setAxisColor(const QColor& iColor)
{
    m_axisColor = iColor;
}

void SKGTableWithGraph::setGridColor(const QColor& iColor)
{
    m_gridColor = iColor;
}

void SKGTableWithGraph::setMaxColor(const QColor& iColor)
{
    m_maxColor = iColor;
}

void SKGTableWithGraph::setMinColor(const QColor& iColor)
{
    m_minColor = iColor;
}

void SKGTableWithGraph::setTendencyColor(const QColor& iColor)
{
    m_tendencyColor = iColor;
}

void SKGTableWithGraph::setOutlineColor(const QColor& iColor)
{
    m_outlineColor = iColor;
}

void SKGTableWithGraph::setBackgroundColor(const QColor& iColor)
{
    m_backgroundColor = iColor;
}

void SKGTableWithGraph::setTextColor(const QColor& iColor)
{
    m_textColor = iColor;
}

void SKGTableWithGraph::setAntialiasing(bool iAntialiasing)
{
    ui.graphicView->setAntialiasing(iAntialiasing);
}

void SKGTableWithGraph::setGraphTypeSelectorVisible(bool iVisible)
{
    m_graphTypeVisible = iVisible;
    if(m_displayMode) m_displayMode->setVisible(m_graphTypeVisible);
    if(m_displayModeWidget) m_displayModeWidget->setVisible(m_graphTypeVisible);
}

bool SKGTableWithGraph::isGraphTypeSelectorVisible() const
{
    return m_graphTypeVisible;
}

void SKGTableWithGraph::onSwitchLimits()
{
    m_limitVisible = !m_limitVisible;
    refresh();
}

void SKGTableWithGraph::onSwitchLegend()
{
    m_legendVisible = !m_legendVisible;
    redrawGraphDelayed();
}

void SKGTableWithGraph::onSwitchZero()
{
    m_zeroVisible = !m_zeroVisible;
    redrawGraphDelayed();
}

void SKGTableWithGraph::onSwitchLinearRegression()
{
    m_linearRegressionVisible = !m_linearRegressionVisible;
    refresh();
}

KMenu* SKGTableWithGraph::getTableContextualMenu() const
{
    return m_mainMenu;
}

KMenu* SKGTableWithGraph::getGraphContextualMenu() const
{
    return ui.graphicView->getContextualMenu();
}

void SKGTableWithGraph::showMenu(const QPoint& pos)
{
    if(m_mainMenu) m_mainMenu->popup(ui.kTable->mapToGlobal(pos));
}

QTableWidget* SKGTableWithGraph::table() const
{
    return ui.kTable;
}

SKGGraphicsView* SKGTableWithGraph::graph() const
{
    return ui.graphicView;
}

SKGWebView* SKGTableWithGraph::textReport() const
{
    return ui.kTextEdit;
}

SKGStringListList SKGTableWithGraph::getTable()
{
    //Build table
    SKGStringListList table;

    //Get header names
    int nb = ui.kTable->columnCount();
    QStringList cols;
    for(int i = 0; i < nb; ++i) {
        cols.append(ui.kTable->horizontalHeaderItem(i)->text());
    }
    table.append(cols);

    //Get content
    int nb2 = ui.kTable->rowCount();
    for(int i = 0; i < nb2; ++i) {
        QStringList row;

        for(int j = 0; j < nb; j++) {
            SKGColorButton* button = qobject_cast<SKGColorButton*>(ui.kTable->cellWidget(i, j));
            if(button) row.append(button->text());
            else row.append(ui.kTable->item(i, j)->text());
        }
        table.append(row);
    }
    return table;
}
QString SKGTableWithGraph::getState()
{
    SKGTRACEIN(10, "SKGTableWithGraph::getState");
    QDomDocument doc("SKGML");
    QDomElement root = doc.createElement("parameters");
    doc.appendChild(root);

    if(ui.graphicView->isVisible() && ui.kTable->isVisible()) root.setAttribute("splitterState", QString(ui.splitter->saveState().toHex()));
    root.setAttribute("graphMode", SKGServices::intToString((int) getGraphType()));
    root.setAttribute("allPositive", m_allPositiveMenu->isChecked() ? "Y" : "N");
    root.setAttribute("filter", ui.kFilterEdit->text());
    root.setAttribute("limitVisible", m_limitVisible ? "Y" : "N");
    root.setAttribute("linearRegressionVisible", m_linearRegressionVisible ? "Y" : "N");
    root.setAttribute("legendVisible", m_legendVisible ? "Y" : "N");
    root.setAttribute("zeroVisible", m_zeroVisible ? "Y" : "N");
    QMapIterator<QString, QColor> i(m_mapTitleColor);
    while(i.hasNext()) {
        i.next();
        QDomElement color = doc.createElement("color");
        root.appendChild(color);
        color.setAttribute("key", i.key());
        color.setAttribute("value", i.value().name());
    }

    QHeaderView* horizontalHeader = ui.kTable->horizontalHeader();
    root.setAttribute("sortOrder", SKGServices::intToString(horizontalHeader->sortIndicatorOrder()));
    root.setAttribute("sortColumn", SKGServices::intToString(horizontalHeader->sortIndicatorSection()));
    root.setAttribute("graphicViewState", ui.graphicView->getState());
    root.setAttribute("web", ui.kTextEdit->getState());
    root.setAttribute("show", ui.kShow->getState());

    return doc.toString();
}

void SKGTableWithGraph::setState(const QString& iState)
{
    SKGTRACEIN(10, "SKGTableWithGraph::setState");
    m_timer.stop();
    m_timerRedraw.stop();

    QDomDocument doc("SKGML");
    doc.setContent(iState);
    QDomElement root = doc.documentElement();

    m_mapTitleColor.clear();

    QDomNodeList colors = root.elementsByTagName("color");
    int nb = colors.count();
    for(int i = 0; i < nb; ++i) {
        QDomElement c = colors.at(i).toElement();
        m_mapTitleColor[c.attribute("key")] = QColor(c.attribute("value"));
    }

    QString splitterState = root.attribute("splitterState");
    if(!splitterState.isEmpty()) ui.splitter->restoreState(QByteArray::fromHex(splitterState.toAscii()));
    QString graphMode = root.attribute("graphMode");
    QString allPositive = root.attribute("allPositive");
    QString limitVisibleString = root.attribute("limitVisible");
    QString legendVisibleString = root.attribute("legendVisible");
    QString zeroVisibleString = root.attribute("zeroVisible");
    QString linearRegressionVisibleString = root.attribute("linearRegressionVisible");
    QString sortOrder = root.attribute("sortOrder");
    QString sortColumn = root.attribute("sortColumn");
    QString graphicViewState = root.attribute("graphicViewState");
    QString web = root.attribute("web");
    QString show = root.attribute("show");

    //Defaut value in case of reset
    if(graphMode.isEmpty()) graphMode = '0';
    if(allPositive.isEmpty()) allPositive = 'N';
    if(limitVisibleString.isEmpty()) limitVisibleString = 'Y';
    if(legendVisibleString.isEmpty()) legendVisibleString = 'N';
    if(linearRegressionVisibleString.isEmpty()) linearRegressionVisibleString = 'Y';
    if(sortOrder.isEmpty()) sortOrder = '0';
    if(sortColumn.isEmpty()) sortColumn = '0';

    //Set
    setGraphType(SKGTableWithGraph::HISTOGRAM);
    if(m_displayMode) m_displayMode->setCurrentIndex(1);
    setGraphType((SKGTableWithGraph::GraphType) SKGServices::stringToInt(graphMode));
    m_allPositiveMenu->setChecked(allPositive == "Y");
    ui.kFilterEdit->setText(root.attribute("filter"));
    m_limitVisible = (limitVisibleString == "Y");
    if(m_actShowLimits) m_actShowLimits->setChecked(m_limitVisible);
    m_legendVisible = (legendVisibleString == "Y");
    if(m_actShowLegend) m_actShowLegend->setChecked(m_legendVisible);
    m_zeroVisible = (zeroVisibleString != "N");
    if(m_actShowZero) m_actShowZero->setChecked(m_zeroVisible);
    m_linearRegressionVisible = (linearRegressionVisibleString == "Y");
    if(m_actShowLinearRegression) m_actShowLinearRegression->setChecked(m_linearRegressionVisible);
    ui.kTable->setColumnCount(SKGServices::stringToInt(sortColumn) + 1);
    QHeaderView* horizontalHeader = ui.kTable->horizontalHeader();
    if(horizontalHeader) {
        bool previous = horizontalHeader->blockSignals(true);
        horizontalHeader->setSortIndicator(SKGServices::stringToInt(sortColumn), (Qt::SortOrder) SKGServices::stringToInt(sortOrder));
        horizontalHeader->blockSignals(previous);
    }
    ui.graphicView->setState(graphicViewState);
    ui.kTextEdit->setState(web);
    if(!show.isEmpty()) ui.kShow->setState(show);
}

void SKGTableWithGraph::setFilterVisibility(bool iVisibility) const
{
    ui.kToolbar->setVisible(iVisibility);
}

void SKGTableWithGraph::onFilterModified()
{
    m_timerRedraw.stop();
    m_timer.start(300);
}

void SKGTableWithGraph::onDisplayModeChanged()
{
    QStringList mode = SKGServices::splitCSVLine(ui.kShow->getState());

    //Hide all
    if(m_scene) {
        m_scene->clear();
        delete m_scene;
    }
    m_scene = new SKGGraphicsScene();
    ui.graphicView->setScene(m_scene);
    ui.graphicView->hide();
    ui.kTextEdit->hide();
    bool p = ui.kTable->blockSignals(true);
    ui.kTable->hide();
    ui.kTable->blockSignals(p);
    m_graphVisible = false;
    m_tableVisible = false;
    m_textVisible = false;
    m_mapItemGraphic.clear();

    //Show needed widget
    if(mode.contains("table")) {
        ui.kTable->show();
        m_tableVisible = true;
    }
    if(mode.contains("graph")) {
        ui.graphicView->show();
        m_graphVisible = true;
        redrawGraphDelayed();
    }
    if(mode.contains("text")) {
        QTimer::singleShot(100, ui.kTextEdit, SLOT(show()));
        m_textVisible = true;
        redrawText();
    }
}

void SKGTableWithGraph::setData(const SKGStringListList& iData, SKGServices::SKGUnitInfo iPrimaryUnit,
                                SKGServices::SKGUnitInfo iSecondaryUnit,
                                DisplayAdditionalFlag iAdditionalInformation,
                                int iNbVirtualColumn)
{
    SKGTRACEIN(10, "SKGTableWithGraph::setData");
    m_data = iData;
    m_primaryUnit = iPrimaryUnit;
    m_secondaryUnit = iSecondaryUnit;
    m_additionalInformation = iAdditionalInformation;
    m_nbVirtualColumns = iNbVirtualColumn;

    onFilterModified();
}

SKGTableWithGraph::DisplayAdditionalFlag SKGTableWithGraph::getAdditionalDisplayMode() const
{
    return m_additionalInformation;
}

bool SKGTableWithGraph::listSort(const QStringList &s1, const QStringList &s2)
{
    if(m_sortColumn >= s1.count()) m_sortColumn = s1.count() - 1;
    if(m_sortColumn >= 0) {
        QString v1 = s1.at(m_sortColumn);
        QString v2 = s2.at(m_sortColumn);
        if(m_sortColumn == 0) {
            int v = KStringHandler::naturalCompare(v1, v2, Qt::CaseInsensitive);
            return (m_sortOrder ? v > 0 : v < 0);
        }

        double vd1 = SKGServices::stringToDouble(v1);
        double vd2 = SKGServices::stringToDouble(v2);
        return (m_sortOrder ? vd1 > vd2 : vd1 < vd2);
    } else return false;
}

void SKGTableWithGraph::refresh()
{
    SKGTRACEIN(10, "SKGTableWithGraph::refresh");
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

    //Set parameters for static method
    QHeaderView* horizontalHeader = ui.kTable->horizontalHeader();
    m_sortOrder = horizontalHeader->sortIndicatorOrder();
    m_sortColumn = horizontalHeader->sortIndicatorSection();

    SKGStringListList groupedTable = m_data;
    int nbCols = -1;
    if(groupedTable.count()) nbCols = groupedTable.at(0).count();
    int nbRealCols = nbCols;

    //Create filtered table
    {
        //Build list of criterias
        QList< SKGServices::SKGSearchCriteria > criterias = SKGServices::stringToSearchCriterias(ui.kFilterEdit->text());

        //Check all lines
        int nblists = criterias.count();
        if(nblists && nbCols) {
            for(int i = groupedTable.count() - 1; i >= 1; --i) {  //The first line is not filtered because it's the title
                QStringList line = groupedTable.at(i);

                //Get title of the line
                QString val = line.at(0);

                //Filtered
                bool ok = false;
                for(int l = 0; l < nblists; ++l) {
                    QStringList words = criterias[l].words;
                    QChar mode = criterias[l].mode;
                    int nbwords = words.count();

                    bool validateAllWords = true;
                    for(int w = 0; validateAllWords && w < nbwords; ++w) {
                        validateAllWords = val.contains(words.at(w) , Qt::CaseInsensitive);
                    }
                    if(mode == '+') ok |= validateAllWords;
                    else if(mode == '-' && validateAllWords) ok = false;
                }

                if(!ok) groupedTable.removeAt(i);
            }
        }
    }

    //Initialise sumRows
    int nblines = groupedTable.count();
    m_sumRows.clear();
    for(int j = 0; j < nblines; ++j) {
        m_sumRows.push_back(false);
    }

    //Compute sums line
    if(m_additionalInformation & SUM) {
        QStringList sums;
        sums.push_back("");
        for(int i = 1; i < nbCols; ++i) {
            double sum = 0;
            for(int j = 1; j < nblines; ++j) {
                QString valstring = groupedTable.at(j).at(i);
                if(!valstring.isEmpty()) sum += SKGServices::stringToDouble(valstring);
            }
            sums.push_back(SKGServices::doubleToString(sum));
        }
        groupedTable.push_back(sums);
        m_sumRows.push_back(true);
        ++nblines;
    }

    //Compute sub sums lines
    if(m_additionalInformation & SUM && m_sortColumn == 0 && m_sortOrder == Qt::AscendingOrder) {
        //Create a list of sums lines associted to the index where to add them
        QMap<int, QStringList> sums;
        for(int i = 1; i < nbCols; ++i) {
            QString previousHeaderSum;
            double sum = 0;
            int nbElem = 0;

            for(int j = 1; j < nblines - 1; ++j) {
                QString currentHeaderSum = groupedTable.at(j).at(0);
                int posSep = currentHeaderSum.indexOf(OBJECTSEPARATOR);
                if(posSep != -1) currentHeaderSum = currentHeaderSum.left(posSep);

                if(currentHeaderSum != previousHeaderSum) {
                    //Add this sum
                    if(nbElem > 1) {
                        QStringList thisSum;
                        if(i == 1) thisSum.push_back(previousHeaderSum);
                        else thisSum = sums[j];
                        thisSum.push_back(SKGServices::doubleToString(sum));

                        sums[j] = thisSum;
                    }

                    //Reset sum
                    sum = 0;
                    nbElem = 0;
                }
                previousHeaderSum = currentHeaderSum;

                QString valstring = groupedTable.at(j).at(i);
                if(!valstring.isEmpty()) sum += SKGServices::stringToDouble(valstring);
                ++nbElem;
            }

            //Add last sum
            if(nbElem > 1) {
                QStringList thisSum;
                if(i == 1) thisSum.push_back(previousHeaderSum);
                else thisSum = sums[nblines-1];
                thisSum.push_back(SKGServices::doubleToString(sum));
                sums[nblines-1] = thisSum;
            }
        }

        //Add all sums in table
        QList<int> keys = sums.keys();
        qSort(keys.begin(), keys.end());
        int nbkey = keys.count();
        for(int i = nbkey - 1; i >= 0; --i) {
            int key = keys.at(i);
            groupedTable.insert(key, sums[key]);
            m_sumRows.insert(key, true);
            ++nblines;
        }
    }

    //Compute sum and average column
    m_indexSum = -1;
    m_indexAverage = -1;
    m_indexMin = -1;
    m_indexLinearRegression = -1;
    if(nbCols > 2 && (m_additionalInformation & SUM || (m_limitVisible && m_additionalInformation & AVERAGE) || (m_limitVisible && m_additionalInformation & LIMITS))) {
        SKGTRACEIN(10, "SKGTableWithGraph::refresh-computation");
        //Add title
        QStringList newLine = groupedTable.at(0);
        if(m_additionalInformation & SUM) {
            m_indexSum = newLine.count();
            newLine.push_back(i18nc("Noun, the numerical sum of a list of values", "Sum"));
        }
        if(m_limitVisible && m_additionalInformation & AVERAGE) {
            m_indexAverage = newLine.count();
            newLine.push_back(i18nc("Noun, the numerical average of a list of values", "Average"));
        }
        if(m_limitVisible && m_additionalInformation & LIMITS) {
            m_indexMin = newLine.count();
            newLine.push_back(i18nc("Noun, the minimum value of a list of values", "Min"));
            newLine.push_back(i18nc("Noun, the maximum value of a list of values", "Max"));
        }

        if(m_linearRegressionVisible) {
            m_indexLinearRegression = newLine.count();
            newLine.push_back(i18nc("Noun", "Tendency line"));
        }

        groupedTable.replace(0, newLine);

        for(int i = 1; i < nblines; ++i) {
            QStringList newLine = groupedTable.at(i);
            double sumy = 0;
            double sumx = 0;
            double sumx2 = 0;
            double sumxy = 0;
            double min = 10e20;
            double max = -10e20;
            int nbVals = 0;
            for(int j = 1; j < nbCols - m_nbVirtualColumns; ++j) {
                QString valstring = newLine.at(j);
                if(!valstring.isEmpty()) {
                    double v = SKGServices::stringToDouble(valstring);
                    sumx += j;
                    sumx2 += j * j;
                    sumy += v;
                    sumxy += j * v;
                    min = qMin(min, v);
                    max = qMax(max, v);
                    ++nbVals;
                }
            }
            if(m_additionalInformation & SUM) newLine.push_back(SKGServices::doubleToString(sumy));
            if(m_limitVisible && m_additionalInformation & AVERAGE && nbVals) newLine.push_back(SKGServices::doubleToString(sumy / nbVals));
            if(m_limitVisible && m_additionalInformation & LIMITS) {
                newLine.push_back(SKGServices::doubleToString(min));
                newLine.push_back(SKGServices::doubleToString(max));
            }

            if(nbVals) {
                double s2x = sumx2 / nbVals - sumx * sumx / (nbVals * nbVals);
                double sxy = sumxy / nbVals - (sumx / nbVals) * (sumy / nbVals);

                double a = (s2x ? sxy / s2x : 0.0);
                double b = sumy / nbVals - a * sumx / nbVals;

                newLine.push_back("y=" % SKGServices::doubleToString(a) % "*x+" % SKGServices::doubleToString(b));
            } else newLine.push_back("y=0");

            groupedTable.replace(i, newLine);
        }
        if(m_linearRegressionVisible) ++nbCols;
        if(m_additionalInformation & SUM) ++nbCols;
        if(m_limitVisible && m_additionalInformation & AVERAGE) ++nbCols;
        if(m_limitVisible && m_additionalInformation & LIMITS) nbCols += 2;
    }

    //Sort lines
    if(m_sortColumn != 0  || m_sortOrder != Qt::AscendingOrder) {
        SKGTRACEIN(10, "SKGTableWithGraph::refresh-sort");
        //Extract title and sums
        if(groupedTable.count()) {
            QStringList fist = groupedTable.takeFirst();
            QStringList last;
            if(m_additionalInformation & SUM && groupedTable.count()) last = groupedTable.takeLast();

            //Sort
            qSort(groupedTable.begin(), groupedTable.end(), listSort);

            //Add title and sums
            groupedTable.insert(0, fist);
            if(m_additionalInformation & SUM) groupedTable.push_back(last);
        }
    }

    IFSKGTRACEL(10) {
        QStringList dump = SKGServices::tableToDump(groupedTable, SKGServices::DUMP_TEXT);
        int nbl = dump.count();
        for(int i = 0; i < nbl; ++i) {
            SKGTRACE << dump[i] << endl;
        }
    }

    //Fill table
    {
        SKGTRACEIN(10, "SKGTableWithGraph::refresh-fill");

        int nbRealColumns = nbRealCols - m_nbVirtualColumns;
        KLocale* locale = KGlobal::locale();
        ui.kTable->hide();
        QHeaderView* horizontalHeader = ui.kTable->horizontalHeader();
        if(horizontalHeader) horizontalHeader->setResizeMode(QHeaderView::Fixed);       //Needed to improve performances of setHorizontalHeaderLabels
        ui.kTable->clear();
        ui.kTable->setRowCount(nblines - 1);
        ui.kTable->setColumnCount(nbCols);
        for(int i = 0; i < nblines; ++i) {
            QStringList line = groupedTable.at(i);
            if(i == 0) {
                SKGTRACEIN(10, "SKGTableWithGraph::refresh-sethorizontalheaderlabels");
                //Set header
                ui.kTable->setHorizontalHeaderLabels(line);
            } else {
                for(int j = 0; j < nbCols; ++j) {
                    QString val = line.at(j);

                    QTableWidgetItem* item;
                    if(j == 0) {
                        //Create the line header
                        if(m_sumRows[i]) {
                            item = new QTableWidgetItem(val.isEmpty() ?
                                                        i18nc("Noun, the numerical sum of a list of values", "Sum")
                                                        : i18nc("Noun, the numerical sum of a list of values", "Sum of '%1'", val));
                            item->setData(1, val);
                            QFont f = item->font();
                            f.setBold(true);
                            item->setFont(f);
                            if(m_indexLinearRegression != -1) item->setData(DATA_VALUE, line.at(m_indexLinearRegression));
                            ui.kTable->setItem(i - 1, j, item);
                        } else {
                            //New color selector
                            QColor defaultColor;
                            int color_h = (240 + 360 * (i - 1) / nblines) % 360; //First color is blue to be compliant with min (red) and max (green)
                            defaultColor = QColor::fromHsv(color_h, 255, 255);

                            QColor color;
                            if(m_mapTitleColor.contains(val)) {
                                color = m_mapTitleColor[val];
                            } else {
                                color = defaultColor;
                                m_mapTitleColor[val] = color;
                            }

                            SKGColorButton* colorSelector = new SKGColorButton(this);
                            colorSelector->setText(val);
                            colorSelector->setToolTip(val);
                            colorSelector->setColor(color);
                            colorSelector->setDefaultColor(defaultColor);
                            connect(colorSelector, SIGNAL(changed(QColor)), this, SLOT(onChangeColor()));

                            ui.kTable->setCellWidget(i - 1, 0, colorSelector);
                        }
                    } else {
                        //Add a value
                        QString tooltip = line.at(0) % '\n' % groupedTable.at(0).at(j);
                        if(!val.isEmpty()) {
                            if(j == m_indexLinearRegression) {
                                //A linear regression value
                                item = new QTableWidgetItem(val);
                            } else {
                                //A single value
                                double vald = SKGServices::stringToDouble(val);
                                QString vals = locale->formatMoney(vald, m_primaryUnit.Symbol, m_primaryUnit.NbDecimal);
                                tooltip += '\n' % vals;

                                item = new QTableWidgetItem(vals);
                                item->setToolTip(tooltip);
                                if(!m_secondaryUnit.Symbol.isEmpty() && m_secondaryUnit.Value) {
                                    item->setToolTip(tooltip % '\n' % locale->formatMoney(vald / m_secondaryUnit.Value, m_secondaryUnit.Symbol, m_secondaryUnit.NbDecimal));
                                }

                                item->setData(DATA_VALUE, vald);
                                item->setTextAlignment(Qt::AlignRight);
                                if(vald < 0) item->setForeground(m_NegativeColor);
                                if(m_sumRows[i]) {
                                    QFont f = item->font();
                                    f.setBold(true);
                                    item->setFont(f);
                                }
                            }
                        } else {
                            //An empty value
                            item = new QTableWidgetItem("");
                            item->setToolTip(tooltip);
                            item->setData(DATA_VALUE, "");
                        }
                        item->setFlags((j >= nbRealColumns && j != m_indexSum) || ui.kTable->horizontalHeaderItem(j)->text() == "0000" ? Qt::NoItemFlags : Qt::ItemIsEnabled | Qt::ItemIsSelectable);
                        ui.kTable->setItem(i - 1, j, item);
                    }
                }
            }
        }

        //Refresh graphic view
        redrawGraph();

        //Refresh text area
        redrawText();

        if(horizontalHeader) horizontalHeader->setResizeMode(QHeaderView::ResizeToContents);

        if(m_tableVisible) ui.kTable->show();

        if(horizontalHeader && horizontalHeader->count()) {
            horizontalHeader->resizeSection(0, 100);
            horizontalHeader->setResizeMode(0, QHeaderView::Interactive);
        }
    }

    QApplication::restoreOverrideCursor();
}

void SKGTableWithGraph::onChangeColor()
{
    SKGColorButton* colorButton = qobject_cast<SKGColorButton*>(sender());
    if(colorButton) {
        m_mapTitleColor[colorButton->text()] = colorButton->color();
        refresh();
    }
}

void SKGTableWithGraph::onResetColors()
{
    m_mapTitleColor.clear();
    refresh();
}

void SKGTableWithGraph::onSelectionChanged()
{
    _SKGTRACEIN(10, "SKGTableWithGraph::onSelectionChanged");
    if(m_graphVisible) {
        //Unset color on previous selection
        int nbRow = ui.kTable->rowCount();
        int nbCol = ui.kTable->columnCount();
        for(int r = 0; r < nbRow; ++r) {
            for(int c = 0; c < nbCol; ++c) {
                QTableWidgetItem* previous = ui.kTable->item(r, c);
                if(previous) {
                    QGraphicsItem* val = m_mapItemGraphic[previous];
                    if(val) {
                        QAbstractGraphicsShapeItem * graphicItem = dynamic_cast<QAbstractGraphicsShapeItem *>(val);
                        if(graphicItem) {
                            QColor color = QColor::fromHsv(graphicItem->data(DATA_COLOR_H).toInt(),
                                                           graphicItem->data(DATA_COLOR_S).toInt(),
                                                           graphicItem->data(DATA_COLOR_V).toInt());
                            color.setAlpha(ALPHA);

                            graphicItem->setBrush(QBrush(color));
                            graphicItem->setZValue(graphicItem->data(DATA_Z_VALUE).toReal());
                            if(graphicItem->isSelected()) graphicItem->setSelected(false);
                        } else {
                            QGraphicsLineItem * graphicItem = dynamic_cast<QGraphicsLineItem *>(val);
                            if(graphicItem) {
                                QPen pen = graphicItem->pen();
                                QColor color = QColor::fromHsv(graphicItem->data(DATA_COLOR_H).toInt(),
                                                               graphicItem->data(DATA_COLOR_S).toInt(),
                                                               graphicItem->data(DATA_COLOR_V).toInt());
                                color.setAlpha(ALPHA);
                                pen.setColor(color);
                                graphicItem->setZValue(graphicItem->data(DATA_Z_VALUE).toReal());
                                graphicItem->setPen(pen);
                            }
                        }
                    }
                }
            }
        }

        //Set highlight color on current selection
        QList<QTableWidgetItem*> selected = ui.kTable->selectedItems();
        int nb = selected.count();
        for(int i = 0; i < nb; ++i) {
            QTableWidgetItem* current = selected.at(i);
            if(current) {
                QGraphicsItem* val = m_mapItemGraphic[current];
                QAbstractGraphicsShapeItem * graphicItem = dynamic_cast<QAbstractGraphicsShapeItem *>(val);
                if(graphicItem) {
                    graphicItem->setBrush(QBrush(QApplication::palette().color(QPalette::Highlight)));
                    graphicItem->setZValue(15);
                    graphicItem->setSelected(true);
                    graphicItem->ensureVisible();
                } else {
                    QGraphicsLineItem * graphicItem = dynamic_cast<QGraphicsLineItem *>(val);
                    if(graphicItem) {
                        QPen pen = graphicItem->pen();
                        pen.setColor(QApplication::palette().color(QPalette::Highlight));
                        graphicItem->setPen(pen);
                        graphicItem->setZValue(15);
                        graphicItem->ensureVisible();
                    }
                }
            }
        }
    }

    emit selectionChanged();
}

void SKGTableWithGraph::onDoubleClickGraph()
{
    if(m_scene) {
        //Get selection
        QList<QGraphicsItem *> selectedGraphItems = m_scene->selectedItems();
        if(selectedGraphItems.count()) {
            Q_EMIT cellDoubleClicked(selectedGraphItems[0]->data(1).toInt(), selectedGraphItems[0]->data(2).toInt());
        }
    }
}

void SKGTableWithGraph::onDoubleClick(int row, int column)
{
    Q_EMIT cellDoubleClicked(row, column);
}

void SKGTableWithGraph::onSelectionChangedInGraph()
{
    _SKGTRACEIN(10, "SKGTableWithGraph::onSelectionChangedInGraph");
    if(m_scene) {
        bool previous = ui.kTable->blockSignals(true);
        ui.kTable->clearSelection();

        //Get selection
        QList<QGraphicsItem *> selectedGraphItems = m_scene->selectedItems();
        int nb = selectedGraphItems.count();
        for(int i = 0; i < nb; ++i) {
            ui.kTable->setCurrentCell(selectedGraphItems[i]->data(1).toInt(), selectedGraphItems[i]->data(2).toInt(), QItemSelectionModel::Select);
        }
        ui.kTable->blockSignals(previous);

        previous = m_scene->blockSignals(true);
        onSelectionChanged();
        m_scene->blockSignals(previous);
    }
}

void SKGTableWithGraph::redrawGraphDelayed()
{
    m_timerRedraw.start(300);
}

void SKGTableWithGraph::redrawText()
{
    if(!m_textVisible) return;
    SKGTRACEIN(10, "SKGTableWithGraph::redrawText");
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

    QString html = "<? xml version = \"1.0\" encoding=\"utf-8\"?>"
                   "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Strict//EN\" \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd\">"
                   "<html xmlns=\"http://www.w3.org/1999/xhtml\">"
                   "<head>"
                   "<meta http-equiv=\"content-type\" content=\"text/html;charset=utf-8\" />"
                   "<meta http-equiv=\"Content-Style-Type\" content=\"text/css\" />"
                   "<style type=\"text/css\">"
                   "body{background-color: #FFFFFF; font-size : small;} h1{text-decoration: underline; color: #FF3333;} h2{text-decoration: underline; color: #FF9933;} .table{border: thin solid #000000; border-collapse: collapse; background-color: #000000;} .tabletitle{background-color: #6495ed; color : #FFFF33; font-weight : bold; font-size : normal} .tabletotal{background-color: #D0E3FA;font-weight : bold;} tr{background-color: #FFFFFF;padding: 2px;} td{padding: 2px;}"
                   "</style>"
                   "</head>"
                   "<body>"
                   "<table class=\"table\"><tr class=\"tabletitle\">";
    //Dump header
    int nbCols = ui.kTable->columnCount();
    for(int i = 0; i < nbCols; ++i) {
        QTableWidgetItem* item = ui.kTable->horizontalHeaderItem(i);
        if(item) html += "<td align=\"center\"><b>" % item->text() % "</b></td>";
    }
    html += "</tr>";

    //Dump values
    int nbLines = ui.kTable->rowCount();
    for(int j = 0; j < nbLines; ++j) {
        html += QString("<tr") % (m_sumRows[j+1] ? " class=\"tabletotal\"" : "") % '>';
        for(int i = 0; i < nbCols; ++i) {
            QTableWidgetItem* item = ui.kTable->item(j, i);
            if(item) {
                bool red = (item->data(DATA_VALUE).toDouble() < 0);
                html += QString("<td align=\"right\">") % (red ? "<font color=\"red\">" : "") % item->text() % (red ? "</font>" : "") % "</td>";
            } else {
                SKGColorButton* item = qobject_cast<SKGColorButton*>(ui.kTable->cellWidget(j, i));
                if(item) html += "<td>" % item->text() % "</td>";
            }
        }
        html += "</tr>";
    }
    html += "</table>";
    html += "</body></html>";
    ui.kTextEdit->setHtml(html);
    QApplication::restoreOverrideCursor();
}

void SKGTableWithGraph::redrawGraph()
{
    SKGTRACEIN(10, "SKGTableWithGraph::redrawGraph");
    m_mapItemGraphic.clear();
    if(!m_graphVisible) return;
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

    ui.graphicView->hide();
    ui.kTable->hide();

    //Recreate scene
    if(m_scene) {
        SKGTRACEIN(10, "SKGTableWithGraph::redrawGraph-remove scene");
        m_scene->clear();
        delete m_scene;
    }

    m_scene = new SKGGraphicsScene();
    {
        SKGTRACEIN(10, "SKGTableWithGraph::redrawGraph-create scene");
        m_scene->setBackgroundBrush(m_backgroundColor);

        //Get current selection
        int crow = ui.kTable->currentRow();
        int ccolumn = ui.kTable->currentColumn();

        //Get nb columns and rows
        int nbRows = ui.kTable->rowCount();
        int nbRealRows = nbRows;
        for(int y = 0; y < nbRows; ++y) {
            if(m_sumRows[y+1]) --nbRealRows;
        }

        int nbColumns = getNbColumns(false);
        int nbRealColumns = nbColumns - m_nbVirtualColumns;

        //Get graphic mode
        GraphType mode =   getGraphType();

        //Get in positive
        bool inPositive = false;
        if(mode == STACK || mode == STACKCOLUMNS || mode == HISTOGRAM || mode == POINT || mode == LINE || mode == STACKAREA) {
            m_allPositiveMenu->setEnabled(true);
            inPositive = (m_allPositiveMenu->isChecked());
        } else {
            m_allPositiveMenu->setEnabled(false);
            if(mode == CONCENTRICPIE || mode == PIE) inPositive = true;
        }

        //Get show origin
        bool showOrigin = true;
        if(mode == HISTOGRAM || mode == POINT || mode == LINE) {
            m_actShowZero->setEnabled(true);
            showOrigin = (m_actShowZero->isChecked());
        } else m_actShowZero->setEnabled(false);

        //Compute y limits
        double minLimit = (showOrigin ? 0 : 9999999);
        double maxLimit = (showOrigin ? 0 : -9999999);
        int nbLevel = 0;
        SKGTRACEL(3) << "mode=" << mode << endl;
        SKGTRACEL(3) << "nb rows        =" << nbRows << endl;
        SKGTRACEL(3) << "nb real rows   =" << nbRealRows << endl;
        SKGTRACEL(3) << "nb columns     =" << nbColumns << endl;
        SKGTRACEL(3) << "nb real columns=" << nbRealColumns << endl;
        SKGTRACEL(3) << "selected row   =" << crow << endl;
        SKGTRACEL(3) << "selected column=" << ccolumn << endl;
        if(mode == STACK) {
            //STACK
            for(int x = 0; x < nbRows; ++x) {
                if(!m_sumRows[x+1]) {
                    double sumPositive = 0;
                    double sumNegative = 0;
                    for(int y = 0; y < nbColumns; ++y) {
                        QTableWidgetItem* tableItem = ui.kTable->item(x, y);
                        if(tableItem) {
                            QVariant valQ = tableItem->data(DATA_VALUE);
                            if(valQ.type() == QVariant::Double) {
                                double val = valQ.toDouble();
                                if(inPositive || val >= 0) {
                                    sumPositive += qAbs(val);
                                } else {
                                    sumNegative += val;
                                }
                            }
                        }
                    }

                    minLimit = qMin(minLimit, sumNegative);
                    maxLimit = qMax(maxLimit, sumPositive);
                }
            }
        } else if(mode == STACKAREA || mode == STACKCOLUMNS || mode == PIE || mode == CONCENTRICPIE) {
            //STACKAREA
            for(int y = 0; y < nbColumns; ++y) {
                double sumPositive = 0;
                double sumNegative = 0;
                for(int x = 0; x < nbRows; ++x) {
                    if(!m_sumRows[x+1]) {
                        QTableWidgetItem* tableItem = ui.kTable->item(x, y);
                        if(tableItem) {
                            QVariant valQ = tableItem->data(DATA_VALUE);
                            if(valQ.type() == QVariant::Double) {
                                double val = valQ.toDouble();
                                if(inPositive || val >= 0) {
                                    sumPositive += qAbs(val);
                                } else {
                                    sumNegative += val;
                                }
                            }
                        }
                    }

                }
                minLimit = qMin(minLimit, sumNegative);
                maxLimit = qMax(maxLimit, sumPositive);
            }
        } else if(mode == HISTOGRAM || mode == POINT || mode == LINE) {
            //HISTOGRAM or POINTS or LINES
            for(int x = 0; x < nbRows; ++x) {
                if(!m_sumRows[x+1]) {
                    for(int y = 0; y < nbColumns; ++y) {
                        QTableWidgetItem* tableItem = ui.kTable->item(x, y);
                        if(tableItem) {
                            QVariant valQ = tableItem->data(DATA_VALUE);
                            if(valQ.type() == QVariant::Double) {
                                double val = valQ.toDouble();
                                if(inPositive) {
                                    maxLimit = qMax(maxLimit, qAbs(val));
                                    minLimit = qMin(minLimit, qAbs(val));
                                } else {
                                    minLimit = qMin(minLimit, val);
                                    maxLimit = qMax(maxLimit, val);
                                }
                            }
                        }
                    }
                }
            }
        } else if(mode == BUBBLE) {
            for(int x = 0; x < nbRows; ++x) {
                if(!m_sumRows[x+1]) {
                    for(int y = 0; y < nbColumns; ++y) {
                        QTableWidgetItem* tableItem = ui.kTable->item(x, y);
                        if(tableItem) {
                            QVariant valQ = tableItem->data(DATA_VALUE);
                            if(valQ.type() == QVariant::Double) {
                                double val = valQ.toDouble();
                                maxLimit = qMax(maxLimit, qAbs(val));
                            }
                        }
                    }
                }
            }
        }

        if(mode == CONCENTRICPIE) {
            for(int x = 0; x < nbRows; ++x) {
                SKGColorButton* btn = qobject_cast<SKGColorButton*>(ui.kTable->cellWidget(x, 0));
                if(btn) {
                    QString xname = btn->text();
                    QStringList vals = xname.split(OBJECTSEPARATOR);
                    int nbvals = vals.count();
                    nbLevel = qMax(nbLevel, nbvals - 1);
                }
            }
        }

        //Compute
        double yorigin = 0.0;
        double width = 10;
        double maxX = 0;
        double margin = 0;
        double marginLeft = 0;
        double xstep = 0.0;
        double radius = 0.0;
        int jstep = qMax(computeStepSize(nbColumns, 10), double(1.0));
        double ystep = computeStepSize(maxLimit - minLimit, 10);
        if(mode != BUBBLE && mode != PIE && mode != CONCENTRICPIE && ystep) {
            double newMinLimit = ystep * qRound(minLimit / ystep);
            minLimit = (minLimit - newMinLimit < -EPSILON ? newMinLimit - ystep : newMinLimit);
            double newMaxLimit = ystep * qRound(maxLimit / ystep);
            maxLimit = (maxLimit - newMaxLimit > EPSILON  ? newMaxLimit + ystep : newMaxLimit);
        }

        if(nbRealRows) {
            QRect vSize = ui.graphicView->rect();
            if(mode == STACK) {
                margin = (maxLimit - minLimit) * 0.3;

                if(!showOrigin) {
                    if(minLimit > 0) yorigin = ystep * (qRound(minLimit / ystep));
                    else if(maxLimit < 0) yorigin = ystep * (qRound(maxLimit / ystep));
                }

                width = (maxLimit - minLimit) / (nbRealRows + 1);
                width = width * ((double) vSize.width()) / ((double) vSize.height());

                xstep = width * (nbRealRows + 1);
                marginLeft = width;

                maxX = width * nbRealRows + margin;
            } else if(mode == HISTOGRAM || mode == POINT || mode == LINE || mode == STACKAREA || mode == STACKCOLUMNS) {
                margin = (maxLimit - minLimit) * 0.3;

                if(!showOrigin) {
                    if(minLimit > 0) yorigin = ystep * (qRound(minLimit / ystep));
                    else if(maxLimit < 0) yorigin = ystep * (qRound(maxLimit / ystep));
                }

                width = (maxLimit - minLimit) / ((nbRealRows + 1) * (nbColumns - 1));
                width = width * ((double) vSize.width()) / ((double) vSize.height());

                xstep = width * (nbRealRows + 1);
                marginLeft = qMin(jstep * xstep, width * (nbColumns - 1));

                maxX = xstep * (nbColumns - 1) ;
                if(mode == POINT || mode == LINE || mode == STACKAREA) maxX -= xstep;
                radius = qMax(margin / 100.0, qMin(width, qMin(ystep / 4, jstep * xstep / 4)));
            } else if(mode == BUBBLE) {
                margin = ystep * nbRealRows * 0.3;

                width = ystep * nbRealRows / (nbRealRows * (nbColumns - 1));
                width = width * ((double) vSize.width()) / ((double) vSize.height());

                xstep = width * (nbRealRows + 1);
                marginLeft = jstep * xstep;

                maxX = width * (nbColumns - 1) * (nbRealRows + 1) ;
            }
        }

        SKGTRACEL(10) << "minLimit=" << minLimit << endl;
        SKGTRACEL(10) << "maxLimit=" << maxLimit << endl;
        SKGTRACEL(10) << "ystep=" << ystep << endl;
        SKGTRACEL(10) << "yorigin=" << yorigin << endl;
        SKGTRACEL(10) << "width=" << width << endl;
        SKGTRACEL(10) << "maxX=" << maxX << endl;
        SKGTRACEL(10) << "margin=" << margin << endl;
        SKGTRACEL(10) << "xstep=" << xstep << endl;
        SKGTRACEL(10) << "marginLeft=" << marginLeft << endl;

        //Initialise pens
        QPen axisPen = QPen(Qt::SolidLine);
        axisPen.setColor(m_axisColor);
        axisPen.setWidthF(margin / 30.0);
        axisPen.setJoinStyle(Qt::RoundJoin);

        QPen gridPen = QPen(Qt::SolidLine);
        gridPen.setColor(m_gridColor);
        gridPen.setWidthF(margin / 100.0);

        QPen dotPen = QPen(Qt::SolidLine);  //WARNING: Qt::DotLine is very bad for performances
        dotPen.setWidthF(margin / 50.0);

        //Set rect
        int nbColInMode2 = (nbColumns > 2 ? sqrt(nbColumns - 1) + .5 : 1);
        m_scene->setItemIndexMethod(QGraphicsScene::NoIndex);
        SKGTRACEL(10) << "Items:" << nbRealRows << "x" << (nbColumns - 1) << "=" << nbRealRows*(nbColumns - 1) << endl;
        SKGTRACEL(10) << "nbColInMode2=" << nbColInMode2 << endl;

        //Redraw scene
        double x0 = 0;
        double x1 = 0;
        double ymin = (showOrigin || mode == STACK || mode == STACKCOLUMNS ? 0 : 9999999);
        double ymax = (showOrigin || mode == STACK || mode == STACKCOLUMNS ? 0 : -9999999);
        int x = 0;
        int nbRowsDrawed = 0;
        QMap<int, double> previousStackedPlus;
        QMap<int, double> previousStackedMoins;
        for(int xx = 0; xx < nbRows; ++xx) {
            SKGColorButton* colorButton = qobject_cast<SKGColorButton*>(ui.kTable->cellWidget(xx, 0));
            if(colorButton) {
                QString xname = colorButton->text();
                QColor initialColor = colorButton->color();
                double yPlus = 0;
                double yMoins = 0;
                int jprevious = -1;

                ++nbRowsDrawed;
                for(int j = 1; j < nbColumns; ++j) {
                    //Get column name
                    QString yname = ui.kTable->horizontalHeaderItem(j)->text();

                    //Get cell value
                    QTableWidgetItem* tableItem = ui.kTable->item(xx, j);
                    if(tableItem) {
                        QVariant valQ = tableItem->data(DATA_VALUE);
                        if(valQ.type() == QVariant::Double) {
                            double val = valQ.toDouble();
                            QString valstring = tableItem->text();

                            if(inPositive)  val = qAbs(val);

                            int color_h;
                            int color_s;
                            int color_v;
                            initialColor.getHsv(&color_h, &color_s, &color_v);
                            color_s = color_s - color_s * 155 / 255 * (nbColumns - 1 - j) / nbColumns;
                            color_v = color_v - color_v * 155 / 255 * (nbColumns - 1 - j) / nbColumns;
                            if(j >= nbRealColumns) {
                                //Yellow
                                color_h = 60;
                                color_s = 255;
                                color_v = 255;
                            }

                            QColor color;
                            if(x == crow && j == ccolumn) color = QApplication::palette().color(QPalette::Highlight);
                            else color = QColor::fromHsv(color_h, color_s, color_v);

                            color.setAlpha(ALPHA);
                            QBrush brush(color);

                            QGraphicsItem* graphItem = NULL;
                            if(mode == STACK) {
                                //STACK
                                if(val >= 0) {
                                    graphItem = m_scene->addRect(width * x, -yPlus - qAbs(val), width, qAbs(val), QPen(m_outlineColor), brush);
                                    yPlus += qAbs(val);
                                    if(yPlus > ymax) ymax = yPlus;
                                } else {
                                    graphItem = m_scene->addRect(width * x, -yMoins, width, -val, QPen(m_outlineColor), brush);
                                    yMoins += val;
                                    if(yMoins < ymin) ymin = yMoins;
                                }
                                if(graphItem) graphItem->setZValue(5 + nbColumns - j);
                            } if(mode == STACKAREA) {
                                //STACKAREA
                                //Empty cells are ignored
                                if(j == 1) x0 = width * (j - 1) * (nbRealRows + 1);
                                if(j == nbColumns - m_nbVirtualColumns - 1) x1 = width * (j - 1) * (nbRealRows + 1);

                                if(!valstring.isEmpty()) {
                                    if(jprevious == -1) jprevious = j;
                                    QTableWidgetItem* tableItem2 = ui.kTable->item(xx, jprevious);
                                    if(tableItem2) {
                                        QString val2string = tableItem->text();
                                        if(!val2string.isEmpty()) {
                                            double val2 = tableItem2->data(DATA_VALUE).toDouble();
                                            if(j == jprevious) val2 = 0;
                                            if(inPositive) val2 = qAbs(val2);

                                            if(val > EPSILON || (qAbs(val) <= EPSILON && val2 >= EPSILON)) {
                                                double xp = width * (jprevious - 1) * (nbRealRows + 1) - (jprevious == j ? width / 20.0 : 0);
                                                double xn = width * (j - 1) * (nbRealRows + 1);
                                                double yp = (previousStackedPlus.contains(jprevious) ? previousStackedPlus[jprevious] : -val2);
                                                double yn = previousStackedPlus[j] - val;

                                                QPolygonF polygon;
                                                polygon << QPointF(xp, yp + val2) << QPointF(xp, yp)
                                                        << QPointF(xn, yn) << QPointF(xn, previousStackedPlus[j]);
                                                graphItem = m_scene->addPolygon(polygon, QPen(m_outlineColor), brush);
                                                previousStackedPlus[j] = yn;
                                                if(-yn > ymax) ymax = -yn;
                                            } else {
                                                double xp = width * (jprevious - 1) * (nbRealRows + 1) - (jprevious == j ? width / 20.0 : 0);
                                                double xn = width * (j - 1) * (nbRealRows + 1);
                                                double yp = (previousStackedMoins.contains(jprevious) ? previousStackedMoins[jprevious] : -val2);
                                                double yn = previousStackedMoins[j] - val;

                                                QPolygonF polygon;
                                                polygon << QPointF(xp, yp + val2) << QPointF(xp, yp)
                                                        << QPointF(xn, yn) << QPointF(xn, previousStackedMoins[j]);
                                                graphItem = m_scene->addPolygon(polygon, QPen(m_outlineColor), brush);
                                                previousStackedMoins[j] = yn;
                                                if(-yn < ymin) ymin = -yn;
                                            }
                                            jprevious = j;
                                        }
                                    }
                                }
                            } if(mode == STACKCOLUMNS) {
                                //STACKCOLUMNS
                                if(val >= 0) {
                                    graphItem = m_scene->addRect(width * (j - 1) * (nbRealRows + 1), -previousStackedPlus[j] - qAbs(val), width * (nbRealRows + 1), qAbs(val), QPen(m_outlineColor), brush);
                                    previousStackedPlus[j] += qAbs(val);
                                    if(previousStackedPlus[j] > ymax) ymax = previousStackedPlus[j];
                                } else {
                                    graphItem = m_scene->addRect(width * (j - 1) * (nbRealRows + 1), -previousStackedMoins[j], width * (nbRealRows + 1), -val, QPen(m_outlineColor), brush);
                                    previousStackedMoins[j] += val;
                                    if(previousStackedMoins[j] < ymin) ymin = previousStackedMoins[j];
                                }
                                if(graphItem) graphItem->setZValue(5 + nbRows - x);
                            } else if(mode == HISTOGRAM) {
                                //HISTOGRAM
                                if(j == 1) x0 = width * ((j - 1) * (nbRealRows + 1) + x) + width;
                                if(j == nbColumns - m_nbVirtualColumns - 1) x1 = width * ((j - 1) * (nbRealRows + 1) + x) + width;
                                graphItem = m_scene->addRect(width * ((j - 1) * (nbRealRows + 1) + x) + width / 2.0, (val < 0 ? -yorigin : -val), width, (val < 0 ? yorigin - val : -yorigin + val), QPen(m_outlineColor), brush);
                                if(val > ymax) ymax = val;
                                if(val < ymin) ymin = val;
                            } else if(mode == POINT) {
                                //POINTS
                                double xmin = width * (j - 1) * (nbRealRows + 1) - radius;
                                if(j == 1) x0 = xmin + radius;
                                if(j == nbColumns - m_nbVirtualColumns - 1) x1 = xmin + radius;
                                graphItem = drawPoint(xmin, -val, radius, x, brush);
                                if(val > ymax) ymax = val;
                                if(val < ymin) ymin = val;
                            } else if(mode == LINE) {
                                //LINES
                                //Empty cells are ignored
                                if(j == 1) x0 = width * (j - 1) * (nbRealRows + 1);
                                if(j == nbColumns - m_nbVirtualColumns - 1) x1 = width * (j - 1) * (nbRealRows + 1);

                                if(!valstring.isEmpty()) {
                                    if(jprevious == -1) jprevious = j;

                                    //Create pen
                                    QPen pen = QPen(color);
                                    pen.setWidthF(radius);
                                    pen.setCapStyle(Qt::RoundCap);
                                    pen.setJoinStyle(Qt::RoundJoin);

                                    QTableWidgetItem* tableItem2 = ui.kTable->item(xx, jprevious);
                                    if(tableItem2) {
                                        QString val2string = tableItem->text();
                                        if(!val2string.isEmpty()) {
                                            double val2 = tableItem2->data(DATA_VALUE).toDouble();
                                            if(inPositive) val2 = qAbs(val2);

                                            graphItem = m_scene->addLine(width * (jprevious - 1) * (nbRealRows + 1) - (jprevious == j ? width / 20.0 : 0), -val2, width * (j - 1) * (nbRealRows + 1), -val, pen);
                                            if(val > ymax) ymax = val;
                                            if(val < ymin) ymin = val;
                                            jprevious = j;
                                        }
                                    }
                                }
                            } else if(mode == BUBBLE) {
                                //BUBBLE
                                ymin = 0;
                                ymax = ystep * nbRealRows;

                                radius =  sqrt(qAbs(val) / maxLimit) * qMin(ystep , jstep * xstep);
                                double xmin = width * (j - 1) * (nbRealRows + 1) - radius;

                                graphItem = m_scene->addEllipse(xmin, -ystep * x - radius, 2 * radius, 2 * radius, QPen(m_outlineColor), brush);
                                if(graphItem) graphItem->setZValue(5 + 5 *(maxLimit - qAbs(val)) / maxLimit);
                            }  else if(mode == PIE || mode == CONCENTRICPIE) {
                                //PIE
                                val = qAbs(val);

                                //Compute absolute sum of the column
                                double previousSum = 0;
                                for(int x2 = 0; x2 < nbRows; ++x2) {
                                    if(!m_sumRows[x2+1]) {
                                        QTableWidgetItem* tableItem = ui.kTable->item(x2, j);
                                        if(tableItem) {
                                            double absVal = qAbs(tableItem->data(DATA_VALUE).toDouble());
                                            if(x2 < xx) previousSum += absVal;
                                        }
                                    }
                                }

                                if(maxLimit) {
                                    int nbvals = xname.split(OBJECTSEPARATOR).count();
                                    double step = 0;
                                    double p = 0;
                                    if(mode == CONCENTRICPIE) {
                                        step = 100.0 / double(nbLevel + 1);
                                        p = step * (nbLevel + 1 - nbvals);
                                    }

                                    QPainterPath path;
                                    path.moveTo(BOX_SIZE*((j - 1) % nbColInMode2) + BOX_SIZE / 2.0, BOX_SIZE * floor((j - 1) / nbColInMode2) + BOX_SIZE / 2.0);
                                    path.arcTo(BOX_SIZE*((j - 1) % nbColInMode2) + BOX_MARGIN + p / 2.0, BOX_SIZE * floor((j - 1) / nbColInMode2) + BOX_MARGIN + p / 2.0, BOX_SIZE - 2 * BOX_MARGIN - p, BOX_SIZE - 2 * BOX_MARGIN - p, 360.0 * previousSum / maxLimit, 360.0 * val / maxLimit);
                                    path.closeSubpath();
                                    if(mode == CONCENTRICPIE && nbvals <= nbLevel + 1 && nbvals > 1) {
                                        p = step * (nbLevel + 1 - nbvals + 1);
                                        QPainterPath path2;
                                        path2.addEllipse(BOX_SIZE*((j - 1) % nbColInMode2) + BOX_MARGIN + p / 2.0,
                                                         BOX_SIZE * floor((j - 1) / nbColInMode2) + BOX_MARGIN + p / 2.0,
                                                         BOX_SIZE - 2 * BOX_MARGIN - p,
                                                         BOX_SIZE - 2 * BOX_MARGIN - p);
                                        path -= path2;
                                    }
                                    graphItem = m_scene->addPath(path, QPen(m_outlineColor), brush);
                                }
                            }

                            if(graphItem) {
                                if(graphItem->zValue() == 0) graphItem->setZValue(5);
                                graphItem->setToolTip(tableItem->toolTip());
                                bool isSelect = (isSelectable() && tableItem->flags() & Qt::ItemIsEnabled);
                                graphItem->setFlag(QGraphicsItem::ItemIsSelectable, isSelect);
                                if(isSelect) graphItem->setCursor(Qt::PointingHandCursor);
                                graphItem->setData(1, xx);
                                graphItem->setData(2, j);
                                graphItem->setData(DATA_COLOR_H, color_h);
                                graphItem->setData(DATA_COLOR_S, color_s);
                                graphItem->setData(DATA_COLOR_V, color_v);
                                graphItem->setData(DATA_Z_VALUE, graphItem->zValue());

                                /*TODO
                                QGraphicsDropShadowEffect* shadow=new QGraphicsDropShadowEffect(this);
                                shadow->setBlurRadius(10);
                                                graphItem->setGraphicsEffect(shadow);
                                                */

                                m_mapItemGraphic[tableItem] = graphItem;
                            }
                        }
                    } else {
                        SKGTRACE << "WARNING: cell " << x << "," << j << " null" << endl;
                    }
                }

                ++x;
            }
        }

        //Draw axis
        double scaleText = 0.0;
        int nbRowInMode2 = ((int)((nbColumns - 1) / nbColInMode2));
        if(nbRealRows) {
            if(mode == PIE || mode == CONCENTRICPIE) {
                //PIE
                if(nbRowInMode2 * nbColInMode2 < nbColumns - 1) ++nbRowInMode2;
                for(int i = 0; i <= nbColInMode2; ++i) {
                    QGraphicsLineItem* item = m_scene->addLine(BOX_SIZE * i, 0, BOX_SIZE * i, BOX_SIZE * nbRowInMode2);
                    item->setPen(axisPen);
                    item->setFlag(QGraphicsItem::ItemIsSelectable, false);
                }
                for(int i = 0; i <= nbRowInMode2; ++i) {
                    QGraphicsLineItem* item = m_scene->addLine(0, BOX_SIZE * i, BOX_SIZE * nbColInMode2, BOX_SIZE * i);
                    item->setPen(axisPen);
                    item->setFlag(QGraphicsItem::ItemIsSelectable, false);
                }

                for(int j = 1; j < nbColumns; ++j) {
                    //Get column name
                    QString yname = ui.kTable->horizontalHeaderItem(j)->text();

                    QGraphicsTextItem* textItem = m_scene->addText(yname);
                    textItem->setDefaultTextColor(m_textColor);
                    textItem->setPos(BOX_SIZE*((j - 1) % nbColInMode2) + 2, BOX_SIZE * floor((j - 1) / nbColInMode2) + 2);
                    textItem->scale(.5, .5);
                    textItem->setFlag(QGraphicsItem::ItemIsSelectable, false);
                    textItem->setZValue(20);
                }
            } else {
                //STACK & HISTOGRAMM
                QGraphicsLineItem* item;

                //Compute scale text
                KLocale* locale = KGlobal::locale();
                if(mode != BUBBLE) {
                    QGraphicsTextItem* t = m_scene->addText(locale->formatMoney(-qMax(qAbs(ymax), qAbs(ymin)) , m_primaryUnit.Symbol, m_primaryUnit.NbDecimal));
                    QRectF rect = t->boundingRect();
                    scaleText = 0.8 * qMin(marginLeft / rect.width(), qMin(marginLeft / rect.height(), ystep / rect.height()));
                    m_scene->removeItem(t);
                } else maxLimit = ystep * nbRealRows;

                if(ystep > 0) {
                    //Draw
                    int i = 1;
                    for(double y = yorigin + ystep ; y <= maxLimit ; y = y + ystep) {
                        item = m_scene->addLine(0, -y, maxX, -y);
                        item->setPen(gridPen);
                        item->setFlag(QGraphicsItem::ItemIsSelectable, false);
                        item->setZValue(1);
                        QGraphicsTextItem* textItem;
                        if(mode == BUBBLE) {
                            SKGColorButton* cel = qobject_cast<SKGColorButton*>(ui.kTable->cellWidget(i, 0));
                            ++i;
                            if(!cel) {
                                cel = qobject_cast<SKGColorButton*>(ui.kTable->cellWidget(i, 0));
                                ++i;
                            }
                            textItem = m_scene->addText(cel ? cel->text() : "");
                        } else {
                            textItem = m_scene->addText(locale->formatMoney(y, m_primaryUnit.Symbol, m_primaryUnit.NbDecimal));
                        }

                        QRectF rect = textItem->boundingRect();
                        if(scaleText == 0)  scaleText = 0.8 * qMin(marginLeft / rect.width(), qMin(marginLeft / rect.height(), ystep / rect.height()));
                        textItem->setDefaultTextColor(m_textColor);
                        textItem->scale(scaleText, scaleText);
                        textItem->setFlag(QGraphicsItem::ItemIsSelectable, false);
                        textItem->setZValue(20);
                        double delta = scaleText * rect.height() / 2.0;

                        textItem->setPos(-marginLeft, -y - delta);
                    }

                    i = 0;
                    for(double y = yorigin ; (y == yorigin) || y >= minLimit; y = y - ystep) {
                        item = m_scene->addLine(0, -y, maxX, -y);
                        item->setPen(gridPen);
                        item->setFlag(QGraphicsItem::ItemIsSelectable, false);
                        item->setZValue(1);

                        QGraphicsTextItem* textItem;
                        if(mode == BUBBLE) {
                            SKGColorButton* cel = qobject_cast<SKGColorButton*>(ui.kTable->cellWidget(i, 0));
                            textItem = m_scene->addText(cel ? cel->text() : "");
                        } else {
                            textItem = m_scene->addText(locale->formatMoney(y, m_primaryUnit.Symbol, m_primaryUnit.NbDecimal));
                        }

                        QRectF rect = textItem->boundingRect();
                        if(scaleText == 0)  scaleText = 0.8 * qMin(marginLeft / rect.width(), qMin(marginLeft / rect.height(), ystep / rect.height()));
                        textItem->setDefaultTextColor(m_textColor);
                        textItem->scale(scaleText, scaleText);
                        textItem->setFlag(QGraphicsItem::ItemIsSelectable, false);
                        textItem->setZValue(20);
                        double delta = scaleText * rect.height() / 2.0;

                        textItem->setPos(-marginLeft, -y - delta);
                    }
                }

                if(mode == HISTOGRAM || mode == POINT || mode == LINE || mode == BUBBLE || mode == STACKAREA || mode == STACKCOLUMNS) {
                    //Line y
                    for(int j = 1; j < nbColumns; j += jstep) {
                        QString yname = ui.kTable->horizontalHeaderItem(j)->text();
                        double x = xstep + (j - 2) * xstep;

                        QGraphicsTextItem* textItem = m_scene->addText(yname);

                        QRectF rect = textItem->boundingRect();
                        if(scaleText == 0)  scaleText = 0.8 * qMin(marginLeft / rect.width(), qMin(marginLeft / rect.height(), ystep / rect.height()));
                        textItem->setDefaultTextColor(m_textColor);
                        textItem->scale(scaleText, scaleText);
                        textItem->setFlag(QGraphicsItem::ItemIsSelectable, false);
                        textItem->setZValue(20);
                        double delta = scaleText * rect.height() / 2.0;
                        textItem->rotate(90);
                        textItem->moveBy(rect.height() *scaleText, 0);

                        textItem->setPos(x + delta, -yorigin);

                        QGraphicsLineItem* item = m_scene->addLine(x, qMax(-yorigin, -minLimit), x, qMin(-yorigin, -maxLimit));
                        item->setPen(gridPen);
                        item->setFlag(QGraphicsItem::ItemIsSelectable, false);
                    }
                }

                //Axis x
                if(yorigin == 0.0) {
                    item = m_scene->addLine(0, -yorigin, maxX, -yorigin, axisPen);
                    item->setFlag(QGraphicsItem::ItemIsSelectable, false);
                    item->setZValue(2);
                }

                //Rect
                {
                    QGraphicsRectItem* item = m_scene->addRect(0, qMax(-yorigin, -minLimit), maxX, qMin(-yorigin, -maxLimit) - (qMax(-yorigin, -minLimit)), axisPen);
                    item->setFlag(QGraphicsItem::ItemIsSelectable, false);
                    item->setZValue(2);
                }
            }
        }

        //Draw Average
        bool lineCondition = m_limitVisible && (mode == HISTOGRAM || mode == POINT || mode == LINE) && scaleText && nbRowsDrawed == 1;
        int averageCol = getAverageColumnIndex();
        if(lineCondition && averageCol != -1) {
            QTableWidgetItem* tableItem = ui.kTable->item(0, averageCol);
            if(tableItem) {
                double y = tableItem->data(DATA_VALUE).toDouble();
                if(inPositive) y = qAbs(y);
                KLocale* locale = KGlobal::locale();
                QGraphicsLineItem* item = m_scene->addLine(0, -y, maxX, -y);
                dotPen.setColor(m_averageColor);
                item->setPen(dotPen);
                item->setFlag(QGraphicsItem::ItemIsSelectable, false);
                item->setZValue(1);
                item->setToolTip(tableItem->toolTip());

                QGraphicsTextItem* textItem = m_scene->addText(locale->formatMoney(y, m_primaryUnit.Symbol, m_primaryUnit.NbDecimal));
                QRectF rect = textItem->boundingRect();
                double delta = scaleText * rect.height() / 2.0;
                textItem->setPos(maxX, -y - delta);
                textItem->setDefaultTextColor(m_averageColor);
                textItem->scale(scaleText, scaleText);
                textItem->setFlag(QGraphicsItem::ItemIsSelectable, false);
                textItem->setZValue(20);
                textItem->setToolTip(tableItem->toolTip());
            }
        }

        //Draw Min & Max limites
        int minCol = getMinColumnIndex();
        if(lineCondition && minCol != -1) {
            //Min
            {
                QTableWidgetItem* tableItem = ui.kTable->item(0, minCol);
                if(tableItem) {
                    double y = tableItem->data(DATA_VALUE).toDouble();
                    if(inPositive) y = qAbs(y);
                    KLocale* locale = KGlobal::locale();
                    QGraphicsLineItem* item = m_scene->addLine(0, -y, maxX, -y);
                    dotPen.setColor(m_minColor);
                    item->setPen(dotPen);
                    item->setFlag(QGraphicsItem::ItemIsSelectable, false);
                    item->setZValue(1);
                    item->setToolTip(tableItem->toolTip());

                    QGraphicsTextItem* textItem = m_scene->addText(locale->formatMoney(y, m_primaryUnit.Symbol, m_primaryUnit.NbDecimal));
                    QRectF rect = textItem->boundingRect();
                    double delta = scaleText * rect.height() / 2.0;
                    textItem->setPos(maxX, -y - delta);
                    textItem->setDefaultTextColor(m_minColor);
                    textItem->scale(scaleText, scaleText);
                    textItem->setFlag(QGraphicsItem::ItemIsSelectable, false);
                    textItem->setZValue(20);
                    textItem->setToolTip(tableItem->toolTip());
                }
            }

            //Max
            {
                QTableWidgetItem* tableItem = ui.kTable->item(0, minCol + 1);
                if(tableItem) {
                    double y = tableItem->data(DATA_VALUE).toDouble();
                    if(inPositive) y = qAbs(y);
                    KLocale* locale = KGlobal::locale();
                    QGraphicsLineItem* item = m_scene->addLine(0, -y, maxX, -y);
                    dotPen.setColor(m_maxColor);
                    item->setPen(dotPen);
                    item->setFlag(QGraphicsItem::ItemIsSelectable, false);
                    item->setZValue(1);
                    item->setToolTip(tableItem->toolTip());

                    QGraphicsTextItem* textItem = m_scene->addText(locale->formatMoney(y, m_primaryUnit.Symbol, m_primaryUnit.NbDecimal));
                    QRectF rect = textItem->boundingRect();
                    double delta = scaleText * rect.height() / 2.0;
                    textItem->setPos(maxX, -y - delta);
                    textItem->setDefaultTextColor(m_maxColor);
                    textItem->scale(scaleText, scaleText);
                    textItem->setFlag(QGraphicsItem::ItemIsSelectable, false);
                    textItem->setZValue(20);
                    textItem->setToolTip(tableItem->toolTip());
                }
            }
        }

        //Draw Linear Regression
        bool lineCondition2 = m_linearRegressionVisible && (mode == HISTOGRAM || mode == POINT || mode == LINE) && scaleText && nbRowsDrawed == 1 ;
        if(lineCondition2 && m_indexLinearRegression != -1) {
            QTableWidgetItem* tableItem = ui.kTable->item(0, m_indexLinearRegression);
            if(tableItem) {
                QString f = tableItem->text();

                QScriptEngine myEngine;
                QString f0 = f;
                f0 = f0.remove("y=");
                f0 = f0.replace('x', '1');
                f0 = f0.replace(',', '.'); // Replace comma by a point in case of typo
                if(inPositive) f0 = "Math.abs(" % f0 % ')';
                double y0 = myEngine.evaluate(f0).toNumber();

                QString f1 = f;
                f1 = f1.remove("y=");
                f1 = f1.replace('x', SKGServices::intToString(nbRealColumns - 1));
                f1 = f1.replace(',', '.'); // Replace comma by a point in case of typo
                if(inPositive) f1 = "Math.abs(" % f1 % ')';
                double y1 = myEngine.evaluate(f1).toNumber();

                QGraphicsLineItem* item = m_scene->addLine(x0, -y0, x1, -y1);
                dotPen.setColor(m_tendencyColor);
                item->setPen(dotPen);
                item->setFlag(QGraphicsItem::ItemIsSelectable, false);
                item->setZValue(1);
                item->setToolTip(f);
            }
        }

        //Draw legend
        if(m_legendVisible) {
            QPointF legendPosition;
            double maxY;
            if(mode == PIE || mode == CONCENTRICPIE) {
                int nbRowInMode2 = ((int)((nbColumns - 1) / nbColInMode2));
                if(nbRowInMode2 * nbColInMode2 < nbColumns - 1) ++nbRowInMode2;

                scaleText = 0.2;
                margin = BOX_SIZE / 10;
                legendPosition = QPointF(BOX_SIZE * nbColInMode2 + margin, 0);
                maxY = BOX_SIZE * nbRowInMode2;
            } else {
                legendPosition = QPointF(maxX + margin, qMin(-yorigin, -maxLimit) - margin / 6);
                maxY = qMax(-yorigin, -minLimit);
            }
            addLegend(legendPosition , margin / 4, scaleText, maxY);
        }
    }
    {
        SKGTRACEIN(10, "SKGTableWithGraph::redrawGraph-add scene to view");
        m_scene->setSceneRect(QRectF());
        ui.graphicView->setScene(m_scene);
        ui.graphicView->show();
        if(m_tableVisible) ui.kTable->show();
        ui.graphicView->initializeZoom();

        //Add selection event on scene
        connect(m_scene, SIGNAL(selectionChanged()), this, SLOT(onSelectionChangedInGraph()), Qt::QueuedConnection);
        connect(m_scene, SIGNAL(doubleClicked()), this, SLOT(onDoubleClickGraph()), Qt::QueuedConnection);
    }
    QApplication::restoreOverrideCursor();
}

int SKGTableWithGraph::getNbColumns(bool iWithComputed) const
{
    int nbColumns = ui.kTable->columnCount();
    if(!iWithComputed) {
        if(m_indexMin != -1) nbColumns -= 2;
        if(m_indexAverage != -1) --nbColumns;
        if(m_indexSum != -1) --nbColumns;
        if(m_indexLinearRegression != -1) --nbColumns;
    }
    return nbColumns;
}

int SKGTableWithGraph::getAverageColumnIndex() const
{
    return m_indexAverage;
}

int SKGTableWithGraph::getMinColumnIndex() const
{
    return m_indexMin;
}

double SKGTableWithGraph::computeStepSize(double iRange, double iTargetSteps)
{
    // Calculate an initial guess at step size
    double tempStep = iRange / iTargetSteps;
    // Get the magnitude of the step size
    double mag = floor(log10(tempStep));
    double magPow = pow((double) 10.0, mag);
    // Calculate most significant digit of the new step size
    double magMsd = ((int)(tempStep / magPow + .5));
    // promote the MSD to either 1, 2, or 5
    if(magMsd > 5.0) magMsd = 10.0;
    else if(magMsd > 2.0) magMsd = 5.0;
    else if(magMsd > 1.0) magMsd = 2.0;
    return magMsd * magPow;
}

void SKGTableWithGraph::addLegend(const QPointF& iPosition, double iSize, double iScaleText, double iMaxY)
{
    SKGTRACEIN(10, "SKGTableWithGraph::addLegend");

    if(m_scene) {
        GraphType mode =  getGraphType();
        int nbRows = ui.kTable->rowCount();
        int nbRealRows = nbRows;
        for(int y = 0; y < nbRows; ++y) {
            if(m_sumRows[y+1]) --nbRealRows;
        }
        int nbColumns = getNbColumns(false);
        if(nbColumns > 1) {
            double margin = 1.2;
            double currentYPos = iPosition.y();
            double currentXPos = iPosition.x();
            double currentXMaxSize = 0;
            for(int i = 0; i < nbRows; ++i) {
                SKGColorButton* btn = qobject_cast<SKGColorButton*>(ui.kTable->cellWidget(i, 0));
                if(btn) {
                    //Get title
                    QString title = btn->text();

                    //Build brush
                    QColor color1;
                    QTableWidgetItem* it = ui.kTable->item(i, 1);
                    if(it) {
                        QGraphicsItem * graphicItem = m_mapItemGraphic[it];
                        if(graphicItem) {
                            //Draw box
                            color1 = QColor::fromHsv(graphicItem->data(DATA_COLOR_H).toInt(),
                                                     graphicItem->data(DATA_COLOR_S).toInt(),
                                                     graphicItem->data(DATA_COLOR_V).toInt());
                            color1.setAlpha(ALPHA);
                        }
                    }
                    QColor color2;
                    it = ui.kTable->item(i, nbColumns - 1 - m_nbVirtualColumns);
                    if(it) {
                        QGraphicsItem * graphicItem = m_mapItemGraphic[it];
                        if(graphicItem) {
                            //Draw box
                            color2 = QColor::fromHsv(graphicItem->data(DATA_COLOR_H).toInt(),
                                                     graphicItem->data(DATA_COLOR_S).toInt(),
                                                     graphicItem->data(DATA_COLOR_V).toInt());
                            color2.setAlpha(ALPHA);
                        }
                    }

                    QLinearGradient grandient(currentXPos, currentYPos, currentXPos + 2.0 * iSize, currentYPos);
                    grandient.setColorAt(0, color1);
                    grandient.setColorAt(1, color2);

                    //Draw legend item
                    QGraphicsItem* item = NULL;
                    if(mode == POINT) {
                        item = drawPoint(currentXPos, currentYPos + iSize / 2.0, iSize / 2.0, i, QBrush(grandient));
                    } else if(mode == BUBBLE) {
                        item = m_scene->addEllipse(currentXPos, currentYPos , iSize, iSize, QPen(m_outlineColor), QBrush(grandient));
                    } else if(mode == PIE || mode == CONCENTRICPIE) {
                        QPainterPath path;
                        path.moveTo(currentXPos + iSize / 2.0, currentYPos + iSize / 2.0);
                        path.arcTo(currentXPos, currentYPos , iSize, iSize, 45, 270);
                        path.closeSubpath();
                        if(mode == CONCENTRICPIE) {
                            QPainterPath path2;
                            double p = iSize / 3.0;
                            path2.addEllipse(currentXPos + p, currentYPos + p, iSize -  2.0 * p, iSize -  2.0 * p);
                            path -= path2;
                        }
                        item = m_scene->addPath(path, QPen(m_outlineColor), QBrush(grandient));
                    } else {
                        item = m_scene->addRect(currentXPos, currentYPos , iSize, iSize, QPen(m_outlineColor), QBrush(grandient));
                    }
                    if(item) {
                        item->setFlag(QGraphicsItem::ItemIsSelectable, false);
                        item->setToolTip(title);
                    }

                    //Draw text
                    QGraphicsTextItem* textItem = m_scene->addText(title);
                    textItem->setDefaultTextColor(m_textColor);
                    textItem->scale(iScaleText, iScaleText);
                    textItem->setPos(currentXPos + margin * iSize, currentYPos + iSize / 2.0 - textItem->boundingRect().height()*iScaleText / 2.0);
                    textItem->setFlag(QGraphicsItem::ItemIsSelectable, false);

                    //Compute next position
                    currentYPos += margin * iSize;
                    QRectF rect = textItem->boundingRect();
                    currentXMaxSize = qMax(currentXMaxSize, (double)(iScaleText * rect.width()));
                    if(currentYPos > iMaxY) {
                        currentYPos = iPosition.y();
                        currentXPos += currentXMaxSize + 2 * margin * iSize;
                        currentXMaxSize = 0;
                    }
                }
            }
        }
    }
}

void SKGTableWithGraph::addArrow(const QPointF& iPeak, double iSize, double iArrowAngle, double iDegree)
{
    if(m_scene) {
        QPolygonF pol;
        double radian = 3.14 * iArrowAngle / 360.0;
        pol << QPointF(0, 0) << QPointF(iSize * cos(radian), iSize * sin(radian)) << QPointF(iSize * cos(radian), -iSize * sin(radian)) << QPointF(0, 0);
        QGraphicsPolygonItem * item = m_scene->addPolygon(pol, QPen(m_axisColor, iSize / 20.0), QBrush(m_axisColor));
        item->rotate(iDegree);
        item->moveBy(iPeak.x(), iPeak.y());
        item->setFlag(QGraphicsItem::ItemIsSelectable, false);
        item->setZValue(2);
    }
}

QGraphicsItem* SKGTableWithGraph::drawPoint(qreal iX, qreal iY, qreal iRadius, int iMode, const QBrush& iBrush)
{
    QGraphicsItem* graphItem = NULL;
    if(m_scene) {
        switch(iMode % 8) {
        case 0: {
            graphItem = m_scene->addEllipse(iX, iY - iRadius, 2 * iRadius, 2 * iRadius, QPen(m_outlineColor), iBrush);
            break;
        }
        case 1: {
            graphItem = m_scene->addRect(iX, iY - iRadius, 2 * iRadius, 2 * iRadius, QPen(m_outlineColor), iBrush);
            break;
        }
        case 2: {
            QPolygonF polygon;
            polygon << QPointF(iX, iY - iRadius) << QPointF(iX + 2 * iRadius, iY + iRadius)
                    << QPointF(iX + 2 * iRadius, iY - iRadius) << QPointF(iX, iY + iRadius);
            graphItem = m_scene->addPolygon(polygon, QPen(m_outlineColor), iBrush);
            break;
        }
        case 3: {
            QPolygonF polygon;
            polygon << QPointF(iX + iRadius, iY + iRadius) << QPointF(iX + 2 * iRadius, iY - iRadius)
                    << QPointF(iX , iY - iRadius) << QPointF(iX + iRadius, iY + iRadius);
            graphItem = m_scene->addPolygon(polygon, QPen(m_outlineColor), iBrush);
            break;
        }
        case 4: {
            QPolygonF polygon;
            polygon << QPointF(iX , iY) << QPointF(iX + iRadius, iY + iRadius)
                    << QPointF(iX + 2 * iRadius, iY) << QPointF(iX + iRadius, iY - iRadius) << QPointF(iX, iY);
            graphItem = m_scene->addPolygon(polygon, QPen(m_outlineColor), iBrush);
            break;
        }
        case 5: {
            QPolygonF polygon;
            polygon << QPointF(iX, iY - iRadius) << QPointF(iX + 2 * iRadius, iY - iRadius)
                    << QPointF(iX, iY + iRadius) << QPointF(iX + 2 * iRadius, iY + iRadius);
            graphItem = m_scene->addPolygon(polygon, QPen(m_outlineColor), iBrush);
            break;
        }
        case 6: {
            QPolygonF polygon;
            polygon << QPointF(iX + iRadius, iY - iRadius) << QPointF(iX + 2 * iRadius, iY + iRadius)
                    << QPointF(iX , iY + iRadius) << QPointF(iX + iRadius, iY - iRadius);
            graphItem = m_scene->addPolygon(polygon, QPen(m_outlineColor), iBrush);
            break;
        }
        case 7: {
            QPainterPath path;
            path.addEllipse(iX, iY - iRadius, 2 * iRadius, 2 * iRadius);
            path.closeSubpath();

            QPainterPath path2;
            path2.addEllipse(iX + iRadius / 2.0, iY - iRadius / 2.0, iRadius, iRadius);
            path -= path2;
            graphItem = m_scene->addPath(path, QPen(m_outlineColor), iBrush);
            break;
        }
        }
    }

    return graphItem;
}

void SKGTableWithGraph::onExport()
{
    _SKGTRACEIN(10, "SKGTableWithGraph::onExport");
    QString lastCodecUsed = QTextCodec::codecForLocale()->name();
    QString fileName = SKGMainPanel::getSaveFileName("kfiledialog:///IMPEXP", "text/csv text/plain" , this, QString(), &lastCodecUsed);
    if(fileName.isEmpty()) return;

    SKGError err;
    QString extension = QFileInfo(fileName).suffix().toUpper();
    if(extension == "CSV") {
        //Write file
        KSaveFile file(fileName);
        if(!file.open()) {
            err.setReturnCode(ERR_INVALIDARG);
            err.setMessage(i18nc("Error message",  "Save file '%1' failed", fileName));
        } else {
            QTextStream out(&file);
            out.setCodec(lastCodecUsed.toAscii().constData());
            QStringList dump = SKGServices::tableToDump(getTable(), SKGServices::DUMP_CSV);
            int nbl = dump.count();
            for(int i = 0; i < nbl; ++i) {
                out << dump[i] << endl;
            }
        }

        //Close file
        file.finalize();
        file.close();
    } else {
        SKGError err;

        //Write file
        KSaveFile file(fileName);
        if(!file.open()) {
            err.setReturnCode(ERR_INVALIDARG);
            err.setMessage(i18nc("Error message",  "Save file '%1' failed", fileName));
        } else {
            QTextStream out(&file);
            out.setCodec(lastCodecUsed.toAscii().constData());
            QStringList dump = SKGServices::tableToDump(getTable(), SKGServices::DUMP_TEXT);
            int nbl = dump.count();
            for(int i = 0; i < nbl; ++i) {
                out << dump[i] << endl;
            }
        }

        //Close file
        file.finalize();
        file.close();
    }
    SKGMainPanel::displayErrorMessage(err);
    QDesktopServices::openUrl(QUrl(fileName));
}

void SKGTableWithGraph::setGraphType(SKGTableWithGraph::GraphType iType) const
{
    if(m_displayMode) m_displayMode->setCurrentIndex(m_displayMode->findData(iType));
}

SKGTableWithGraph::GraphType SKGTableWithGraph::getGraphType() const
{
    GraphType mode = (GraphType) m_displayMode->itemData(m_displayMode->currentIndex()).toInt();
    return mode;
}

void SKGTableWithGraph::setSelectable(bool iSelectable)
{
    m_selectable = iSelectable;
}

bool SKGTableWithGraph::isSelectable() const
{
    return m_selectable;
}

#include "skgtablewithgraph.moc"

