/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is a proxy model with better filter mechanism.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgsortfilterproxymodel.h"
#include "skgtraces.h"
#include "skgservices.h"
#include "skgobjectmodelbase.h"

#include <QRegExp>

SKGSortFilterProxyModel::SKGSortFilterProxyModel(QObject * parent)
    : QSortFilterProxyModel(parent)
{
    _SKGTRACEIN(10, "SKGSortFilterProxyModel::SKGSortFilterProxyModel");

    setSortCaseSensitivity(Qt::CaseInsensitive);
    setSortLocaleAware(true);
    setFilterKeyColumn(0);
}

SKGSortFilterProxyModel::~SKGSortFilterProxyModel()
{
    _SKGTRACEIN(10, "SKGSortFilterProxyModel::~SKGSortFilterProxyModel");
}

bool SKGSortFilterProxyModel::lessThan(const QModelIndex &left,
                                       const QModelIndex &right) const
{
    //Get source
    SKGObjectModelBase* model = qobject_cast<SKGObjectModelBase*>(this->sourceModel());
    if(model) {
        QVariant leftData = model->data(left, Qt::UserRole);
        QVariant rightData = model->data(right, Qt::UserRole);
        SKGObjectBase* leftObj = model->getObjectPointer(left);
        if(leftData == rightData && leftObj) {
            if(leftObj->getTable().isEmpty() && left.column() != 0) {
                //Groups
                return SKGSortFilterProxyModel::lessThan(model->index(left.row(), 0), model->index(right.row(), 0));
            }
            SKGObjectBase* rightObj = model->getObjectPointer(right);
            return (rightObj && leftObj->getID() < rightObj->getID());
        } else return QSortFilterProxyModel::lessThan(left, right);
    }
    return false;
}

bool SKGSortFilterProxyModel::filterAcceptsRow(int source_row, const QModelIndex & source_parent) const
{
    _SKGTRACEIN(10, "SKGSortFilterProxyModel::filterAcceptsRow");

    //Initialisation
    bool output = (filterRegExp().isEmpty());
    if(!output) {
        //Build list of criterias
        QList< SKGServices::SKGSearchCriteria > criterias = SKGServices::stringToSearchCriterias(filterRegExp().pattern());

        //Check if at least one group validate the line
        int nbList = criterias.count();
        output = false;
        for(int i = 0; i < nbList; ++i) {
            QChar mode = criterias[i].mode;

            bool validateAllWords =  filterAcceptsRowWords(source_row, source_parent, criterias[i].words);
            if(mode == '+') output |= validateAllWords;
            else if(mode == '-' && validateAllWords) output = false;
        }

        if(!output) {
            QAbstractItemModel* model = this->sourceModel();
            if(model) {
                QModelIndex index0 = model->index(source_row, 0, source_parent);

                int nb = model->rowCount(index0);
                for(int i = 0; !output && i < nb; ++i) {
                    output = filterAcceptsRow(i, index0);
                }
            }
        }
    }
    return output;
}

bool SKGSortFilterProxyModel::filterAcceptsRowWords(int source_row, const QModelIndex & source_parent, const QStringList& iWords) const
{
    _SKGTRACEIN(10, "SKGSortFilterProxyModel::filterAcceptsRowWords");

    //Initialisation
    bool output = true;

    //Get source
    QAbstractItemModel* model = this->sourceModel();
    if(model) {
        int nbwords = iWords.count();
        for(int w = 0; output && w < nbwords; ++w) {
            QString word = iWords.at(w);

            //Is this word validated by at least one column ?
            output = false;
            int nbcol = model->columnCount();
            for(int i = 0; !output && i < nbcol; ++i) {
                QModelIndex index0 = model->index(source_row, i, source_parent);
                if(index0.isValid()) {
                    output = model->data(index0).toString().contains(word , Qt::CaseInsensitive);
                    if(!output)  output = model->data(index0, Qt::UserRole).toString().contains(word , Qt::CaseInsensitive);
                }
            }
        }
    }
    return output;
}
