# encoding: utf-8
=begin

 * Name: SiSU

 * Description: a framework for document structuring, publishing and search

 * Author: Ralph Amissah

 * Copyright: (C) 1997 - 2012, Ralph Amissah, All Rights Reserved.

 * License: GPL 3 or later:

   SiSU, a framework for document structuring, publishing and search

   Copyright (C) Ralph Amissah

   This program is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation, either version 3 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program. If not, see <http://www.gnu.org/licenses/>.

   If you have Internet connection, the latest version of the GPL should be
   available at these locations:
   <http://www.fsf.org/licensing/licenses/gpl.html>
   <http://www.gnu.org/licenses/gpl.html>

   <http://www.jus.uio.no/sisu/gpl.fsf/toc.html>
   <http://www.jus.uio.no/sisu/gpl.fsf/doc.html>
   <http://www.jus.uio.no/sisu/gpl.fsf/plain.txt>

 * SiSU uses:
   * Standard SiSU markup syntax,
   * Standard SiSU meta-markup syntax, and the
   * Standard SiSU object citation numbering and system

 * Hompages:
   <http://www.jus.uio.no/sisu>
   <http://www.sisudoc.org>

 * Download:
   <http://www.jus.uio.no/sisu/SiSU/download.html>

 * Ralph Amissah
   <ralph@amissah.com>
   <ralph.amissah@gmail.com>

 ** Description: epub formating, css template

=end
module SiSU_EPUB_Format
  include SiSU_Viz
  class ParagraphNumber
    def initialize(md,ocn)
      @md,@ocn=md,ocn.to_s
      @ocn ||=''
      vz=SiSU_Env::GetInit.instance.skin
      @skin_no_ocn=if defined? vz.ocn_display_off \
      and vz.ocn_display_off==true
        true
      else false
      end
    end
    def ocn_display
      if @md.markup.inspect =~/no_ocn/ \
      or @md.opt.mod.inspect =~/--no-ocn/ \
      or @skin_no_ocn
        ocn_class='ocn_off'
        @ocn.gsub(/^(\d+|)$/,
        %{<label class="#{ocn_class}">&nbsp;</label>})
      elsif @ocn.to_i==0
        @ocn.gsub(/^(\d+|)$/,
        %{<label class="#{ocn_class}">&nbsp;</label>})
      else
        ocn_class='ocn'
        @ocn.gsub(/^(\d+|)$/,
        %{<label class="#{ocn_class}"><a href="#o\\1" class="lnk#{ocn_class}">\\1</a></label>})
      end
    end
    def name
      %{<a name="#{@ocn}"></a>}
    end
    def id #w3c? "tidy" complains about numbers as identifiers ! annoying
      %{id="o#{@ocn}"}
    end
    def goto
      %{<a href="##{@ocn}">}
    end
  end
  class CSS
    def css_epub_xhtml
<<WOK
/* SiSU epub css default stylesheet */
  body {
    color: black;
    background: #ffffff;
    background-color: #ffffff;
  }
/*
    table {
      margin-left: 5%;
      display: block;
    }
    tr {
      display: block;
    }
    th,td {
      display: inline;
      vertical-align: top;
    }
*/
  a:link {
    color: #003399;
    text-decoration: none;
  }
  a:visited {
    color: #003399;
    text-decoration: none;
  }
  a:hover {
    color: #000000;
    background-color: #f9f9aa;
  }
/*
  a:hover {
    border-bottom: 2px solid #777777;
    background-color: #fff3b6;
  }
*/
  a:hover img {
    background-color: #ffffff;
  }
  a:active {
    color: #003399;
    text-decoration: underline;
  }
  a.lnkocn:link {
    color: #777777;
    text-decoration: none;
  }
  div {
    margin-left: 0;
    margin-right: 0;
  }
  div.p {
    margin-left: 5%;
    margin-right: 1%;
  }

  .norm, .bold, .verse, .group, .block, .alt {
    line-height: 150%;
    margin-left: 0em;
    margin-right: 2em;
    margin-top: 10px;
    margin-bottom: 0px;
    padding-left: 0em;
    text-indent: 0mm;
  }
  p, h0, h1, h2, h3, h4, h5, h6 {
    display: block;
    font-family: verdana, arial, georgia, tahoma, sans-serif, helvetica, times, roman;
    font-size: 100%;
    font-weight: normal;
    line-height: 150%;
    text-align: justify;
    margin-left: 0em;
    margin-right: 2em;
    text-indent: 0mm;
    margin-top: 3px;
    margin-bottom: 3px;
  }
  p.norm { }
  p.i1 {padding-left: 1em;}
  p.i2 {padding-left: 2em;}
  p.i3 {padding-left: 3em;}
  p.i4 {padding-left: 4em;}
  p.i5 {padding-left: 5em;}
  p.i6 {padding-left: 6em;}
  p.i7 {padding-left: 7em;}
  p.i8 {padding-left: 8em;}
  p.i9 {padding-left: 9em;}

  p.h0i0 {
    padding-left: 0em;
    text-indent:  0em;
  }
  p.h0i1 {
    padding-left: 1em;
    text-indent: -1em;
  }
  p.h0i2 {
    padding-left: 2em;
    text-indent: -2em;
  }
  p.h0i3 {
    padding-left: 3em;
    text-indent: -3em;
  }
  p.h0i4 {
    padding-left: 4em;
    text-indent: -4em;
  }
  p.h0i5 {
    padding-left: 5em;
    text-indent: -5em;
  }
  p.h0i6 {
    padding-left: 6em;
    text-indent: -6em;
  }
  p.h0i7 {
    padding-left: 7em;
    text-indent: -7em;
  }
  p.h0i8 {
    padding-left: 8em;
    text-indent: -8em;
  }
  p.h0i9 {
    padding-left: 9em;
    text-indent: -9em;
  }

  p.h1i0 {
    padding-left: 0em;
    text-indent:  1em;
  }
  p.h1i1 {
    padding-left: 1em;
    text-indent:  0em;
  }
  p.h1i2 {
    padding-left: 2em;
    text-indent: -1em;
  }
  p.h1i3 {
    padding-left: 3em;
    text-indent: -2em;
  }
  p.h1i4 {
    padding-left: 4em;
    text-indent: -3em;
  }
  p.h1i5 {
    padding-left: 5em;
    text-indent: -4em;
  }
  p.h1i6 {
    padding-left: 6em;
    text-indent: -5em;
  }
  p.h1i7 {
    padding-left: 7em;
    text-indent: -6em;
  }
  p.h1i8 {
    padding-left: 8em;
    text-indent: -7em;
  }
  p.h1i9 {
    padding-left: 9em;
    text-indent: -8em;
  }

  p.h2i0 {
    padding-left: 0em;
    text-indent:  2em;
  }
  p.h2i1 {
    padding-left: 1em;
    text-indent:  1em;
  }
  p.h2i2 {
    padding-left: 2em;
    text-indent:  0em;
  }
  p.h2i3 {
    padding-left: 3em;
    text-indent: -1em;
  }
  p.h2i4 {
    padding-left: 4em;
    text-indent: -2em;
  }
  p.h2i5 {
    padding-left: 5em;
    text-indent: -3em;
  }
  p.h2i6 {
    padding-left: 6em;
    text-indent: -4em;
  }
  p.h2i7 {
    padding-left: 7em;
    text-indent: -5em;
  }
  p.h2i8 {
    padding-left: 8em;
    text-indent: -6em;
  }
  p.h2i9 {
    padding-left: 9em;
    text-indent: -7em;
  }

  p.h3i0 {
    padding-left: 0em;
    text-indent:  3em;
  }
  p.h3i1 {
    padding-left: 1em;
    text-indent:  2em;
  }
  p.h3i2 {
    padding-left: 2em;
    text-indent:  1em;
  }
  p.h3i3 {
    padding-left: 3em;
    text-indent:  0em;
  }
  p.h3i4 {
    padding-left: 4em;
    text-indent: -1em;
  }
  p.h3i5 {
    padding-left: 5em;
    text-indent: -2em;
  }
  p.h3i6 {
    padding-left: 6em;
    text-indent: -3em;
  }
  p.h3i7 {
    padding-left: 7em;
    text-indent: -4em;
  }
  p.h3i8 {
    padding-left: 8em;
    text-indent: -5em;
  }
  p.h3i9 {
    padding-left: 9em;
    text-indent: -6em;
  }

  p.h4i0 {
    padding-left: 0em;
    text-indent:  4em;
  }
  p.h4i1 {
    padding-left: 1em;
    text-indent:  3em;
  }
  p.h4i2 {
    padding-left: 2em;
    text-indent:  2em;
  }
  p.h4i3 {
    padding-left: 3em;
    text-indent:  1em;
  }
  p.h4i4 {
    padding-left: 4em;
    text-indent:  0em;
  }
  p.h4i5 {
    padding-left: 5em;
    text-indent: -1em;
  }
  p.h4i6 {
    padding-left: 6em;
    text-indent: -2em;
  }
  p.h4i7 {
    padding-left: 7em;
    text-indent: -3em;
  }
  p.h4i8 {
    padding-left: 8em;
    text-indent: -4em;
  }
  p.h4i9 {
    padding-left: 9em;
    text-indent: -5em;
  }

  p.h5i0 {
    padding-left: 0em;
    text-indent:  5em;
  }
  p.h5i1 {
    padding-left: 1em;
    text-indent:  4em;
  }
  p.h5i2 {
    padding-left: 2em;
    text-indent:  3em;
  }
  p.h5i3 {
    padding-left: 3em;
    text-indent:  2em;
  }
  p.h5i4 {
    padding-left: 4em;
    text-indent:  1em;
  }
  p.h5i5 {
    padding-left: 5em;
    text-indent:  0em;
  }
  p.h5i6 {
    padding-left: 6em;
    text-indent: -1em;
  }
  p.h5i7 {
    padding-left: 7em;
    text-indent: -2em;
  }
  p.h5i8 {
    padding-left: 8em;
    text-indent: -3em;
  }
  p.h5i9 {
    padding-left: 9em;
    text-indent: -4em;
  }

  p.h6i0 {
    padding-left: 0em;
    text-indent:  6em;
  }
  p.h6i1 {
    padding-left: 1em;
    text-indent:  5em;
  }
  p.h6i2 {
    padding-left: 2em;
    text-indent:  4em;
  }
  p.h6i3 {
    padding-left: 3em;
    text-indent:  3em;
  }
  p.h6i4 {
    padding-left: 4em;
    text-indent:  2em;
  }
  p.h6i5 {
    padding-left: 5em;
    text-indent:  1em;
  }
  p.h6i6 {
    padding-left: 6em;
    text-indent:  0em;
  }
  p.h6i7 {
    padding-left: 7em;
    text-indent: -1em;
  }
  p.h6i8 {
    padding-left: 8em;
    text-indent: -2em;
  }
  p.h6i9 {
    padding-left: 9em;
    text-indent: -3em;
  }

  p.h7i0 {
    padding-left: 0em;
    text-indent:  7em;
  }
  p.h7i1 {
    padding-left: 1em;
    text-indent:  6em;
  }
  p.h7i2 {
    padding-left: 2em;
    text-indent:  5em;
  }
  p.h7i3 {
    padding-left: 3em;
    text-indent:  4em;
  }
  p.h7i4 {
    padding-left: 4em;
    text-indent:  3em;
  }
  p.h7i5 {
    padding-left: 5em;
    text-indent:  2em;
  }
  p.h7i6 {
    padding-left: 6em;
    text-indent:  1em;
  }
  p.h7i7 {
    padding-left: 7em;
    text-indent:  0em;
  }
  p.h7i8 {
    padding-left: 8em;
    text-indent: -1em;
  }
  p.h7i9 {
    padding-left: 9em;
    text-indent: -2em;
  }

  p.h8i0 {
    padding-left: 0em;
    text-indent:  8em;
  }
  p.h8i1 {
    padding-left: 1em;
    text-indent:  7em;
  }
  p.h8i2 {
    padding-left: 2em;
    text-indent:  6em;
  }
  p.h8i3 {
    padding-left: 3em;
    text-indent:  5em;
  }
  p.h8i4 {
    padding-left: 4em;
    text-indent:  4em;
  }
  p.h8i5 {
    padding-left: 5em;
    text-indent:  3em;
  }
  p.h8i6 {
    padding-left: 6em;
    text-indent:  2em;
  }
  p.h8i7 {
    padding-left: 7em;
    text-indent:  1em;
  }
  p.h8i8 {
    padding-left: 8em;
    text-indent:  0em;
  }
  p.h8i9 {
    padding-left: 9em;
    text-indent: -1em;
  }

  p.h9i0 {
    padding-left: 0em;
    text-indent:  9em;
  }
  p.h9i1 {
    padding-left: 1em;
    text-indent:  8em;
  }
  p.h9i2 {
    padding-left: 2em;
    text-indent:  7em;
  }
  p.h9i3 {
    padding-left: 3em;
    text-indent:  6em;
  }
  p.h9i4 {
    padding-left: 4em;
    text-indent:  5em;
  }
  p.h9i5 {
    padding-left: 5em;
    text-indent:  4em;
  }
  p.h9i6 {
    padding-left: 6em;
    text-indent:  3em;
  }
  p.h9i7 {
    padding-left: 7em;
    text-indent:  2em;
  }
  p.h9i8 {
    padding-left: 8em;
    text-indent:  1em;
  }
  p.h9i9 {
    padding-left: 9em;
    text-indent:  0em;
  }

  p.it0 {
    margin-left: 0em;
    margin-top: 6px;
    margin-bottom: 0px;
    line-height: 100%;
  }
  p.it1 {
    margin-left: 1em;
    margin-top: 0px;
    margin-bottom: 0px;
    line-height: 100%;
  }
  p.it2 {
    margin-left: 2em;
    margin-top: 0px;
    margin-bottom: 0px;
    line-height: 100%;
  }
  p.it3 {
    margin-left: 3em;
    margin-top: 0px;
    margin-bottom: 0px;
    line-height: 100%;
  }
  p.it4 {
    margin-left: 4em;
    margin-top: 0px;
    margin-bottom: 0px;
    line-height: 100%;
  }
  p.it5 {
    margin-left: 5em;
    margin-top: 0px;
    margin-bottom: 0px;
    line-height: 100%;
  }
  p.it6 {
    margin-left: 6em;
    margin-top: 0px;
    margin-bottom: 0px;
    line-height: 100%;
  }
  p.it7 {
    margin-left: 7em;
    margin-top: 0px;
    margin-bottom: 0px;
    line-height: 100%;
  }
  p.it8 {
    margin-left: 8em;
    margin-top: 0px;
    margin-bottom: 0px;
    line-height: 100%;
  }
  p.it9 {
    margin-left: 9em;
    margin-bottom: 0px;
    margin-top: 0px;
    line-height: 100%;
  }

  p.group { }

  p.block { }

  p.alt { }

  p.verse {
    margin-bottom: 6px;
  }

  p.code {
    font-family: inconsolata, andale mono, courier new, courier, monospace;
    font-size: 90%;
    text-align: left;
    background-color: #eeeeee;
  }

  p.caption {
    text-align: left;
    font-size: 80%;
    display: inline;
  }

  p.endnote {
    font-size: 96%;
    line-height: 120%;
    text-align: left;
    margin-right: 2em;
  }
  p.endnote_indent {
    font-size: 96%;
    line-height: 120%;
    text-align: left;
    margin-left: 2em;
    margin-right: 2em;
  }

  p.center {
    text-align: center;
  }
  p.align_right {
    text-align: right;
  }
  p.bold {
    font-weight: bold;
  }
  p.bold_left {
    font-weight: bold;
    text-align: left;
  }
  p.centerbold {
    text-align: center;
    font-weight: bold;
  }
  p.em {
    font-weight: bold;
    font-style: normal;
    background: #fff3b6;
  }

  p.small {
    font-size: 80%;
    margin-top: 0px;
    margin-bottom: 0px;
    margin-right: 6px;
    text-align: left;
  }

  .tiny, .tiny_left, .tiny_right, .tiny_center {
    font-size: 10px;
    margin-top: 0px;
    margin-bottom: 0px;
    color: #777777;
    margin-right: 6px;
    text-align: left;
  }
  p.tiny { }
  p.tiny_left {
    margin-left: 0px;
    margin-right: 0px;
    text-align: left;
  }
  p.tiny_right {
    margin-right: 1em;
    text-align: right;
  }
  p.tiny_center {
    margin-left: 0px;
    margin-right: 0px;
    text-align: center;
  }

  p.concordance_word {
    line-height: 150%;
    font-weight: bold;
    display: inline;
    margin-top: 4px;
    margin-bottom: 1px;
  }
  p.concordance_count {
    font-size: 80%;
    color: #777777;
    display: inline;
    margin-left: 0em;
  }
  p.concordance_object {
    font-size: 80%;
    line-height: 120%;
    text-align: left;
    margin-left: 3em;
    margin-top: 1px;
    margin-bottom: 3px;
  }
  p.book_index_lev1 {
    line-height: 100%;
    margin-top: 4px;
    margin-bottom: 1px;
  }
  p.book_index_lev2 {
    line-height: 100%;
    text-align: left;
    margin-left: 3em;
    margin-top: 1px;
    margin-bottom: 3px;
  }

  p.quickref {
    font-size: 10px;
    font-style: italic;
    margin-top: 0px;
    margin-bottom: 0px;
    color: #777777;
    margin-right: 5px;
    text-align: left;
  }
  p.bigref {
    font-size: 11px;
    font-weight: bold;
    margin-top: 0px;
    margin-bottom: 0px;
    color: #777777;
    margin-right: 5px;
    text-align: center;
  }

  p.letter {
    font-weight: bold;
    font-size: 80%;
    margin-left: 0em;
    margin-top: 2px;
    margin-bottom: 2px;
    margin-right: 6px;
    text-align: left;
    color: white;
    background: #880000;
  }

  tt {
    font-family: inconsolata, andale mono, courier new, courier, monospace;
    background-color: #eeeeee;
  }

  label.ocn {
    width: 2%;
    float: right;
    top: 0;
    font-size: 10px;
    margin-top: 0px;
    margin-bottom: 5px;
    color: #777777;
    margin-right: 5px;
    text-align: right;
    background-color: #ffffff;
  }

  table { }
  tr { }
  th,td {
    vertical-align: top;
    text-align: left;
  }
  th {
    font-weight: bold;
  }

  p.left, th.left, td.left {
    text-align: left;
  }
  p.small_left, th.small_left, td.small_left {
    text-align: left;
    font-size: 80%;
  }
  p.right, th.right, td.right {
    text-align: right;
  }

  #horizontal_links {
    background: #eeeeee;
    margin-left: 5%;
    margin-right: 5%;
  }
  #horizontal {
    margin: 0;
    padding: 0 0 0 10px;
    border-top: 1px solid #000077;
    border-bottom: 1px solid #000077;
  }
  #horizontal li {
    margin: 0 0 0 0;
    padding: 0 16px 0 0;
    display: inline;
    list-style-type: none;
    text-align: left;
    background: none;
  }
  #horizontal a {
    line-height: 12px;
    margin: 0 0 0 0;
    text-decoration: none;
    color: #000077;
  }
  #horizontal a.active, #horizontal a:hover {
    border-bottom: 2px solid #777777;
    padding-bottom: 2px;
    color: #000077;
  }
  #horizontal a:hover {
    color: #000077;
  }

  #document_versions {
    position: absolute;
    top: 10mm;
    right: 2%;
    width: 12%;
    float: right;
  }

  #vertical_links {
    position: absolute;
    top: 10mm;
    right: 0px;
    width: 20%;
    background: #dddddd;
    float: right;
  }
  #vertical {
    padding: 0 12px 0px 0px;
    margin-left: 2%;
    margin-right: 2%;
  }
  #vertical li {
    display: block;
    list-style-type: none;
  }
  #vertical a {
    line-height: 12px;
    text-decoration: none;
    color: #000077;
  }
  #vertical a.active, #vertical a:hover {
    border-bottom: 2px solid #777777;
    padding-bottom: 2px;
    color: #000077;
  }

  ul, li {
    list-style-type: none;
    list-style: none;
    padding-left: 20px;
    display: block;
    font-family: verdana, arial, georgia, tahoma, sans-serif, helvetica, times, roman;
    font-weight: normal;
    line-height: 150%;
    text-align: justify;
    text-indent: 0mm;
    margin-left: 1em;
    margin-right: 2em;
    margin-top: 3px;
    margin-bottom: 3px;
  }

  li {
    background: url(../image/bullet_09.png) no-repeat 0px 6px;
  }

  ul {
  }
  li.bullet { margin-left: 0em; }
  li.i1 { margin-left: 1em; }
  li.i2 { margin-left: 2em; }
  li.i3 { margin-left: 3em; }
  li.i4 { margin-left: 4em; }
  li.i5 { margin-left: 5em; }
  li.i6 { margin-left: 6em; }
  li.i7 { margin-left: 7em; }
  li.i8 { margin-left: 8em; }
  li.i9 { margin-left: 9em; }

  li.doc, li.ref, li.refcenter {
    margin-top: 0px;
    margin-bottom: 0px;
    margin-right: 0px;
    font-size: 8px;
    font-style: normal;
    text-align: left;
  }
  li.doc {
    background: url(../image/bullet_09.png) no-repeat 0px 6px;
    padding-left: 16px;
    margin-left: 10px;
    margin-right: 0px;
  }
  li.ref {
    background: none;
    padding-left: 0;
    margin-left: 0;
    color: #777777;
  }
  li.refcenter {
    background: url(../image/bullet_09.png) no-repeat 0px 6px;
    padding-left: 20px;
    margin-left: 10%;
    font-size: 9px;
    color: #777777;
    text-align: center;
  }
  li.refbold {
    list-style-type: none;
    padding-left: 16px;
    margin-left: 0;
    margin-right: 10mm;
    font-weight: bold;
  }

  h0, h1, h2, h3, h4, h5, h6 {
    font-weight: bold;
    line-height: 120%;
    text-align: left;
    margin-top: 20px;
    margin-bottom: 10px;
  }
  h4.norm, h5.norm, h6.norm {
    margin-top: 10px;
    margin-bottom: 0px;
  }
  h1.center, h2.center, h3.center, h4.center, h5.center, h6.center {
    text-align: center;
  }
  h1 { font-size: 120%; }
  h2 { font-size: 115%; }
  h3 { font-size: 110%; }
  h4 { font-size: 105%; }
  h5 { font-size: 100%; }
  h6 { font-size: 100%; }
  h0 { font-size: 80%; }

  h1.i {margin-left: 2em;}
  h2.i {margin-left: 3em;}
  h3.i {margin-left: 4em;}
  h4.i {margin-left: 5em;}
  h5.i {margin-left: 6em;}
  h6.i {margin-left: 7em;}
  h7.i {margin-left: 8em;}
  h8.i {margin-left: 9em;}
  h9.i {margin-left: 10em;}

  .toc {
    font-weight: normal;
    margin-top: 6px;
    margin-bottom: 6px;
  }
  h1.toc {
    margin-left: 1em;
    font-size: 115%;
    line-height: 150%;
  }
  h2.toc {
    margin-left: 2em;
    font-size: 110%;
    line-height: 140%;
  }
  h3.toc {
    margin-left: 3em;
    font-size: 105%;
    line-height: 120%;
  }
  h4.toc {
    margin-left: 4em;
    font-size: 100%;
    line-height: 120%;
  }
  h5.toc {
    margin-left: 5em;
    font-size: 95%;
    line-height: 110%;
  }
  h6.toc {
    margin-left: 6em;
    font-size: 90%;
    line-height: 110%;
  }

  .microtoc {
    margin-top: 2px;
    margin-bottom: 2px;
  }

  h1.microtoc {
    margin-left: 0mm;
    font-size: 115%;
  }
  h2.microtoc {
    margin-left: 5mm;
    font-size: 110%;
  }
  h3.microtoc {
    margin-left: 10mm;
    font-size: 105%;
  }
  h4.microtoc {
    margin-left: 15mm;
    font-weight: normal;
    font-size: 100%;
  }
  h5.microtoc {
    margin-left: 20mm;
    font-weight: normal;
    font-size: 95%;
  }
  h6.microtoc {
    margin-left: 25mm;
    font-weight: normal;
    font-size: 90%;
  }

  .subtoc {
    margin-right: 34%;
    font-weight: normal;
  }
  h5.subtoc {
    margin-left: 2em;
    font-size: 80%;
    margin-top: 2px;
    margin-bottom: 2px;
  }
  h6.subtoc {
    margin-left: 3em;
    font-size: 75%;
    margin-top: 0px;
    margin-bottom: 0px;
  }

  div.substance {
    width: 100%;
    background-color: #ffffff;
  }
  div.ocn {
    width: 5%;
    float: right;
    top: 0;
    background-color: #ffffff;
  }
  div.endnote {
    width: 100%;
    background-color: #fffffff;
  }
  div.toc {
    position: absolute;
    float: left;
    margin: 0;
    padding: 0;
    padding-top: 0.5em;
    border: 0;
    width: 5%;
    background-color: #eeeeee;
    margin-right:1em;
  }
  div.summary {
    margin: 0;
    padding: 0;
    border-left: 2em solid #eeeeee;
    padding-left: 0em;
    background-color: #eeeeee;
  }
  div.content, div.main_column {
    margin: 0;
    padding: 0;
    border-left: 0% solid #ffffff;
    padding-left: 5%;
  }
  div.content:after {
    content:' ';
    clear:both;
    display:block;
    height:0;
    overflow:hidden
  }
  div.footer {
    clear:left;
    padding: 0.5em;
    font-size: 80%;
    margin: 0;
  }
  div.toc ul {
    list-style: none;
    padding: 0;
    margin: 0;
  }
  div.toc li ul a, li ul span.currentlink
  {
    font-weight: normal;
    font-size: 90%;
    padding-left: 2em;
    background-color: #eeeeee;
  }
  div.toc a, span.currentlink{
    display:block;
    text-decoration: none;
    padding-left: 0.5em;
    color: #0000aa;
  }
  hr {
    width: 90%;
  }

  span.currentlink {
    text-decoration: none;
    background-color: #aaaaf9;
  }

  div.toc a:visited {
    color: #0000aa;
  }
  div.toc a:hover {
    color: #000000;
    background-color: #f9f9aa;
  }

  .minitoc {
    font-weight: normal;
    margin-top: 2px;
    margin-bottom: 2px;
  }
  h1.minitoc, h2.minitoc, h3.minitoc {
    margin-left: 0em;
    font-weight: bold;
    text-align: left;
    font-size: 90%;
    margin-top: 4px;
    margin-bottom: 4px;
  }
  h4.minitoc {
    margin-left: 0em;
    font-size: 90%;
  }
  h5.minitoc {
    margin-left: 1em;
    font-size: 85%;
  }
  h6.minitoc {
    margin-left: 2em;
    font-size: 85%;
  }
  h0.minitoc {
    margin-left: 0em;
    font-size: 90%;
  }

  h1.c, h2.c, h3.c, h4.c, h5.c, h6.c, p.c {
    text-align: center
  }
  h1.red, h2.red, h3.red, h4.red, h5.red, h6.red {
    text-align: center;
    color: #ff0000;
    margin-left: 5mm;
    text-indent: 5mm;
    margin-top: 30px;
    margin-bottom: 20px;
    margin-right: 15mm;
  }
  h1.ruby, h2.ruby, h3.ruby, h4.ruby, h5.ruby, h6.ruby {
    text-align: center;
    color: #990000;
    margin-left: 5mm;
    text-indent: 5mm;
    margin-top: 30px;
    margin-bottom: 20px;
    margin-right: 15mm;
  }
WOK
    end
  end
  class HeadInformation
    include SiSU_Viz
    attr_reader :md,:rdf,:vz
    def initialize(md)
      @md=md
      # DublinCore 1 - title
      @vz=SiSU_Env::GetInit.instance.skin
      @css=SiSU_Env::CSS_Stylesheet.new(md)
      @seg_name_xhtml=(SiSU_EPUB::Source::Seg.new.seg_name_xhtml || [])
      @seg_name_xhtml_tracker=(SiSU_EPUB::Source::Seg.new.seg_name_xhtml_tracker || [])
      @index='index'
      @metalink='#metadata'
      @tocband_scroll,@tocband_segtoc=nil,nil
    end
    def doc_type_xhtml
<<WOK
<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.1//EN"
   "http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd">
<html xmlns="http://www.w3.org/1999/xhtml">
WOK
    end
=begin
~/epub
  |-- META-INF
  |   `-- container.xml                 #✓ simple, make sure full-path of rootfile points to metadata.opf
  |-- content
  |   |-- 1.xhtml
  |   |-- 2.xhtml
  |   |-- 3.xhtml
  |   |-- ... .xhtml
  |   |-- concordance.xhtml
  |   |-- css
  |   |   `-- xhtml.css
  |   |-- endnotes.xhtml
  |   |-- image
  |   |   |-- arrow_next_red.png
  |   |   |-- arrow_prev_red.png
  |   |   |-- arrow_up_red.png
  |   |   `-- bullet_09.png
  |   |-- index.xhtml
  |   |-- meta.xhtml
  |   |-- metadata.xhtml
  |   `-- toc.xhtml
  |-- metadata.opf                     #(i) metadata dc; (ii) manifest (contents); (iii) spine (mimetypes)
  |-- mimetype                         #✓ application/epub+zip
  `-- toc.ncx                          #✓ (i) head (ii) doc title (iii) navmap, list of navigation points (like chapters)
=end
    def doc_type
      doc_type_xhtml
    end
    def mimetype
      <<WOK
application/epub+zip
WOK
    end
    def metainf_container #container.xml file in META-INF directory
      #simple, make sure full-path of rootfile points to metadata.opf
      #epub_metadata.opf content.opf
      <<WOK
<?xml version="1.0" encoding="UTF-8"?>
<container version="1.0"
  xmlns="urn:oasis:names:tc:opendocument:xmlns:container">
  <rootfiles>
    <rootfile full-path="#{Ep[:d_oebps]}/#{Ep[:f_opf]}"
      media-type="application/oebps-package+xml" />
  </rootfiles>
</container>
WOK
    end
    def sections(dob,name)
      filename="#{name}#{Sfx[:epub_xhtml]}"
      dir_epub_cont="#{@md.env.processing_path.epub}/#{Ep[:d_oebps]}"
      segfilename="#{dir_epub_cont}/#{filename}"
      output_epub_cont_seg=File.new(segfilename,'w')
      output_epub_cont_seg << %{#{doc_type}
<head>
  <title>
    #{dob.obj} -
    #{@md.html_title}
  </title>
<meta http-equiv="Content-Type" content="text/html;charset=utf-8" />
#{@css.xhtml_epub}
</head>
#{@vz.color_body}
<div class="content">
<div class="substance">
  <label class="ocn"><a href="#o8" class="lnkocn">8</a></label>
  <h1 class="norm" id="o8">
    #{dob.obj}
  </h1>
</div>
</body>
</html>}
output_epub_cont_seg.close
    end
    def toc_ncx #list of navigation points (like chapters), table of contents, listing each navigation point (chapters and such) under the navigation map
      def structure
        open
        head_open
        head
        head_close
        doc_title
        doc_author
        navmap_open
       #navmap ...
        navmap_close
        close
      end
      def open
      <<WOK
<?xml version="1.0"?>
<!DOCTYPE ncx PUBLIC "-//NISO//DTD ncx 2005-1//EN"
  "http://www.daisy.org/z3986/2005/ncx-2005-1.dtd">
<ncx xmlns="http://www.daisy.org/z3986/2005/ncx/" version="2005-1">
WOK
      end
      def close
      <<WOK
</ncx>
WOK
      end
      def head_open
      <<WOK
  <head>
WOK
      end
      def head
        depth=@md.lvs[1] + @md.lvs[2] + @md.lvs[3] + @md.lvs[4]
        <<WOK
    <!-- four required metadata items (for all NCX documents,
      (including the relaxed constraints of OPS 2.0) -->
    <title>#{@md.title.full} by #{@md.author}</title>
    <link href="css/xhtml.css" rel="stylesheet" type="text/css" id="main-css" />
    <meta name="dtb:uid" content="urn:uuid:#{@md.dgst[1]}" />
    <!-- <meta name="epub-creator" content="#{@md.publisher}" /> -->
    <meta name="dtb:depth" content="#{depth}" />
    <meta name="dtb:totalPageCount" content="0" />
    <meta name="dtb:maxPageNumber" content="0" />
WOK
      end
      def head_close
        <<WOK
  </head>
WOK
      end
      def doc_title
        <<WOK
  <docTitle>
    <text>#{@md.title.full}</text>
  </docTitle>
WOK
      end
      def doc_author
        <<WOK
  <docAuthor>
    <text>#{@md.author}</text>
  </docAuthor>
WOK
      end
      def navmap_open
        <<WOK
  <navMap>
WOK
      end
      def navmap_sisu_toc(no)
        id_u=DISABLE[:epub][:ncx_navpoint_unique_id] \
        ? ''
        : "-#{no}"
        <<WOK
    <navPoint id="navpoint#{id_u}" playOrder="#{no}">
      <navLabel>
        <text>Table of Contents</text>
      </navLabel>
      <content src="index#{Sfx[:epub_xhtml]}" />
    </navPoint>
WOK
      end
      def navpoint(dob,no,name=nil)
        name=name ? name : dob.name
        id_u=DISABLE[:epub][:ncx_navpoint_unique_id] \
        ? ''
        : "-#{no}"
        <<WOK
    <navPoint id="navpoint#{id_u}" playOrder="#{no}">
      <navLabel>
        <text>#{dob.obj}</text>
      </navLabel>
      <content src="#{name}#{Sfx[:epub_xhtml]}" />
WOK
      end
      def navpoint_close
        <<WOK
    </navPoint>
WOK
      end
      def navmap_close
        <<WOK
  </navMap>
WOK
      end
      self
    end
    def metadata_opf #(i) metadata dc; (ii) manifest (contents); (iii) spine (mimetypes)
      def structure
        package_open
        metadata_open
        metadata_close
        manifest_open
        manifest_close
        spine_open
        spine_close
        guide_open
        guide_close
        package_close
      end
      def package_open
        <<WOK
<?xml version="1.0" encoding="UTF-8"?>
<package xmlns="http://www.idpf.org/2007/opf" unique-identifier="EPB-UUID" version="2.0">
WOK
      end
      def package_close
        <<WOK
</package>
WOK
      end
      def metadata #metadata dc
        author=if defined? @md.creator.author \
        and @md.creator.author =~/\S+/
          m=''
          @md.creator.author_detail.each do |x|
            surname=x[:the] \
            ? x[:the]
            : ''
            other_names=x[:others] \
            ? ', ' + x[:others]
            : ''
            m=(m.empty?) \
            ? (surname + other_names)
            : (m + '; ' + surname + ', ' + other_names)
            m=m.gsub(/</,'&lt;').gsub(/>/,'&gt;').
              gsub(/&lt;br(?: \/)?&gt;/,';')
          end
          x=@md.creator.author.dup
          x=x.gsub(/</,'&lt;').gsub(/>/,'&gt;').
            gsub(/&lt;br(?: \/)?&gt;/,'<br />')
          %{\n    <dc:creator opf:file-as="#{m}" opf:role="aut">#{x}</dc:creator>}
        else ''
        end
        editor=if defined? @md.creator.editor \
        and @md.creator.editor =~/\S+/
          m=''
          @md.creator.editor_detail.each do |x|
            surname=x[:the] \
            ? x[:the]
            : ''
            other_names=x[:others] \
            ? ', ' + x[:others]
            : ''
            m=(m.empty?) \
            ? (surname + other_names)
            : (m + '; ' + surname + ', ' + other_names)
            m=m.gsub(/</,'&lt;').gsub(/>/,'&gt;').
              gsub(/&lt;br(?: \/)?&gt;/,';')
          end
          x=@md.creator.editor.dup
          x=x.gsub(/</,'&lt;').gsub(/>/,'&gt;').
            gsub(/&lt;br(?: \/)?&gt;/,'<br />')
          %{\n    <dc:creator opf:file-as="#{m}" opf:role="edt">#{x}</dc:creator>}
        else ''
        end
        translator=if defined? @md.creator.translator \
        and @md.creator.translator =~/\S+/
          m=''
          @md.creator.translator_detail.each do |x|
            surname=x[:the] \
            ? x[:the]
            : ''
            other_names=x[:others] \
            ? ', ' + x[:others]
            : ''
            m=(m.empty?) \
            ? (surname + other_names)
            : (m + '; ' + surname + ', ' + other_names)
            m=m.gsub(/</,'&lt;').gsub(/>/,'&gt;').
              gsub(/&lt;br(?: \/)?&gt;/,';')
          end
          x=@md.creator.translator.dup
          x=x.gsub(/</,'&lt;').gsub(/>/,'&gt;').
            gsub(/&lt;br(?: \/)?&gt;/,'<br />')
          %{\n    <dc:creator opf:file-as="#{m}" opf:role="trl">#{x}</dc:creator>}
        else ''
        end
        illustrator=if defined? @md.creator.illustrator \
        and @md.creator.illustrator =~/\S+/
          m=''
          @md.creator.illustrator_detail.each do |x|
            surname=x[:the] \
            ? x[:the]
            : ''
            other_names=x[:others] \
            ? ', ' + x[:others]
            : ''
            m=(m.empty?) \
            ? (surname + other_names)
            : (m + '; ' + surname + ', ' + other_names)
            m=m.gsub(/</,'&lt;').gsub(/>/,'&gt;').
              gsub(/&lt;br(?: \/)?&gt;/,';')
          end
          x=@md.creator.illustrator.dup
          x=x.gsub(/</,'&lt;').gsub(/>/,'&gt;').
            gsub(/&lt;br(?: \/)?&gt;/,'<br />')
          %{\n    <dc:creator opf:file-as="#{m}" opf:role="ill">#{x}</dc:creator>}
        else ''
        end
        date_published=if defined? @md.date.published \
        and @md.date.published =~/\S+/
          x=@md.date.published.dup
          x=x.gsub(/</,'&lt;').gsub(/>/,'&gt;').
            gsub(/&lt;br(?: \/)?&gt;/,'<br />')
          %{\n    <dc:date opf:event="published">#{x}</dc:date>}
        else ''
        end
        subject=if defined? @md.classify.subject \
        and @md.classify.subject =~/\S+/
          x=@md.classify.subject.dup
          x=x.gsub(/</,'&lt;').gsub(/>/,'&gt;').
            gsub(/&lt;br(?: \/)?&gt;/,'<br />')
          %{\n    <dc:subject>#{x}</dc:subject>}
        else ''
        end
        language=if defined? @md.opt.lng \
        and @md.opt.lng =~/\S+/
          language=@md.opt.lng.gsub(/<br>/,'<br />')
          %{\n    <dc:language>#{language}</dc:language>}
        else ''
        end
        rights=if defined? @md.rights.all \
        and @md.rights.all =~/\S+/
          rights=@md.rights.all.gsub(/<br>/,'<br />')
          %{\n    <dc:rights>#{rights}</dc:rights>}
        else ''
        end
        f=SiSU_Env::FileOp.new(@md)
        <<WOK
  <metadata xmlns:dc="http://purl.org/dc/elements/1.1/"
    xmlns:opf="http://www.idpf.org/2007/opf"
    unique-identifier="urn:uuid:#{@md.dgst[1]}" version="2.0">
    <dc:title>#{@md.title.full}</dc:title>
    #{author}#{editor}#{translator}#{illustrator}#{language}#{date_published}#{subject}#{rights}
    <dc:identifier opf:scheme="URI">#{f.output_path.epub.url}/#{f.base_filename.epub}</dc:identifier>
    <dc:identifier id="bookid">urn:uuid:#{@md.dgst[1]}</dc:identifier>
    <!-- <dc:identifier id="EPB-UUID">urn:uuid:#{@md.dgst[1]}</dc:identifier> -->
  </metadata>
WOK
      end
      def manifest_open
        <<WOK
  <manifest>
    <!-- NCX -->
    <item id="ncx" href="#{Ep[:f_ncx]}" media-type="application/x-dtbncx+xml" />
    <!-- CSS Style Sheets -->
    <item id="main-css" href="css/xhtml.css" media-type="text/css" />
    <!-- Content Documents -->
WOK
      end
      def manifest_content_sisu_toc
        <<WOK
    <item id="index" href="index.xhtml" media-type="application/xhtml+xml" />
WOK
      end
      def manifest_content(dob,name=nil)
        name=name ? name : dob.name
        <<WOK
    <item id="#{name}" href="#{name}#{Sfx[:epub_xhtml]}" media-type="application/xhtml+xml" />
WOK
      end
      def manifest_images(imgs)
        imgs=imgs + ['arrow_next_red.png','arrow_prev_red.png','arrow_up_red.png','bullet_09.png']
        images=["    <!-- Images -->\n"]
        imgs.each do |i|
          image,type=/(\S+?)\.(png|jpg|gif)/.match(i)[1,2]
          images<<<<WOK
    <item id="#{image}" href="image/#{image}.#{type}" media-type="image/#{type}" />
WOK
        end
        images=images.join('')
        images
    end
    def manifest_close
      <<WOK
  </manifest>
WOK
      end
      def spine_open
        #spine: reading order of XHTML files from manifest, idref attribute refers back to id in manifest (exclude images, CSS etc.).
        <<WOK
  <spine toc="ncx">
WOK
      end
      def spine_sisu_toc
      <<WOK
    <itemref idref="index" linear="yes" />
WOK
      end
      def spine(dob,name=nil)
        name=name ? name : dob.name
        <<WOK
    <itemref idref="#{name}" linear="yes" />
WOK
      end
      def spine_close
        <<WOK
  </spine>
WOK
      end
      def guide_open
        #guide: presentation order of XHTML files by reader).
        <<WOK
  <guide>
WOK
      end
      def guide_sisu_toc
      <<WOK
    <reference type="index" href="index#{Sfx[:epub_xhtml]}" />
WOK
      end
      def guide(dob,name=nil)
        name=name ? name : dob.name
        <<WOK
    <reference type="text" href="#{name}#{Sfx[:epub_xhtml]}" />
WOK
      end
      def guide_close
        <<WOK
  </guide>
WOK
      end
      self
    end
    def toc_head_escript
    end
    def seg_head_escript
    end
    def table_close
      %{  </font>
#{@vz.table_close}}
    end
    def buttons_home
      %{<!- home and index buttons -!>
  #{@vz.banner_home_and_index_buttons}
<!- home and index buttons -!>}
    end
    def copyat
      %{<p class="tiny">copy @
  <a href="#{@vz.url_home}" #{@vz.js_home}>
    #{@vz.txt_home}
  </a></p>}
    end
    def xhtml_close
    %{#{@vz.credits_sisu_epub}
  </body>
</html>}
    end
  end
  class Widget < HeadInformation
    def initialize(md)
      super(md)
      @md=md
      @cf_defaults=SiSU_Env::InfoProcessingFlag.new
    end
    def home
      %{<td align="center" bgcolor=#{@vz.color_band2}>
  <a href="../index.html" target="_top">
  #{@vz.png_homepage}</a>
</td>
}
    end
    def search
      env=SiSU_Env::InfoEnv.new(@md.fns,@md)
      env.widget.search_form('sisusearch',nil,nil,true)
    end
    def manifest
      %{<td align="center" bgcolor=#{@vz.color_band2}>
  <a href="#{@md.fn[:manifest]}" target="_top" #{@vz.js_manifest}>
    #{@vz.nav_txt_manifest}
  </a>
</td>}
    end
  end
  class XML
  end
  class HeadToc < HeadInformation
    def initialize(md)
      super(md)
      @md=md
      @tocband_segtoc=make_seg
    end
    def concordance_navigation_band(type='')
      if type=~/pdf/
        @tocband_concordance=make_concordance
      end
      %{<p class="align_right">
  <a href="index.html" target="_top">
    #{@vz.epub_png_nav_toc}
  </a>
</p>
}
    end
    def seg_head_navigation_band
      firstseg=%{<a href="#{@md.firstseg}#{Sfx[:epub_xhtml]}" target="_top">
      #{@vz.epub_png_nav_nxt}
    </a>} if @md.firstseg =~/\S+/
      DISABLE[:epub][:internal_navigation] \
      ? ''
      : %{<p class="align_right">#{firstseg}</p>}
    end
    def seg_head_navigation_band_bottom
      firstseg=%{<a href="#{@md.firstseg}#{Sfx[:epub_xhtml]}" target="_top">
      #{@vz.epub_png_nav_nxt}
    </a>} if @md.firstseg =~/\S+/
      DISABLE[:epub][:internal_navigation] \
      ? ''
      : %{<p class="align_right">#{firstseg}</p>}
    end
    def manifest_link(text) #watch fix removed font size 2
  %{ <a href="#{@md.fn[:manifest]}" target="_top" #{@vz.js_manifest}>#{text}</a>}
    end
    def concordance_link(text) #watch fix removed font size 2
      if @md.concord_make
  %{<a href="#{@md.fn[:concordance]}" target="_top" #{@vz.js_concordance}>
      #{text}
    </a>}
      else ''
      end
    end
    def make_concordance
      manifest=scroll=seg=''
      wgt=SiSU_EPUB_Format::Widget.new(@md)
      %{<table summary="toc scroll and segment with pdf" border="0" cellpadding="3" cellspacing="0">
<tr>
  #{wgt.seg(@vz.nav_txt_toc_link)}#{wgt.scroll(@vz.nav_txt_doc_link)}#{wgt.pdf}#{wgt.odf}
<td align="center" bgcolor=#{@vz.color_band2}>
  #{wgt.concordance(@vz.nav_txt_concordance)}
  #{wgt.manifest}
  #{wgt.search}
  #{wgt.home}
</tr></table>}
    end
    def head
      %{#{doc_type}
<head>
<meta http-equiv="Content-Type" content="text/html;charset=utf-8" />
#{@css.xhtml_epub}
</head>
#{@vz.color_body}
#{@vz.js_top}}
    end
    def concordance
      if @md.concord_make
      %{#{@vz.margin_css}
  <h4 class="toc">
    <a href="./#{@md.fn[:concordance]}" #{@vz.js_concordance}>
      <i>Concordance</i>
    </a>
  </h4>
#{@vz.table_close}}
      else
      %{#{@vz.margin_css}
#{@vz.table_close}}
      end
    end
    def links_guide_open(type='horizontal')
      if type=='vertical'; links_guide_vertical_open
      else                 links_guide_horizontal_open
      end
    end
    def prefix_a
    end
    def rights
      def all
        rghts=@md.rights.all.gsub(/<br>/,'<br />')
        rghts=rghts.gsub(/^\s*Copyright\s+\(C\)/,'Copyright <sup>&copy;</sup>&nbsp;')
        %{<p class="small_left">Rights: #{rghts}</p>}
      end
      self
    end
    def prefix_b
      %{<p class="small_left">Prefix: #{@md.prefix_b}}
    end
    def make_seg
      concord=concordance_link(@vz.nav_txt_concordance)
      %{<table summary="toc segment" border="0" cellpadding="3" cellspacing="0">
<tr><td align="center" bgcolor="white">
  #{@vz.nav_txt_toc_link}
</td>
<td align="center" bgcolor="white">
  <font size=2>
   #{concord}
#{@vz.table_close}}
    end
    def manifest #check structure
      manifest=manifest_link(@vz.nav_txt_manifest)
      %{#{@vz.margin_txt_3}
  #{@vz.paragraph_font_small}
   #{manifest}
    </font>
#{@vz.table_close}}
    end
    def concordance #check structure
      concord=concordance_link(@vz.nav_txt_concordance)
      %{#{@vz.margin_txt_3}
  #{@vz.paragraph_font_small}
   #{concord}
    </font>
#{@vz.table_close}}
    end
    def metadata
      %{#{@vz.margin_css}
  <h4 class="toc">
    <a href="#{@metalink}" #{@vz.js_metalink}>
      <i>MetaData</i>
    </a>
  </h4>
#{@vz.table_close}}
    end
    def seg_metadata
      @metalink=%{./#{@md.fn[:metadata]}}
      metadata
    end
  end
  class HeadSeg < HeadInformation
    def initialize(md)
      super(md)
    end
    def head
      %{#{doc_type}
<head>
  <title>
    #{@seg_name_xhtml[@seg_name_xhtml_tracker]} -
    #{@md.html_title}
  </title>
<meta http-equiv="Content-Type" content="text/html;charset=utf-8" />
#{@vz.font_css_table_file}
</head>
#{@vz.color_body}
#{@vz.js_top}}
    end
    def title_banner(title,subtitle,creator)
    end
    def dot_control_pre_next
      %{<table summary="segment hidden control pre and next" width="100%" border="0" cellpadding="0" bgcolor=#{@vz.color_grey_pale} align="center">
<tr><td align="left">
  <a href="#{@seg_name_xhtml[@seg_name_xhtml_tracker-1]}#{Sfx[:epub_xhtml]}" target="_top" #{@vz.js_prev}>
    #{@vz.png_nav_dot_pre}
  </a>
</td>
<td align="center">
  <a href="#{@index}#{Sfx[:epub_xhtml]}" target="_top" #{@vz.js_toc}>
    #{@vz.png_nav_dot_toc}
  </a>
</td>
<td align="right">
  <a href="#{@seg_name_xhtml[@seg_name_xhtml_tracker+1]}#{Sfx[:epub_xhtml]}" target="_top" #{@vz.js_next}>
    #{@vz.png_nav_dot_nxt}
  </a>
#{@vz.table_close}}
    end
    def dot_control_pre
      %{<table summary="segment hidden control pre" width="100%" border="0" cellpadding="0" bgcolor=#{@vz.color_grey_pale} align="center">
<tr><td align="left">
  <a href="#{@seg_name_xhtml[@seg_name_xhtml_tracker-1]}#{Sfx[:epub_xhtml]}" target="_top" #{@vz.js_prev}>
    #{@vz.png_nav_dot_pre}
  </a>
</td>
<td align="center">
  <a href="#{@index}#{Sfx[:epub_xhtml]}" target="_top" #{@vz.js_toc}>
    #{@vz.png_nav_dot_toc}
  </a>
</td>
<td align="right">
  <a href="#{@md.fn[:toc]}" target="_top" #{@vz.js_next}>
    #{@vz.png_nav_dot_nxt}
  </a>
#{@vz.table_close}}
    end
    def toc_nav(f_pre=false,f_nxt=false,use=1)
      pre=nxt=''
      toc=%{<a href="#{@index}#{Sfx[:epub_xhtml]}" target="_top" #{@vz.js_toc}>
  #{@vz.epub_png_nav_toc}
</a>
}
      pre=%{<a href="#{@seg_name_xhtml[@seg_name_xhtml_tracker-use]}#{Sfx[:epub_xhtml]}" target="_top" #{@vz.js_prev}>
  #{@vz.epub_png_nav_pre}
</a>
} if f_pre==true
      nxt=%{<a href="#{@seg_name_xhtml[@seg_name_xhtml_tracker+1]}#{Sfx[:epub_xhtml]}" target="_top" #{@vz.js_next}>
    #{@vz.epub_png_nav_nxt}
  </a>
} if f_nxt==true
      DISABLE[:epub][:internal_navigation] \
      ? ''
      : %{<p class="align_right">
#{pre}
#{toc}
#{nxt}
</p>
}
    end
    def toc_next2
      pre,nxt=false,true
      toc_nav(false,true).dup
    end
    def toc_pre_next2
      toc_nav(true,true).dup
    end
    def toc_pre2
      toc_nav(true,false).dup
    end
    def manifest_link(text)
  %{<a href="#{@md.fn[:manifest]}" target="_top" #{@vz.js_manifest}>
      #{text}
    </a>}
    end
    def concordance_link(text)
      if @md.concord_make
  %{<a href="#{@md.fn[:concordance]}" target="_top" #{@vz.js_concordance}>
      #{text}
    </a>}
      else ''
      end
    end
    def doc_types #used in seg_nav_band ###
      scroll=seg=''
      wgt=SiSU_EPUB_Format::Widget.new(@md)
      x=if @md.concord_make
        %{
<table summary="segment navigation available documents types: toc,doc,pdf,concordance" border="0" cellpadding="3" cellspacing="0">
<tr>
  #{wgt.seg(@vz.nav_txt_toc_link)}
  #{wgt.scroll(@vz.nav_txt_doc_link)}
  #{wgt.pdf}#{wgt.odf}
<td align="center" bgcolor=#{@vz.color_band2}>
  #{wgt.concordance(@vz.nav_txt_concordance)}
  #{wgt.manifest}
  #{wgt.search}
  #{wgt.home}
</tr></table>}
      else
        %{
<table summary="segment navigation available documents types: toc,doc,pdf,concordance" border="0" cellpadding="3" cellspacing="0">
<tr>
  #{wgt.seg(@vz.nav_txt_toc_link)}
  #{wgt.scroll(@vz.nav_txt_doc_link)}
  #{wgt.pdf}#{wgt.odf}
<td align="center" bgcolor=#{@vz.color_band2}>
  #{wgt.manifest}
  #{wgt.search}
  #{wgt.home}
</tr></table>}
      end
    end
    def navigation_table
      %{<table summary="navigation segment table" width=#{@vz.table_width_1} border="0" bgcolor="white" cellpadding="0">
<tr><th width="#{@@indent['leve_1']}" align="right">
</td>
<td valign="top">
  <font size=2>}
    end
    def navigation_table1
      %{<table summary="navigation segment table1" width=#{@vz.table_width_1} border="0" cellpadding=#{@vz.table_cellpad_box} bgcolor=#{@vz.color_table1} align="left">
<tr><td valign="top">
  <font size="2">}
    end
    def navigation_table2
      %{<table summary="navigation segment table2" width=#{@vz.table_width_2} border="0" cellpadding=#{@vz.table_cellpad_box} bgcolor=#{@vz.color_table2} align="left">
<tr><td valign="top">
  <font size="2">}
    end
    def heading_advert_local_1
      dir=SiSU_Env::InfoEnv.new(@fns)
      %{  <p class="center">
<table summary="image" border="0" cellpadding="3" cellspacing="0" align="center">
<tr><td align="center" bgcolor="white">
  <a href="http://#{@md.ad_url}#{Sfx[:epub_xhtml]}" target="_top">
    <img border="0" src="#{dir.url.images_epub}/#{@md.ad_png}" alt="#{@md.ad_alt}" />
  </a>
#{@vz.table_close}
</p>}
    end
    def heading_advert_local_2
      dir=SiSU_Env::InfoEnv.new(@fns)
      %{  <p class="center">
<table summary="image" border="0" cellpadding="3" cellspacing="0" align="center">
<tr><td align="center" bgcolor="white">
  <a href="#{@md.ad_url}#{Sfx[:epub_xhtml]}" target="_top">
    <img border="0" src="#{dir.url.images_epub}/#{@md.ad_png}" alt="#{@md.ad_alt}" />
  </a>
#{@vz.table_close}
</p>}
    end
    def heading_advert_external
      dir=SiSU_Env::InfoEnv.new(@fns)
      %{  <p class="center">
<table summary="image" border="0" cellpadding="3" cellspacing="0" align="center">
<tr><td align="center" bgcolor="white">
  <a href="#{@md.ad_url}#{Sfx[:epub_xhtml]}" target="external">
    <img border="0" src="#{dir.url.images_epub}/#{@md.ad_png}" alt="#{@md.ad_alt}" />
  </a>
#{@vz.table_close}
</p>}
    end
    def navigation_band(segtocband) #change name to navigaion_band_banner
      %{#{segtocband}
}
    end
    def navigation_band_bottom(segtocband,seg_table_top_control) #change name to navigaion_band_bannerless
      %{
<div class="main_column">
  <table summary="segment navigation band" bgcolor=#{@vz.color_band1} width="100%"><tr>
  <td width="70%" align="center">
    #{doc_types}
  </td>
  <td width="30%" align="right">
    #{segtocband}
  </td></tr>
  </table>
  #{seg_table_top_control}
</div>
}
    end
    def endnote_mark
%{
  <hr class="endnote" />
}
    end
    def endnote_section_open
%{
<div class="endnote">
}
    end
    def endnote_section_close
%{
</div>
} #revisit
    end
    def head
      %{#{doc_type}
<head>
  <title>
    #{@seg_name_xhtml[@seg_name_xhtml_tracker]} -
    #{@md.html_title}
  </title>
<meta http-equiv="Content-Type" content="text/html;charset=utf-8" />
#{@css.xhtml_epub}
</head>
#{@vz.color_body}}
    end
    def toc_metadata
      @metalink=%{./#{@md.fn[:metadata]}}
      toc_metadata
    end
    def title_banner(title,subtitle,creator)
    end
  end
  class HeadScroll < HeadToc
    def initialize(md)
      super(md)
    end
    def toc_owner_details
      %{#{@vz.margin_txt_3}
#{@vz.paragraph_font_small}
  <a href="#owner.details">
    Owner Details
    <font size="1" color="#777777">
      &nbsp;&nbsp;&nbsp;
    </font>
  </a>
  </font>
#{@vz.table_close}}
    end
  end
  class FormatTextObject
    @vz=SiSU_Env::GetInit.instance.skin
    attr_accessor :md,:t_o,:txt,:ocn,:format,:table,:link,:linkname,:paranum,:p_num,:headname,:banner,:url
    def initialize(md,t_o)
      @md,@t_o=md,t_o
      if t_o.class==Hash
        @txt            =t_o[:txt]            || nil
        @ocn            =t_o[:ocn]            || nil
        @ocn_display    =t_o[:ocn_display]    || nil
        @headname       =t_o[:headname]       || nil
        @trailer        =t_o[:trailer]        || nil
        @endnote_part_a =t_o[:endnote_part_a] || nil
        @endnote_part_b =t_o[:endnote_part_b] || nil
        @lnk_url        =t_o[:lnk_url]        || nil
        @lnk_txt        =t_o[:lnk_txt]        || nil
        @format         =t_o[:format]         || nil
        @target         =t_o[:target]         || nil #occasionally passed but not used
        if @format and not @format.empty?
          if @format=~/^\d:(\S+)/ #need more reliable marker #if @format =~ /#{Rx[:lv]}/
            headname=$1 #format[/\d~(\S+)/m,1]
            @headname=if headname =~/^[a-zA-Z]/; %{<id="#{headname}">} #consider: h_#{headname}
            else %{<id="h#{headname}"></a>}
            end
            @headname=if headname =~/^[a-zA-Z]/; %{<a name="#{headname}" id="#{headname}"></a>} #consider: h_#{headname}
            else %{<a name="h#{headname}" id="h#{headname}"></a>}
            end
          end
        end
      elsif t_o.class.inspect =~/Object/
        @dob=t_o if defined? t_o.is
        @named=nametags_seg(@dob)
        @txt=((defined? t_o.obj) ? t_o.obj : nil)
        @ocn=((defined? t_o.ocn) ? t_o.ocn.to_s : nil)
        @headname=((t_o.is=='heading' and defined? t_o.name) ? t_o.name : nil)
      else
        if @md.opt.cmd =~/M/
          p __FILE__ +':'+ __LINE__.to_s
          p t_o.class
          p caller
        end
      end
      if @txt and not @txt.empty?
        @txt=@txt.gsub(/#{Mx[:mk_o]}[-~]##{Mx[:mk_c]}/,'')
      end
      @p_num=ParagraphNumber.new(@md,@ocn)
      @vz=SiSU_Env::GetInit.instance.skin
    end
    def nametags_seg(dob) #FIX
      tags=''
      if defined? dob.tags \
      and dob.tags.length > 0 # insert tags "hypertargets"
        dob.tags.each do |t|
          tags=tags +%{<a name="#{t}" />}
        end
      end
      tags
    end
    def endnote_body
      %{
<p class="endnote">
  #{@txt}
</p>
}
    end
    def endnote_body_indent
      %{
  <p class="endnote_indent">
    #{@txt}
  </p>
}
    end
    def no_paranum
      %{
<div class="substance">
  <label class="ocn">&nbsp;</label>
  <p class="norm">
    #{@txt}
  </p>
</div>
}
    end
    def para_form_css(tag,attrib,txt)                                                    # regular paragraphs shaped here
      ul=ulc=''
      ul,ulc="<ul>\n  ","\n  </ul>" if @tag =~/li/
      %{
<div class="substance">
  #{@p_num.ocn_display}
  #{ul}<#{tag} class="#{attrib}" #{@p_num.id}>
    #{@named}#{txt}
  </#{tag}>#{ulc}
</div>
}
    end
    def para
      para_form_css('p','norm',@txt)
    end
    def group
      para_form_css('p','group',@txt)
    end
    def block
      para_form_css('p','block',@txt)
    end
    def alt
      para_form_css('p','alt',@txt)
    end
    def verse
      para_form_css('p','verse',@txt)
    end
    def code
      para_form_css('p','code',@txt)
    end
    def center
      para_form_css('p','center',@txt)
    end
    def bold
      para_form_css('p','bold',@txt)
    end
    def bullet
      para_form_css('li','bullet',@txt)
    end
    def table
      @txt=if @t_o.obj !~/^<table\s/
        table=SiSU_XHTML_Shared::TableXHTML.new(@t_o) #move, make happen earlier
        table.table.obj
      else @txt
      end
      para_form_css('p','norm',@txt)
    end
    def break
      @txt=@txt.gsub(/#{Mx[:br_page_new]}|#{Mx[:br_page]}/,'<hr /><br />').
        gsub(/#{Mx[:br_obj]}/,'<hr style="width:30%" /><br />')
      para_form_css('p','norm',@txt)
    end
    def format(tag,attrib)
      para_form_css(tag,attrib,@txt)
    end
    def title_heading(tag,attrib)
      %{
<div class="content">
<#{tag} class="#{attrib}">
    #{@named}#{@txt}
  </#{tag}>
</div>
}
    end
    def title_heading1
      id_u=DISABLE[:epub][:per_section_title] \
      ? ''
      : title_heading('h1','tiny')
    end
    def title_heading2
      id_u=DISABLE[:epub][:per_section_title] \
      ? ''
      : title_heading('h2','tiny')
    end
    def title_heading3
      id_u=DISABLE[:epub][:per_section_title] \
      ? ''
      : title_heading('h3','tiny')
    end
    def title_heading4
      ''
    end
    def seg_heading_sub(tag,attrib,txt)
      txt=txt.gsub(/(?:#{Mx[:en_a_o]}.+?#{Mx[:en_a_c]}|#{Mx[:en_b_o]}.+?#{Mx[:en_b_c]})\s*/m,' ')
      %{
<div class="substance">
  #{@p_num.ocn_display}
  <#{tag} class="#{attrib}" #{@p_num.id}>#{@p_num.name}
    #{@named}#{@txt}
  </#{tag}>
</div>
}
    end
    def seg_heading4
      %{
<div class="substance">
  #{@p_num.ocn_display}
  <h1 class="norm" #{@p_num.id}>
    #{@txt}
  </h1>
</div>
}
    end
    def seg_heading5
      seg_heading_sub('p','bold',@txt)
    end
    def seg_heading6
      seg_heading_sub('p','bold',@txt)
    end
    def dl #check :trailer
      "<dl><b>#{@txt}</b> #{@trailer}</dl>"
    end
    def table_css_end
      '</table>
    </p>
  </div>'
    end
    def gsub_body #unused
      @txt=case @txt
      when /^(?:#{Mx[:pa_o]}i[1-9]#{Mx[:pa_c]}\s*)?\((i+|iv|v|vi+|ix|x|xi+)\)/
        @txt.gsub(/^\((i+|iv|v|vi+|ix|x|xi+)\)/,'<b>(\1)</b>').
          gsub(/^(#{Mx[:pa_o]}i[1-9]#{Mx[:pa_c]})\s*\((i+|iv|v|vi+|ix|x|xi+)\)/,'\1<b>(\2)</b>')
      when /^(?:#{Mx[:pa_o]}i[1-9]#{Mx[:pa_c]}\s*)?\(?(\d|[a-z])+\)/
        @txt.gsub(/^\((\d+|[a-z])+\)/,'<b>(\1)</b>').
          gsub(/^(#{Mx[:pa_o]}i[1-9]#{Mx[:pa_c]})\s*\((\d+|[a-z])+\)/,'\1<b>(\2)</b>')
      when /^\s*\d{1,3}\.\s/
        @txt.gsub(/^\s*(\d+\.)/,'<b>\1</b>')
      when /^\s*[A-Z]\.\s/
        @txt.gsub(/^\s*([A-Z]\.)/,'<b>\1</b>')
      else @txt
      end
    end
    def bold_para
      %{#{@vz.margin_txt_0}
  <p class="bold">
    #{@txt}
  </p>
#{@vz.margin_num_css}
  &nbsp;&nbsp;&nbsp;
#{@vz.table_close}}
    end
    def bold_heading #unused
      @txt=@txt.gsub(/[1-9]~\S+/,'').
        gsub(/[1-9]~/,'')
      %{<p class="bold">
    #{@txt}
  </p>
#{@vz.margin_num_css}
  &nbsp;&nbsp;&nbsp;
#{@vz.table_close}}
    end
    def toc_head_copy_at
      %{<p class="center">#{@txt}</p>\n}
    end
    def center
      %{<p class="center">#{@txt}</p>\n}
    end
    def bold
      %{<p class="bold">#{@txt}</p>\n}
    end
    def center_bold
      %{<p class="centerbold">#{@txt}</p>\n}
    end
  end
  class FormatScroll < FormatTextObject
    def initialize(md,txt)
      super(md,txt)
      @vz=SiSU_Env::GetInit.instance.skin
    end
  end
  class FormatSeg < FormatTextObject
    def initialize(md,txt)
      super(md,txt)
    end
    def navigation_toc_lev1_advert
      %{#{@banner.home_button}\n
<p class="center">
#{@txt}
#{@two}
</a></p>}
    end
    def navigation_toc_lev1
      %{#{@banner.nav_toc}}
    end
    def navigation_toc_lev2                                                      #change bold use css
      %{<table summary="navigation segment level 2">
<tr><td width ="20">
</td>
<td>
  <font size="3" #{@vz.font_face}>
    <b>#{@txt}</b>
  </font>
  </p>
#{@vz.table_close}}
    end
    def navigation_toc_lev3                                                      #change bold use css
      %{<table summary="navigation segment level 3">
<tr><td width ="20">
</td>
<td>
  <font size="3" #{@vz.font_face}>
    <b>#{@txt}</b>
  </font>
  </p>
#{@vz.table_close}}
    end
    def navigation_toc_lev4
      %{<table summary="navigation segment level 4">
<tr><td width ="80">
</td>
<td>
<p>
  #{@txt}
</p>
#{@vz.table_close}}
    end
    def navigation_toc_lev5
    end
    def navigation_toc_lev6
    end
    def endnote_seg_body(fn='')  #FIX                                                #url construction keep within single line... BUG WATCH 200408
      fn='doc' if fn.to_s.empty? #you may wish to reconsider, sends to 'doc' where no segment info
      %{
  <p class="endnote">
    #{@endnote_part_a}#{fn}#{Sfx[:epub_xhtml]}#{@endnote_part_b}
  </p>
}
    end
    def clean(txt)
      txt=txt.gsub(/#{Mx[:en_a_o]}.+?#{Mx[:en_a_c]}/,'').
        gsub(/#{Mx[:en_b_o]}.+?#{Mx[:en_b_c]}/,'')
    end
    def subtoc_lev(tag,attrib)
      @txt=clean(@txt)
      txt=if @txt \
      and @txt =~/<\/?i>|<a\s+name="\S+?">/mi
        @txt.gsub(/<\/?i>|<a\s+name="\S+?">/mi,'') #removes name markers from subtoc, go directly to substantive text
      else @txt
      end
      note=''
      if txt =~/(#{Mx[:en_a_o]}.+?#{Mx[:en_a_c]}|#{Mx[:en_b_o]}.+?#{Mx[:en_b_c]})/m # had \s* at end
        note=$1
        note=note.gsub(/[\n\s]+/m,' ')
        txt=txt.gsub(/(?:#{Mx[:en_a_o]}.+?#{Mx[:en_a_c]}|#{Mx[:en_b_o]}.+?#{Mx[:en_b_c]})\s*/m,' ').
          gsub(/<a[\n\s]+"[\n\s]+href="#note_ref\d+">&nbsp;<sup id="note\d+">\d+<\/sup>&nbsp;/m,'').
          gsub(/<a[\n\s]+"[\n\s]+href="#note_ref\d+">#{Mx[:nbsp]}<sup id="note\d+">\d+<\/sup>#{Mx[:nbsp]}/m,'') #remove
      end
      %{<#{tag} class="#{attrib}">
    <a href="#o#{@ocn}"><i>#{txt}</i></a> #{note}
  </#{tag}>}
    end
    def subtoc_lev5
      subtoc_lev('h5','subtoc') if @txt
    end
    def subtoc_lev6
      subtoc_lev('h6','subtoc') if @txt
    end
    def heading_sub(tag,attrib,txt)
      txt=txt.gsub(/(?:#{Mx[:en_a_o]}.+?#{Mx[:en_a_c]}|#{Mx[:en_b_o]}.+?#{Mx[:en_b_c]})\s*/m,' ')
      %{
<div class="substance">
  #{@p_num.ocn_display}
  <#{tag} class="#{attrib}" #{@p_num.id}> #{@headname}
    #{@txt}
  </#{tag}>
</div>
}
    end
    def heading4
      %{
<div class="substance">
  #{@p_num.ocn_display}
  <h1 class="norm" #{@p_num.id}>
    #{@t_o[:format]}
    #{@txt}
  </h1>
</div>
}
    end
    def heading5
      heading_sub('p','bold',@txt)
    end
    def heading6
      heading_sub('p','bold',@txt)
    end
    def navigation_heading4
      %{<table summary="navigation segment heading 4" width=100% bgcolor="#08163f" border="0">
<tr><td align="center">
<p class="bold">
  #{@txt}
</p>
#{@vz.table_close}}
    end
    def navigation_heading5
      %{<p class="bold">
  #{@txt}
</p>}
    end
    def navigation_heading6
      %{<p class="bold">
  #{@txt}
</p>}
    end
    def navigation_center
      %{<p class="centerbold">#{@txt}</p>}
    end
  end
  class FormatToc < FormatTextObject
    def initialize(md,txt)
      super(md,txt)
    end
    def links_guide
      %{  <li class="doc">
    <a href="#{@lnk_url}" target="_top">
      #{@lnk_txt}
    </a>
  </li>
}
    end
    def lev(tag,attrib)
      if @txt
        %{<#{tag} class="#{attrib}">
    #{@txt}
  </#{tag}>
}
      else ''
      end
    end
    def lev1
      lev('h1','toc')
    end
    def lev2
      lev('h2','toc')
    end
    def lev3
      lev('h3','toc')
    end
    def lev4
      lev('h4','toc')
    end
    def lev5
      lev('h5','toc')
    end
    def lev6
      lev('h6','toc')
    end
    def lev0 #docinfo
      lev('h0','toc')
    end
    def mini_lev1
      lev('h1','minitoc')
    end
    def mini_lev2
      lev('h2','minitoc')
    end
    def mini_lev3
      lev('h3','minitoc')
    end
    def mini_lev4
      lev('h4','minitoc')
    end
    def mini_lev5
      lev('h5','minitoc')
    end
    def mini_lev6
      lev('h6','minitoc')
    end
    def mini_lev0 #docinfo
      lev('h0','minitoc')
    end
    def mini_tail
  %{
  <h4 class="minitoc">
    <a href="sisu_manifest.html">Manifest (alternative outputs)</a>
  </h4>
}
    end
    def mini_concord_tail
  %{
  <h4 class="minitoc">
    <a href="concordance.html">Concordance (wordlist)</a>
  </h4>
  <h4 class="minitoc">
    <a href="sisu_manifest.html">Manifest (alternative outputs)</a>
  </h4>
}
    end
  end
end
__END__
