# encoding: utf-8
=begin

 * Name: SiSU

 * Description: a framework for document structuring, publishing and search

 * Author: Ralph Amissah

 * Copyright: (C) 1997 - 2012, Ralph Amissah, All Rights Reserved.

 * License: GPL 3 or later:

   SiSU, a framework for document structuring, publishing and search

   Copyright (C) Ralph Amissah

   This program is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation, either version 3 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program. If not, see <http://www.gnu.org/licenses/>.

   If you have Internet connection, the latest version of the GPL should be
   available at these locations:
   <http://www.fsf.org/licensing/licenses/gpl.html>
   <http://www.gnu.org/copyleft/gpl.html>

   <http://www.jus.uio.no/sisu/gpl.fsf/toc.html>
   <http://www.jus.uio.no/sisu/gpl.fsf/doc.html>
   <http://www.jus.uio.no/sisu/gpl.fsf/plain.txt>

 * SiSU uses:
   * Standard SiSU markup syntax,
   * Standard SiSU meta-markup syntax, and the
   * Standard SiSU object citation numbering and system

 * Hompages:
   <http://www.jus.uio.no/sisu>
   <http://www.sisudoc.org>

 * Download:
   <http://www.jus.uio.no/sisu/SiSU/download.html>

 * Ralph Amissah
   <ralph@amissah.com>
   <ralph.amissah@gmail.com>

 ** Description: system environment, resource control and configuration details

=end
module SiSU_Metadata
  class Summary
    attr_accessor :tag,:inf,:class,:attrib
    def initialize(md,display_heading=false)
      @md,@display_heading=md,display_heading
      @tag,@inf,@class,@attrib=nil
    end
    def metadata_base
      meta,tag=[],[]
      dir=SiSU_Env::InfoEnv.new(@md.fns)
      base_html="#{dir.url.root}/#{@md.fnb}"
      l=SiSU_Env::StandardiseLanguage.new(@md.opt.lng).language
      language=l[:n]
      tr=SiSU_Translate::Source.new(@md,language)
      @attrib='md'
      if @display_heading
        @tag,@inf=%{<b><u>Document Metadata</u></b>},''
        meta << self.meta_para
      end
      if defined? @md.title.full \
      and @md.title.full=~/\S+/
        @tag,@inf,@class=tr.full_title,@md.title.full,'dc' #1
        meta << self.meta_para
      end
      if defined? @md.creator.author \
      and @md.creator.author=~/\S+/
        @tag,@inf,@class=tr.author,@md.creator.author,'dc' #2
        meta << self.meta_para
      end
      if defined? @md.creator.translator \
      and @md.creator.translator=~/\S+/
        @tag,@inf,@class=tr.translator,@md.creator.translator,'ext'
        meta << self.meta_para
      end
      if defined? @md.creator.illustrator \
      and @md.creator.illustrator=~/\S+/
        @tag,@inf,@class=tr.illustrator,@md.creator.illustrator,'ext'
        meta << self.meta_para
      end
      if defined? @md.creator.prepared_by \
      and @md.creator.prepared_by=~/\S+/
        @tag,@inf,@class=tr.prepared_by,@md.creator.prepared_by,'ext'
        meta << self.meta_para
      end
      if defined? @md.creator.digitized_by \
      and @md.creator.digitized_by=~/\S+/
        @tag,@inf,@class=tr.digitized_by,@md.creator.digitized_by,'ext'
        meta << self.meta_para
      end
      if defined? @md.rights.all \
      and @md.rights.all=~/\S+/
        @tag,@inf,@class=tr.rights,@md.rights.all,'dc' #15
        meta << self.meta_para
      end
      if defined? @md.classify.subject \
      and @md.classify.subject=~/\S+/
        @tag,@inf,@class=tr.subject,@md.classify.subject,'dc' #3
        meta << self.meta_para
      end
      if defined? @md.notes.description \
      and @md.notes.description=~/\S+/
        @tag,@inf,@class=tr.description,@md.notes.description,'dc' #4
        meta << self.meta_para
      end
      if defined? @md.publisher \
      and @md.publisher=~/\S+/
        @tag,@inf,@class=tr.publisher,@md.publisher,'dc' #5
        meta << self.meta_para
      end
      if defined? @md.creator.contributor \
      and @md.creator.contributor=~/\S+/
        @tag,@inf,@class=tr.contributor,@md.creator.contributor,'dc' #6
        meta << self.meta_para
      end
      if defined? @md.notes.abstract \
      and @md.notes.abstract=~/\S+/
        @tag,@inf,@class=tr.abstract,@md.notes.abstract,'ext'
        meta << self.meta_para
      end
      if defined? @md.date.created \
      and @md.date.created=~/\S+/
        @tag,@inf,@class=tr.date_created,@md.date.created,'dc' #7
        meta << self.meta_para
      end
      if defined? @md.date.issued \
      and @md.date.issued=~/\S+/
        @tag,@inf,@class=tr.date_issued,@md.date.issued,'dc' #7
        meta << self.meta_para
      end
      if defined? @md.date.available \
      and @md.date.available=~/\S+/
        @tag,@inf,@class=tr.date_available,@md.date.available,'dc' #7
        meta << self.meta_para
      end
      if defined? @md.date.modified \
      and @md.date.modified=~/\S+/
        @tag,@inf,@class=tr.date_modified,@md.date.modified,'dc' #7
        meta << self.meta_para
      end
      if defined? @md.date.valid \
      and @md.date.valid=~/\S+/
        @tag,@inf,@class=tr.date_valid,@md.date.valid,'dc' #7
        meta << self.meta_para
      end
      if defined? @md.date.published \
      and @md.date.published=~/\S+/
        @tag,@inf,@class=tr.date,@md.date.published,'dc' #7
        meta << self.meta_para
      end
      if defined? @md.type \
      and @md.date=~/\S+/
        @tag,@inf,@class=tr.type,@md.type,'dc' #8
        meta << self.meta_para
      end
      if defined? @md.classify.loc \
      and @md.classify.loc=~/\S+/
        @tag,@inf,@class=tr.cls_loc,@md.classify.loc,'id'
        meta << self.meta_para
      end
      if defined? @md.classify.dewey \
      and @md.classify.dewey=~/\S+/
        @tag,@inf,@class=tr.cls_dewey,@md.classify.dewey,'id'
        meta << self.meta_para
      end
      if defined? @md.classify.pg \
      and @md.classify.pg=~/\S+/
        @tag,@inf,@class=tr.cls_gutenberg,@md.classify.pg,'id'
        meta << self.meta_para
      end
      if defined? @md.classify.isbn \
      and @md.classify.isbn=~/\S+/
        @tag,@inf,@class=tr.cls_isbn,@md.classify.isbn,'id'
        meta << self.meta_para
      end
      if defined? @md.notes.comment \
      and @md.notes.comment=~/\S+/
        @tag,@inf,@class=tr.comments,@md.notes.comment,'ext'
        meta << self.meta_para
      end
      if defined? @md.notes.prefix_a \
      and @md.notes.prefix_a=~/\S+/
        @tag,@inf,@class=tr.prefix_a,@md.notes.prefix_a,'inf'
        meta << self.meta_para
      end
      if defined? @md.notes.prefix_b \
      and @md.notes.prefix_b=~/\S+/
        @tag,@inf,@class=tr.prefix_b,@md.notes.prefix_b,'inf'
        meta << self.meta_para
      end
      if defined? @md.classify.identifier \
      and @md.classify.identifier=~/\S+/
        @tag,@inf,@class=tr.identifier,@md.classify.identifier,'dc' #10
        meta << self.meta_para
      end
      if defined? @md.original.source \
      and @md.original.source=~/\S+/
        @tag,@inf,@class=tr.source,@md.original.source,'dc' #11
        meta << self.meta_para
      end
      if defined? @md.title.language \
      and @md.title.language=~/\S+/
        @tag,@inf,@class=tr.language,@md.title.language,'dc' #12
        meta << self.meta_para
      end
      if defined? @md.original.language \
      and @md.original.language=~/\S+/
        @tag,@inf,@class=tr.language_original,@md.original.language,'ext'
        meta << self.meta_para
      end
      if defined? @md.classify.format \
      and @md.classify.format=~/\S+/
        @tag,@inf,@class=tr.format,@md.classify.format,'dc' #9
        meta << self.meta_para
      end
      if defined? @md.classify.relation \
      and @md.classify.relation=~/\S+/
        @tag,@inf,@class=tr.relation,@md.classify.relation,'dc' #13
        meta << self.meta_para
      end
      if defined? @md.classify.coverage \
      and @md.classify.coverage=~/\S+/
        @tag,@inf,@class=tr.coverage,@md.classify.coverage,'dc' #14
        meta << self.meta_para
      end
      if defined? @md.classify.keywords \
      and @md.classify.keywords=~/\S+/
        @tag,@inf,@class=tr.keywords,@md.classify.keywords,'ext'
        meta << self.meta_para
      end
      if @display_heading
        @tag,@inf=%{<b><u>Version Information</u></b>},''
        meta << self.meta_para
      end
      if defined? @md.fns \
      and @md.fns=~/\S+/
        @tag,@inf,@class=tr.sourcefile,@md.fns,'src'
        meta << self.meta_para
      end
      if defined? @md.file_encoding \
      and @md.file_encoding=~/\S+/
        @tag,@inf,@class='Filetype',@md.file_encoding,'src'
        meta << self.meta_para
      end
      if defined? @md.dgst \
      and @md.dgst.class==Array
        @tag,@inf,@class='Source Digest',"#{@md.dgst[0]} #{@md.dgst[1]}",'src'
        meta << self.meta_para
      end
      if defined? @md.dgst_skin \
      and @md.dgst_skin.class==Array
        @tag,@inf,@class='Skin Digest',"#{@md.dgst_skin[0]} #{@md.dgst_skin[1]}",'src'
        meta << self.meta_para
      end
      if @display_heading
        @tag,@inf=%{<b><u>Generated</u></b>},''
        meta << self.meta_para
      end
      if defined? @md.sisu_version \
      and @md.sisu_version[:version]=~/\S+/
        v="#{tr.sisu_version}: " +
          "#{@md.sisu_version[:project]} " +
          "#{@md.sisu_version[:version]} " +
          "of #{@md.sisu_version[:date_stamp]} " +
          "(#{@md.sisu_version[:date]})"
        @tag,@inf,@class='Generated by',v,'ver'
        meta << self.meta_para
      end
      if defined? @md.ruby_version \
      and @md.ruby_version=~/\S+/
        @tag,@inf,@class=tr.ruby_version,@md.ruby_version,'ver'
        meta << self.meta_para
      end
      if defined? @md.generated \
      and @md.generated.class==Time
        @tag,@inf,@class=tr.last_generated,@md.generated,'date'
        meta << self.meta_para
      end
      meta
    end
    def processing_tags
      def make
        def language
          if defined? @md.make.language \
          and @md.make.language
            ' :language: ' + @md.make.language.join(', ')
          else nil
          end
        end
        def headings
          if defined? @md.make.headings \
          and @md.make.headings
            ' :headings: ' + @md.make.headings[0].join('; ')
          else nil
          end
        end
        def num_top
          if defined? @md.make.num_top \
          and @md.make.num_top
            ' :num_top: ' + @md.make.num_top
          else nil
          end
        end
        def breaks
          x=if defined? @md.make.breaks \
            and @md.make.breaks
            x=' :breaks:'
            if @md.make.breaks[:page_break]
              x +=' break=' + @md.make.breaks[:page_break] + ';'
            end
            if @md.make.breaks[:page_new]
              x +=' new=' + @md.make.breaks[:page_new] + ';'
            end
          else nil
          end
        end
        def emphasis
          if defined? @md.make.emphasis \
          and @md.make.emphasis
            ' :emphasis: ' + @md.make.emphasis[:regx].inspect
          else nil
          end
        end
        def bold
          if defined? @md.make.bold \
          and @md.make.bold
            ' :bold: ' + @md.make.bold[:regx].inspect
          else nil
          end
        end
        def italics
          if defined? @md.make.italics \
          and  @md.make.italics
            ' :italics: ' + @md.make.italics[:regx].inspect
          else nil
          end
        end
        def texpdf_font
          if defined? @md.make.texpdf_font \
          and @md.make.texpdf_font
            ' :texpdf_font: ' + @md.make.texpdf_font
          else nil
          end
        end
        def skin
          if defined? @md.make.skin \
          and @md.make.skin
            ' :skin: ' + @md.make.skin
          else nil
          end
        end
        self
      end
      self
    end
    def metadata_tags
      def title
        def main
          if defined? @md.title.main \
          and @md.title.main
            '@title: ' + @md.title.main
          else '@title:'
          end
        end
        def sub
          if defined? @md.title.sub \
          and @md.title.sub
            ' :subtitle: ' + @md.title.sub
          else nil
          end
        end
        def edition
          if defined? @md.title.edition \
          and @md.title.edition
            ' :edition: ' + @md.title.edition
          else nil
          end
        end
        def note
          if defined? @md.title.note \
          and @md.title.note
            ' :note: ' + @md.title.note
          else nil
          end
        end
        def short
          if defined? @md.title.short \
          and @md.title.short
            ' :short: ' + @md.title.short
          else nil
          end
        end
        def language
          if defined? @md.title.language \
          and @md.title.language
            ' :language: ' + @md.title.language.join(', ')
          else nil
          end
        end
        def language_char
          if defined? @md.title.language_char \
          and @md.title.language_char
            ' :language_char: ' + @md.title.language_char
          else nil
          end
        end
        self
      end
      def creator
        def head
          '@creator:'
        end
        def author
          x=if defined? @md.creator.author_detail \
            and @md.creator.author_detail
            x=''
            @md.creator.author_detail.each do |n|
              x += "#{n[:the]}, #{n[:others]}; "
            end
            x.gsub!(/;\s*$/,'')
            ' :author: ' + x
          else nil
          end
        end
        def contributor
          x=if defined? @md.creator.contributor_detail \
            and @md.creator.contributor_detail
            x=''
            @md.creator.contributor_detail.each do |n|
              x += "#{n[:the]}, #{n[:others]}; "
            end
            x.gsub!(/;\s*$/,'')
            ' :contributor: ' + x
          else nil
          end
        end
        def illustrator
          x=if defined? @md.creator.illustrator_detail \
            and @md.creator.illustrator_detail
            x=''
            @md.creator.illustrator_detail.each do |n|
              x += "#{n[:the]}, #{n[:others]}; "
            end
            x.gsub!(/;\s*$/,'')
            ' :illustrator: ' + x
          else nil
          end
        end
        def photographer
          x=if defined? @md.creator.photographer_detail \
            and @md.creator.photographer_detail
            x=''
            @md.creator.photographer_detail.each do |n|
              x += "#{n[:the]}, #{n[:others]}; "
            end
            x.gsub!(/;\s*$/,'')
            ' :photographer: ' + x
          else nil
          end
        end
        def translator
          x=if defined? @md.creator.translator_detail \
            and @md.creator.translator_detail
            x=''
            @md.creator.translator_detail.each do |n|
              x += "#{n[:the]}, #{n[:others]}; "
            end
            x.gsub!(/;\s*$/,'')
            ' :translator: ' + x
          else nil
          end
        end
        def audio
          x=if defined? @md.creator.audio_detail \
            and @md.creator.audio_detail
            x=''
            @md.creator.audio_detail.each do |n|
              x += "#{n[:the]}, #{n[:others]}; "
            end
            x.gsub!(/;\s*$/,'')
            ' :audio: ' + x
          else nil
          end
        end
        def digitized_by
          x=if defined? @md.creator.digitized_by_detail \
            and @md.creator.digitized_by_detail
            x=''
            @md.creator.digitized_by_detail.each do |n|
              x += "#{n[:the]}, #{n[:others]}; "
            end
            x.gsub!(/;\s*$/,'')
            ' :digitized_by: ' + x
          else nil
          end
        end
        def prepared_by
          x=if defined? @md.creator.prepared_by_detail \
            and @md.creator.prepared_by_detail
            x=''
            @md.creator.prepared_by_detail.each do |n|
              x += "#{n[:the]}, #{n[:others]}; "
            end
            x.gsub!(/;\s*$/,'')
            ' :prepared_by: ' + x
          else nil
          end
        end
        self
      end
      def rights
        def head
          '@rights:'
        end
        def copyright
          def text
            if defined? @md.rights.copyright.text \
            and @md.rights.copyright.text
              ' :copyright: ' + @md.rights.copyright.text
            else nil
            end
          end
          def translation
            if defined? @md.rights.copyright.translation \
            and @md.rights.copyright.translation
              ' :translation: ' + @md.rights.copyright.translation
            else nil
            end
          end
          def illustrations
            if defined? @md.rights.copyright.illustrations \
            and @md.rights.copyright.illustrations
              ' :illustrations: ' + @md.rights.copyright.illustrations
            else nil
            end
          end
          def photographs
            if defined? @md.rights.copyright.photographs \
            and @md.rights.copyright.photographs
              ' :photographs: ' + @md.rights.copyright.photographs
            else nil
            end
          end
          def digitization
            if defined? @md.rights.copyright.digitization \
            and @md.rights.copyright.digitization
              ' :digitization: ' + @md.rights.copyright.digitization
            else nil
            end
          end
          def audio
            if defined? @md.rights.copyright.audio \
            and @md.rights.copyright.audio
              ' :audio: ' + @md.rights.copyright.audio
            else nil
            end
          end
          self
        end
        def license
          if defined? @md.rights.license \
          and @md.rights.license
            ' :license: ' + @md.rights.license
          else nil
          end
        end
        self
      end
      def classify
        def head
          '@classify:'
        end
        def coverage
          if defined? @md.classify.coverage \
          and @md.classify.coverage
            ' :coverage: ' + @md.classify.coverage
          else nil
          end
        end
        def relation
          if defined? @md.classify.relation \
          and @md.classify.relation
            ' :relation: ' + @md.classify.relation
          else nil
          end
        end
        def subject
          if defined? @md.classify.subject \
          and @md.classify.subject
            ' :subject: ' + @md.classify.subject
          else nil
          end
        end
        def topic_register
          if defined? @md.classify.topic_register \
          and @md.classify.topic_register
            ' :topic_register: ' + @md.classify.topic_register
          else nil
          end
        end
        def type
#         if defined? @md.classify.type \
#         and @md.classify.type
#           ' :type: ' + @md.classify.type
#         else nil
#         end
          nil
        end
        def identifier
          if defined? @md.classify.identifier \
          and @md.classify.identifier
            ' :identifier: ' + @md.classify.identifier
          else nil
          end
        end
        def loc
          if defined? @md.classify.loc \
          and @md.classify.loc
            ' :loc: ' + @md.classify.loc
          else nil
          end
        end
        def dewey
          if defined? @md.classify.dewey \
          and @md.classify.dewey
            ' :dewey: ' + @md.classify.dewey
          else nil
          end
        end
        def oclc
          if defined? @md.classify.oclc \
          and @md.classify.oclc
            ' :oclc: ' + @md.classify.oclc
          else nil
          end
        end
        def pg
          if defined? @md.classify.pg \
          and @md.classify.pg
            ' :pg: ' + @md.classify.pg
          else nil
          end
        end
        def isbn
          if defined? @md.classify.isbn \
          and @md.classify.isbn
            ' :isbn: ' + @md.classify.isbn
          else nil
          end
        end
        self
      end
      def date
        def head
          '@date:'
        end
        def added_to_site
          if defined? @md.date.added_to_site \
          and @md.date.added_to_site
            ' :added_to_site: ' + @md.date.added_to_site
          else nil
          end
        end
        def available
          if defined? @md.date.available \
          and @md.date.available
            ' :available: ' + @md.date.available
          else nil
          end
        end
        def created
          if defined? @md.date.created \
          and @md.date.created
            ' :created: ' + @md.date.created
          else nil
          end
        end
        def issued
          if defined? @md.date.issued \
          and @md.date.issued
            ' :issued: ' + @md.date.issued
          else nil
          end
        end
        def modified
          if defined? @md.date.modified \
          and @md.date.modified
            ' :modified: ' + @md.date.modified
          else nil
          end
        end
        def published
          if defined? @md.date.published \
          and @md.date.published
            ' :published: ' + @md.date.published
          else nil
          end
        end
        def valid
          if defined? @md.date.valid \
          and @md.date.valid
            ' :valid: ' + @md.date.valid
          else nil
          end
        end
        self
      end
      #def make
      #  def headings
      #    @md.make.headings \
      #    ? (' :headings: ' + @md.make.headings) \
      #    : nil
      #  end
      #end
      self
    end
    def char_enc(str)
      @s=str
      def utf8
        if @s \
        and @s.class==String
          @s.gsub!(/<br(?: \/)?>/u,Mx[:br_paragraph])
          @s.gsub!(/</um,'&#60;') # '&lt;'   # &#060;
          @s.gsub!(/</um,'&#60;') # '&lt;'   # &#060;
          @s.gsub!(/>/um,'&#62;') # '&gt;'   # &#062;
          @s.gsub!(/¢/um,'&#162;') # '&cent;'   # &#162;
          @s.gsub!(/£/um,'&#163;') # '&pound;'  # &#163;
          @s.gsub!(/¥/um,'&#165;') # '&yen;'    # &#165;
          @s.gsub!(/§/um,'&#167;') # '&sect;'   # &#167;
          @s.gsub!(/©/um,'&#169;') # '&copy;'   # &#169;
          @s.gsub!(/ª/um,'&#170;') # '&ordf;'   # &#170;
          @s.gsub!(/«/um,'&#171;') # '&laquo;'  # &#171;
          @s.gsub!(/®/um,'&#174;') # '&reg;'    # &#174;
          @s.gsub!(/°/um,'&#176;') # '&deg;'    # &#176;
          @s.gsub!(/±/um,'&#177;') # '&plusmn;' # &#177;
          @s.gsub!(/²/um,'&#178;') # '&sup2;'   # &#178;
          @s.gsub!(/³/um,'&#179;') # '&sup3;'   # &#179;
          @s.gsub!(/µ/um,'&#181;') # '&micro;'  # &#181;
          @s.gsub!(/¶/um,'&#182;') # '&para;'   # &#182;
          @s.gsub!(/¹/um,'&#185;') # '&sup1;'   # &#185;
          @s.gsub!(/º/um,'&#186;') # '&ordm;'   # &#186;
          @s.gsub!(/»/um,'&#187;') # '&raquo;'  # &#187;
          @s.gsub!(/¼/um,'&#188;') # '&frac14;' # &#188;
          @s.gsub!(/½/um,'&#189;') # '&frac12;' # &#189;
          @s.gsub!(/¾/um,'&#190;') # '&frac34;' # &#190;
          @s.gsub!(/×/um,'&#215;') # '&times;'  # &#215;
          @s.gsub!(/÷/um,'&#247;') # '&divide;' # &#247;
          @s.gsub!(/¿/um,'&#191;') # '&iquest;' # &#191;
          @s.gsub!(/À/um,'&#192;') # '&Agrave;' # &#192;
          @s.gsub!(/Á/um,'&#193;') # '&Aacute;' # &#193;
          @s.gsub!(/Â/um,'&#194;') # '&Acirc;'  # &#194;
          @s.gsub!(/Ã/um,'&#195;') # '&Atilde;' # &#195;
          @s.gsub!(/Ä/um,'&#196;') # '&Auml;'   # &#196;
          @s.gsub!(/Å/um,'&#197;') # '&Aring;'  # &#197;
          @s.gsub!(/Æ/um,'&#198;') # '&AElig;'  # &#198;
          @s.gsub!(/Ç/um,'&#199;') # '&Ccedil;' # &#199;
          @s.gsub!(/È/um,'&#200;') # '&Egrave;' # &#200;
          @s.gsub!(/É/um,'&#201;') # '&Eacute;' # &#201;
          @s.gsub!(/Ê/um,'&#202;') # '&Ecirc;'  # &#202;
          @s.gsub!(/Ë/um,'&#203;') # '&Euml;'   # &#203;
          @s.gsub!(/Ì/um,'&#204;') # '&Igrave;' # &#204;
          @s.gsub!(/Í/um,'&#205;') # '&Iacute;' # &#205;
          @s.gsub!(/Î/um,'&#206;') # '&Icirc;'  # &#206;
          @s.gsub!(/Ï/um,'&#207;') # '&Iuml;'   # &#207;
          @s.gsub!(/Ð/um,'&#208;') # '&ETH;'    # &#208;
          @s.gsub!(/Ñ/um,'&#209;') # '&Ntilde;' # &#209;
          @s.gsub!(/Ò/um,'&#210;') # '&Ograve;' # &#210;
          @s.gsub!(/Ó/um,'&#211;') # '&Oacute;' # &#211;
          @s.gsub!(/Ô/um,'&#212;') # '&Ocirc;'  # &#212;
          @s.gsub!(/Õ/um,'&#213;') # '&Otilde;' # &#213;
          @s.gsub!(/Ö/um,'&#214;') # '&Ouml;'   # &#214;
          @s.gsub!(/Ø/um,'&#216;') # '&Oslash;' # &#216;
          @s.gsub!(/Ù/um,'&#217;') # '&Ugrave;' # &#217;
          @s.gsub!(/Ú/um,'&#218;') # '&Uacute;' # &#218;
          @s.gsub!(/Û/um,'&#219;') # '&Ucirc;'  # &#219;
          @s.gsub!(/Ü/um,'&#220;') # '&Uuml;'   # &#220;
          @s.gsub!(/Ý/um,'&#221;') # '&Yacute;' # &#221;
          @s.gsub!(/Þ/um,'&#222;') # '&THORN;'  # &#222;
          @s.gsub!(/ß/um,'&#223;') # '&szlig;'  # &#223;
          @s.gsub!(/à/um,'&#224;') # '&agrave;' # &#224;
          @s.gsub!(/á/um,'&#225;') # '&aacute;' # &#225;
          @s.gsub!(/â/um,'&#226;') # '&acirc;'  # &#226;
          @s.gsub!(/ã/um,'&#227;') # '&atilde;' # &#227;
          @s.gsub!(/ä/um,'&#228;') # '&auml;'   # &#228;
          @s.gsub!(/å/um,'&#229;') # '&aring;'  # &#229;
          @s.gsub!(/æ/um,'&#230;') # '&aelig;'  # &#230;
          @s.gsub!(/ç/um,'&#231;') # '&ccedil;' # &#231;
          @s.gsub!(/è/um,'&#232;') # '&egrave;' # &#232;
          @s.gsub!(/é/um,'&#233;') # '&acute;'  # &#233;
          @s.gsub!(/ê/um,'&#234;') # '&circ;'   # &#234;
          @s.gsub!(/ë/um,'&#235;') # '&euml;'   # &#235;
          @s.gsub!(/ì/um,'&#236;') # '&igrave;' # &#236;
          @s.gsub!(/í/um,'&#237;') # '&acute;'  # &#237;
          @s.gsub!(/î/um,'&#238;') # '&icirc;'  # &#238;
          @s.gsub!(/ï/um,'&#239;') # '&iuml;'   # &#239;
          @s.gsub!(/ð/um,'&#240;') # '&eth;'    # &#240;
          @s.gsub!(/ñ/um,'&#241;') # '&ntilde;' # &#241;
          @s.gsub!(/ò/um,'&#242;') # '&ograve;' # &#242;
          @s.gsub!(/ó/um,'&#243;') # '&oacute;' # &#243;
          @s.gsub!(/ô/um,'&#244;') # '&ocirc;'  # &#244;
          @s.gsub!(/õ/um,'&#245;') # '&otilde;' # &#245;
          @s.gsub!(/ö/um,'&#246;') # '&ouml;'   # &#246;
          @s.gsub!(/ø/um,'&#248;') # '&oslash;' # &#248;
          @s.gsub!(/ù/um,'&#250;') # '&ugrave;' # &#250;
          @s.gsub!(/ú/um,'&#251;') # '&uacute;' # &#251;
          @s.gsub!(/û/um,'&#252;') # '&ucirc;'  # &#252;
          @s.gsub!(/ü/um,'&#253;') # '&uuml;'   # &#253;
          @s.gsub!(/þ/um,'&#254;') # '&thorn;'  # &#254;
          @s.gsub!(/ÿ/um,'&#255;') # '&yuml;'   # &#255;
          @s.gsub!(/‘/um,'&#8216;') # '&lsquo;' # &#8216;
          @s.gsub!(/’/um,'&#8217;') # '&rsquo;' # &#8217;
          @s.gsub!(/“/um,'&#8220;') # &ldquo;   # &#8220;
          @s.gsub!(/”/um,'&#8221;') # &rdquo;   # &#8221;
          @s.gsub!(/–/um,'&#8211;') # &ndash;   # &#8211;
          @s.gsub!(/—/um,'&#8212;') # &mdash;   # &#8212;
          @s.gsub!(/∝/um,'&#8733;') # &prop;    # &#8733;
          @s.gsub!(/∞/um,'&#8734;') # &infin;   # &#8734;
          @s.gsub!(/™/um,'&#8482;') # &trade;   # &#8482;
          @s.gsub!(/✠/um,'&#10016;') # &cross;  # &#10016;
          @s.gsub!(/ /um,' ')       # space identify
          @s.gsub!(/ /um,' ')       # space identify
          @s.gsub!(/#{Mx[:br_paragraph]}/u,'<br />')
        end
        @s
      end
      self
    end
    def xml_sax
      def meta_para
        inf_xml=char_enc(@inf).utf8
        <<WOK
<metadata>
#{Ax[:tab]}<meta>#{@tag.capitalize}:</meta>
#{Ax[:tab]}<data class="#{@attrib}">
#{Ax[:tab]*2}#{inf_xml}
#{Ax[:tab]}</data>
</metadata>
WOK
      end
      def metadata
        SiSU_Metadata::Summary.new(@md).metadata_base
      end
      self
    end
    def xml_dom
      def meta_para
        inf_xml=char_enc(@inf).utf8
        <<WOK
#{Ax[:tab]}<header>
#{Ax[:tab]*2}<meta>#{@tag.capitalize}:</meta>
#{Ax[:tab]*2}<#{@attrib}>
#{Ax[:tab]*3}#{inf_xml}
#{Ax[:tab]*2}</#{@attrib}>
#{Ax[:tab]}</header>
WOK
      end
      def metadata
        SiSU_Metadata::Summary.new(@md).metadata_base
      end
      self
    end
    def xhtml_scroll
      def meta_para
        inf_xml=char_enc(@inf).utf8
        <<WOK
#{Ax[:tab]}<metadata>
#{Ax[:tab]}<meta>#{@tag.capitalize}:</meta>
#{Ax[:tab]}<#{@attrib} class="#{@class}">
#{Ax[:tab]*2}#{inf_xml}
#{Ax[:tab]}</#{@attrib}>
#{Ax[:tab]}</metadata>
	<br />
WOK
      end
      def metadata
        SiSU_Metadata::Summary.new(@md).metadata_base
      end
      self
    end
    def xhtml_display
      def meta_para
        inf_xml=char_enc(@inf).utf8
        %{<p class="norm">
  <b>#{@tag}</b>: #{inf_xml}
</p>}
      end
      def metadata
        SiSU_Metadata::Summary.new(@md,true).metadata_base
      end
      self
    end
    def odf
      def meta_para
        url_brace=SiSU_Viz::Skin.new.url_decoration
        if @inf.class==String
          @inf.gsub!(/</,'&lt;'); @inf.gsub!(/>/,'&gt;')
          @inf.gsub!(/&lt;br(?: \/)?&gt;/,'<br />')
          if @inf =~/&/
            inf_array=[]
            word=@inf.scan(/\S+|\n/)
            word.each do |w| # _ - / # | : ! ^ ~
              w.gsub!(/&nbsp;/,'&#160;')
              if w !~/&\S{2,7}?;/
                w.gsub!(/&/,'&amp;')
              end
              inf_array << w
            end
            @inf=inf_array.join(' ')
          end
          @inf.gsub!(/#{Mx[:url_o]}_(\S+?)#{Mx[:url_c]}/,
            '<text:a xlink:type="simple" xlink:href="\1">\1</text:a>') #http ftp matches escaped, no decoration
          @inf.gsub!(/(#{Mx[:lnk_c]})#{Mx[:url_o]}(\S+?)#{Mx[:url_c]}/,
            '\1<text:a xlink:type="simple" xlink:href="\2">\2</text:a>') #special case \{ e.g. \}http://url
          if @inf =~/#{Mx[:url_o]}(\S+?)#{Mx[:url_c]}/
            @inf.gsub!(/#{Mx[:url_o]}(\S+?)#{Mx[:url_c]}/,
              %{#{url_brace.xml_open}<text:a xlink:type="simple" xlink:href="\\1">\\1</text:a>#{url_brace.xml_close}}) #http ftp matches with decoration
          else
            @inf.gsub!(/(https?:\/\/[^<>()'"\s]+)/,
              %{#{url_brace.xml_open}<text:a xlink:type="simple" xlink:href="\\1">\\1</text:a>#{url_brace.xml_close}}) #http ftp matches with decoration
          end
          @inf.gsub!(/([a-zA-Z0-9._-]+@[a-zA-Z0-9._-]+)/,
            %{#{url_brace.xml_open}<text:a xlink:type="simple" xlink:href="mailto:\\1">\\1</text:a>#{url_brace.xml_close}}) if @inf !~/http:\/\// # improve upon, document crash where url contains '@' symbol
        end
        <<WOK
<text:p text:style-name="P1">#{@tag.capitalize}: #{@inf}</text:p>
WOK
      end
      def metadata
        SiSU_Metadata::Summary.new(@md).metadata_base
      end
      self
    end
    def plaintext
      def meta_para
        <<WOK

#{@tag.capitalize}: #{@inf}
WOK
      end
      def metadata
        SiSU_Metadata::Summary.new(@md).metadata_base
      end
      self
    end
    def manpage
      def meta_para
        <<WOK

.TP
#{@tag.capitalize}:
.I #{@inf}
WOK
      end
      def metadata
        SiSU_Metadata::Summary.new(@md).metadata_base
      end
      self
    end
  end
  class TeX_Metadata
    def initialize(md)
      @md=md
      @br="\\\\\n"
    end
    def meta_para(tag,inf,sc=true)
      inf=((inf.class==String && sc) ? spec_char(inf) : inf)
      %{\\begin\{bfseries\}#{tag}:\\end\{bfseries\} #{inf}
}
    end
    def spec_char(inf)
      SiSU_TeX_Pdf::SpecialCharacters.new(@md,inf).special_characters
    end
    def word_break_points(inf)
      SiSU_TeX_Pdf::SpecialCharacters.new(@md,inf).special_word_break_points
    end
    def number_break_points(inf)
      SiSU_TeX_Pdf::SpecialCharacters.new(@md,inf).special_number_break_points
    end
    def metadata_tex
      meta=[]
      dir=SiSU_Env::InfoEnv.new(@md.fns)
      base_html="#{dir.url.root}/#{@md.fnb}"
      l=SiSU_Env::StandardiseLanguage.new(@md.opt.lng).language
      language=l[:n]
      tr=SiSU_Translate::Source.new(@md,language)
      tag="Document Manifest @"
      inf="#{@br}#{base_html}/#{@md.fn[:manifest]}"
      meta << meta_para(tag,inf)
      if defined? @md.title.full \
      and @md.title.full=~/\S+/
        tag,inf=tr.full_title,@md.title.full
        meta << meta_para(tag,inf)
      end
      if defined? @md.creator.author \
      and @md.creator.author=~/\S+/
        tag,inf=tr.author,@md.creator.author
        meta << meta_para(tag,inf)
      end
      if defined? @md.creator.translator \
      and @md.creator.translator=~/\S+/
        tag,inf=tr.translator,@md.creator.translator
        meta << meta_para(tag,inf)
      end
      if defined? @md.creator.illustrator \
      and @md.creator.illustrator=~/\S+/
        tag,inf=tr.illustrator,@md.creator.illustrator
        meta << meta_para(tag,inf)
      end
      if defined? @md.creator.prepared_by \
      and @md.creator.prepared_by=~/\S+/
        tag,inf=tr.prepared_by,@md.creator.prepared_by
        meta << meta_para(tag,inf)
      end
      if defined? @md.creator.digitized_by \
      and @md.creator.digitized_by=~/\S+/
        tag,inf=tr.digitized_by,@md.creator.digitized_by
        meta << meta_para(tag,inf)
      end
      if defined? @md.rights.all \
      and @md.rights.all=~/\S+/
        tag,inf=tr.rights,@md.rights.all
        meta << meta_para(tag,inf)
      end
      if defined? @md.notes.description \
      and @md.notes.description=~/\S+/
        tag,inf=tr.description,@md.notes.description
        meta << meta_para(tag,inf)
      end
      if defined? @md.classify.subject \
      and @md.classify.subject=~/\S+/
        tag,inf=tr.subject,@md.classify.subject
        meta << meta_para(tag,inf)
      end
      if defined? @md.publisher \
      and @md.publisher=~/\S+/
        tag,inf=tr.publisher,@md.publisher
        meta << meta_para(tag,inf)
      end
      if defined? @md.creator.contributor \
      and @md.creator.contributor=~/\S+/
        tag,inf=tr.contributor,@md.creator.contributor
        meta << meta_para(tag,inf)
      end
      if defined? @md.notes.abstract \
      and @md.notes.abstract=~/\S+/
        tag,inf=tr.abstract,@md.notes.abstract
        meta << meta_para(tag,inf)
      end
      if defined? @md.date.created \
      and @md.date.created=~/\S+/
        tag,inf=tr.date_created,@md.date.created
        meta << meta_para(tag,inf)
      end
      if defined? @md.date.issued \
      and @md.date.issued=~/\S+/
        tag,inf=tr.date_issued,@md.date.issued
        meta << meta_para(tag,inf)
      end
      if defined? @md.date.available \
      and @md.date.available=~/\S+/
        tag,inf=tr.date_available,@md.date.available
        meta << meta_para(tag,inf)
      end
      if defined? @md.date.modified \
      and @md.date.modified=~/\S+/
        tag,inf=tr.date_modified,@md.date.modified
        meta << meta_para(tag,inf)
      end
      if defined? @md.date.valid \
      and @md.date.valid=~/\S+/
        tag,inf=tr.date_valid,@md.date.valid
        meta << meta_para(tag,inf)
      end
      if defined? @md.date.published \
      and @md.date.published=~/\S+/
        tag,inf=tr.date,@md.date.published
        meta << meta_para(tag,inf)
      end
      if defined? @md.classify.topic_register \
      and @md.classify.topic_register=~/\S+/
        tag,inf=tr.topic_register,@md.classify.topic_register
        inf=word_break_points(inf)
        meta << meta_para(tag,inf)
      end
      if defined? @md.classify.loc \
      and @md.classify.loc=~/\S+/
        tag,inf=tr.cls_loc,@md.classify.loc
        meta << meta_para(tag,inf)
      end
      if defined? @md.classify.dewey \
      and @md.classify.dewey=~/\S+/
        tag,inf=tr.cls_dewey,@md.classify.dewey
        meta << meta_para(tag,inf)
      end
      if defined? @md.classify.oclc \
      and @md.classify.oclc=~/\S+/
        tag,inf=tr.cls_oclc,@md.classify.oclc
        meta << meta_para(tag,inf)
      end
      if defined? @md.classify.pg \
      and @md.classify.pg=~/\S+/
        tag,inf=tr.cls_gutenberg,@md.classify.pg
        meta << meta_para(tag,inf)
      end
      if defined? @md.classify.isbn \
      and @md.classify.isbn=~/\S+/
        tag,inf=tr.cls_isbn,@md.classify.isbn
        meta << meta_para(tag,inf)
      end
      if defined? @md.notes.comment \
      and @md.notes.comment=~/\S+/
        tag,inf=tr.comments,@md.notes.comment
        meta << meta_para(tag,inf)
      end
      if defined? @md.notes.prefix_a \
      and @md.notes.prefix_a=~/\S+/
        tag,inf=tr.prefix_a,@md.notes.prefix_a
        meta << meta_para(tag,inf)
      end
      if defined? @md.notes.prefix_b \
      and @md.notes.prefix_b=~/\S+/
        tag,inf=tr.prefix_b,@md.notes.prefix_b
        meta << meta_para(tag,inf)
      end
      if defined? @md.classify.identifier \
      and @md.classify.identifier=~/\S+/
        tag,inf=tr.identifier,@md.classify.identifier
        meta << meta_para(tag,inf)
      end
      if defined? @md.original.source \
      and @md.original.source=~/\S+/
        tag,inf=tr.source,@md.original.source
        meta << meta_para(tag,inf)
      end
      if defined? @md.title.language \
      and @md.title.language=~/\S+/
        tag,inf=tr.language,@md.title.language
        meta << meta_para(tag,inf)
      end
      if defined? @md.original.language \
      and @md.original.language=~/\S+/
        tag,inf=tr.language_original,@md.original.language
        meta << meta_para(tag,inf)
      end
      if defined? @md.classify.format \
      and @md.classify.format=~/\S+/
        tag,inf=tr.format,@md.classify.format
        meta << meta_para(tag,inf)
      end
      if defined? @md.classify.relation \
      and @md.classify.relation=~/\S+/
        tag,inf=tr.relation,@md.classify.relation
        meta << meta_para(tag,inf)
      end
      if defined? @md.classify.coverage \
      and @md.classify.coverage=~/\S+/
        tag,inf=tr.coverage,@md.classify.coverage
        meta << meta_para(tag,inf)
      end
      if defined? @md.classify.keywords \
      and @md.classify.keywords=~/\S+/
        tag,inf=tr.keywords,@md.classify.keywords
        meta << meta_para(tag,inf)
      end
      meta << %{#{@br}\\begin\{bfseries\}Version Information \\end\{bfseries\}}
      if defined? @md.fns \
      and @md.fns=~/\S+/
        fn=spec_char(@md.fns)
        fn=word_break_points(fn)
        fn="\\begin\{footnotesize\}#{fn}\\end\{footnotesize\}"
        tag,inf=tr.sourcefile,fn
        meta << meta_para(tag,inf,false)
      end
      if defined? @md.file_encoding \
      and @md.file_encoding=~/\S+/
        tag,inf='Filetype',@md.file_encoding
        meta << meta_para(tag,inf)
      end
      if defined? @md.dgst \
      and @md.dgst.class==Array
        hash_of=spec_char(@md.dgst[0])
        hash_of=word_break_points(hash_of)
        dgst=number_break_points(@md.dgst[1])
        tag,inf='Source Digest',"\\begin\{footnotesize\}#{hash_of}\\end\{footnotesize\}\\-\\begin\{scriptsize\}#{dgst}\\end\{scriptsize\}"
        meta << meta_para(tag,inf,false)
      end
      if defined? @md.dgst_skin \
      and @md.dgst_skin.class==Array
        hash_of=spec_char(@md.dgst_skin[0])
        hash_of=word_break_points(hash_of)
        dgst=number_break_points(@md.dgst_skin[1])
        tag,inf='Skin Digest',"\\begin\{footnotesize\}#{hash_of}\\end\{footnotesize\}\\-\\begin\{scriptsize\}#{dgst}\\end\{scriptsize\}"
        meta << meta_para(tag,inf,false)
      end
      meta << %{#{@br}\\begin\{bfseries\}Generated \\end\{bfseries\}}
      if defined? @md.generated \
      and @md.generated.class==Time
        tag,inf=tr.last_generated,@md.generated
        meta << meta_para(tag,inf)
      end
      if defined? @md.sisu_version \
      and @md.sisu_version[:version]=~/\S+/
        tag=tr.sisu_version
        inf="#{@md.sisu_version[:project]} " +
          "#{@md.sisu_version[:version]} " +
          "of #{@md.sisu_version[:date_stamp]} " +
          "(#{@md.sisu_version[:date]})"
        meta << meta_para(tag,inf)
      end
      if defined? @md.ruby_version \
      and @md.ruby_version=~/\S+/
        tag,inf=tr.ruby_version,@md.ruby_version
        meta << meta_para(tag,inf)
      end
      meta
    end
  end
end
__END__
if @md.title
  x=[
    @md.title.main,
    @md.title.sub,
    @md.title.edition,
    @md.title.note,
    @md.title.short,
    @md.title.full,
    @md.title.language,
    @md.title.language_char
  ]
  x.each {|y| p y if y}
end
if @md.creator
  x=[
    @md.creator.author,
    @md.creator.author_detail,
    @md.creator.contributor,
    @md.creator.contributor_detail,
    @md.creator.illustrator,
    @md.creator.illustrator_detail,
    @md.creator.photographer,
    @md.creator.photographer_detail,
    @md.creator.translator,
    @md.creator.translator_detail,
    @md.creator.audio,
    @md.creator.audio_detail,
    @md.creator.digitized_by,
    @md.creator.digitized_by_detail,
    @md.creator.prepared_by,
    @md.creator.prepared_by_detail
  ]
  x.each {|y| p y if y}
end
if @md.rights
  x=[
    @md.rights.copyright.text,
    @md.rights.copyright.translation,
    @md.rights.copyright.illustrations,
    @md.rights.copyright.photographs,
    @md.rights.copyright.digitization,
    @md.rights.copyright.audio,
    @md.rights.license,
    @md.rights.all
  ]
  x.each {|y| p y if y}
end
if @md.classify
  x=[
    @md.classify.coverage,
    @md.classify.relation,
    @md.classify.subject,
    @md.classify.topic_register,
    @md.classify.type,
    @md.classify.identifier,
    @md.classify.loc,
    @md.classify.dewey,
    @md.classify.oclc,
    @md.classify.pg,
    @md.classify.isbn,
  ]
  x.each {|y| p y if y}
end
if @md.date
  x=[
    @md.date.added_to_site,
    @md.date.available,
    @md.date.created,
    @md.date.issued,
    @md.date.modified,
    @md.date.published,
    @md.date.valid
  ]
  x.each {|y| p y if y}
end
#if @md.language
#  p @md.language.document
#  p @md.language.document_char
#  p @md.language.original
#  p @md.language.original_char
#end
if @md.make
  x=[
    @md.make.headings,
    @md.make.num_top,
    @md.make.breaks,
    @md.make.bold,
    @md.make.italics,
    @md.make.emphasis,
    @md.make.plaintext_wrap,
    @md.make.texpdf_font,
    @md.make.skin,
    @md.make.promo,
    @md.make.ad,
    @md.make.manpage
  ]
  x.each {|y| p y if y}
end
if @md.current_publisher # @md.publisher
  x=[
    @md.current_publisher
  ]
  x.each {|y| p y if y}
end
if @md.original
  x=[
    @md.original.publisher,
    @md.original.language,
    @md.original.language_char,
    @md.original.source,
    @md.original.institution,
    @md.original.nationality
  ]
  x.each {|y| p y if y}
end
if @md.notes
  x=[
    @md.notes.abstract,
    @md.notes.comment,
    @md.notes.description,
    @md.notes.history,
    @md.notes.prefix
  ]
  x.each {|y| p y if y}
end
