#include <math.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>

#include "lib/config.h"

#ifdef HAVE_LAPACK
#include "lib/Mathematics.h"
#include "classifier/svm/Tron.h"

CTron::CTron(const function *f, double e, int it)
: CSGObject()
{
	this->fun_obj=const_cast<function *>(f);
	this->eps=e;
	this->max_iter=it;
}

CTron::~CTron()
{
}

void CTron::tron(double *w)
{
	// Parameters for updating the iterates.
	double eta0 = 1e-4, eta1 = 0.25, eta2 = 0.75;

	// Parameters for updating the trust region size delta.
	double sigma1 = 0.25, sigma2 = 0.5, sigma3 = 4;

	int n = fun_obj->get_nr_variable();
	int i, cg_iter;
	double delta, snorm, one=1.0;
	double alpha, f, fnew, prered, actred, gs;
	int search = 1, iter = 1, inc = 1;
	double *s = new double[n];
	double *r = new double[n];
	double *w_new = new double[n];
	double *g = new double[n];

	for (i=0; i<n; i++)
		w[i] = 0;

        f = fun_obj->fun(w);
	fun_obj->grad(w, g);
	delta = cblas_dnrm2(n, g, inc);
	double gnorm1 = delta;
	double gnorm = gnorm1;

	if (gnorm <= eps*gnorm1)
		search = 0;

	iter = 1;

	while (iter <= max_iter && search)
	{
		cg_iter = trcg(delta, g, s, r);

		memcpy(w_new, w, sizeof(double)*n);
		cblas_daxpy(n, one, s, inc, w_new, inc);

		gs = cblas_ddot(n, g, inc, s, inc);
		prered = -0.5*(gs-cblas_ddot(n, s, inc, r, inc));
                fnew = fun_obj->fun(w_new);

		// Compute the actual reduction.
	        actred = f - fnew;

		// On the first iteration, adjust the initial step bound.
		snorm = cblas_dnrm2(n, s, inc);
		if (iter == 1)
			delta = CMath::min(delta, snorm);

		// Compute prediction alpha*snorm of the step.
		if (fnew - f - gs <= 0)
			alpha = sigma3;
		else
			alpha = CMath::max(sigma1, -0.5*(gs/(fnew - f - gs)));

		// Update the trust region bound according to the ratio of actual to predicted reduction.
		if (actred < eta0*prered)
			delta = CMath::min(CMath::max(alpha, sigma1)*snorm, sigma2*delta);
		else if (actred < eta1*prered)
			delta = CMath::max(sigma1*delta, CMath::min(alpha*snorm, sigma2*delta));
		else if (actred < eta2*prered)
			delta = CMath::max(sigma1*delta, CMath::min(alpha*snorm, sigma3*delta));
		else
			delta = CMath::max(delta, CMath::min(alpha*snorm, sigma3*delta));

		SG_INFO("iter %2d act %5.3e pre %5.3e delta %5.3e f %5.3e |g| %5.3e CG %3d\n", iter, actred, prered, delta, f, gnorm, cg_iter);

		if (actred > eta0*prered)
		{
			iter++;
			memcpy(w, w_new, sizeof(double)*n);
			f = fnew;
		        fun_obj->grad(w, g);

			gnorm = cblas_dnrm2(n, g, inc);
			if (gnorm < eps*gnorm1)
				break;
		}
		if (f < -1.0e+32)
		{
			SG_WARNING("f < -1.0e+32\n");
			break;
		}
		if (CMath::abs(actred) <= 0 && CMath::abs(prered) <= 0)
		{
			SG_WARNING("actred and prered <= 0\n");
			break;
		}
		if (CMath::abs(actred) <= 1.0e-12*CMath::abs(f) &&
		    CMath::abs(prered) <= 1.0e-12*CMath::abs(f))
		{
			SG_WARNING("actred and prered too small\n");
			break;
		}
	}

	delete[] g;
	delete[] r;
	delete[] w_new;
	delete[] s;
}

int CTron::trcg(double delta, double *g, double *s, double *r)
{
	int i, inc = 1;
	int n = fun_obj->get_nr_variable();
	double one = 1;
	double *d = new double[n];
	double *Hd = new double[n];
	double rTr, rnewTrnew, alpha, beta, cgtol;

	for (i=0; i<n; i++)
	{
		s[i] = 0;
		r[i] = -g[i];
		d[i] = r[i];
	}
	cgtol = 0.1*cblas_dnrm2(n, g, inc);

	int cg_iter = 0;
	rTr = cblas_ddot(n, r, inc, r, inc);
	while (1)
	{
		if (cblas_dnrm2(n, r, inc) <= cgtol)
			break;
		cg_iter++;
		fun_obj->Hv(d, Hd);

		alpha = rTr/cblas_ddot(n, d, inc, Hd, inc);
		cblas_daxpy(n, alpha, d, inc, s, inc);
		if (cblas_dnrm2(n, s, inc) > delta)
		{
			SG_INFO("cg reaches trust region boundary\n");
			alpha = -alpha;
			cblas_daxpy(n, alpha, d, inc, s, inc);

			double std = cblas_ddot(n, s, inc, d, inc);
			double sts = cblas_ddot(n, s, inc, s, inc);
			double dtd = cblas_ddot(n, d, inc, d, inc);
			double dsq = delta*delta;
			double rad = sqrt(std*std + dtd*(dsq-sts));
			if (std >= 0)
				alpha = (dsq - sts)/(std + rad);
			else
				alpha = (rad - std)/dtd;
			cblas_daxpy(n, alpha, d, inc, s, inc);
			alpha = -alpha;
			cblas_daxpy(n, alpha, Hd, inc, r, inc);
			break;
		}
		alpha = -alpha;
		cblas_daxpy(n, alpha, Hd, inc, r, inc);
		rnewTrnew = cblas_ddot(n, r, inc, r, inc);
		beta = rnewTrnew/rTr;
		cblas_dscal(n, beta, d, inc);
		cblas_daxpy(n, one, r, inc, d, inc);
		rTr = rnewTrnew;
	}

	delete[] d;
	delete[] Hd;

	return(cg_iter);
}

double CTron::norm_inf(int n, double *x)
{
	double dmax = CMath::abs(x[0]);
	for (int i=1; i<n; i++)
		if (CMath::abs(x[i]) >= dmax)
			dmax = CMath::abs(x[i]);
	return(dmax);
}
#endif //HAVE_LAPACK
