/*
 * Scilab ( http://www.scilab.org/ ) - This file is part of Scilab
 * Copyright (C) 2009-2011 - DIGITEO - Pierre Lando
 *
 * This file must be used under the terms of the CeCILL.
 * This source file is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at
 * http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt
 */

package org.scilab.forge.scirenderer.ruler.graduations;

import java.util.LinkedList;
import java.util.List;

/**
 * @author Pierre Lando
 */
public class LogarithmicGraduations extends AbstractGraduations implements Graduations {

    /**
     * Exponent of the step.
     * Possible value are : 1, 2 and 3*k with k > 0
     */
    private final int stepExponent;

    private List<Double> allValues;
    private Graduations subGraduation;
    private Graduations moreGraduation;
    private Graduations alternativeGraduation;

    public static LogarithmicGraduations create(double lowerBound, double upperBound) {
        return create(lowerBound, true, upperBound, true);
    }

    public static LogarithmicGraduations create(double lowerBound, boolean lowerBoundIncluded, double upperBound, boolean upperBoundIncluded) {
        if (lowerBound < upperBound) {
            return new LogarithmicGraduations(lowerBound, lowerBoundIncluded, upperBound, upperBoundIncluded);
        } else {
            return new LogarithmicGraduations(upperBound, upperBoundIncluded, lowerBound, lowerBoundIncluded);
        }
    }

    private LogarithmicGraduations(double lowerBound, boolean lowerBoundIncluded, double upperBound, boolean upperBoundIncluded) {
        super(lowerBound, lowerBoundIncluded, upperBound, upperBoundIncluded);
        if (lowerBound != upperBound) {
            stepExponent = 1;
        } else {
            stepExponent = 0;
            allValues = new LinkedList<Double>();
            allValues.add(lowerBound);
        }
    }

    private LogarithmicGraduations(Graduations parentGraduations, int stepExponent) {
        super(parentGraduations);
        this.stepExponent = stepExponent;
    }

    @Override
    public List<Double> getAllValues() {
        if (allValues == null) {
            allValues = new LinkedList<Double>();
            int currentExponent = (int) Math.ceil(Math.log10(lowerBound));
            double currentValue = Math.pow(10, currentExponent);

            if ((currentValue == lowerBound) && (!isLowerBoundIncluded)) {
                currentExponent += stepExponent;
                currentValue = Math.pow(10, currentExponent);
            }

            while (contain(currentValue)) {
                allValues.add(currentValue);
                currentExponent += stepExponent;
                currentValue = Math.pow(10, currentExponent);
            }
        }
        return allValues;
    }

    @Override
    public List<Double> getNewValues() {
        return getAllValues();
    }

    @Override
    public Graduations getMore() {
        if (moreGraduation == null) {
            moreGraduation = new LinLogGraduation(this);
        }
        return moreGraduation;
    }

    @Override
    public Graduations getAlternative() {
        if (alternativeGraduation == null) {
            int nextStep = 3 + stepExponent - stepExponent % 3;
            alternativeGraduation = new LogarithmicGraduations(this, nextStep);
        }
        return alternativeGraduation;
    }

    @Override
    public Graduations getSubGraduations() {
        if (subGraduation == null) {
            if (stepExponent > 1) {
                subGraduation = new LogarithmicGraduations(this, stepExponent / 3);
            } else {
                subGraduation = new LinLogGraduation(this).getSubGraduations();
            }
        }
        return subGraduation;
    }

    @Override
    public int getSubDensity() {
        if (stepExponent >= 3) {
            return 3;
        } else if (stepExponent == 2) {
            return stepExponent;
        } else {
            return getSubGraduations().getSubDensity();
        }
    }

    /**
     * This class manage linear graduation between 10^n and 10^(n+1)
     */
    private class LinLogGraduation extends AbstractGraduations implements Graduations {
        private Graduations alternativeLLGraduation;
        private Graduations moreLLGraduation;
        private Graduations subLLGraduation;

        private List<Double> allValues;
        private List<Double> newValues;

        private final List<Graduations> graduationsList;

        public LinLogGraduation(LogarithmicGraduations parentGraduations) {
            super(parentGraduations);
            graduationsList = computeGraduationsList();
        }

        private LinLogGraduation(Graduations parentGraduations, List<Graduations> graduationsList) {
            super(parentGraduations);
            this.graduationsList = graduationsList;
        }

        @Override
        public List<Double> getAllValues() {
            if (allValues == null) {
                allValues = new LinkedList<Double>();
                for (Graduations graduations : graduationsList) {
                    allValues.addAll(graduations.getAllValues());
                }

                allValues.addAll(getLogarithmicParent().getAllValues());
            }
            return allValues;
        }

        @Override
        public List<Double> getNewValues() {
            if (newValues == null) {
                newValues = new LinkedList<Double>();
                if (getParentGraduations() instanceof LogarithmicGraduations) {
                    for (Graduations graduations : graduationsList) {
                        newValues.addAll(graduations.getAllValues());
                    }
                } else {
                    for (Graduations graduations : graduationsList) {
                        newValues.addAll(graduations.getNewValues());
                    }
                }
            }
            return newValues;
        }

        @Override
        public Graduations getMore() {
            if (moreLLGraduation == null) {
                List<Graduations> moreList = new LinkedList<Graduations>();
                for (Graduations graduations : graduationsList) {
                    Graduations more = graduations.getMore();
                    if (more != null) {
                        moreList.add(more);
                    }
                }
                if (!moreList.isEmpty()) {
                    moreLLGraduation = new LinLogGraduation(this, moreList);
                }
            }
            return moreLLGraduation;
        }

        @Override
        public Graduations getAlternative() {
            if (alternativeLLGraduation == null) {
                List<Graduations> alternativeList = new LinkedList<Graduations>();
                for (Graduations graduations : graduationsList) {
                    Graduations alternative = graduations.getAlternative();
                    if (alternative != null) {
                        alternativeList.add(alternative);
                    }
                }
                if (!alternativeList.isEmpty()) {
                    alternativeLLGraduation = new LinLogGraduation(this, alternativeList);
                }
            }
            return alternativeLLGraduation;
        }

        @Override
        public Graduations getSubGraduations() {
            if (subLLGraduation == null) {
                List<Graduations> subList = new LinkedList<Graduations>();
                for (Graduations graduations : graduationsList) {
                    Graduations sub = graduations.getSubGraduations();
                    if (sub != null) {
                        subList.add(sub);
                    }
                }
                if (subList.isEmpty()) {
                    subLLGraduation = getMore().getSubGraduations();
                } else {
                    subLLGraduation = new LinLogGraduation(this, subList);
                }
            }
            return subLLGraduation;
        }

        @Override
        public int getSubDensity() {
            return 0;
        }

        private List<Graduations> computeGraduationsList() {
            List<Graduations> graduationsList = new LinkedList<Graduations>();

            /**
             * Let a and b a power of 10.
             * lowerBound < a < b < upperBound
             */

            double aPower = Math.ceil(Math.log10(lowerBound));
            double bPower = Math.floor(Math.log10(upperBound));
            double a = Math.pow(10, aPower);
            double b = Math.pow(10, bPower);

            if (aPower > bPower) {
                // Case of 10^n <= a < b <= 10^(n+1)
                graduationsList.add(LinearGraduations.create(
                        this,
                        lowerBound, true,
                        upperBound, true
                ));
            } else {
                if (a != lowerBound) {
                    graduationsList.add(LinearGraduations.create(
                            this,
                            lowerBound, true,
                            a, false
                    ));
                }

                if (aPower != bPower) {
                    for (double i = aPower; i < bPower; i++) {
                        graduationsList.add(LinearGraduations.create(
                                this,
                                Math.pow(10, i), false,
                                Math.pow(10, i + 1), false
                        ));
                    }
                }

                if (b != upperBound) {
                    graduationsList.add(LinearGraduations.create(
                            this,
                            b, false,
                            upperBound, true
                    ));
                }
            }

            return graduationsList;
        }

        private Graduations getLogarithmicParent() {
            Graduations currentGraduation = getParentGraduations();
            while(!(currentGraduation instanceof LogarithmicGraduations)) {
                currentGraduation = currentGraduation.getParentGraduations();
            }
            return currentGraduation;
        }
    }
}
