/*
 * Scilab ( http://www.scilab.org/ ) - This file is part of Scilab
 * Copyright (C) 2009-2011 - DIGITEO - Pierre Lando
 *
 * This file must be used under the terms of the CeCILL.
 * This source file is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at
 * http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt
 */

package org.scilab.forge.scirenderer.ruler.graduations;

import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;

/**
 * @author Pierre Lando
 */
public abstract class AbstractGraduations implements Graduations {

    private final Graduations parentGraduations;
    protected final boolean isLowerBoundIncluded;
    protected final boolean isUpperBoundIncluded;
    protected final double lowerBound;
    protected final double upperBound;
    private DecimalFormat numberFormat;

    public AbstractGraduations(Graduations parentGraduations) {
        this.parentGraduations = parentGraduations;
        this.isLowerBoundIncluded = parentGraduations.isLowerBoundIncluded();
        this.isUpperBoundIncluded = parentGraduations.isUpperBoundIncluded();
        this.lowerBound = parentGraduations.getLowerBound();
        this.upperBound = parentGraduations.getUpperBound();
    }

    public AbstractGraduations(double lowerBound, boolean lowerBoundIncluded, double upperBound, boolean upperBoundIncluded) {
        this.parentGraduations = null;
        this.isLowerBoundIncluded = lowerBoundIncluded;
        this.isUpperBoundIncluded = upperBoundIncluded;
        this.lowerBound = lowerBound;
        this.upperBound = upperBound;
    }

    public AbstractGraduations(double lowerBound, double upperBound) {
        this.parentGraduations = null;
        this.isLowerBoundIncluded = true;
        this.isUpperBoundIncluded = true;
        this.lowerBound = lowerBound;
        this.upperBound = upperBound;
    }

    public AbstractGraduations(Graduations parentGraduations, double lowerBound, boolean lowerBoundIncluded, double upperBound, boolean upperBoundIncluded) {
        this.parentGraduations = parentGraduations;
        this.isLowerBoundIncluded = lowerBoundIncluded;
        this.isUpperBoundIncluded = upperBoundIncluded;
        this.lowerBound = lowerBound;
        this.upperBound = upperBound;
    }



    @Override
    public double getLowerBound() {
        return lowerBound;
    }

    @Override
    public boolean isLowerBoundIncluded() {
        return isLowerBoundIncluded;
    }

    @Override
    public double getUpperBound() {
        return upperBound;
    }

    @Override
    public boolean isUpperBoundIncluded() {
        return isUpperBoundIncluded;
    }

    @Override
    public Graduations getParentGraduations() {
        return parentGraduations;
    }

    @Override
    public boolean contain(double value) {
        if (value == lowerBound) {
            return isLowerBoundIncluded;
        }
        if (value == upperBound) {
            return isUpperBoundIncluded;
        }
        return (lowerBound < value) && (value < upperBound);
    }

    @Override
    public DecimalFormat getFormat() {
        if (numberFormat == null) {
            int maxDisplayedExponent = (int) Math.log10(Math.max(Math.abs(lowerBound), Math.abs(upperBound)));
            if ((maxDisplayedExponent <= -5) || (maxDisplayedExponent >= 6)) {
                numberFormat = new DecimalFormat("0.#E00");
            } else {
                numberFormat = new DecimalFormat();
            }

            DecimalFormatSymbols decimalFormatSymbols = numberFormat.getDecimalFormatSymbols();
            decimalFormatSymbols.setExponentSeparator("e");
            numberFormat.setDecimalFormatSymbols(decimalFormatSymbols);
        }
        return numberFormat;
    }

    @Override
    public String toString() {
        return "Graduation " +
                (isLowerBoundIncluded() ? "[" : "]") +
                getFormat().format(lowerBound) + ", " + getFormat().format(upperBound) +
                (isUpperBoundIncluded() ? "]" : "[");
    }
}
