/*
 * Scilab ( http://www.scilab.org/ ) - This file is part of Scilab
 * Copyright (C) 2009-2012 - DIGITEO - Pierre Lando
 *
 * This file must be used under the terms of the CeCILL.
 * This source file is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at
 * http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt
 */

package org.scilab.forge.scirenderer.implementation.jogl.texture;

import com.sun.opengl.util.texture.TextureData;
import org.scilab.forge.scirenderer.SciRendererException;
import org.scilab.forge.scirenderer.implementation.jogl.JoGLCanvas;
import org.scilab.forge.scirenderer.implementation.jogl.JoGLDrawingTools;
import org.scilab.forge.scirenderer.texture.AbstractTexture;
import org.scilab.forge.scirenderer.texture.Texture;
import org.scilab.forge.scirenderer.texture.TextureManager;

import javax.media.opengl.GL;
import java.awt.Dimension;
import java.util.HashSet;
import java.util.Set;

/**
 * @author Pierre Lando
 */
public class JoGLTextureManager implements TextureManager {
    private final Set<JoGLTexture> allTextures = new HashSet<JoGLTexture>();

    public JoGLTextureManager(JoGLCanvas canvas) {
    }

    /**
     * Texture binder.
     * Bind the given texture to the given OpenGl context.
     * @param drawingTools drawing tools.
     * @param texture given texture.
     * @throws org.scilab.forge.scirenderer.SciRendererException if the texture can't be bind.
     */
    public void bind(JoGLDrawingTools drawingTools, Texture texture) throws SciRendererException {
        if ((texture instanceof JoGLTexture) && (allTextures.contains((JoGLTexture) texture))) {
            ((JoGLTexture) texture).bind(drawingTools);
        }
    }

    /**
     * Draw the given texture.
     * @param drawingTools used drawing tools.
     * @param texture the texture too drawn.
     * @throws org.scilab.forge.scirenderer.SciRendererException if the texture is invalid.
     */
    public void draw(JoGLDrawingTools drawingTools, Texture texture) throws SciRendererException {
        if ((texture instanceof JoGLTexture) && (allTextures.contains((JoGLTexture) texture))) {
            ((JoGLTexture) texture).draw(drawingTools);
        }
    }

    @Override
    public Texture createTexture() {
        JoGLTexture texture = new JoGLTexture();
        allTextures.add(texture);
        return texture;
    }

    /**
     * Inner class for {@link Texture} implementation.
     */
    private class JoGLTexture extends AbstractTexture implements Texture {
        private com.sun.opengl.util.texture.Texture textures[];
        private int wCuts;
        private int hCuts;

        /**
         * Bind the texture in the OpenGl context.
         * @param drawingTools current drawing tools.
         * @throws SciRendererException if the texture is invalid.
         */
        public void bind(JoGLDrawingTools drawingTools) throws SciRendererException {
            GL gl = drawingTools.getGl();
            if (isValid()) {
                checkData(drawingTools);
                if (textures.length == 1) {
                    gl.glEnable(GL.GL_TEXTURE_2D);
                    textures[0].setTexParameteri(GL.GL_TEXTURE_MAG_FILTER, getAsGLFilter(getMagnificationFilter(), false));
                    textures[0].setTexParameteri(GL.GL_TEXTURE_MIN_FILTER, getAsGLFilter(getMinifyingFilter(), true));
                    textures[0].setTexParameteri(GL.GL_TEXTURE_WRAP_S, getAsGLWrappingMode(getSWrappingMode()));
                    textures[0].setTexParameteri(GL.GL_TEXTURE_WRAP_T, getAsGLWrappingMode(getTWrappingMode()));
                    gl.glTexEnvi(GL.GL_TEXTURE_ENV, GL.GL_TEXTURE_ENV_MODE, GL.GL_REPLACE);
                    textures[0].bind();
                } else {
                    throw new SciRendererException("Texture is too large");
                }
            } else {
                throw new SciRendererException("Texture have no data.");
            }
        }

        /**
         * Check if the texture data are up to date.
         * @param drawingTools the drawing tools.
         * @throws SciRendererException if the texture is too big.
         */
        private void checkData(JoGLDrawingTools drawingTools) throws SciRendererException {
            if (isValid() && !upToDate) {
                upToDate = true;

                releaseTextures();

                Dimension textureSize = getDataProvider().getTextureSize();
                int maxSize = drawingTools.getGLCapacity().getMaximumTextureSize();
                wCuts = (int) Math.ceil(textureSize.getWidth() / maxSize);
                hCuts = (int) Math.ceil(textureSize.getHeight() / maxSize);

                textures = new com.sun.opengl.util.texture.Texture[wCuts * hCuts];
                int k = 0;
                for (int i = 0 ; i < wCuts ; i++) {
                    for (int j = 0 ; j < hCuts ; j++) {
                        textures[k] = com.sun.opengl.util.texture.TextureIO.newTexture(GL.GL_TEXTURE_2D);
                        int x = i * maxSize;
                        int y = j * maxSize;
                        int width = ((i == (wCuts - 1)) ? textureSize.width % maxSize : maxSize);
                        int height = ((j == (hCuts - 1)) ? textureSize.height % maxSize : maxSize);
                        width = (width == 0) ? maxSize : width;
                        height = (height == 0) ? maxSize : height;

                        TextureData data = new TextureData(
                                GL.GL_RGBA,
                                width,
                                height,
                                0, // no border
                                GL.GL_RGBA, // pixel format.
                                GL.GL_FLOAT, // pixel type.
                                true, // mipmap
                                false, // no compression
                                false, // no vertical flip needed TODO as an option from data provider.
                                getDataProvider().getSubData(x, y, width, height),
                                null   // no flusher
                        );
                        textures[k].updateImage(data);
                        k++;
                    }
                }
            }
        }

        private void releaseTextures() {
            if (textures != null) {
                for (com.sun.opengl.util.texture.Texture texture : textures) {
                    texture.dispose();
                }
                textures = null;
            }
        }

        /**
         * Draw the texture in XY plane.
         * @param drawingTools the drawing tools.
         * @throws org.scilab.forge.scirenderer.SciRendererException if the texture is invalid.
         */
        public void draw(JoGLDrawingTools drawingTools) throws SciRendererException {
            checkData(drawingTools);
            final int maxSize = drawingTools.getGLCapacity().getMaximumTextureSize();
            final Dimension textureSize = getDataProvider().getTextureSize();
            final GL gl = drawingTools.getGl();

            gl.glEnable(GL.GL_TEXTURE_2D);
            int k = 0;
            for (int i = 0 ; i < wCuts ; i++) {
                for (int j = 0 ; j < hCuts ; j++) {
                    int x = i * maxSize;
                    int y = j * maxSize;
                    int width = ((i == (wCuts - 1)) ? textureSize.width % maxSize : maxSize);
                    int height = ((j == (hCuts - 1)) ? textureSize.height % maxSize : maxSize);
                    width = (width == 0) ? maxSize : width;
                    height = (height == 0) ? maxSize : height;

                    textures[k].setTexParameteri(GL.GL_TEXTURE_MAG_FILTER, getAsGLFilter(getMagnificationFilter(), false));
                    textures[k].setTexParameteri(GL.GL_TEXTURE_MIN_FILTER, getAsGLFilter(getMinifyingFilter(), true));
                    textures[k].setTexParameteri(GL.GL_TEXTURE_WRAP_S, getAsGLWrappingMode(getSWrappingMode()));
                    textures[k].setTexParameteri(GL.GL_TEXTURE_WRAP_T, getAsGLWrappingMode(getTWrappingMode()));
                    gl.glTexEnvi(GL.GL_TEXTURE_ENV, GL.GL_TEXTURE_ENV_MODE, GL.GL_REPLACE);
                    textures[k].bind();

                    gl.glBegin(GL.GL_QUADS);
                    gl.glTexCoord2d(0, 1);
                    gl.glVertex2f(x, y);

                    gl.glTexCoord2d(1, 1);
                    gl.glVertex2f(x + width, y);

                    gl.glTexCoord2d(1, 0);
                    gl.glVertex2f(x + width, y + height);

                    gl.glTexCoord2d(0, 0);
                    gl.glVertex2f(x, y + height);
                    gl.glEnd();

                    k++;
                }
            }

            gl.glDisable(GL.GL_TEXTURE_2D);
        }

        private int getAsGLWrappingMode(Texture.Wrap wrappingMode) {
            switch (wrappingMode) {
                case CLAMP:
                    return GL.GL_CLAMP;
                case REPEAT:
                    return GL.GL_REPEAT;
                default:
                    return GL.GL_REPEAT;
            }
        }

        private int getAsGLFilter(Texture.Filter filter, boolean mipmap) {
            if (mipmap) {
                switch (filter) {
                    case LINEAR:
                        return GL.GL_LINEAR_MIPMAP_LINEAR;
                    case NEAREST:
                        return GL.GL_NEAREST_MIPMAP_NEAREST;
                    default:
                        return GL.GL_NEAREST;
                }
            } else {
                switch (filter) {
                    case LINEAR:
                        return GL.GL_LINEAR;
                    case NEAREST:
                        return GL.GL_NEAREST;
                    default:
                        return GL.GL_NEAREST;
                }
            }
        }
    }
}

/**
    private class JoGLTexture implements Texture {
        private Filter magnificationFilter;
        private Filter minifyingFilter;

        private Wrap sWrappingMode;
        private Wrap tWrappingMode;

        private TextureDataProvider textureDataProvider;
        private boolean upToDate = false;
        private Integer glName = null;

        public JoGLTexture() {
            magnificationFilter = Filter.LINEAR;
            minifyingFilter = Filter.LINEAR;

            sWrappingMode = Wrap.CLAMP;
            tWrappingMode = Wrap.CLAMP;
        }

        @Override
        public boolean isValid() {
            return (textureDataProvider != null) && (textureDataProvider.isValid());
        }

        @Override
        public Wrap getSWrappingMode() {
            return sWrappingMode;
        }

        @Override
        public void setSWrappingMode(Wrap sWrappingMode) {
            this.sWrappingMode = sWrappingMode;
        }

        @Override
        public Wrap getTWrappingMode() {
            return tWrappingMode;
        }

        @Override
        public void setTWrappingMode(Wrap tWrappingMode) {
            this.tWrappingMode = tWrappingMode;
        }

        @Override
        public Filter getMinifyingFilter() {
            return minifyingFilter;
        }

        @Override
        public void setMinifyingFilter(Filter minifyingFilter) {
            this.minifyingFilter = minifyingFilter;
        }

        @Override
        public Filter getMagnificationFilter() {
            return magnificationFilter;
        }

        @Override
        public void setMagnificationFilter(Filter magnificationFilter) {
            this.magnificationFilter = magnificationFilter;
        }

        @Override
        public TextureDataProvider getDataProvider() {
            return textureDataProvider;
        }

        @Override
        public void setDataProvider(TextureDataProvider provider) {
            if (textureDataProvider != null) {
                textureDataProvider.removeDataUser(this);
            }

            textureDataProvider = provider;
            upToDate = false;

            if (textureDataProvider != null) {
                textureDataProvider.addDataUser(this);
            }
        }

        @Override
        public void dataUpdated() {
            upToDate = false;
        }

        void bind(GL gl) {
            if (synchronize(gl)) {
                gl.glEnable(GL.GL_TEXTURE_2D);
                gl.glBindTexture(GL.GL_TEXTURE_2D, glName);
                gl.glTexParameteri(GL.GL_TEXTURE_2D, GL.GL_TEXTURE_MAG_FILTER, getAsGLFilter(getMagnificationFilter()));
                gl.glTexParameteri(GL.GL_TEXTURE_2D, GL.GL_TEXTURE_MIN_FILTER, getAsGLFilter(getMinifyingFilter()));
                gl.glTexParameteri(GL.GL_TEXTURE_2D, GL.GL_TEXTURE_WRAP_S, getAsGLWrappingMode(getSWrappingMode()));
                gl.glTexParameteri(GL.GL_TEXTURE_2D, GL.GL_TEXTURE_WRAP_T, getAsGLWrappingMode(getTWrappingMode()));
                gl.glTexEnvi(GL.GL_TEXTURE_ENV, GL.GL_TEXTURE_ENV_MODE, GL.GL_REPLACE);
            }
        }

        private boolean synchronize(GL gl) {
            if ((glName == null) || !gl.glIsTexture(glName)) {
                int[] t = new int[1];
                gl.glGenTextures(1, t, 0);
                glName = t[0];
                upToDate = false;
            }

            if ((!upToDate) && (textureDataProvider != null) && (textureDataProvider.isValid())) {
                gl.glBindTexture(GL.GL_TEXTURE_2D, glName);
                Dimension size = textureDataProvider.getTextureSize();
                gl.glTexImage2D(
                        GL.GL_TEXTURE_2D,
                        0,          // level 0, no MIP mapping.
                        GL.GL_RGBA,
                        size.width,
                        size.height,
                        0,          // no border.
                        GL.GL_RGBA,
                        GL.GL_FLOAT,
                        textureDataProvider.getData()
                );
                gl.glBindTexture(GL.GL_TEXTURE_2D, 0);
                return true;
            } else {
                return false;
            }
        }
    }
**/
