/*
 * Scilab ( http://www.scilab.org/ ) - This file is part of Scilab
 * Copyright (C) 2009-2011 - DIGITEO - Pierre Lando
 *
 * This file must be used under the terms of the CeCILL.
 * This source file is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at
 * http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt
 */

package org.scilab.forge.scirenderer.implementation.jogl.sprite;

import org.scilab.forge.scirenderer.implementation.jogl.utils.G2DShortCuts;
import org.scilab.forge.scirenderer.implementation.jogl.utils.texture.TextureBufferedImage;
import org.scilab.forge.scirenderer.shapes.appearance.Appearance;
import org.scilab.forge.scirenderer.shapes.appearance.Color;
import org.scilab.forge.scirenderer.sprite.SpriteDrawer;
import org.scilab.forge.scirenderer.sprite.SpriteDrawingTools;
import org.scilab.forge.scirenderer.sprite.TextEntity;

import javax.swing.Icon;
import javax.swing.JLabel;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.font.TextLayout;
import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;

/**
 *
 * Implementation of {@link SpriteDrawingTools}.
 * This implementation create a {@link TextureBufferedImage} an fill it with sprite drawing.
 *
 * @author Pierre Lando
 */
class TextureDrawingTools implements SpriteDrawingTools {

    /**
     * The {@link Graphics2D} used to fill the {@link TextureBufferedImage}
     */
    private Graphics2D g2d;
    private final TextureBufferedImage image;
    private final int spriteWidth;
    private final int spriteHeight;
    private final AffineTransform baseTransform;

    /**
     * Default constructor.
     * @param width the image width.
     * @param height the image height.
     * @param useSquareTexture true if square texture is needed.
     */
    TextureDrawingTools(int width, int height, boolean useSquareTexture) {
        if (useSquareTexture) {
            int size = Math.max(width, height);
            image = new TextureBufferedImage(size, size);
        } else {
            image = new TextureBufferedImage(width, height);
        }
        spriteWidth = width;
        spriteHeight = height;

        double deltaW = (image.getWidth() - spriteWidth) / 2.0;
        double deltaH = (image.getHeight() - spriteHeight) / 2.0;
        baseTransform = AffineTransform.getTranslateInstance(deltaW, deltaH);
    }

    /**
     * Ask this image to accept a sprite drawer.
     * This image will contain the drawing of the given drawer.
     * @param spriteDrawer the given sprite drawer.
     */
    void accept(SpriteDrawer spriteDrawer) {
        g2d = image.createGraphics();
        // Set transparent background.
        g2d.setBackground(new java.awt.Color(0, 0, 0, 0));
        g2d.clearRect(0, 0, image.getWidth(), image.getHeight());

        g2d.transform(baseTransform);

        // Change center coordinate to (0, 0).
        if (spriteDrawer.getOriginPosition() == SpriteDrawer.OriginPosition.CENTER) {
            g2d.translate(spriteWidth / 2.0, spriteHeight / 2.0);
        }

        spriteDrawer.draw(this);

        g2d.dispose();
    }

    @Override
    public void drawPolyline(int[] coordinates, Appearance appearance) {
        int nbPoint = coordinates.length / 2;

        int[] xCoordinates = new int[nbPoint];
        int[] yCoordinates = new int[nbPoint];

        int k = 0;
        for (int i = 0; i < coordinates.length; i += 2) {
            xCoordinates[k] = coordinates[i];
            yCoordinates[k] = coordinates[i + 1];
            k++;
        }

        G2DShortCuts.useLineAppearance(g2d, appearance);
        G2DShortCuts.useColor(g2d, appearance.getLineColor());
        g2d.drawPolyline(xCoordinates, yCoordinates, nbPoint);

    }

    @Override
    public void fillPolygon(int[] coordinates, Appearance appearance) {
        int nbPoint = coordinates.length / 2;

        int[] xCoordinates = new int[nbPoint];
        int[] yCoordinates = new int[nbPoint];

        int k = 0;
        for (int i = 0; i < coordinates.length; i += 2) {
            xCoordinates[k] = coordinates[i];
            yCoordinates[k] = coordinates[i + 1];
            k++;
        }

        G2DShortCuts.useColor(g2d, appearance.getFillColor());
        g2d.fillPolygon(xCoordinates, yCoordinates, nbPoint);

        if (!appearance.getLineColor().equals(appearance.getFillColor())) {
            int usedLength = coordinates.length - (coordinates.length % 2);
            int[] borderCoordinate = new int[usedLength + 2];
            System.arraycopy(coordinates, 0, borderCoordinate, 0, usedLength);
            borderCoordinate[usedLength]     = coordinates[0];
            borderCoordinate[usedLength + 1] = coordinates[1];

            drawPolyline(borderCoordinate, appearance);
        }
    }

    @Override
    public void drawCircle(int x, int y, int diameter, Appearance appearance) {
        G2DShortCuts.useLineAppearance(g2d, appearance);
        G2DShortCuts.useColor(g2d, appearance.getLineColor());
        int r = diameter / 2;
        g2d.drawOval(x - r, y - r, diameter, diameter);
    }

    @Override
    public void fillDisc(int x, int y, int diameter, Color color) {
        G2DShortCuts.useColor(g2d, color);
        int r = diameter / 2;
        g2d.fillOval(x - r, y - r, diameter, diameter);
    }

    @Override
    public void draw(TextEntity textEntity, int x, int y) {
        try {
            if (textEntity.isTextAntiAliased()) {
                g2d.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
            } else {
                g2d.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_OFF);
            }

            if (textEntity.isTextUseFractionalMetrics()) {
                g2d.setRenderingHint(RenderingHints.KEY_FRACTIONALMETRICS, RenderingHints.VALUE_FRACTIONALMETRICS_ON);
            } else {
                g2d.setRenderingHint(RenderingHints.KEY_FRACTIONALMETRICS, RenderingHints.VALUE_FRACTIONALMETRICS_OFF);
            }
            g2d.setColor(textEntity.getTextColor());
            TextLayout textLayout = new TextLayout(textEntity.getText(), textEntity.getFont(), g2d.getFontRenderContext());
            Rectangle2D bounds = textLayout.getBounds();
            textLayout.draw(g2d, (float) (x - bounds.getX()), (float) (y - bounds.getY()));
        } catch (Exception ignored) {
        }
    }

    @Override
    public void draw(Icon icon, int x, int y) {
        icon.paintIcon(new JLabel(), g2d, x, y);
    }

    @Override
    public void clear(Color color) {
        AffineTransform oldTransform = g2d.getTransform();
        g2d.setTransform(baseTransform);
        G2DShortCuts.useColor(g2d, color);
        g2d.fillRect(0, 0, spriteWidth + 1, spriteHeight + 1);
        g2d.setTransform(oldTransform);
    }

    /**
     * Return the image.
     * @return the image.
     */
    public TextureBufferedImage getImage() {
        return image;
    }
}
