/*
 * Scilab ( http://www.scilab.org/ ) - This file is part of Scilab
 * Copyright (C) 2009-2011 - DIGITEO - Pierre Lando
 *
 * This file must be used under the terms of the CeCILL.
 * This source file is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at
 * http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt
 */

package org.scilab.forge.scirenderer.implementation.jogl.drawer;

import org.scilab.forge.scirenderer.SciRendererException;
import org.scilab.forge.scirenderer.buffers.IndicesBuffer;
import org.scilab.forge.scirenderer.implementation.jogl.JoGLDrawingTools;
import org.scilab.forge.scirenderer.implementation.jogl.buffers.JoGLBuffersManager;
import org.scilab.forge.scirenderer.implementation.jogl.buffers.JoGLElementsBuffer;
import org.scilab.forge.scirenderer.implementation.jogl.utils.GLShortCuts;
import org.scilab.forge.scirenderer.shapes.appearance.Appearance;
import org.scilab.forge.scirenderer.shapes.geometry.Geometry;
import org.scilab.forge.scirenderer.texture.Texture;

import javax.media.opengl.GL;
import java.nio.FloatBuffer;
import java.nio.IntBuffer;

/**
 * Utility class for drawing shapes.
 * @author Pierre Lando
 */
public final class JoGLShapeDrawer {

    private static JoGLShapeDrawer drawer;

    /**
     * Private constructor : this is an utility class.
     */
    private JoGLShapeDrawer() {
    }

    public static JoGLShapeDrawer getDrawer() {
        if (drawer == null) {
            drawer = new JoGLShapeDrawer();
        }
        return drawer;
    }

    /**
     * Draw a given geometry with given appearance.
     * @param drawingTools the drawing tools.
     * @param geometry the geometry.
     * @param appearance the appearance.
     * @throws org.scilab.forge.scirenderer.SciRendererException if the draw is not possible.
     */
    public void draw(JoGLDrawingTools drawingTools, Geometry geometry, Appearance appearance) throws SciRendererException {
        GL gl = drawingTools.getGl();
        gl.glFrontFace(GL.GL_CCW);
        switch (geometry.getFaceCullingMode()) {
            case CW:
                gl.glEnable(GL.GL_CULL_FACE);
                gl.glCullFace(GL.GL_FRONT);
                break;
            case CCW:
                gl.glEnable(GL.GL_CULL_FACE);
                gl.glCullFace(GL.GL_BACK);
                break;
            case BOTH:
                gl.glDisable(GL.GL_CULL_FACE);
                break;
        }

        if (drawingTools.getCanvas().getJoGLParameters().useVBO()) {
            vboDrawing(drawingTools, geometry, appearance);
        } else {
            directDrawing(drawingTools, geometry, appearance);
        }

        GLShortCuts.useLineAppearance(gl, null);
        gl.glDisable(GL.GL_CULL_FACE);
    }

    /**
     * Perform geometry drawing using VBO.
     * @param drawingTools the drawing tools.
     * @param geometry the geometry to draw.
     * @param appearance the current appearance.
     * @throws org.scilab.forge.scirenderer.SciRendererException if the draw is not possible.
     */
    private void vboDrawing(JoGLDrawingTools drawingTools, Geometry geometry, Appearance appearance) throws SciRendererException {
        final GL gl = drawingTools.getGl();
        final JoGLBuffersManager buffersManager = drawingTools.getCanvas().getBuffersManager();
        final Texture texture = appearance.getTexture();

        int verticesNumber = buffersManager.bindVertexBuffer(gl, geometry.getVertices());
        if (verticesNumber == 0) {
            gl.glDisableClientState(GL.GL_VERTEX_ARRAY);
            return;
        }

        buffersManager.bindNormalsBuffer(gl, geometry.getNormals());

        if ((texture != null) && (geometry.getTextureCoordinates() != null)) {
            drawingTools.bind(texture);
            buffersManager.bindTextureCoordinatesBuffer(gl, geometry.getTextureCoordinates());
        } else {
            buffersManager.bindColorsBuffer(gl, geometry.getColors());
        }

        // We use polygon offset all the time for filled geometry.
        gl.glEnable(GL.GL_POLYGON_OFFSET_FILL);
        gl.glPolygonOffset(1, 1);
        GLShortCuts.useColor(gl, appearance.getFillColor());

        IndicesBuffer indices = geometry.getIndices();
        if (geometry.getFillDrawingMode() != Geometry.FillDrawingMode.NONE) {
            if (indices != null) {
                int indicesSize = buffersManager.bindIndicesBuffer(gl, indices);
                gl.glDrawElements(getGlMode(geometry.getFillDrawingMode()), indicesSize, GL.GL_UNSIGNED_INT, 0);
                gl.glBindBuffer(GL.GL_ELEMENT_ARRAY_BUFFER, 0);
            } else {
                int count = geometry.getVertices().getSize();
                gl.glDrawArrays(getGlMode(geometry.getFillDrawingMode()), 0, count);
            }
        }

        gl.glDisable(GL.GL_POLYGON_OFFSET_FILL);
        gl.glDisableClientState(GL.GL_COLOR_ARRAY);
        gl.glDisableClientState(GL.GL_NORMAL_ARRAY);
        gl.glDisableClientState(GL.GL_TEXTURE_COORD_ARRAY);
        gl.glDisable(GL.GL_TEXTURE_2D);

        if (geometry.getLineDrawingMode() != Geometry.LineDrawingMode.NONE) {
            if ((appearance.getLineColor() != null) || (geometry.getColors() != null)) {
                GLShortCuts.useLineAppearance(gl, appearance);
                if (appearance.getLineColor() == null) {
                    buffersManager.bindColorsBuffer(gl, geometry.getColors());
                }

                if (geometry.getWireIndices() != null) {
                    int edgesIndicesSize = buffersManager.bindIndicesBuffer(gl, geometry.getWireIndices());
                    gl.glDrawElements(getGlMode(geometry.getLineDrawingMode()), edgesIndicesSize, GL.GL_UNSIGNED_INT, 0);
                    gl.glBindBuffer(GL.GL_ELEMENT_ARRAY_BUFFER, 0);
                } else {
                    int count = geometry.getVertices().getSize();
                    gl.glDrawArrays(getGlMode(geometry.getLineDrawingMode()), 0, count);
                }

                gl.glDisableClientState(GL.GL_COLOR_ARRAY);
            }
        }

        gl.glDisableClientState(GL.GL_VERTEX_ARRAY);
    }

    /**
     * Perform geometry drawing by direct OpenGl call.
     * @param drawingTools the drawing tools.
     * @param geometry the geometry to draw.
     * @param appearance the used appearance.
     * @throws org.scilab.forge.scirenderer.SciRendererException if the draw is not possible.
     */
    private void directDrawing(JoGLDrawingTools drawingTools, Geometry geometry, Appearance appearance) throws SciRendererException {
        GL gl = drawingTools.getGl();
        if (geometry.getVertices() == null) {
            return;
        }

        FloatBuffer vertexBuffer = geometry.getVertices().getData();
        IndicesBuffer indices = geometry.getIndices();

        FloatBuffer colorBuffer;
        if (geometry.getColors() != null) {
            colorBuffer = geometry.getColors().getData();
        } else {
            colorBuffer = null;
        }

        FloatBuffer normalBuffer;
        if (geometry.getNormals() != null) {
            normalBuffer = geometry.getNormals().getData();
        } else {
            normalBuffer = null;
        }

        Texture texture = appearance.getTexture();
        FloatBuffer textureCoordinatesBuffer;
        if ((texture != null) && (geometry.getTextureCoordinates() != null)) {
            drawingTools.bind(texture);
            textureCoordinatesBuffer = geometry.getTextureCoordinates().getData();
        } else {
            textureCoordinatesBuffer = null;
        }

        final int elementsSize = JoGLElementsBuffer.ELEMENT_SIZE;
        final float[] buffer = new float[elementsSize];

        gl.glEnable(GL.GL_POLYGON_OFFSET_FILL);
        gl.glPolygonOffset(1, 1);

        if (geometry.getFillDrawingMode() != Geometry.FillDrawingMode.NONE) {
            GLShortCuts.useColor(gl, appearance.getFillColor());
            gl.glBegin(getGlMode(geometry.getFillDrawingMode()));
            if (indices != null) {
                IntBuffer indicesBuffer = indices.getData();
                indicesBuffer.rewind();

                for (int i = 0; i < indicesBuffer.limit(); i++) {
                    int index = indicesBuffer.get(i);
                    if ((index * elementsSize) < vertexBuffer.limit()) {

                        if (colorBuffer != null) {
                            colorBuffer.position(index * elementsSize);
                            colorBuffer.get(buffer, 0, elementsSize);
                            gl.glColor4fv(buffer, 0);
                        }

                        if (normalBuffer != null) {
                            normalBuffer.position(index * elementsSize);
                            normalBuffer.get(buffer, 0, elementsSize);
                            gl.glNormal3fv(buffer, 0);
                        }

                        if (textureCoordinatesBuffer != null) {
                            textureCoordinatesBuffer.position(index * elementsSize);
                            textureCoordinatesBuffer.get(buffer, 0, elementsSize);
                            gl.glTexCoord4fv(buffer, 0);
                        }

                        vertexBuffer.position(index * elementsSize);
                        vertexBuffer.get(buffer, 0, elementsSize);
                        gl.glVertex4fv(buffer, 0);

                    }
                }
            } else {
                vertexBuffer.rewind();

                if (colorBuffer != null) {
                    colorBuffer.rewind();
                }

                if (normalBuffer != null) {
                    normalBuffer.rewind();
                }

                for (int i = 0; i < vertexBuffer.limit(); i += elementsSize) {
                    if (colorBuffer != null) {
                        colorBuffer.get(buffer, 0, elementsSize);
                        gl.glColor4fv(buffer, 0);
                    }

                    if (normalBuffer != null) {
                        normalBuffer.get(buffer, 0, elementsSize);
                        gl.glNormal3fv(buffer, 0);
                    }

                    if (textureCoordinatesBuffer != null) {
                        textureCoordinatesBuffer.get(buffer, 0, elementsSize);
                        gl.glTexCoord4fv(buffer, 0);
                    }

                    vertexBuffer.get(buffer, 0, elementsSize);
                    gl.glVertex4fv(buffer, 0);
                }
            }
            gl.glEnd();
        }

        gl.glDisable(GL.GL_POLYGON_OFFSET_FILL);
        gl.glDisable(GL.GL_TEXTURE_2D);

        // Draw edges if any.
        if (geometry.getLineDrawingMode() != Geometry.LineDrawingMode.NONE) {
            GLShortCuts.useLineAppearance(gl, appearance);
            if (appearance.getLineColor() != null) {
                gl.glBegin(getGlMode(geometry.getLineDrawingMode()));
                if (geometry.getWireIndices() != null) {
                    IntBuffer edgesIndicesBuffer = geometry.getWireIndices().getData();
                    edgesIndicesBuffer.rewind();
                    while (edgesIndicesBuffer.remaining() != 0) {
                        int index = edgesIndicesBuffer.get();
                        if ((index * elementsSize) < vertexBuffer.limit()) {
                            vertexBuffer.position(index * elementsSize);
                            vertexBuffer.get(buffer, 0, elementsSize);
                            gl.glVertex4fv(buffer, 0);

                        }
                    }
                } else {
                    vertexBuffer.rewind();
                    while (vertexBuffer.remaining() >= elementsSize) {
                        vertexBuffer.get(buffer, 0, elementsSize);
                        gl.glVertex4fv(buffer, 0);
                    }
                }
                gl.glEnd();
            } else if (colorBuffer != null) {
                gl.glBegin(getGlMode(geometry.getLineDrawingMode()));
                if (geometry.getWireIndices() != null) {
                    IntBuffer edgesIndicesBuffer = geometry.getWireIndices().getData();
                    edgesIndicesBuffer.rewind();
                    while (edgesIndicesBuffer.remaining() != 0) {
                        int index = edgesIndicesBuffer.get();
                        if ((index * elementsSize) < vertexBuffer.limit()) {
                            colorBuffer.position(index * elementsSize);
                            colorBuffer.get(buffer, 0, elementsSize);
                            gl.glColor4fv(buffer, 0);
                            vertexBuffer.position(index * elementsSize);
                            vertexBuffer.get(buffer, 0, elementsSize);
                            gl.glVertex4fv(buffer, 0);

                        }
                    }
                } else {
                    vertexBuffer.rewind();
                    colorBuffer.rewind();
                    while (vertexBuffer.remaining() >= elementsSize) {
                        colorBuffer.get(buffer, 0, elementsSize);
                        gl.glColor4fv(buffer, 0);
                        vertexBuffer.get(buffer, 0, elementsSize);
                        gl.glVertex4fv(buffer, 0);
                    }
                }
                gl.glEnd();
            }
        }
    }


    /**
     * Return the gl drawing mode corresponding to the given {@link Geometry.FillDrawingMode}.
     * @param drawingMode the given drawing mode..
     * @return the gl drawing mode corresponding to the given {@link Geometry.FillDrawingMode}.
     */
    private int getGlMode(Geometry.FillDrawingMode drawingMode) {
        switch (drawingMode) {
            case TRIANGLE_FAN:
                return GL.GL_TRIANGLE_FAN;
            case TRIANGLE_STRIP:
                return GL.GL_TRIANGLE_STRIP;
            default:
            case TRIANGLES:
                return GL.GL_TRIANGLES;
        }
    }

    /**
     * Return the gl drawing mode corresponding to the given {@link org.scilab.forge.scirenderer.shapes.geometry.Geometry.LineDrawingMode}
     * @param drawingMode the given drawing mode.
     * @return the gl drawing mode corresponding to the given {@link org.scilab.forge.scirenderer.shapes.geometry.Geometry.LineDrawingMode}
     */
    private int getGlMode(Geometry.LineDrawingMode drawingMode) {
        switch (drawingMode) {
            default:
            case SEGMENTS:
                return GL.GL_LINES;
            case SEGMENTS_LOOP:
                return GL.GL_LINE_LOOP;
            case SEGMENTS_STRIP:
                return GL.GL_LINE_STRIP;
        }
    }
}
