/*
 * Scilab ( http://www.scilab.org/ ) - This file is part of Scilab
 * Copyright (C) 2009-2011 - DIGITEO - Pierre Lando
 *
 * This file must be used under the terms of the CeCILL.
 * This source file is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at
 * http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt
 */

package org.scilab.forge.scirenderer.implementation.jogl;

import org.scilab.forge.scirenderer.Canvas;
import org.scilab.forge.scirenderer.Drawer;
import org.scilab.forge.scirenderer.depthtest.DepthTestManager;
import org.scilab.forge.scirenderer.implementation.jogl.buffers.JoGLBuffersManager;
import org.scilab.forge.scirenderer.implementation.jogl.depthtest.JoGLDepthBufferManager;
import org.scilab.forge.scirenderer.implementation.jogl.picking.JoGLPickingManager;
import org.scilab.forge.scirenderer.implementation.jogl.renderer.JoGLRendererManager;
import org.scilab.forge.scirenderer.implementation.jogl.sprite.JoGLSpriteManager;
import org.scilab.forge.scirenderer.implementation.jogl.texture.JoGLTextureManager;
import org.scilab.forge.scirenderer.picking.PickingManager;

import javax.media.opengl.DebugGL;
import javax.media.opengl.GL;
import javax.media.opengl.GLAutoDrawable;
import javax.media.opengl.GLEventListener;
import java.awt.Dimension;

/**
 * JoGL implementation of a Canvas.
 *
 * @author Pierre Lando
 */
public class JoGLCanvas implements Canvas, GLEventListener {

    private final GLAutoDrawable autoDrawable;

    private final JoGLDrawingTools drawingTools;
    private final JoGLParameters parameters;
    private final JoGLSpriteManager spriteManager;
    private final JoGLBuffersManager buffersManager;
    private final JoGLRendererManager rendererManager;
    private final JoGLDepthBufferManager depthBufferManager;
    private final JoGLPickingManager pickingManager;
    private final JoGLTextureManager textureManager;


    /**
     * The current mainDrawer.
     */
    private Drawer mainDrawer;

    /**
     * Default constructor.
     * @param autoDrawable the JoGL autoDrawable this canvas depend on.
     */
    JoGLCanvas(GLAutoDrawable autoDrawable) {
        this.autoDrawable = autoDrawable;
        parameters = new JoGLParameters();
        spriteManager = new JoGLSpriteManager(this);
        buffersManager = new JoGLBuffersManager();
        rendererManager = new JoGLRendererManager();
        depthBufferManager = new JoGLDepthBufferManager();
        drawingTools = new JoGLDrawingTools(this);
        pickingManager = new JoGLPickingManager(this);
        textureManager = new JoGLTextureManager(this);

        autoDrawable.addGLEventListener(this);
    }

    // Implementation of getter & setter from Canvas.

    @Override
    public void setMainDrawer(Drawer mainDrawer) {
        this.mainDrawer = mainDrawer;
    }

    @Override
    public Drawer getMainDrawer() {
        return mainDrawer;
    }

    @Override
    public JoGLSpriteManager getSpriteManager() {
        return spriteManager;
    }

    @Override
    public JoGLRendererManager getRendererManager() {
        return rendererManager;
    }

    @Override
    public JoGLBuffersManager getBuffersManager() {
        return buffersManager;
    }

    @Override
    public DepthTestManager getDepthTestManager() {
        return depthBufferManager;
    }

    @Override
    public PickingManager getPickingManager() {
        return pickingManager;
    }

    @Override
    public JoGLTextureManager getTextureManager() {
        return textureManager;
    }

    @Override
    public int getWidth() {
        return autoDrawable.getWidth();
    }

    @Override
    public int getHeight() {
        return autoDrawable.getHeight();
    }

    @Override
    public Dimension getDimension() {
        return new Dimension(autoDrawable.getWidth(), autoDrawable.getHeight());
    }

    @Override
    public void redraw() {
        autoDrawable.repaint();
    }


    // JoGLCanvas specific getter.

    /**
     * Return the OpenGl context.
     * @return the OpenGl context.
     */
    public GL getGl() {
        return new DebugGL(autoDrawable.getGL());
        //return autoDrawable.getGL();
    }

    /**
     * Package method to get the sprite manager.
     * @return the sprite manager.
     */
    JoGLSpriteManager getJoGLSpriteManager() {
        return spriteManager;
    }

    /**
     * Return the rendering parameters.
     * @return the rendering parameters.
     */
    public JoGLParameters getJoGLParameters() {
        return parameters;
    }


    // Implementation of function from GLEventListener.

    @Override
    public void display(GLAutoDrawable glAutoDrawable) {
        GL gl = getGl();

        drawingTools.glSynchronize(gl);
        buffersManager.glSynchronize(gl);
        rendererManager.glSynchronize(gl);
        spriteManager.glSynchronize(drawingTools);

        if (mainDrawer != null) {
            gl.glEnable(GL.GL_DEPTH_TEST); // TODO add possibility to (de-)activate depth test.
            gl.glDepthFunc(GL.GL_LEQUAL); // Set to less equal to allow last drawn object to be on the top.
            if (depthBufferManager.isAutoClearDepthBufferEnable()) {
                gl.glClear(GL.GL_DEPTH_BUFFER_BIT);
            }

            mainDrawer.draw(drawingTools);
        }
        pickingManager.glConsume(drawingTools);
    }

    @Override
    public void init(GLAutoDrawable glAutoDrawable) {
        GL gl = getGl();

        drawingTools.glSynchronize(gl);
        buffersManager.glReload(gl);
        rendererManager.glReload();
        spriteManager.glReload(drawingTools);
    }

    @Override
    public void reshape(GLAutoDrawable glAutoDrawable, int x, int y, int width, int height) {
    }

    @Override
    public void displayChanged(GLAutoDrawable glAutoDrawable, boolean b, boolean b1) {
    }
}
