//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [mean_cir_t2, osc_cir_t2] = CL_ex_eckHech (t1, mean_cir_t1, t2,er,mu,zonals )
// Eckstein Hechler orbit propagation analytical model
//
// Calling Sequence
// [mean_cir_t2,osc_cir_t2] = CL_ex_eckHech(t1,mean_cir_t1,t2 [,er,mu,zonals])
//
// Description
// <itemizedlist><listitem>
// The function computes the mean orbital elements <emphasis role="bold">mean_cir_t2</emphasis> and the osculating orbital elements <emphasis role="bold">osc_cir_t2</emphasis>
// at time <emphasis role="bold">t2</emphasis> given the mean orbital elements <emphasis role="bold">mean_cir_t1</emphasis> at time <emphasis role="bold">t1</emphasis>.
// <para>Zonals coefficients up to J6 are taken into account. </para>
// <para> There can either be several sets of orbital elements and one final time or one set of orbital elements 
// and multiple final times. </para>
// </listitem>
// <listitem> The orbital elements are the following:  
// <para><inlinemediaobject><imageobject><imagedata fileref="cir_par.gif"/></imageobject></inlinemediaobject></para>
// </listitem>
// <listitem>Warnings :
// <para> - This function does not work for inclinations close to the critical inclinations (63.43494882 deg and 116.5650512 deg)</para>
// <para> - This function nominally works for eccentricities smaller than 5.e-3</para>
// <para> - This function works but with a lesser accuracy for eccentricities between 5.e-3 and 0.1</para>
// <para> - This function does not work for eccentricities greater than 0.1</para></listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-06-03 )</emphasis></para>
//
// Parameters
// t1: Initial time [days] (1xP)
// mean_cir_t1: Circular adapted mean orbital elements at time t1 [sma;ex;ey;inc;raan;pom+anm] (6xP). P can be > 1 if there is only one final time (i.e. if N==1).
// t2: Final time [days] (1xN). N can be > 1 if there is only one set of orbital elements (i.e. if P==1).
// er: (optional) Equatorial radius [m] (default is %CL_eqRad)
// mu: (optional) Gravitational constant [m^3/s^2] (default value is %CL_mu)
// zonals: (optional) Vector of zonal coefficients J1 to Jn (troncated to J6) to be used (default is %CL_j1jn(1:6)) (1 x N)
// mean_cir_t2: Circular adapted mean orbital elements at t2 [sma;ex;ey;inc;raan;pom+anm] (6 x max(P,N))
// osc_cir_t2: Circular adapted osculating orbital elements at t2 [sma;ex;ey;inc;raan;pom+anm] (6 x max(P,N))
//
// Authors
// CNES - DCT/SB
//
// Bibliography
// 1 CNES - MSLIB FORTRAN 90, Volume E (me_eck_hech)
//
// See also
// CL_ex_lyddane
// CL_ex_meanEckHech
//
// Examples
// // Example 1 : one orbit extrapolated at several dates :
// t1 = 12584
// mean_cir_t1 = [42166712;2.9e-3;1.2e-3;CL_deg2rad([97.2;125;5])]
// t2 = 12587:0.2:12588;
// [mean_cir_t2,osc_cir_t2] = CL_ex_eckHech(t1,mean_cir_t1,t2)
//
// // Example 2 : several orbits extrapolated at one date :
// t1 = 12584
// mean_cir_t1 = [42166712;2.9e-3;1.2e-3;CL_deg2rad([97.2;125;5])]
// mean_cir_t1 = [mean_cir_t1,mean_cir_t1*0.95]
// t2 = 12587;
// [mean_cir_t2,osc_cir_t2] = CL_ex_eckHech(t1,mean_cir_t1,t2)


//************************************************************************
//
// But:  Modele analytique d'extrapolation d'orbite de ECKSTEIN-HECHLER.
// ===
//
// Note d'utilisation:  L'excentricite  doit etre superieure >= 0 et < 0.1.
// ==================   Le domaine d'utilisation du modele est .. [ 0E+0 , 5E-3 [.
//                      Le domaine de precision degradee est .... [ 5E-3 , 1E-1 [.
//                      Le domaine d'erreur est ................. [ 1E-1 , +Oo  [.
//                      L'inclinaison  doit etre > 0. et < pi et non proches des inclinaisons critiques
//                      ( pm_i_critique_non_retro, pm_i_critique_retro ) a pm_eps_i_critique pres.
//
// Historique:
// ==========
//   + Version 2.0 (SP 341 ed01 rev00): creation a partir de la routine MEEHJ6 de la MSLIB f77
//   + Version 3.1 (DE globale 439 ed01 rev00) : ajout des champs %biblio et %message pour le code retour
//   + Version 4.1 (DE globale 482 ed01 rev00): Corrections qualite
//   + Version 4.1 (DE globale 482 ed01 rev00): Corrections qualite
//   + Version 6.0 (DE globale 618 ed01 rev00): ajout de tests sur mu et les coefficients CN0
//************************************************************************

//initialisations

// Declarations:
if(~exists('%CL_criticalInc1')) then global %CL_criticalInc1; end;
if(~exists('%CL_criticalInc2')) then global %CL_criticalInc2; end;
if(~exists('%CL_eqRad')) then global %CL_eqRad; end;
if(~exists('%CL_j1jn')) then global %CL_j1jn; end;
if(~exists('%CL_mu')) then global %CL_mu; end;

// Code:

lhs = argn(1);
compute_osc_cir_t2 = %f;
if lhs==2
  compute_osc_cir_t2 = %t;
else
  CL__error('too many outputs');
end
if ~exists('mu','local') then mu=%CL_mu; end
if ~exists('er','local') then er=%CL_eqRad; end
if ~exists('zonals','local') then zonals=%CL_j1jn(1:6); end

if(length(zonals) < 6)
  warning('zonals must be a vector of size 6 (or more), completing with zeros...');
  zonals = [zonals 0 0 0 0 0 0];
end

// initialisation de la valeur epsilon * 100
eps100 = 100 * %eps
eccel = 0.05

//controle des donnees
//demi grand axe
if or( mean_cir_t1(1,:) <= eps100 )
  CL__error("Error : semi major axis < or = 0");
end
//eccentricite
ecc = sqrt(mean_cir_t1(2,:)**2+(mean_cir_t1(3,:)**2));
if or( ecc < 0 )
  CL__error("Error : negative eccentricity");
end
if or( ecc > 0.1 )
  CL__error("Error : eccentricity higher than 0.1");
end
if or( ecc > 5.e-3)
  warning("Warning : lesser precision for eccentricities between 5e-3 and 0.1");
end

// inclinaison
if or( mean_cir_t1(4,:) < 0 )
  CL__error("negative inclination");
end
if or( mean_cir_t1(4,:) > %pi )
  CL__error("inclination > pi");
end

// inclinaison proche de l'inclinaison critique.
eps_i_critique = 4e-5; // obtenu experimentalement : au dela ca plante
if (abs(mean_cir_t1(4,:) - %CL_criticalInc1) < eps_i_critique | ...
     abs(mean_cir_t1(4,:) - %CL_criticalInc2) < eps_i_critique) then
   CL__error("inclination close (less than "+string(eps_i_critique)+" rad) to the critical value");
end

//cn0 = -1. * [0, %CL_j2, %CL_j3, %CL_j4, %CL_j5, %CL_j6 ] ;
cn0 = -zonals;

// parametres d'orbite adaptes circulaires
// recuperation valeurs parametres meanens en entree
// ................................................
a0   = mean_cir_t1(1,:);
i0   = mean_cir_t1(4,:);
om0  = mean_cir_t1(5,:);
ex0  = mean_cir_t1(2,:);                         // e cos(w)
ey0  = mean_cir_t1(3,:);                         // e sin(w)
l0   = mean_cir_t1(6,:);                         // w + M

// calculs preliminaires
// .....................
// calcul de : g(l) = cn0(l) * ( %CL_eqRad / a0 )**l
q  = er ./ a0;
ql = q;
g = zeros(6,size(mean_cir_t1,2));
for i = 2 : 6
  ql   = ql .* q;
  g(i,:) = cn0(i) * ql;
end

// c    = cos(i0)
// b(i) = sin(i0)**i
c    = cos(i0);
b = zeros(6,size(mean_cir_t1,2));
b(1,:) = sin(i0);

for i = 2 : 6
  j    = i-1;
  b(i,:) = b(j,:) .* b(1,:);
end

xnot  = sqrt(mu./a0) ./ a0 .* (t2 - t1) * 86400.;  // VECT

// demi grand axe
am = a0;

// vecteur excentricite
delta_pom  = -0.75 .* g(2,:) .* ( 4. - 5. .* b(2,:) );
delta_pomp = 7.5 .* g(4,:) .* ( 1. - 31. ./ 8. .* b(2,:) + 49. ./ 16. .* b(4,:) );
delta_pomp = delta_pomp - 13.125 .* g(6,:) .* ( 1. - 8. .* b(2,:) + 129. ./ 8. .* b(4,:) -          ...
    297. ./ 32. .* b(6,:) );
x          = ( delta_pom + delta_pomp ) .* xnot; // VECT
//x          = x - int(x/(2..*%pi)) .* (2..*%pi); // VECT
cx         = cos(x); // VECT
sx         = sin(x); // VECT
q          = 3. ./ 32. ./ delta_pom;
eps1       = q .* g(4,:) .* b(2,:) .* ( 30. - 35. .* b(2,:) ) - 175. .* q .* g(6,:) .* b(2,:)                             ...
    .* ( 1. - 3. .* b(2,:) + 33. ./ 16. .* b(4,:) );
q          = 3. ./ 8. .* b(1,:) ./ delta_pom;
eps2       = q .* g(3,:) .* ( 4. - 5. .* b(2,:) ) - q .* g(5,:) .* ( 10. - 35. .* b(2,:) + 26.25 .* b(4,:) );
exm        = ex0 .* cx - ( 1. - eps1 ) .* ey0 .* sx + eps2 .* sx; // VECT
eym        = ( 1. + eps1 ) .* ex0 .* sx + ( ey0 - eps2 ) .* cx + eps2; // VECT

// inclinaison
im = i0;

// ascension droite du noeud
q   = 1.5 .* g(2,:) - 2.25 .* g(2,:) .* g(2,:) .* ( 2.5 - 19. ./ 6. .* b(2,:) );
q   = q + 15. ./ 16. .* g(4,:) .* ( 7. .* b(2,:) - 4. ) + 105. ./ 32. .* g(6,:)              ...
    .* ( 2. - 9. .* b(2,:) + 33. ./ 4. .* b(4,:) );
omm = om0 + q .* c .* xnot; // VECT
//omm = omm - int(omm ./ (2..*%pi)) .* (2..*%pi); // VECT

// argument de latitude
delta_l = 1. - 1.5 .* g(2,:) .* ( 3. - 4. .* b(2,:) )
q   = delta_l + 2.25 .* g(2,:) .* g(2,:) .* ( 9. - 263. ./ 12. .* b(2,:) + 341. ./ 24. .* b(4,:) );
q   = q + 15. ./ 16. .* g(4,:) .* ( 8. - 31. .* b(2,:) + 24.5 .* b(4,:) );
q   = q + 105. ./ 32. .* g(6,:) .* ( -10. ./ 3. + 25. .* b(2,:) - 48.75                   ...
    .* b(4,:) + 27.5 .* b(6,:) );
lm  = l0 + q .* xnot; // VECT
//lm  = lm - int(lm/(2..*%pi)) .* (2..*%pi); // VECT

// termes periodiques
// ..................

// calcul des cl(i)=cos(i.*lm)  sl(i)=sin(i.*lm)
cl = zeros(6, length(lm));
sl = zeros(6, length(lm));

cl(1,:) = cos(lm);
sl(1,:) = sin(lm);
for i = 2 : 6
  j = i-1;
  cl(i,:) = cl(j,:) .* cl(1,:) - sl(j,:) .* sl(1,:);
  sl(i,:) = cl(j,:) .* sl(1,:) + sl(j,:) .* cl(1,:);
end

// autres quantites
qq = -1.5 .* g(2,:) ./ delta_l;
qh =  3. ./ 8. .* ( eym - eps2 ) ./ delta_pom; // VECT
ql =  3. ./ 8. .* exm ./ b(1,:) ./ delta_pom; // VECT

// sortie des elements meanens a la date t2
// ....................
mean_cir_t2 = [am .* ones(t2); exm; eym; im .* ones(t2); pmodulo(omm,2*%pi); pmodulo(lm,2*%pi)];






// MODELE POUR LES OSCULATEURS
//============================
if compute_osc_cir_t2

  // termes periodiques sur a
  // ............
  // contribution de j2
  f       = ( 2. - 3.5 .* b(2,:) ) .* exm .* cl(1,:) + ( 2. - 2.5 .* b(2,:) ) .* eym .* sl(1,:);
  f       = f + b(2,:) .* cl(2,:) + 3.5 .* b(2,:) .* ( exm .* cl(3,:) + eym .* sl(3,:) );
  delta_a = qq .* f;

  // contribution j2 carre
  q       = 0.75 .* g(2,:) .* g(2,:) .* b(2,:);
  f       = 7. .* ( 2. - 3. .* b(2,:) ) .* cl(2,:) + b(2,:) .* cl(4,:);
  delta_a = delta_a + q .* f;

  // contribution de j3
  q       = -0.75 .* g(3,:) .* b(1,:);
  f       = ( 4. - 5. .* b(2,:) ) .* sl(1,:) + 5. ./ 3. .* b(2,:) .* sl(3,:);
  delta_a = delta_a + q .* f;

  // contribution de j4
  q       = 0.25 .* g(4,:) .* b(2,:);
  f       = ( 15. - 17.5 .* b(2,:) ) .* cl(2,:) + 4.375 .* b(2,:) .* cl(4,:);
  delta_a = delta_a + q .* f;

  // contribution de j5
  q       = 3.75 .* g(5,:) .* b(1,:);
  f       = ( 2.625 .* b(4,:) - 3.5 .* b(2,:) + 1. ) .* sl(1,:);
  f       = f + 7. ./ 6. .* b(2,:) .* ( 1. - 1.125 .* b(2,:) ) .* sl(3,:) + 21.                       ..
    ./ 80. .* b(4,:) .* sl(5,:);
  delta_a = delta_a + q .* f;

  // contribution de j6
  q       = 105. ./ 16. .* g(6,:) .* b(2,:);
  f       = ( 3. .* b(2,:) - 1. - 33. ./ 16. .* b(4,:) ) .* cl(2,:);
  f       = f + 0.75 .* ( 1.1 .* b(4,:) - b(2,:) ) .* cl(4,:) - 11. ./ 80. .* b(4,:) .* cl(6,:);
  delta_a = delta_a + q .* f;

  // termes periodiques sur ex
  // .............
  // contribution de j2
  f        = ( 1. - 1.25 .* b(2,:) ) .* cl(1,:) + 0.5 .* ( 3. - 5. .* b(2,:) ) .* exm .* cl(2,:);
  f        = f + ( 2. - 1.5 .* b(2,:) ) .* eym .* sl(2,:) + 7. ./ 12. .* b(2,:) .* cl(3,:);
  f        = f + 17. ./ 8. .* b(2,:) .* ( exm .* cl(4,:) + eym .* sl(4,:) );
  delta_ex = qq .* f;

  // termes periodiques sur ey
  // .............
  // contribution j2
  f        = ( 1. - 1.75 .* b(2,:) ) .* sl(1,:) + ( 1. - 3. .* b(2,:) ) .* exm .* sl(2,:);
  f        = f + ( 2. .* b(2,:) - 1.5 ) .* eym .* cl(2,:) + 7. ./ 12. .* b(2,:) .* sl(3,:);
  f        = f + 17. ./ 8. .* b(2,:) .* ( exm .* sl(4,:) - eym .* cl(4,:) );
  delta_ey = qq .* f;

  // termes periodiques sur om
  // .............
  // contribution de j2
  q         = -qq .* c;
  f         = 3.5 .* exm .* sl(1,:) - 2.5 .* eym .* cl(1,:) - 0.5 .* sl(2,:);
  f         = f + 7. ./ 6. .* ( eym .* cl(3,:) - exm .* sl(3,:) );
  delta_gom = q .* f;

  // contribution de j3
  f         = g(3,:) .* c .* ( 4. - 15. .* b(2,:) );
  delta_gom = delta_gom + ql .* f;

  // contribution de j5
  f         = 2.5 .* g(5,:) .* c .* ( 4. - 42. .* b(2,:) + 52.5 .* b(4,:) );
  delta_gom = delta_gom - ql .* f;

  // termes periodiques sur xi
  // .............
  // contribution de j2
  q       = 0.5 .* qq .* b(1,:) .* c;
  f       = eym .* sl(1,:) - exm .* cl(1,:) + cl(2,:);
  f       = f + 7. ./ 3. .* ( exm .* cl(3,:) + eym .* sl(3,:) );
  delta_i = q .* f;

  // contribution de j3
  f       = g(3,:) .* c .* ( 4. - 5. .* b(2,:) );
  delta_i = delta_i - qh .* f;

  // contribution de j5
  f       = 2.5 .* g(5,:) .* c .* ( 4. - 14. .* b(2,:) + 10.5 .* b(4,:) );
  delta_i = delta_i + qh .* f;

  // termes periodiques sur xl
  // .............
  // contribution de j2
  f        = ( 7. - 77. ./ 8. .* b(2,:) ) .* exm .* sl(1,:) +                                                      ..
    ( 55. ./ 8. .* b(2,:) - 7.5 ) .* eym .* cl(1,:);
  f        = f + ( 1.25 .* b(2,:) - 0.5 ) .* sl(2,:);
  f        = f + ( 77. ./ 24. .* b(2,:) - 7. ./ 6. ) .* ( exm .* sl(3,:) - eym .* cl(3,:) );
  ddelta_l = qq .* f;

  // contribution de j3
  f        = g(3,:) .* ( 53. .* b(2,:) - 4. - 57.5 .* b(4,:) );
  ddelta_l = ddelta_l + ql .* f;

  // contribution de j5
  f        = 2.5 .* g(5,:) .* ( 4. - 96. .* b(2,:) + 269.5 .* b(4,:) - 183.75 .* b(6,:) );
  ddelta_l = ddelta_l + ql .* f;

  // parametres osculateurs
  // ...........
  ex = exm + delta_ex;
  ey = eym + delta_ey;
  om = omm + delta_gom;
  xi = im + delta_i;
  xl = lm + ddelta_l;

  // sortie optionnelle des parametres osculateurs a la date t2
  // .............................
  osc_cir_t2 = [am .* ( 1.  + delta_a ); ex; ey; xi; pmodulo(om,2*%pi); pmodulo(xl,2*%pi)];
end


endfunction
