module Schleuder
  class Mail < TMail::Mail
    # The keywords (X-commands) collected from incoming mail
    attr_accessor :keywords
    # Schleuder::Member's the mail shall be sent to
    attr_accessor :recipients
    # Additional data that is to be stored into internally sent
    # mails. Must be a Hash.
    attr_accessor :metadata
    # Indicator if the incoming mail was encrypted
    attr_accessor :in_encrypted
    # Indicator if (and by whom) the incoming mail was signed. Is +false+ or a
    # GPGME::Signature.
    attr_accessor :in_signed
    attr_accessor :resend_to
    # An instance of Schleuder::Crypt
    attr_accessor :crypt
    # The incoming  mail in raw form (string). Needed to verify detached
    # signatures (see +decrypt!+)
    attr_accessor :original_message
    # Shall we consider this email to be sent by the list-admin?
    attr_reader :from_admin
    # Shall we consider this email to be sent by a list-member?
    attr_reader :from_member

    # internal message_id we'll use for all outgoing mails
    @@schleuder_message_id = nil

    # Instantiates some variables (a.o. @crypt)
    def initialize(*data)
      super(*data)
      @resend_to = []
      @metadata = {}
      @keywords = {}
      @crypt = Crypt.new(Schleuder.list.config.gpg_password)
    end

    # Overwrites TMail.parse to store the original incoming data. We possibly
    # need that to verify pgp-signatures (TMail changes headers which
    # invalidates the signature)
    def self.parse(string)
      foo = super(string)
      foo.original_message = string
      foo
    end

    # Desctructivly decrypt/verify +self+.
    def decrypt!
      # Note: We don't recurse into nested Mime-parts. Only the first level
      # will be touched
      if self.multipart? and !self.type_param('protocol').nil? and self.type_param('protocol').match(/^application\/pgp.*/)
        Schleuder.log.debug 'mail is pgp/mime-formatted'
        if self.sub_type == 'signed'
          Schleuder.log.debug 'mail is signed but not encrypted'
          Schleuder.log.debug 'Parsing original_message to split content from signature'

          # first, identify the boundary
          bm = self.original_message.match(/boundary="?'?([^"';]*)/)
          boundary = "--" + Regexp.escape(bm[1])
          Schleuder.log.debug "Identified boundary: #{boundary}"
          # next, find the signed string between the first and the second-last boundary
          signed_string = self.original_message.match(/.*#{boundary}\r?\n(.*?)\r?\n#{boundary}.*?#{boundary}.*?/m)[1]
          # Add CRs, which probably have been stripped by the MTA
          signed_string.gsub!(/\n/, "\r\n") unless signed_string.include?("\r")
          Schleuder.log.debug "Identified signed string"
          # third, find the pgp-signature
          signature = self.original_message.match(/.*(-----BEGIN PGP SIGNATURE-----.*?-----END PGP SIGNATURE-----).*/m)[1] rescue ''
          Schleuder.log.debug "Identified signature"
          # finally, verify
          Schleuder.log.info "Verifying"
          foo, self.in_signed = self.crypt.verify(signature, signed_string)
          plaintext = self.parts[0].to_s
        elsif self.sub_type == 'encrypted'
          Schleuder.log.debug 'mail is encrypted (and possibly signed)'
          plaintext, self.in_encrypted, self.in_signed = self.crypt.decrypt(self.parts[1].body)
        else
          Schleuder.log.warn "Strange: message claims to be pgp/mime but neither to be signed nor encrypted"
        end
        # replace encrypted content with cleartext content
        plainmsg = Mail.parse(plaintext)
        #Schleuder.log.debug "plainmsg: #{plainmsg.to_s.inspect}"
        # test for signed content within the previously encrypted content
        if plainmsg._content_type_stripped == 'multipart/signed'
          Schleuder.log.debug "Found signed message as cleartext, recurseing once"
          plainmsg.original_message = plaintext
          plainmsg.decrypt!
          Schleuder.log.debug "End of recursion"
          #Schleuder.log.debug "plainmsg after decrypt: #{plainmsg.to_s.inspect}"
          self.in_signed = plainmsg.in_signed
        end
        # get headers and body(parts) into self
        if plainmsg.multipart?
          Schleuder.log.debug "plainmsg.multipart? => true"
          self.parts.clear
          plainmsg.parts.each do |p|
            self.parts.push(Mail.parse(p.to_s))
          end
        else
          Schleuder.log.debug "plainmsg.multipart? => false"
          self.body = plainmsg.body
        end
        self.content_type = plainmsg._content_type
        self.disposition = plainmsg._disposition
        #Schleuder.log.debug "plainmsg: #{plainmsg.to_s}"
        #Schleuder.log.debug "self after decrypt: #{self.to_s}"
      else
        Schleuder.log.debug 'no pgp-mime found, looking for pgp-inline'
        # Do we simply push everything to crypt as that should return plain
        # text if it is feeded plain text?  Or does crypt return only an error
        # if there is nothing to do for it?
        if self.multipart?
          Schleuder.log.debug 'multipart found, looking for pgp content in each part'
          self.parts.each do |part|
            _decrypt_pgp_inline(part)
          end
        else
          Schleuder.log.debug 'single inline content found, looking for pgp content'
          _decrypt_pgp_inline(self)
        end
      end
      _parse_signature
    end

    # Destructivly encrypt and sign. +receiver+ must be a string or a Schleuder::Member
    def encrypt!(receiver)
      if receiver.kind_of?(String)
        receiver = Member.new({'email' => receiver})
      elsif ! receiver.kind_of?(Member)
        raise "need a Member-object as argument, got '#{receiver.inspect}'"
      end

      Schleuder.log.info "Encrypting for #{receiver.inspect}"

      keystring = receiver.key || receiver.email
      unless @crypt.get_key(keystring)
        Schleuder.log.warn "No public key found for #{keystring}"
        return false
      end

      # choose pgp-variant from addresses value oder config.default_mime
      pgpvariant = receiver.mime || Schleuder.list.config.default_mime

      case pgpvariant.upcase
      when 'PLAIN'
        Schleuder.log.debug "encrypting as text/plain"
        if self.multipart?
          self.parts.each_with_index do |p,i|
            self.parts[i] = _encrypt_pgp_inline(p, receiver)
          end
          self.content_type = 'multipart/mixed'
          self.encoding = nil
        else
          foo =  _encrypt_pgp_inline(self, receiver)
          self.body = foo.body
          foo.header.each do |k,v|
            self[k] = v.to_s rescue nil
          end
        end
      when 'MIME'
        Schleuder.log.debug "encrypting pgp-mime"
        gpgcontent = Mail.new
        if self.multipart?
          self.parts.each do |p|
            gpgcontent.parts.push(p)
          end
        else
          gpgcontent.body = self.body
          gpgcontent.content_type = self._content_type
          gpgcontent.charset = self.charset || 'UTF-8'
        end

        #Schleuder.log.debug "mail right before encrypting: #{gpgcontent.to_s}"
        gpgcontainer = _new_part(@crypt.encrypt_str(gpgcontent.to_s, receiver), 'application/octet-stream', self.charset)
        gpgcontainer.set_disposition('inline', {:filename => 'message.asc'})

        mimecontainer = _new_part("Version: 1\n", 'application/pgp-encrypted')
        mimecontainer.disposition = 'attachment'

        self.encoding = nil
        self.body = ""
        self.parts.clear
        self.parts.push(mimecontainer)
        self.parts.push(gpgcontainer)
        self.set_content_type('multipart', 'encrypted', {:protocol => 'application/pgp-encrypted'})

      when 'APPL'
        Schleuder.log.error "mime-setting 'APPL' is deprecated, use 'MIME' instead for user '#{receiver.email}'"
      else
        Schleuder.log.error "Unknown mime-setting for user '#{receiver.email}': #{pgpvariant}"
      end
      return true
    end

    # Clearsign the message. Returns a string (raw msg, which must not be
    # changed anymore) or a Schleuder::Mail.
    def sign
      Schleuder.log.debug "signing message"
      if self.to.length > 1
        Schleuder.log.error "More than one recipient in To-header"
        return false
      end
      member = Schleuder.list.find_member_by_email(self.to.to_s)
      # ugly but neccessary: member might be false and member.mime might be nil
      pgpvariant = member.mime || Schleuder.list.config.default_mime rescue Schleuder.list.config.default_mime
      case pgpvariant.upcase
      when 'MIME'
        Schleuder.log.debug "signing MIME"
        gpgcontainer = Mail.new
        if self.multipart?
          self.parts.each do |p|
            gpgcontainer.parts.push(Mail.parse(p.to_s))
          end
          self.parts.clear
        else
          gpgcontainer.body = self.body.to_s
          gpgcontainer.content_type = self._content_type
          gpgcontainer.disposition = self._disposition
          self.body = ''
        end

        self.encoding = ''
        self.parts.push gpgcontainer

        # This following part is quite complicated. We have to do it this way
        # because TMail changes the mime-boundary on every encoded() (implicit
        # in to_s()), which invalidates the signature

        # TODO: refactor with encrypt!()

        # First we create the signature-attachment and fill it with a dummy
        dummy = "dummytobereplaced-#{Time.now.to_f}"
        sigpart = _new_part(dummy, 'application/pgp-signature')
        self.parts.push sigpart

        # TODO: take care of micalg: the digest used for hashing the plaintext.
        # RFC 3156 requires it to be set in the content-type. ruby-gpgme
        # doesn't provide it to us, though.
        # (Don't use symbols with TMail, it expects strings.)
        self.set_content_type('multipart', 'signed', {'protocol' => 'application/pgp-signature', 'micalg' => 'pgp-sha1'})

        # Then we dump the crafted msg
        rawmsg = self.encoded

        # get mime boundary from raw mail
        bm = rawmsg.match(/boundary="?'?([^"';]*)/)
        boundary = "--" + Regexp.escape(bm[1])
        # Now get the to be signed string from the raw message
        tobesigned = rawmsg.match(/.*#{boundary}\r?\n(.*?)\r?\n#{boundary}.*?#{boundary}.*?/m)[1]
        # For some reason I don't manage to do this in one gsub-call... *snif*
        tobesigned_crlf = tobesigned.gsub(/\n/, "\r\n").gsub(/\r\r/, "\r")
        # For another stupid reason tobesigned doesn't match -- although it was
        # extracted from rawmsg just a few seconds ago...
        rawmsg.gsub!(/(.*#{boundary}\r?\n).*?(\r?\n#{boundary}.*?#{boundary}.*?)/m, "\\1#{tobesigned_crlf}\\2").inspect

        # sign the extracted part
        sig = self.crypt.sign(tobesigned_crlf)
        # replace dummy with signature
        rawmsg.gsub!(/(.*)#{dummy}(.*?#{boundary}.*?)/m, "\\1#{sig}\\2")
        rawmsg

      when 'PLAIN'
        Schleuder.log.debug "signing PLAIN"
        if self.multipart?
          self.parts.each_with_index do |p,i|
            if p.disposition == 'attachment' or ! p.disposition_param('filename').nil?
              Schleuder.log.debug "found attachment, creating detached signature"
              # attachment, need detached sig
              container = Mail.new
              container.parts.push Mail.parse(p.to_s)

              sig = self.crypt.sign(p.body)
              Schleuder.log.debug "sig: #{sig.inspect}"
              sigpart = _new_part(sig, 'text/plain')
              sigpart.set_disposition('attachment', {:filename => p.disposition_param('filename') + '.asc'})
              container.parts.push Mail.parse(sigpart.to_s)

              container.content_type = 'multipart/mixed'

              self.parts[i] = Mail.parse(container.to_s)
            else
              p.body = self.crypt.clearsign(p.body)
            end
          end
        else
          self.body = self.crypt.clearsign(self.body)
        end
        self

      else
        Schleuder.log.error "Strange mime-setting: #{pgpvariant}. Don't know what to do with that, ignoring it."
        self
      end
    end

    # Create a new Schleuder::Mail-instance and copy/set selected headers.
    def individualize(recv)
      new = Mail.new
      new.crypt = @crypt
      # add some headers we want to keep
      new.content_type = self['content-type'].to_s if self['content-type']
      new.disposition = self._disposition
      new.encoding = self.encoding.to_s
      new.subject = _quote_if_necessary(self.subject.to_s,'UTF-8')

      new.message_id = Mail._schleuder_message_id

      new.to = recv.email.to_s
      new.date = Time.now
      new.from = Schleuder.list.config.myaddr
      new = _add_openpgp_header(new) if Schleuder.list.config.include_openpgp_header
      if self.multipart?
        self.parts.each do |p|
          new.parts.push(Mail.parse(p.to_s))
        end
      else
        new.body = self.body
      end
      new
    end

    def individualize_member(recv)
      new = self.individualize(recv)
      _message_ids(new) if Schleuder.list.config.keep_msgid
      _list_headers(new) if Schleuder.list.config.include_list_headers
    end

    # Strips keywords from self and stores them in @+keywords+
    def collect_keywords!
      # we're looking for keywords only in the first mime-part
      b = self
      while b.multipart?
        b = b.parts.first
      end
      # split to array to ease parsing
      a = b.body.to_a

      a.map! do |line|
        if line.match(/^X-.*/)
          Schleuder.log.debug "found keyword: #{line.chomp}"
          key, val = $&.split(':')
          if not self.from_admin and Schleuder.list.config.keywords_admin_only.include? key.slice(2..-1)
            Schleuder.log.info "keyword is listed as admin only and mail is not from admin, skipping"
            # return the line to have it stay in the body
            line
          else
            Schleuder.log.info "Storing keyword #{key}"
            val = '' if val.nil?
            if @keywords[key]
              @keywords[key] += ",#{val.strip}"
            else
              @keywords[key] = val.strip
            end
            nil
          end
        elsif line.chomp.empty?
          line
        else
          # break on the first non-empty and non-command line so we don't parse
          # the whole message (could be much data)
          break
        end
      end
      # delete nil's (lines formerly containing X-Commands) from array
      a.compact!
      # rejoin to string
      b.body = a.join("")
      # The whole procedure makes TMail parse and reformat the message, so now
      # it's decoded utf-8
      b.charset = 'UTF-8'
      b.encoding = nil
    end

    # +require+s all found plugins, tests SomePlugin.match and if that returns
    # true runs SomePlugin.process
    def process_plugins!
      if self.keywords.empty?
        Schleuder.log.info 'No keywords present, skipping plugins'
      elsif File.directory? Schleuder.config.plugins_dir
        Dir[Schleuder.config.plugins_dir + '/*_plugin.rb'].each do |plugfile|
          Schleuder.log.debug "processing file #{plugfile} as plugin"
          require plugfile
          # interpreting class name from file name
          classname = File.basename(plugfile, '.rb').split('_').collect { |p| p.capitalize }.join
          plugin = instance_eval(classname).new
          Schleuder.log.debug "Testing #{plugin.class}.match..."
          if plugin.match(self)
            Schleuder.log.debug "#{plugin.class}.match matched -- executing #{plugin.class}.process"
            plugin.process(self)
          else
            Schleuder.log.debug "#{plugin.class}.match didn't match"
          end
        end
      else
        Schleuder.log.error "#{Schleuder.config.plugins_dir} does not exist or is not readable!"
      end
    end

    def add_prefix!
      # only add if it's not already present
      unless self.subject.index(Schleuder.list.config.prefix)
        Schleuder.log.debug "adding prefix"
        self.subject = Schleuder.list.config.prefix + " " + self.subject.to_s
      end
    end

    def add_prefix_in!
      # only add if it's not already present
      unless self.subject.index(Schleuder.list.config.prefix_in)
        Schleuder.log.debug "adding prefix_in"
        self.subject = Schleuder.list.config.prefix_in + " " + self.subject.to_s
      end
    end

    def add_prefix_out!
      # only add if it's not already present
      unless self.subject.index(Schleuder.list.config.prefix_out)
        Schleuder.log.debug "adding prefix_out"
        self.subject = Schleuder.list.config.prefix_out + " " + self.subject.to_s
      end
    end

    def add_metadata!
      Schleuder.log.info "Adding meta-information on old mail to new mail"
      Schleuder.log.debug 'Generating meta-information'
      meta = ''
      Schleuder.list.config.headers_to_meta.each do |h|
        h = h.to_s.capitalize
        val = self.header_string(h) rescue '(not parsable)'
        next if val.nil?
        val = TMail::Unquoter.unquote_and_convert_to(val,'utf-8')
        meta << "#{h}: #{val}\n"
      end
      meta << "Enc: #{_enc_str @in_encrypted}\n"
      meta << "Sig: #{_sig_str @in_signed}\n"
      if self.multipart?
        self.parts.each_with_index do |p,i|
          unless p.in_encrypted.nil? and p.in_signed.nil?
            meta << "part #{i+1}:\n"
            meta << "  enc: #{_enc_str p.in_encrypted}\n"
            meta << "  sig: #{_sig_str p.in_signed}\n"
          end
        end
      end

      unless @metadata.empty?
        Schleuder.log.debug 'Adding extra meta data'
        @metadata.each do |name, content|
          meta << "#{name.gsub(/_/, '-').capitalize}: #{content.to_s}\n"
        end
      end

      meta << "\n"

      # insert oder prepend to the message
      if self.first_part._content_type_stripped == 'text/plain'
        Schleuder.log.debug "Glueing meta data into first part of message"
        self.first_part.body = meta + self.first_part.body
        # body is now utf-8 and decoded!
        self.first_part.charset = 'UTF-8'
        self.first_part.encoding = ''
      else
        # make the message multipart and prepend the meta-part 
        self.to_multipart! unless self.multipart?
        Schleuder.log.debug "Prepending meta data as own mime part to message"
        self.parts.unshift _new_part(meta, 'text/plain', 'UTF-8')
      end

    end

    # Adds Schleuder::ListConfig.public_footer to the end of the body of self
    # or the body of the first mimepart (if one of those is text/plain) or
    # appends it as a new mimepart.
    def add_public_footer!
      Schleuder.log.debug "appending public footer"
      footer = "\n\n-- \n" + Schleuder.list.config.public_footer

      if self.first_part._content_type_stripped == 'text/plain'
        self.first_part.body = self.first_part.body.to_s + footer
      else
        self.to_multipart! unless self.multipart?
        self.parts.push _new_part(footer, 'text/plain', 'UTF-8')
      end
    end


    def to_multipart!
      # skip if already multipart
      return false if self.multipart?
      Schleuder.log.debug "Making message multipart"
      # else move the body into a mime-part
      p = _new_part(self.body, self['content-type'].to_s, self.charset, self.encoding)
      p.disposition = self._disposition
      self.parts.push p
      self.body = ''
      self.set_content_type 'multipart', 'mixed', {:boundary => TMail.new_boundary}
      self.disposition = ''
      #Schleuder.log.debug self.to_s
    end

    def first_part
      if self.multipart?
        self.parts.first
      else
        self
      end
    end

    def _content_type(default = 'text/plain')
      (self['content-type'] || default).to_s rescue default
    end

    def _content_type_stripped(default = nil)
      ( default and _content_type(default) or _content_type() ).to_s.split(';').first
    end

    def _disposition
      (self['content-disposition'] || '').to_s rescue ''
    end

    private

    def _decrypt_pgp_inline(msg)
      if msg.body =~ /^-----BEGIN PGP.*/
        Schleuder.log.debug 'found pgp-inline in input'
        # Look for charset-armor-header. In most cases useless but nobody shall say we didn't try.
        msg.body.each_line do |l|
          break if l.strip.empty?
          next unless m = l.match(/^Charset:\s(.*)$/)
          @charset = m[1]
        end
        msg.body, msg.in_encrypted, msg.in_signed = @crypt.decrypt(msg.body)
        if msg.content_type and msg['content-type'].params['x-action'] =~ /^pgp/
          msg['content-type'].params['x-action'] = nil
        end
        # RFC 2440 defines that the armored text by default is utf-8
        msg.charset = @charset || 'UTF-8'
      else
        Schleuder.log.debug 'no pgp-inline-data found, doing nothing'
      end
    end

    def _encrypt_pgp_inline(msg, receiver)
      #Schleuder.log.debug "encoding: #{msg.encoding}"
      msg.body = @crypt.encrypt_str(msg.body.to_s, receiver)
      # reset encoding, TMail has converted the content
      msg.encoding = nil
      # if attachment add '.gpg'-suffix to attachments file names
      # (The query for 'filename' is a work around against buggy client
      # formatting that doesn't disposition attachments as attachments. This
      # variant works ok.)
      unless msg.disposition_param('filename').nil?
        msg.set_content_type('application', 'octet-stream', {'x-action' => 'pgp-encrypted'})
        msg.set_disposition('attachment', {:filename => msg.disposition_param('filename') + '.gpg'})
      else
        msg.set_content_type('text', 'plain', {'x-action' => 'pgp-encrypted'})
        msg.charset = 'UTF-8'
      end
      msg
    end

    def _enc_str arg
      arg and 'encrypted' or 'unencrypted'
    end

    def _sig_str arg
      if arg
        if self.crypt.get_key(arg.fpr)
          arg.to_s
        else
          #Schleuder.log.debug arg.inspect
          "Unknown signature from #{arg.fpr} (public key not present)"
        end
      else
        'No signature'
      end
    end

    # Tests if the signature of the incoming mail (if any) belongs to a
    # list-member or adminaddr by testing all gpg-key-ids of the signing
    # key for matches
    def _parse_signature
      @from_admin = false
      @from_member = false

      Schleuder.log.debug 'Testing for valid signature'
      if ! @in_signed
        Schleuder.log.info 'No signature found'
      elsif @in_signed.status != 0
        Schleuder.log.info 'Invalid or unknown signature found'
      else
        Schleuder.log.info 'Valid signature found'
        if key = @crypt.get_key(@in_signed.fpr)
          Schleuder.log.debug 'Testing UIDs from key for membership or matching adminaddr'
          key.uids.each do |u|
            if Schleuder.list.is_admin?(u.email) or Schleuder.list.is_admin?(u.name)
              Schleuder.log.debug "#{u.inspect} matched as adminaddr"
              @from_admin = true
            end
            if Schleuder.list.is_member?(u.email) or Schleuder.list.is_member?(u.name)
              Schleuder.log.debug "#{u.inspect} matches as member"
              @from_member = true
            end
          end
        end
      end
    end

    # Convert the given text into quoted printable format, with an instruction
    # that the text be eventually interpreted in the given charset.
    def _quoted_printable(text, charset)
      text = text.gsub( /[^a-z ]/i ) { _quoted_printable_encode($&) }.
                  gsub( / /, "_" )
      "=?#{charset}?Q?#{text}?="
    end

    # Convert the given character to quoted printable format, taking into
    # account multi-byte characters (if executing with $KCODE="u", for instance)
    def _quoted_printable_encode(character)
      result = ""
      character.each_byte { |b| result << "=%02X" % b }
      result
    end

    # A quick-and-dirty regexp for determining whether a string contains any
    # characters that need escaping.
    if !defined?(CHARS_NEEDING_QUOTING)
      CHARS_NEEDING_QUOTING = /[\000-\011\013\014\016-\037\177-\377]/
    end

    # Quote the given text if it contains any "illegal" characters
    def _quote_if_necessary(text, charset)
      text = text.dup.force_encoding(Encoding::ASCII_8BIT) if text.respond_to?(:force_encoding)

      (text =~ CHARS_NEEDING_QUOTING) ?
        _quoted_printable(text, charset) :
        text
    end

    # Helper for creating new message-parts
    def _new_part(body, content_type, charset='', encoding='')
      p = Mail.new
      p.body = body.to_s
      p.content_type = content_type
      p.charset = charset.to_s unless charset.to_s.empty?
      p.encoding = encoding.to_s unless encoding.to_s.empty?
      p
    end

    def _collect_message_ids(ids)
      return nil if ids.nil? or not ids.is_a?(Array) or ids.empty?
      ids.select{ |id| Utils.schleuder_id?(id,Schleuder.list.listname) }
    end

    def self._schleuder_message_id
      @@schleuder_message_id = Utils.generate_message_id(Schleuder.list.listname) unless @@schleuder_message_id
      @@schleuder_message_id
    end

    def _message_ids(mail)
      Schleuder.log.debug "Copying msgid to in-reply-to/references"
      mail.in_reply_to = _collect_message_ids(self.in_reply_to)
      mail.references = _collect_message_ids(self.references)
      mail
    end

    def _list_headers(mail)
      Schleuder.log.debug "Generating list-ids"
      mail['List-Id'] = _list_id
      mail
    end

    def _list_id
      "<#{Schleuder.list.config.myaddr.gsub(/@/,'.')}>"
    end

    def _add_openpgp_header(mail)
      Schleuder.log.debug "Add OpenPGP-Headers"
      mail['OpenPGP'] = "id=#{@crypt.get_key(Schleuder.list.config.myaddr).subkeys.first.keyid} "+
                        "(Send an email to #{Schleuder.list.sendkey_addr} to receive the public-key)"+
                        _gen_openpgp_pref_header
      mail
    end

    def _gen_openpgp_pref_header
      unless Schleuder.list.config.openpgp_header_preference == 'none'
        pref_str = "; preference=#{Schleuder.list.config.openpgp_header_preference} ("
        if !Schleuder.list.config.receive_authenticated_only
          if Schleuder.list.config.receive_encrypted_only \
              && Schleuder.list.config.receive_signed_only
            pref_str << 'Only encrypted and signed emails are accepted'
          elsif Schleuder.list.config.receive_encrypted_only \
                  && !Schleuder.list.config.receive_signed_only
            pref_str << 'Only encrypted emails are accepted'
          elsif !Schleuder.list.config.receive_encrypted_only \
                  && Schleuder.list.config.receive_signed_only
            pref_str << 'Only signed emails are accepted'
          else
            pref_str << 'All kind of emails are accepted'
          end
        elsif Schleuder.list.config.receive_authenticated_only
          if Schleuder.list.config.receive_encrypted_only
            pref_str << 'Only encrypted and signed emails by list-members are accepted'
          else
            pref_str << 'Only signed emails by list-members are accepted'
          end
        else
          pref_str << 'All kind of emails are accepted'
        end
        pref_str << ')'
      end
      pref_str || ''
    end
  end
end
