/*
 * scamper_ping.h
 *
 * $Id: scamper_ping.h,v 1.12 2006/12/12 01:16:03 mjl Exp $
 *
 * Copyright (C) 2005-2006 The University of Waikato
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef __SCAMPER_PING_H
#define __SCAMPER_PING_H

#define SCAMPER_PING_REPLY_IS_ICMP_ECHO_REPLY(reply) ( \
 (reply->addr->type == SCAMPER_ADDR_TYPE_IPV4 && reply->icmp_type == 0) || \
 (reply->addr->type == SCAMPER_ADDR_TYPE_IPV6 && reply->icmp_type == 129))

#define SCAMPER_PING_REPLY_IS_ICMP_UNREACH(reply) ( \
 (reply->addr->type == SCAMPER_ADDR_TYPE_IPV4 && reply->icmp_type == 3) || \
 (reply->addr->type == SCAMPER_ADDR_TYPE_IPV6 && reply->icmp_type == 1))

#define SCAMPER_PING_REPLY_IS_ICMP_TTL_EXP(reply) ( \
 (reply->addr->af == SCAMPER_ADDR_TYPE_IPV4 && reply->icmp_type == 11) || \
 (reply->addr->af == SCAMPER_ADDR_TYPE_IPV6 && reply->icmp_type == 3))

#define SCAMPER_PING_STOP_NONE      0x00 /* null reason */
#define SCAMPER_PING_STOP_COMPLETED 0x01 /* sent all probes */
#define SCAMPER_PING_STOP_ERROR     0x02 /* error occured during ping */

#define SCAMPER_PING_REPLY_FLAG_REPLY_TTL  0x01 /* reply ttl included */

/*
 * scamper_ping_reply
 *
 * a ping reply structure keeps track of how a ping packet was responded to.
 * the default structure has enough fields for interesting pieces out of an
 * echo reply packet.
 *
 * if the icmp type/code is not an ICMP echo reply packet, then the TLVs
 * defined above may be present in the response.
 */
typedef struct scamper_ping_reply
{
  /* where the response came from */
  scamper_addr_t            *addr;

  /* flags defined by SCAMPER_PING_REPLY_FLAG_* */
  uint8_t                    flags;

  /* the TTL / size of the packet that is returned */
  uint8_t                    reply_ttl;
  uint16_t                   reply_size;
  uint16_t                   probe_id;

  /* the icmp type / code returned */
  uint8_t                    icmp_type;
  uint8_t                    icmp_code;

  /* the time elapsed between sending the probe and getting this response */
  struct timeval             rtt;

  /* if a single probe gets more than one response, they get chained */
  struct scamper_ping_reply *next;

} scamper_ping_reply_t;

/*
 * scamper_ping
 *
 * this structure contains details of a ping between a source and a
 * destination.  is specifies the parameters to the ping and the
 * replies themselves.
 */
typedef struct scamper_ping
{
  /* the list / cycle that this ping is in relation to */
  scamper_list_t        *list;
  scamper_cycle_t       *cycle;

  /* source and destination addresses of the ping */
  scamper_addr_t        *src;
  scamper_addr_t        *dst;

  /* when the ping started */
  struct timeval         start;

  /* why the ping finished */
  uint8_t                stop_reason;
  uint8_t                stop_data;

  /* the pattern to use inside of a probe.  if null then all zeros */
  uint16_t               pattern_len; /* -p option to ping */
  uint8_t               *pattern_bytes;

  /* ping options */
  uint16_t               probe_count; /* -c option to ping */
  uint16_t               probe_size;  /* -s option to ping */
  uint8_t                probe_wait;  /* -i option to ping */
  uint8_t                probe_ttl;   /* -m option to ping */
  uint8_t                probe_tos;   /* -z option to ping */
  uint16_t               reply_count; /* -o option to ping */

  /* actual data collected with the ping */
  scamper_ping_reply_t **ping_replies;
  uint16_t               ping_sent;
} scamper_ping_t;

/* basic routines to allocate and free scamper_ping structures */
scamper_ping_t *scamper_ping_alloc(void);
void scamper_ping_free(scamper_ping_t *ping);
int scamper_ping_setpattern(scamper_ping_t *ping,uint8_t *bytes,uint16_t len);

/* utility function for allocating an array for recording replies */
int scamper_ping_replies_alloc(scamper_ping_t *ping, int count);

/* basic routines to allocate and free scamper_ping_reply structures */
scamper_ping_reply_t *scamper_ping_reply_alloc(void);
void scamper_ping_reply_free(scamper_ping_reply_t *reply);
int scamper_ping_reply_append(scamper_ping_t *p, scamper_ping_reply_t *reply);
uint32_t scamper_ping_reply_count(const scamper_ping_t *ping);

/* routine to return basic stats for the measurement */
int scamper_ping_stats(const scamper_ping_t *ping,
		       uint32_t *nreplies, uint32_t *ndups, uint16_t *nloss,
		       struct timeval *min_rtt, struct timeval *max_rtt,
		       struct timeval *avg_rtt, struct timeval *stddev_rtt);

#endif /* __SCAMPER_PING_H */
